/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagent.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the flow version.
 * </p>
 * <p>
 * This data type is used in the following API operations:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href=
 * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_ListFlowVersions.html#API_agent_ListFlowVersions_ResponseSyntax"
 * >ListFlowVersions response</a>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class FlowVersionSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<FlowVersionSummary.Builder, FlowVersionSummary> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(FlowVersionSummary::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(FlowVersionSummary::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(FlowVersionSummary::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(FlowVersionSummary::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("version")
            .getter(getter(FlowVersionSummary::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, CREATED_AT_FIELD,
            ID_FIELD, STATUS_FIELD, VERSION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final Instant createdAt;

    private final String id;

    private final String status;

    private final String version;

    private FlowVersionSummary(BuilderImpl builder) {
        this.arn = builder.arn;
        this.createdAt = builder.createdAt;
        this.id = builder.id;
        this.status = builder.status;
        this.version = builder.version;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the flow that the version belongs to.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the flow that the version belongs to.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The time at the flow version was created.
     * </p>
     * 
     * @return The time at the flow version was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The unique identifier of the flow.
     * </p>
     * 
     * @return The unique identifier of the flow.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The status of the flow.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link FlowStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the flow.
     * @see FlowStatus
     */
    public final FlowStatus status() {
        return FlowStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the flow.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link FlowStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the flow.
     * @see FlowStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The version of the flow.
     * </p>
     * 
     * @return The version of the flow.
     */
    public final String version() {
        return version;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof FlowVersionSummary)) {
            return false;
        }
        FlowVersionSummary other = (FlowVersionSummary) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(id(), other.id()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(version(), other.version());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("FlowVersionSummary").add("Arn", arn()).add("CreatedAt", createdAt()).add("Id", id())
                .add("Status", statusAsString()).add("Version", version()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<FlowVersionSummary, T> g) {
        return obj -> g.apply((FlowVersionSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, FlowVersionSummary> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the flow that the version belongs to.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the flow that the version belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The time at the flow version was created.
         * </p>
         * 
         * @param createdAt
         *        The time at the flow version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The unique identifier of the flow.
         * </p>
         * 
         * @param id
         *        The unique identifier of the flow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The status of the flow.
         * </p>
         * 
         * @param status
         *        The status of the flow.
         * @see FlowStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FlowStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the flow.
         * </p>
         * 
         * @param status
         *        The status of the flow.
         * @see FlowStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FlowStatus
         */
        Builder status(FlowStatus status);

        /**
         * <p>
         * The version of the flow.
         * </p>
         * 
         * @param version
         *        The version of the flow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private Instant createdAt;

        private String id;

        private String status;

        private String version;

        private BuilderImpl() {
        }

        private BuilderImpl(FlowVersionSummary model) {
            arn(model.arn);
            createdAt(model.createdAt);
            id(model.id);
            status(model.status);
            version(model.version);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(FlowStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        @Override
        public FlowVersionSummary build() {
            return new FlowVersionSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
