/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagent.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Settings for customizing steps in the data source content ingestion pipeline.
 * </p>
 * <p>
 * You can configure the data source to process documents with a Lambda function after they are parsed and converted
 * into chunks. When you add a post-chunking transformation, the service stores chunked documents in an S3 bucket and
 * invokes a Lambda function to process them.
 * </p>
 * <p>
 * To process chunked documents with a Lambda function, define an S3 bucket path for input and output objects, and a
 * transformation that specifies the Lambda function to invoke. You can use the Lambda function to customize how chunks
 * are split, and the metadata for each chunk.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CustomTransformationConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<CustomTransformationConfiguration.Builder, CustomTransformationConfiguration> {
    private static final SdkField<IntermediateStorage> INTERMEDIATE_STORAGE_FIELD = SdkField
            .<IntermediateStorage> builder(MarshallingType.SDK_POJO).memberName("intermediateStorage")
            .getter(getter(CustomTransformationConfiguration::intermediateStorage)).setter(setter(Builder::intermediateStorage))
            .constructor(IntermediateStorage::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("intermediateStorage").build())
            .build();

    private static final SdkField<List<Transformation>> TRANSFORMATIONS_FIELD = SdkField
            .<List<Transformation>> builder(MarshallingType.LIST)
            .memberName("transformations")
            .getter(getter(CustomTransformationConfiguration::transformations))
            .setter(setter(Builder::transformations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("transformations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Transformation> builder(MarshallingType.SDK_POJO)
                                            .constructor(Transformation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INTERMEDIATE_STORAGE_FIELD,
            TRANSFORMATIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final IntermediateStorage intermediateStorage;

    private final List<Transformation> transformations;

    private CustomTransformationConfiguration(BuilderImpl builder) {
        this.intermediateStorage = builder.intermediateStorage;
        this.transformations = builder.transformations;
    }

    /**
     * <p>
     * An S3 bucket path for input and output objects.
     * </p>
     * 
     * @return An S3 bucket path for input and output objects.
     */
    public final IntermediateStorage intermediateStorage() {
        return intermediateStorage;
    }

    /**
     * For responses, this returns true if the service returned a value for the Transformations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTransformations() {
        return transformations != null && !(transformations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A Lambda function that processes documents.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTransformations} method.
     * </p>
     * 
     * @return A Lambda function that processes documents.
     */
    public final List<Transformation> transformations() {
        return transformations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(intermediateStorage());
        hashCode = 31 * hashCode + Objects.hashCode(hasTransformations() ? transformations() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CustomTransformationConfiguration)) {
            return false;
        }
        CustomTransformationConfiguration other = (CustomTransformationConfiguration) obj;
        return Objects.equals(intermediateStorage(), other.intermediateStorage())
                && hasTransformations() == other.hasTransformations()
                && Objects.equals(transformations(), other.transformations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CustomTransformationConfiguration").add("IntermediateStorage", intermediateStorage())
                .add("Transformations", hasTransformations() ? transformations() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "intermediateStorage":
            return Optional.ofNullable(clazz.cast(intermediateStorage()));
        case "transformations":
            return Optional.ofNullable(clazz.cast(transformations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CustomTransformationConfiguration, T> g) {
        return obj -> g.apply((CustomTransformationConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CustomTransformationConfiguration> {
        /**
         * <p>
         * An S3 bucket path for input and output objects.
         * </p>
         * 
         * @param intermediateStorage
         *        An S3 bucket path for input and output objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder intermediateStorage(IntermediateStorage intermediateStorage);

        /**
         * <p>
         * An S3 bucket path for input and output objects.
         * </p>
         * This is a convenience method that creates an instance of the {@link IntermediateStorage.Builder} avoiding the
         * need to create one manually via {@link IntermediateStorage#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IntermediateStorage.Builder#build()} is called immediately and
         * its result is passed to {@link #intermediateStorage(IntermediateStorage)}.
         * 
         * @param intermediateStorage
         *        a consumer that will call methods on {@link IntermediateStorage.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #intermediateStorage(IntermediateStorage)
         */
        default Builder intermediateStorage(Consumer<IntermediateStorage.Builder> intermediateStorage) {
            return intermediateStorage(IntermediateStorage.builder().applyMutation(intermediateStorage).build());
        }

        /**
         * <p>
         * A Lambda function that processes documents.
         * </p>
         * 
         * @param transformations
         *        A Lambda function that processes documents.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transformations(Collection<Transformation> transformations);

        /**
         * <p>
         * A Lambda function that processes documents.
         * </p>
         * 
         * @param transformations
         *        A Lambda function that processes documents.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transformations(Transformation... transformations);

        /**
         * <p>
         * A Lambda function that processes documents.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.bedrockagent.model.Transformation.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.bedrockagent.model.Transformation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.bedrockagent.model.Transformation.Builder#build()} is called
         * immediately and its result is passed to {@link #transformations(List<Transformation>)}.
         * 
         * @param transformations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.bedrockagent.model.Transformation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #transformations(java.util.Collection<Transformation>)
         */
        Builder transformations(Consumer<Transformation.Builder>... transformations);
    }

    static final class BuilderImpl implements Builder {
        private IntermediateStorage intermediateStorage;

        private List<Transformation> transformations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CustomTransformationConfiguration model) {
            intermediateStorage(model.intermediateStorage);
            transformations(model.transformations);
        }

        public final IntermediateStorage.Builder getIntermediateStorage() {
            return intermediateStorage != null ? intermediateStorage.toBuilder() : null;
        }

        public final void setIntermediateStorage(IntermediateStorage.BuilderImpl intermediateStorage) {
            this.intermediateStorage = intermediateStorage != null ? intermediateStorage.build() : null;
        }

        @Override
        public final Builder intermediateStorage(IntermediateStorage intermediateStorage) {
            this.intermediateStorage = intermediateStorage;
            return this;
        }

        public final List<Transformation.Builder> getTransformations() {
            List<Transformation.Builder> result = TransformationsCopier.copyToBuilder(this.transformations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTransformations(Collection<Transformation.BuilderImpl> transformations) {
            this.transformations = TransformationsCopier.copyFromBuilder(transformations);
        }

        @Override
        public final Builder transformations(Collection<Transformation> transformations) {
            this.transformations = TransformationsCopier.copy(transformations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder transformations(Transformation... transformations) {
            transformations(Arrays.asList(transformations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder transformations(Consumer<Transformation.Builder>... transformations) {
            transformations(Stream.of(transformations).map(c -> Transformation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public CustomTransformationConfiguration build() {
            return new CustomTransformationConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
