/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagent.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details about an agent.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Agent implements SdkPojo, Serializable, ToCopyableBuilder<Agent.Builder, Agent> {
    private static final SdkField<String> AGENT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("agentArn").getter(getter(Agent::agentArn)).setter(setter(Builder::agentArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("agentArn").build()).build();

    private static final SdkField<String> AGENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("agentId").getter(getter(Agent::agentId)).setter(setter(Builder::agentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("agentId").build()).build();

    private static final SdkField<String> AGENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("agentName").getter(getter(Agent::agentName)).setter(setter(Builder::agentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("agentName").build()).build();

    private static final SdkField<String> AGENT_RESOURCE_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("agentResourceRoleArn").getter(getter(Agent::agentResourceRoleArn))
            .setter(setter(Builder::agentResourceRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("agentResourceRoleArn").build())
            .build();

    private static final SdkField<String> AGENT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("agentStatus").getter(getter(Agent::agentStatusAsString)).setter(setter(Builder::agentStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("agentStatus").build()).build();

    private static final SdkField<String> AGENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("agentVersion").getter(getter(Agent::agentVersion)).setter(setter(Builder::agentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("agentVersion").build()).build();

    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("clientToken").getter(getter(Agent::clientToken)).setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clientToken").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(Agent::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> CUSTOMER_ENCRYPTION_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("customerEncryptionKeyArn").getter(getter(Agent::customerEncryptionKeyArn))
            .setter(setter(Builder::customerEncryptionKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("customerEncryptionKeyArn").build())
            .build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(Agent::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<List<String>> FAILURE_REASONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("failureReasons")
            .getter(getter(Agent::failureReasons))
            .setter(setter(Builder::failureReasons))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("failureReasons").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> FOUNDATION_MODEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("foundationModel").getter(getter(Agent::foundationModel)).setter(setter(Builder::foundationModel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("foundationModel").build()).build();

    private static final SdkField<GuardrailConfiguration> GUARDRAIL_CONFIGURATION_FIELD = SdkField
            .<GuardrailConfiguration> builder(MarshallingType.SDK_POJO).memberName("guardrailConfiguration")
            .getter(getter(Agent::guardrailConfiguration)).setter(setter(Builder::guardrailConfiguration))
            .constructor(GuardrailConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("guardrailConfiguration").build())
            .build();

    private static final SdkField<Integer> IDLE_SESSION_TTL_IN_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("idleSessionTTLInSeconds")
            .getter(getter(Agent::idleSessionTTLInSeconds)).setter(setter(Builder::idleSessionTTLInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("idleSessionTTLInSeconds").build())
            .build();

    private static final SdkField<String> INSTRUCTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instruction").getter(getter(Agent::instruction)).setter(setter(Builder::instruction))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instruction").build()).build();

    private static final SdkField<MemoryConfiguration> MEMORY_CONFIGURATION_FIELD = SdkField
            .<MemoryConfiguration> builder(MarshallingType.SDK_POJO).memberName("memoryConfiguration")
            .getter(getter(Agent::memoryConfiguration)).setter(setter(Builder::memoryConfiguration))
            .constructor(MemoryConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memoryConfiguration").build())
            .build();

    private static final SdkField<Instant> PREPARED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("preparedAt")
            .getter(getter(Agent::preparedAt))
            .setter(setter(Builder::preparedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("preparedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<PromptOverrideConfiguration> PROMPT_OVERRIDE_CONFIGURATION_FIELD = SdkField
            .<PromptOverrideConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("promptOverrideConfiguration")
            .getter(getter(Agent::promptOverrideConfiguration))
            .setter(setter(Builder::promptOverrideConfiguration))
            .constructor(PromptOverrideConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("promptOverrideConfiguration")
                    .build()).build();

    private static final SdkField<List<String>> RECOMMENDED_ACTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("recommendedActions")
            .getter(getter(Agent::recommendedActions))
            .setter(setter(Builder::recommendedActions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recommendedActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt")
            .getter(getter(Agent::updatedAt))
            .setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AGENT_ARN_FIELD,
            AGENT_ID_FIELD, AGENT_NAME_FIELD, AGENT_RESOURCE_ROLE_ARN_FIELD, AGENT_STATUS_FIELD, AGENT_VERSION_FIELD,
            CLIENT_TOKEN_FIELD, CREATED_AT_FIELD, CUSTOMER_ENCRYPTION_KEY_ARN_FIELD, DESCRIPTION_FIELD, FAILURE_REASONS_FIELD,
            FOUNDATION_MODEL_FIELD, GUARDRAIL_CONFIGURATION_FIELD, IDLE_SESSION_TTL_IN_SECONDS_FIELD, INSTRUCTION_FIELD,
            MEMORY_CONFIGURATION_FIELD, PREPARED_AT_FIELD, PROMPT_OVERRIDE_CONFIGURATION_FIELD, RECOMMENDED_ACTIONS_FIELD,
            UPDATED_AT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String agentArn;

    private final String agentId;

    private final String agentName;

    private final String agentResourceRoleArn;

    private final String agentStatus;

    private final String agentVersion;

    private final String clientToken;

    private final Instant createdAt;

    private final String customerEncryptionKeyArn;

    private final String description;

    private final List<String> failureReasons;

    private final String foundationModel;

    private final GuardrailConfiguration guardrailConfiguration;

    private final Integer idleSessionTTLInSeconds;

    private final String instruction;

    private final MemoryConfiguration memoryConfiguration;

    private final Instant preparedAt;

    private final PromptOverrideConfiguration promptOverrideConfiguration;

    private final List<String> recommendedActions;

    private final Instant updatedAt;

    private Agent(BuilderImpl builder) {
        this.agentArn = builder.agentArn;
        this.agentId = builder.agentId;
        this.agentName = builder.agentName;
        this.agentResourceRoleArn = builder.agentResourceRoleArn;
        this.agentStatus = builder.agentStatus;
        this.agentVersion = builder.agentVersion;
        this.clientToken = builder.clientToken;
        this.createdAt = builder.createdAt;
        this.customerEncryptionKeyArn = builder.customerEncryptionKeyArn;
        this.description = builder.description;
        this.failureReasons = builder.failureReasons;
        this.foundationModel = builder.foundationModel;
        this.guardrailConfiguration = builder.guardrailConfiguration;
        this.idleSessionTTLInSeconds = builder.idleSessionTTLInSeconds;
        this.instruction = builder.instruction;
        this.memoryConfiguration = builder.memoryConfiguration;
        this.preparedAt = builder.preparedAt;
        this.promptOverrideConfiguration = builder.promptOverrideConfiguration;
        this.recommendedActions = builder.recommendedActions;
        this.updatedAt = builder.updatedAt;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the agent.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the agent.
     */
    public final String agentArn() {
        return agentArn;
    }

    /**
     * <p>
     * The unique identifier of the agent.
     * </p>
     * 
     * @return The unique identifier of the agent.
     */
    public final String agentId() {
        return agentId;
    }

    /**
     * <p>
     * The name of the agent.
     * </p>
     * 
     * @return The name of the agent.
     */
    public final String agentName() {
        return agentName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IAM role with permissions to invoke API operations on the agent.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IAM role with permissions to invoke API operations on the agent.
     */
    public final String agentResourceRoleArn() {
        return agentResourceRoleArn;
    }

    /**
     * <p>
     * The status of the agent and whether it is ready for use. The following statuses are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * CREATING – The agent is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * PREPARING – The agent is being prepared.
     * </p>
     * </li>
     * <li>
     * <p>
     * PREPARED – The agent is prepared and ready to be invoked.
     * </p>
     * </li>
     * <li>
     * <p>
     * NOT_PREPARED – The agent has been created but not yet prepared.
     * </p>
     * </li>
     * <li>
     * <p>
     * FAILED – The agent API operation failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * UPDATING – The agent is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * DELETING – The agent is being deleted.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #agentStatus} will
     * return {@link AgentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #agentStatusAsString}.
     * </p>
     * 
     * @return The status of the agent and whether it is ready for use. The following statuses are possible:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         CREATING – The agent is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PREPARING – The agent is being prepared.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PREPARED – The agent is prepared and ready to be invoked.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         NOT_PREPARED – The agent has been created but not yet prepared.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FAILED – The agent API operation failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         UPDATING – The agent is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DELETING – The agent is being deleted.
     *         </p>
     *         </li>
     * @see AgentStatus
     */
    public final AgentStatus agentStatus() {
        return AgentStatus.fromValue(agentStatus);
    }

    /**
     * <p>
     * The status of the agent and whether it is ready for use. The following statuses are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * CREATING – The agent is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * PREPARING – The agent is being prepared.
     * </p>
     * </li>
     * <li>
     * <p>
     * PREPARED – The agent is prepared and ready to be invoked.
     * </p>
     * </li>
     * <li>
     * <p>
     * NOT_PREPARED – The agent has been created but not yet prepared.
     * </p>
     * </li>
     * <li>
     * <p>
     * FAILED – The agent API operation failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * UPDATING – The agent is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * DELETING – The agent is being deleted.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #agentStatus} will
     * return {@link AgentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #agentStatusAsString}.
     * </p>
     * 
     * @return The status of the agent and whether it is ready for use. The following statuses are possible:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         CREATING – The agent is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PREPARING – The agent is being prepared.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         PREPARED – The agent is prepared and ready to be invoked.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         NOT_PREPARED – The agent has been created but not yet prepared.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         FAILED – The agent API operation failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         UPDATING – The agent is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DELETING – The agent is being deleted.
     *         </p>
     *         </li>
     * @see AgentStatus
     */
    public final String agentStatusAsString() {
        return agentStatus;
    }

    /**
     * <p>
     * The version of the agent.
     * </p>
     * 
     * @return The version of the agent.
     */
    public final String agentVersion() {
        return agentVersion;
    }

    /**
     * <p>
     * A unique, case-sensitive identifier to ensure that the API request completes no more than one time. If this token
     * matches a previous request, Amazon Bedrock ignores the request, but does not return an error. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
     * idempotency</a>.
     * </p>
     * 
     * @return A unique, case-sensitive identifier to ensure that the API request completes no more than one time. If
     *         this token matches a previous request, Amazon Bedrock ignores the request, but does not return an error.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
     *         idempotency</a>.
     */
    public final String clientToken() {
        return clientToken;
    }

    /**
     * <p>
     * The time at which the agent was created.
     * </p>
     * 
     * @return The time at which the agent was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the KMS key that encrypts the agent.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the KMS key that encrypts the agent.
     */
    public final String customerEncryptionKeyArn() {
        return customerEncryptionKeyArn;
    }

    /**
     * <p>
     * The description of the agent.
     * </p>
     * 
     * @return The description of the agent.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the FailureReasons property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFailureReasons() {
        return failureReasons != null && !(failureReasons instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains reasons that the agent-related API that you invoked failed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFailureReasons} method.
     * </p>
     * 
     * @return Contains reasons that the agent-related API that you invoked failed.
     */
    public final List<String> failureReasons() {
        return failureReasons;
    }

    /**
     * <p>
     * The foundation model used for orchestration by the agent.
     * </p>
     * 
     * @return The foundation model used for orchestration by the agent.
     */
    public final String foundationModel() {
        return foundationModel;
    }

    /**
     * <p>
     * Details about the guardrail associated with the agent.
     * </p>
     * 
     * @return Details about the guardrail associated with the agent.
     */
    public final GuardrailConfiguration guardrailConfiguration() {
        return guardrailConfiguration;
    }

    /**
     * <p>
     * The number of seconds for which Amazon Bedrock keeps information about a user's conversation with the agent.
     * </p>
     * <p>
     * A user interaction remains active for the amount of time specified. If no conversation occurs during this time,
     * the session expires and Amazon Bedrock deletes any data provided before the timeout.
     * </p>
     * 
     * @return The number of seconds for which Amazon Bedrock keeps information about a user's conversation with the
     *         agent.</p>
     *         <p>
     *         A user interaction remains active for the amount of time specified. If no conversation occurs during this
     *         time, the session expires and Amazon Bedrock deletes any data provided before the timeout.
     */
    public final Integer idleSessionTTLInSeconds() {
        return idleSessionTTLInSeconds;
    }

    /**
     * <p>
     * Instructions that tell the agent what it should do and how it should interact with users.
     * </p>
     * 
     * @return Instructions that tell the agent what it should do and how it should interact with users.
     */
    public final String instruction() {
        return instruction;
    }

    /**
     * <p>
     * Contains memory configuration for the agent.
     * </p>
     * 
     * @return Contains memory configuration for the agent.
     */
    public final MemoryConfiguration memoryConfiguration() {
        return memoryConfiguration;
    }

    /**
     * <p>
     * The time at which the agent was last prepared.
     * </p>
     * 
     * @return The time at which the agent was last prepared.
     */
    public final Instant preparedAt() {
        return preparedAt;
    }

    /**
     * <p>
     * Contains configurations to override prompt templates in different parts of an agent sequence. For more
     * information, see <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html">Advanced
     * prompts</a>.
     * </p>
     * 
     * @return Contains configurations to override prompt templates in different parts of an agent sequence. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html">Advanced prompts</a>.
     */
    public final PromptOverrideConfiguration promptOverrideConfiguration() {
        return promptOverrideConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the RecommendedActions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRecommendedActions() {
        return recommendedActions != null && !(recommendedActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains recommended actions to take for the agent-related API that you invoked to succeed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecommendedActions} method.
     * </p>
     * 
     * @return Contains recommended actions to take for the agent-related API that you invoked to succeed.
     */
    public final List<String> recommendedActions() {
        return recommendedActions;
    }

    /**
     * <p>
     * The time at which the agent was last updated.
     * </p>
     * 
     * @return The time at which the agent was last updated.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(agentArn());
        hashCode = 31 * hashCode + Objects.hashCode(agentId());
        hashCode = 31 * hashCode + Objects.hashCode(agentName());
        hashCode = 31 * hashCode + Objects.hashCode(agentResourceRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(agentStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(agentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(customerEncryptionKeyArn());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasFailureReasons() ? failureReasons() : null);
        hashCode = 31 * hashCode + Objects.hashCode(foundationModel());
        hashCode = 31 * hashCode + Objects.hashCode(guardrailConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(idleSessionTTLInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(instruction());
        hashCode = 31 * hashCode + Objects.hashCode(memoryConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(preparedAt());
        hashCode = 31 * hashCode + Objects.hashCode(promptOverrideConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasRecommendedActions() ? recommendedActions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Agent)) {
            return false;
        }
        Agent other = (Agent) obj;
        return Objects.equals(agentArn(), other.agentArn()) && Objects.equals(agentId(), other.agentId())
                && Objects.equals(agentName(), other.agentName())
                && Objects.equals(agentResourceRoleArn(), other.agentResourceRoleArn())
                && Objects.equals(agentStatusAsString(), other.agentStatusAsString())
                && Objects.equals(agentVersion(), other.agentVersion()) && Objects.equals(clientToken(), other.clientToken())
                && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(customerEncryptionKeyArn(), other.customerEncryptionKeyArn())
                && Objects.equals(description(), other.description()) && hasFailureReasons() == other.hasFailureReasons()
                && Objects.equals(failureReasons(), other.failureReasons())
                && Objects.equals(foundationModel(), other.foundationModel())
                && Objects.equals(guardrailConfiguration(), other.guardrailConfiguration())
                && Objects.equals(idleSessionTTLInSeconds(), other.idleSessionTTLInSeconds())
                && Objects.equals(instruction(), other.instruction())
                && Objects.equals(memoryConfiguration(), other.memoryConfiguration())
                && Objects.equals(preparedAt(), other.preparedAt())
                && Objects.equals(promptOverrideConfiguration(), other.promptOverrideConfiguration())
                && hasRecommendedActions() == other.hasRecommendedActions()
                && Objects.equals(recommendedActions(), other.recommendedActions())
                && Objects.equals(updatedAt(), other.updatedAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("Agent")
                .add("AgentArn", agentArn())
                .add("AgentId", agentId())
                .add("AgentName", agentName())
                .add("AgentResourceRoleArn", agentResourceRoleArn())
                .add("AgentStatus", agentStatusAsString())
                .add("AgentVersion", agentVersion())
                .add("ClientToken", clientToken())
                .add("CreatedAt", createdAt())
                .add("CustomerEncryptionKeyArn", customerEncryptionKeyArn())
                .add("Description", description())
                .add("FailureReasons", hasFailureReasons() ? failureReasons() : null)
                .add("FoundationModel", foundationModel())
                .add("GuardrailConfiguration", guardrailConfiguration())
                .add("IdleSessionTTLInSeconds", idleSessionTTLInSeconds())
                .add("Instruction", instruction() == null ? null : "*** Sensitive Data Redacted ***")
                .add("MemoryConfiguration", memoryConfiguration())
                .add("PreparedAt", preparedAt())
                .add("PromptOverrideConfiguration",
                        promptOverrideConfiguration() == null ? null : "*** Sensitive Data Redacted ***")
                .add("RecommendedActions", hasRecommendedActions() ? recommendedActions() : null).add("UpdatedAt", updatedAt())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "agentArn":
            return Optional.ofNullable(clazz.cast(agentArn()));
        case "agentId":
            return Optional.ofNullable(clazz.cast(agentId()));
        case "agentName":
            return Optional.ofNullable(clazz.cast(agentName()));
        case "agentResourceRoleArn":
            return Optional.ofNullable(clazz.cast(agentResourceRoleArn()));
        case "agentStatus":
            return Optional.ofNullable(clazz.cast(agentStatusAsString()));
        case "agentVersion":
            return Optional.ofNullable(clazz.cast(agentVersion()));
        case "clientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "customerEncryptionKeyArn":
            return Optional.ofNullable(clazz.cast(customerEncryptionKeyArn()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "failureReasons":
            return Optional.ofNullable(clazz.cast(failureReasons()));
        case "foundationModel":
            return Optional.ofNullable(clazz.cast(foundationModel()));
        case "guardrailConfiguration":
            return Optional.ofNullable(clazz.cast(guardrailConfiguration()));
        case "idleSessionTTLInSeconds":
            return Optional.ofNullable(clazz.cast(idleSessionTTLInSeconds()));
        case "instruction":
            return Optional.ofNullable(clazz.cast(instruction()));
        case "memoryConfiguration":
            return Optional.ofNullable(clazz.cast(memoryConfiguration()));
        case "preparedAt":
            return Optional.ofNullable(clazz.cast(preparedAt()));
        case "promptOverrideConfiguration":
            return Optional.ofNullable(clazz.cast(promptOverrideConfiguration()));
        case "recommendedActions":
            return Optional.ofNullable(clazz.cast(recommendedActions()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Agent, T> g) {
        return obj -> g.apply((Agent) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Agent> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the agent.
         * </p>
         * 
         * @param agentArn
         *        The Amazon Resource Name (ARN) of the agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArn(String agentArn);

        /**
         * <p>
         * The unique identifier of the agent.
         * </p>
         * 
         * @param agentId
         *        The unique identifier of the agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentId(String agentId);

        /**
         * <p>
         * The name of the agent.
         * </p>
         * 
         * @param agentName
         *        The name of the agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentName(String agentName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IAM role with permissions to invoke API operations on the agent.
         * </p>
         * 
         * @param agentResourceRoleArn
         *        The Amazon Resource Name (ARN) of the IAM role with permissions to invoke API operations on the agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentResourceRoleArn(String agentResourceRoleArn);

        /**
         * <p>
         * The status of the agent and whether it is ready for use. The following statuses are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * CREATING – The agent is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * PREPARING – The agent is being prepared.
         * </p>
         * </li>
         * <li>
         * <p>
         * PREPARED – The agent is prepared and ready to be invoked.
         * </p>
         * </li>
         * <li>
         * <p>
         * NOT_PREPARED – The agent has been created but not yet prepared.
         * </p>
         * </li>
         * <li>
         * <p>
         * FAILED – The agent API operation failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * UPDATING – The agent is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * DELETING – The agent is being deleted.
         * </p>
         * </li>
         * </ul>
         * 
         * @param agentStatus
         *        The status of the agent and whether it is ready for use. The following statuses are possible:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        CREATING – The agent is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PREPARING – The agent is being prepared.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PREPARED – The agent is prepared and ready to be invoked.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        NOT_PREPARED – The agent has been created but not yet prepared.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FAILED – The agent API operation failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        UPDATING – The agent is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DELETING – The agent is being deleted.
         *        </p>
         *        </li>
         * @see AgentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AgentStatus
         */
        Builder agentStatus(String agentStatus);

        /**
         * <p>
         * The status of the agent and whether it is ready for use. The following statuses are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * CREATING – The agent is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * PREPARING – The agent is being prepared.
         * </p>
         * </li>
         * <li>
         * <p>
         * PREPARED – The agent is prepared and ready to be invoked.
         * </p>
         * </li>
         * <li>
         * <p>
         * NOT_PREPARED – The agent has been created but not yet prepared.
         * </p>
         * </li>
         * <li>
         * <p>
         * FAILED – The agent API operation failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * UPDATING – The agent is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * DELETING – The agent is being deleted.
         * </p>
         * </li>
         * </ul>
         * 
         * @param agentStatus
         *        The status of the agent and whether it is ready for use. The following statuses are possible:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        CREATING – The agent is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PREPARING – The agent is being prepared.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        PREPARED – The agent is prepared and ready to be invoked.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        NOT_PREPARED – The agent has been created but not yet prepared.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        FAILED – The agent API operation failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        UPDATING – The agent is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DELETING – The agent is being deleted.
         *        </p>
         *        </li>
         * @see AgentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AgentStatus
         */
        Builder agentStatus(AgentStatus agentStatus);

        /**
         * <p>
         * The version of the agent.
         * </p>
         * 
         * @param agentVersion
         *        The version of the agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentVersion(String agentVersion);

        /**
         * <p>
         * A unique, case-sensitive identifier to ensure that the API request completes no more than one time. If this
         * token matches a previous request, Amazon Bedrock ignores the request, but does not return an error. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
         * idempotency</a>.
         * </p>
         * 
         * @param clientToken
         *        A unique, case-sensitive identifier to ensure that the API request completes no more than one time. If
         *        this token matches a previous request, Amazon Bedrock ignores the request, but does not return an
         *        error. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
         *        idempotency</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        /**
         * <p>
         * The time at which the agent was created.
         * </p>
         * 
         * @param createdAt
         *        The time at which the agent was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the KMS key that encrypts the agent.
         * </p>
         * 
         * @param customerEncryptionKeyArn
         *        The Amazon Resource Name (ARN) of the KMS key that encrypts the agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerEncryptionKeyArn(String customerEncryptionKeyArn);

        /**
         * <p>
         * The description of the agent.
         * </p>
         * 
         * @param description
         *        The description of the agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Contains reasons that the agent-related API that you invoked failed.
         * </p>
         * 
         * @param failureReasons
         *        Contains reasons that the agent-related API that you invoked failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReasons(Collection<String> failureReasons);

        /**
         * <p>
         * Contains reasons that the agent-related API that you invoked failed.
         * </p>
         * 
         * @param failureReasons
         *        Contains reasons that the agent-related API that you invoked failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReasons(String... failureReasons);

        /**
         * <p>
         * The foundation model used for orchestration by the agent.
         * </p>
         * 
         * @param foundationModel
         *        The foundation model used for orchestration by the agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder foundationModel(String foundationModel);

        /**
         * <p>
         * Details about the guardrail associated with the agent.
         * </p>
         * 
         * @param guardrailConfiguration
         *        Details about the guardrail associated with the agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder guardrailConfiguration(GuardrailConfiguration guardrailConfiguration);

        /**
         * <p>
         * Details about the guardrail associated with the agent.
         * </p>
         * This is a convenience method that creates an instance of the {@link GuardrailConfiguration.Builder} avoiding
         * the need to create one manually via {@link GuardrailConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link GuardrailConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #guardrailConfiguration(GuardrailConfiguration)}.
         * 
         * @param guardrailConfiguration
         *        a consumer that will call methods on {@link GuardrailConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #guardrailConfiguration(GuardrailConfiguration)
         */
        default Builder guardrailConfiguration(Consumer<GuardrailConfiguration.Builder> guardrailConfiguration) {
            return guardrailConfiguration(GuardrailConfiguration.builder().applyMutation(guardrailConfiguration).build());
        }

        /**
         * <p>
         * The number of seconds for which Amazon Bedrock keeps information about a user's conversation with the agent.
         * </p>
         * <p>
         * A user interaction remains active for the amount of time specified. If no conversation occurs during this
         * time, the session expires and Amazon Bedrock deletes any data provided before the timeout.
         * </p>
         * 
         * @param idleSessionTTLInSeconds
         *        The number of seconds for which Amazon Bedrock keeps information about a user's conversation with the
         *        agent.</p>
         *        <p>
         *        A user interaction remains active for the amount of time specified. If no conversation occurs during
         *        this time, the session expires and Amazon Bedrock deletes any data provided before the timeout.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder idleSessionTTLInSeconds(Integer idleSessionTTLInSeconds);

        /**
         * <p>
         * Instructions that tell the agent what it should do and how it should interact with users.
         * </p>
         * 
         * @param instruction
         *        Instructions that tell the agent what it should do and how it should interact with users.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instruction(String instruction);

        /**
         * <p>
         * Contains memory configuration for the agent.
         * </p>
         * 
         * @param memoryConfiguration
         *        Contains memory configuration for the agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memoryConfiguration(MemoryConfiguration memoryConfiguration);

        /**
         * <p>
         * Contains memory configuration for the agent.
         * </p>
         * This is a convenience method that creates an instance of the {@link MemoryConfiguration.Builder} avoiding the
         * need to create one manually via {@link MemoryConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MemoryConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #memoryConfiguration(MemoryConfiguration)}.
         * 
         * @param memoryConfiguration
         *        a consumer that will call methods on {@link MemoryConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #memoryConfiguration(MemoryConfiguration)
         */
        default Builder memoryConfiguration(Consumer<MemoryConfiguration.Builder> memoryConfiguration) {
            return memoryConfiguration(MemoryConfiguration.builder().applyMutation(memoryConfiguration).build());
        }

        /**
         * <p>
         * The time at which the agent was last prepared.
         * </p>
         * 
         * @param preparedAt
         *        The time at which the agent was last prepared.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preparedAt(Instant preparedAt);

        /**
         * <p>
         * Contains configurations to override prompt templates in different parts of an agent sequence. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html">Advanced prompts</a>.
         * </p>
         * 
         * @param promptOverrideConfiguration
         *        Contains configurations to override prompt templates in different parts of an agent sequence. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html">Advanced
         *        prompts</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder promptOverrideConfiguration(PromptOverrideConfiguration promptOverrideConfiguration);

        /**
         * <p>
         * Contains configurations to override prompt templates in different parts of an agent sequence. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html">Advanced prompts</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link PromptOverrideConfiguration.Builder}
         * avoiding the need to create one manually via {@link PromptOverrideConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PromptOverrideConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #promptOverrideConfiguration(PromptOverrideConfiguration)}.
         * 
         * @param promptOverrideConfiguration
         *        a consumer that will call methods on {@link PromptOverrideConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #promptOverrideConfiguration(PromptOverrideConfiguration)
         */
        default Builder promptOverrideConfiguration(Consumer<PromptOverrideConfiguration.Builder> promptOverrideConfiguration) {
            return promptOverrideConfiguration(PromptOverrideConfiguration.builder().applyMutation(promptOverrideConfiguration)
                    .build());
        }

        /**
         * <p>
         * Contains recommended actions to take for the agent-related API that you invoked to succeed.
         * </p>
         * 
         * @param recommendedActions
         *        Contains recommended actions to take for the agent-related API that you invoked to succeed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendedActions(Collection<String> recommendedActions);

        /**
         * <p>
         * Contains recommended actions to take for the agent-related API that you invoked to succeed.
         * </p>
         * 
         * @param recommendedActions
         *        Contains recommended actions to take for the agent-related API that you invoked to succeed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendedActions(String... recommendedActions);

        /**
         * <p>
         * The time at which the agent was last updated.
         * </p>
         * 
         * @param updatedAt
         *        The time at which the agent was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);
    }

    static final class BuilderImpl implements Builder {
        private String agentArn;

        private String agentId;

        private String agentName;

        private String agentResourceRoleArn;

        private String agentStatus;

        private String agentVersion;

        private String clientToken;

        private Instant createdAt;

        private String customerEncryptionKeyArn;

        private String description;

        private List<String> failureReasons = DefaultSdkAutoConstructList.getInstance();

        private String foundationModel;

        private GuardrailConfiguration guardrailConfiguration;

        private Integer idleSessionTTLInSeconds;

        private String instruction;

        private MemoryConfiguration memoryConfiguration;

        private Instant preparedAt;

        private PromptOverrideConfiguration promptOverrideConfiguration;

        private List<String> recommendedActions = DefaultSdkAutoConstructList.getInstance();

        private Instant updatedAt;

        private BuilderImpl() {
        }

        private BuilderImpl(Agent model) {
            agentArn(model.agentArn);
            agentId(model.agentId);
            agentName(model.agentName);
            agentResourceRoleArn(model.agentResourceRoleArn);
            agentStatus(model.agentStatus);
            agentVersion(model.agentVersion);
            clientToken(model.clientToken);
            createdAt(model.createdAt);
            customerEncryptionKeyArn(model.customerEncryptionKeyArn);
            description(model.description);
            failureReasons(model.failureReasons);
            foundationModel(model.foundationModel);
            guardrailConfiguration(model.guardrailConfiguration);
            idleSessionTTLInSeconds(model.idleSessionTTLInSeconds);
            instruction(model.instruction);
            memoryConfiguration(model.memoryConfiguration);
            preparedAt(model.preparedAt);
            promptOverrideConfiguration(model.promptOverrideConfiguration);
            recommendedActions(model.recommendedActions);
            updatedAt(model.updatedAt);
        }

        public final String getAgentArn() {
            return agentArn;
        }

        public final void setAgentArn(String agentArn) {
            this.agentArn = agentArn;
        }

        @Override
        public final Builder agentArn(String agentArn) {
            this.agentArn = agentArn;
            return this;
        }

        public final String getAgentId() {
            return agentId;
        }

        public final void setAgentId(String agentId) {
            this.agentId = agentId;
        }

        @Override
        public final Builder agentId(String agentId) {
            this.agentId = agentId;
            return this;
        }

        public final String getAgentName() {
            return agentName;
        }

        public final void setAgentName(String agentName) {
            this.agentName = agentName;
        }

        @Override
        public final Builder agentName(String agentName) {
            this.agentName = agentName;
            return this;
        }

        public final String getAgentResourceRoleArn() {
            return agentResourceRoleArn;
        }

        public final void setAgentResourceRoleArn(String agentResourceRoleArn) {
            this.agentResourceRoleArn = agentResourceRoleArn;
        }

        @Override
        public final Builder agentResourceRoleArn(String agentResourceRoleArn) {
            this.agentResourceRoleArn = agentResourceRoleArn;
            return this;
        }

        public final String getAgentStatus() {
            return agentStatus;
        }

        public final void setAgentStatus(String agentStatus) {
            this.agentStatus = agentStatus;
        }

        @Override
        public final Builder agentStatus(String agentStatus) {
            this.agentStatus = agentStatus;
            return this;
        }

        @Override
        public final Builder agentStatus(AgentStatus agentStatus) {
            this.agentStatus(agentStatus == null ? null : agentStatus.toString());
            return this;
        }

        public final String getAgentVersion() {
            return agentVersion;
        }

        public final void setAgentVersion(String agentVersion) {
            this.agentVersion = agentVersion;
        }

        @Override
        public final Builder agentVersion(String agentVersion) {
            this.agentVersion = agentVersion;
            return this;
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getCustomerEncryptionKeyArn() {
            return customerEncryptionKeyArn;
        }

        public final void setCustomerEncryptionKeyArn(String customerEncryptionKeyArn) {
            this.customerEncryptionKeyArn = customerEncryptionKeyArn;
        }

        @Override
        public final Builder customerEncryptionKeyArn(String customerEncryptionKeyArn) {
            this.customerEncryptionKeyArn = customerEncryptionKeyArn;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Collection<String> getFailureReasons() {
            if (failureReasons instanceof SdkAutoConstructList) {
                return null;
            }
            return failureReasons;
        }

        public final void setFailureReasons(Collection<String> failureReasons) {
            this.failureReasons = FailureReasonsCopier.copy(failureReasons);
        }

        @Override
        public final Builder failureReasons(Collection<String> failureReasons) {
            this.failureReasons = FailureReasonsCopier.copy(failureReasons);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder failureReasons(String... failureReasons) {
            failureReasons(Arrays.asList(failureReasons));
            return this;
        }

        public final String getFoundationModel() {
            return foundationModel;
        }

        public final void setFoundationModel(String foundationModel) {
            this.foundationModel = foundationModel;
        }

        @Override
        public final Builder foundationModel(String foundationModel) {
            this.foundationModel = foundationModel;
            return this;
        }

        public final GuardrailConfiguration.Builder getGuardrailConfiguration() {
            return guardrailConfiguration != null ? guardrailConfiguration.toBuilder() : null;
        }

        public final void setGuardrailConfiguration(GuardrailConfiguration.BuilderImpl guardrailConfiguration) {
            this.guardrailConfiguration = guardrailConfiguration != null ? guardrailConfiguration.build() : null;
        }

        @Override
        public final Builder guardrailConfiguration(GuardrailConfiguration guardrailConfiguration) {
            this.guardrailConfiguration = guardrailConfiguration;
            return this;
        }

        public final Integer getIdleSessionTTLInSeconds() {
            return idleSessionTTLInSeconds;
        }

        public final void setIdleSessionTTLInSeconds(Integer idleSessionTTLInSeconds) {
            this.idleSessionTTLInSeconds = idleSessionTTLInSeconds;
        }

        @Override
        public final Builder idleSessionTTLInSeconds(Integer idleSessionTTLInSeconds) {
            this.idleSessionTTLInSeconds = idleSessionTTLInSeconds;
            return this;
        }

        public final String getInstruction() {
            return instruction;
        }

        public final void setInstruction(String instruction) {
            this.instruction = instruction;
        }

        @Override
        public final Builder instruction(String instruction) {
            this.instruction = instruction;
            return this;
        }

        public final MemoryConfiguration.Builder getMemoryConfiguration() {
            return memoryConfiguration != null ? memoryConfiguration.toBuilder() : null;
        }

        public final void setMemoryConfiguration(MemoryConfiguration.BuilderImpl memoryConfiguration) {
            this.memoryConfiguration = memoryConfiguration != null ? memoryConfiguration.build() : null;
        }

        @Override
        public final Builder memoryConfiguration(MemoryConfiguration memoryConfiguration) {
            this.memoryConfiguration = memoryConfiguration;
            return this;
        }

        public final Instant getPreparedAt() {
            return preparedAt;
        }

        public final void setPreparedAt(Instant preparedAt) {
            this.preparedAt = preparedAt;
        }

        @Override
        public final Builder preparedAt(Instant preparedAt) {
            this.preparedAt = preparedAt;
            return this;
        }

        public final PromptOverrideConfiguration.Builder getPromptOverrideConfiguration() {
            return promptOverrideConfiguration != null ? promptOverrideConfiguration.toBuilder() : null;
        }

        public final void setPromptOverrideConfiguration(PromptOverrideConfiguration.BuilderImpl promptOverrideConfiguration) {
            this.promptOverrideConfiguration = promptOverrideConfiguration != null ? promptOverrideConfiguration.build() : null;
        }

        @Override
        public final Builder promptOverrideConfiguration(PromptOverrideConfiguration promptOverrideConfiguration) {
            this.promptOverrideConfiguration = promptOverrideConfiguration;
            return this;
        }

        public final Collection<String> getRecommendedActions() {
            if (recommendedActions instanceof SdkAutoConstructList) {
                return null;
            }
            return recommendedActions;
        }

        public final void setRecommendedActions(Collection<String> recommendedActions) {
            this.recommendedActions = RecommendedActionsCopier.copy(recommendedActions);
        }

        @Override
        public final Builder recommendedActions(Collection<String> recommendedActions) {
            this.recommendedActions = RecommendedActionsCopier.copy(recommendedActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendedActions(String... recommendedActions) {
            recommendedActions(Arrays.asList(recommendedActions));
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        @Override
        public Agent build() {
            return new Agent(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
