/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagent.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Defines parameters that the agent needs to invoke from the user to complete the function. Corresponds to an action in
 * an action group.
 * </p>
 * <p>
 * This data type is used in the following API operations:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href=
 * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_CreateAgentActionGroup.html#API_agent_CreateAgentActionGroup_RequestSyntax"
 * >CreateAgentActionGroup request</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href=
 * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_CreateAgentActionGroup.html#API_agent_CreateAgentActionGroup_ResponseSyntax"
 * >CreateAgentActionGroup response</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href=
 * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_UpdateAgentActionGroup.html#API_agent_UpdateAgentActionGroup_RequestSyntax"
 * >UpdateAgentActionGroup request</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href=
 * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_UpdateAgentActionGroup.html#API_agent_UpdateAgentActionGroup_ResponseSyntax"
 * >UpdateAgentActionGroup response</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href=
 * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetAgentActionGroup.html#API_agent_GetAgentActionGroup_ResponseSyntax"
 * >GetAgentActionGroup response</a>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Function implements SdkPojo, Serializable, ToCopyableBuilder<Function.Builder, Function> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(Function::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(Function::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<Map<String, ParameterDetail>> PARAMETERS_FIELD = SdkField
            .<Map<String, ParameterDetail>> builder(MarshallingType.MAP)
            .memberName("parameters")
            .getter(getter(Function::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("parameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<ParameterDetail> builder(MarshallingType.SDK_POJO)
                                            .constructor(ParameterDetail::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> REQUIRE_CONFIRMATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("requireConfirmation").getter(getter(Function::requireConfirmationAsString))
            .setter(setter(Builder::requireConfirmation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("requireConfirmation").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD, NAME_FIELD,
            PARAMETERS_FIELD, REQUIRE_CONFIRMATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String description;

    private final String name;

    private final Map<String, ParameterDetail> parameters;

    private final String requireConfirmation;

    private Function(BuilderImpl builder) {
        this.description = builder.description;
        this.name = builder.name;
        this.parameters = builder.parameters;
        this.requireConfirmation = builder.requireConfirmation;
    }

    /**
     * <p>
     * A description of the function and its purpose.
     * </p>
     * 
     * @return A description of the function and its purpose.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * A name for the function.
     * </p>
     * 
     * @return A name for the function.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parameters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The parameters that the agent elicits from the user to fulfill the function.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameters} method.
     * </p>
     * 
     * @return The parameters that the agent elicits from the user to fulfill the function.
     */
    public final Map<String, ParameterDetail> parameters() {
        return parameters;
    }

    /**
     * <p>
     * Contains information if user confirmation is required to invoke the function.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #requireConfirmation} will return {@link RequireConfirmation#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #requireConfirmationAsString}.
     * </p>
     * 
     * @return Contains information if user confirmation is required to invoke the function.
     * @see RequireConfirmation
     */
    public final RequireConfirmation requireConfirmation() {
        return RequireConfirmation.fromValue(requireConfirmation);
    }

    /**
     * <p>
     * Contains information if user confirmation is required to invoke the function.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #requireConfirmation} will return {@link RequireConfirmation#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #requireConfirmationAsString}.
     * </p>
     * 
     * @return Contains information if user confirmation is required to invoke the function.
     * @see RequireConfirmation
     */
    public final String requireConfirmationAsString() {
        return requireConfirmation;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(requireConfirmationAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Function)) {
            return false;
        }
        Function other = (Function) obj;
        return Objects.equals(description(), other.description()) && Objects.equals(name(), other.name())
                && hasParameters() == other.hasParameters() && Objects.equals(parameters(), other.parameters())
                && Objects.equals(requireConfirmationAsString(), other.requireConfirmationAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Function").add("Description", description()).add("Name", name())
                .add("Parameters", hasParameters() ? parameters() : null)
                .add("RequireConfirmation", requireConfirmationAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "requireConfirmation":
            return Optional.ofNullable(clazz.cast(requireConfirmationAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> java.util.function.Function<Object, T> getter(java.util.function.Function<Function, T> g) {
        return obj -> g.apply((Function) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Function> {
        /**
         * <p>
         * A description of the function and its purpose.
         * </p>
         * 
         * @param description
         *        A description of the function and its purpose.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A name for the function.
         * </p>
         * 
         * @param name
         *        A name for the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The parameters that the agent elicits from the user to fulfill the function.
         * </p>
         * 
         * @param parameters
         *        The parameters that the agent elicits from the user to fulfill the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, ParameterDetail> parameters);

        /**
         * <p>
         * Contains information if user confirmation is required to invoke the function.
         * </p>
         * 
         * @param requireConfirmation
         *        Contains information if user confirmation is required to invoke the function.
         * @see RequireConfirmation
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RequireConfirmation
         */
        Builder requireConfirmation(String requireConfirmation);

        /**
         * <p>
         * Contains information if user confirmation is required to invoke the function.
         * </p>
         * 
         * @param requireConfirmation
         *        Contains information if user confirmation is required to invoke the function.
         * @see RequireConfirmation
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RequireConfirmation
         */
        Builder requireConfirmation(RequireConfirmation requireConfirmation);
    }

    static final class BuilderImpl implements Builder {
        private String description;

        private String name;

        private Map<String, ParameterDetail> parameters = DefaultSdkAutoConstructMap.getInstance();

        private String requireConfirmation;

        private BuilderImpl() {
        }

        private BuilderImpl(Function model) {
            description(model.description);
            name(model.name);
            parameters(model.parameters);
            requireConfirmation(model.requireConfirmation);
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Map<String, ParameterDetail.Builder> getParameters() {
            Map<String, ParameterDetail.Builder> result = ParameterMapCopier.copyToBuilder(this.parameters);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setParameters(Map<String, ParameterDetail.BuilderImpl> parameters) {
            this.parameters = ParameterMapCopier.copyFromBuilder(parameters);
        }

        @Override
        public final Builder parameters(Map<String, ParameterDetail> parameters) {
            this.parameters = ParameterMapCopier.copy(parameters);
            return this;
        }

        public final String getRequireConfirmation() {
            return requireConfirmation;
        }

        public final void setRequireConfirmation(String requireConfirmation) {
            this.requireConfirmation = requireConfirmation;
        }

        @Override
        public final Builder requireConfirmation(String requireConfirmation) {
            this.requireConfirmation = requireConfirmation;
            return this;
        }

        @Override
        public final Builder requireConfirmation(RequireConfirmation requireConfirmation) {
            this.requireConfirmation(requireConfirmation == null ? null : requireConfirmation.toString());
            return this;
        }

        @Override
        public Function build() {
            return new Function(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
