/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagent.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details about a data ingestion job. Data sources are ingested into a knowledge base so that Large Language
 * Models (LLMs) can use your data.
 * </p>
 * <p>
 * This data type is used in the following API operations:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href=
 * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_StartIngestionJob.html#API_agent_StartIngestionJob_ResponseSyntax"
 * >StartIngestionJob response</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href=
 * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_GetIngestionJob.html#API_agent_GetIngestionJob_ResponseSyntax"
 * >GetIngestionJob response</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href=
 * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_ListIngestionJobs.html#API_agent_ListIngestionJobs_ResponseSyntax"
 * >ListIngestionJob response</a>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IngestionJob implements SdkPojo, Serializable, ToCopyableBuilder<IngestionJob.Builder, IngestionJob> {
    private static final SdkField<String> DATA_SOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("dataSourceId").getter(getter(IngestionJob::dataSourceId)).setter(setter(Builder::dataSourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dataSourceId").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(IngestionJob::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<List<String>> FAILURE_REASONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("failureReasons")
            .getter(getter(IngestionJob::failureReasons))
            .setter(setter(Builder::failureReasons))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("failureReasons").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> INGESTION_JOB_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ingestionJobId").getter(getter(IngestionJob::ingestionJobId)).setter(setter(Builder::ingestionJobId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ingestionJobId").build()).build();

    private static final SdkField<String> KNOWLEDGE_BASE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("knowledgeBaseId").getter(getter(IngestionJob::knowledgeBaseId)).setter(setter(Builder::knowledgeBaseId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("knowledgeBaseId").build()).build();

    private static final SdkField<Instant> STARTED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("startedAt")
            .getter(getter(IngestionJob::startedAt))
            .setter(setter(Builder::startedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("startedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<IngestionJobStatistics> STATISTICS_FIELD = SdkField
            .<IngestionJobStatistics> builder(MarshallingType.SDK_POJO).memberName("statistics")
            .getter(getter(IngestionJob::statistics)).setter(setter(Builder::statistics))
            .constructor(IngestionJobStatistics::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statistics").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(IngestionJob::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt")
            .getter(getter(IngestionJob::updatedAt))
            .setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DATA_SOURCE_ID_FIELD,
            DESCRIPTION_FIELD, FAILURE_REASONS_FIELD, INGESTION_JOB_ID_FIELD, KNOWLEDGE_BASE_ID_FIELD, STARTED_AT_FIELD,
            STATISTICS_FIELD, STATUS_FIELD, UPDATED_AT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String dataSourceId;

    private final String description;

    private final List<String> failureReasons;

    private final String ingestionJobId;

    private final String knowledgeBaseId;

    private final Instant startedAt;

    private final IngestionJobStatistics statistics;

    private final String status;

    private final Instant updatedAt;

    private IngestionJob(BuilderImpl builder) {
        this.dataSourceId = builder.dataSourceId;
        this.description = builder.description;
        this.failureReasons = builder.failureReasons;
        this.ingestionJobId = builder.ingestionJobId;
        this.knowledgeBaseId = builder.knowledgeBaseId;
        this.startedAt = builder.startedAt;
        this.statistics = builder.statistics;
        this.status = builder.status;
        this.updatedAt = builder.updatedAt;
    }

    /**
     * <p>
     * The unique identifier of the data source for the data ingestion job.
     * </p>
     * 
     * @return The unique identifier of the data source for the data ingestion job.
     */
    public final String dataSourceId() {
        return dataSourceId;
    }

    /**
     * <p>
     * The description of the data ingestion job.
     * </p>
     * 
     * @return The description of the data ingestion job.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the FailureReasons property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFailureReasons() {
        return failureReasons != null && !(failureReasons instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of reasons that the data ingestion job failed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFailureReasons} method.
     * </p>
     * 
     * @return A list of reasons that the data ingestion job failed.
     */
    public final List<String> failureReasons() {
        return failureReasons;
    }

    /**
     * <p>
     * The unique identifier of the data ingestion job.
     * </p>
     * 
     * @return The unique identifier of the data ingestion job.
     */
    public final String ingestionJobId() {
        return ingestionJobId;
    }

    /**
     * <p>
     * The unique identifier of the knowledge for the data ingestion job.
     * </p>
     * 
     * @return The unique identifier of the knowledge for the data ingestion job.
     */
    public final String knowledgeBaseId() {
        return knowledgeBaseId;
    }

    /**
     * <p>
     * The time the data ingestion job started.
     * </p>
     * <p>
     * If you stop a data ingestion job, the <code>startedAt</code> time is the time the job was started before the job
     * was stopped.
     * </p>
     * 
     * @return The time the data ingestion job started.</p>
     *         <p>
     *         If you stop a data ingestion job, the <code>startedAt</code> time is the time the job was started before
     *         the job was stopped.
     */
    public final Instant startedAt() {
        return startedAt;
    }

    /**
     * <p>
     * Contains statistics about the data ingestion job.
     * </p>
     * 
     * @return Contains statistics about the data ingestion job.
     */
    public final IngestionJobStatistics statistics() {
        return statistics;
    }

    /**
     * <p>
     * The status of the data ingestion job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link IngestionJobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the data ingestion job.
     * @see IngestionJobStatus
     */
    public final IngestionJobStatus status() {
        return IngestionJobStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the data ingestion job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link IngestionJobStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the data ingestion job.
     * @see IngestionJobStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The time the data ingestion job was last updated.
     * </p>
     * <p>
     * If you stop a data ingestion job, the <code>updatedAt</code> time is the time the job was stopped.
     * </p>
     * 
     * @return The time the data ingestion job was last updated.</p>
     *         <p>
     *         If you stop a data ingestion job, the <code>updatedAt</code> time is the time the job was stopped.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(dataSourceId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasFailureReasons() ? failureReasons() : null);
        hashCode = 31 * hashCode + Objects.hashCode(ingestionJobId());
        hashCode = 31 * hashCode + Objects.hashCode(knowledgeBaseId());
        hashCode = 31 * hashCode + Objects.hashCode(startedAt());
        hashCode = 31 * hashCode + Objects.hashCode(statistics());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IngestionJob)) {
            return false;
        }
        IngestionJob other = (IngestionJob) obj;
        return Objects.equals(dataSourceId(), other.dataSourceId()) && Objects.equals(description(), other.description())
                && hasFailureReasons() == other.hasFailureReasons() && Objects.equals(failureReasons(), other.failureReasons())
                && Objects.equals(ingestionJobId(), other.ingestionJobId())
                && Objects.equals(knowledgeBaseId(), other.knowledgeBaseId()) && Objects.equals(startedAt(), other.startedAt())
                && Objects.equals(statistics(), other.statistics()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(updatedAt(), other.updatedAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IngestionJob").add("DataSourceId", dataSourceId()).add("Description", description())
                .add("FailureReasons", hasFailureReasons() ? failureReasons() : null).add("IngestionJobId", ingestionJobId())
                .add("KnowledgeBaseId", knowledgeBaseId()).add("StartedAt", startedAt()).add("Statistics", statistics())
                .add("Status", statusAsString()).add("UpdatedAt", updatedAt()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "dataSourceId":
            return Optional.ofNullable(clazz.cast(dataSourceId()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "failureReasons":
            return Optional.ofNullable(clazz.cast(failureReasons()));
        case "ingestionJobId":
            return Optional.ofNullable(clazz.cast(ingestionJobId()));
        case "knowledgeBaseId":
            return Optional.ofNullable(clazz.cast(knowledgeBaseId()));
        case "startedAt":
            return Optional.ofNullable(clazz.cast(startedAt()));
        case "statistics":
            return Optional.ofNullable(clazz.cast(statistics()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<IngestionJob, T> g) {
        return obj -> g.apply((IngestionJob) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IngestionJob> {
        /**
         * <p>
         * The unique identifier of the data source for the data ingestion job.
         * </p>
         * 
         * @param dataSourceId
         *        The unique identifier of the data source for the data ingestion job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataSourceId(String dataSourceId);

        /**
         * <p>
         * The description of the data ingestion job.
         * </p>
         * 
         * @param description
         *        The description of the data ingestion job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A list of reasons that the data ingestion job failed.
         * </p>
         * 
         * @param failureReasons
         *        A list of reasons that the data ingestion job failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReasons(Collection<String> failureReasons);

        /**
         * <p>
         * A list of reasons that the data ingestion job failed.
         * </p>
         * 
         * @param failureReasons
         *        A list of reasons that the data ingestion job failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReasons(String... failureReasons);

        /**
         * <p>
         * The unique identifier of the data ingestion job.
         * </p>
         * 
         * @param ingestionJobId
         *        The unique identifier of the data ingestion job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ingestionJobId(String ingestionJobId);

        /**
         * <p>
         * The unique identifier of the knowledge for the data ingestion job.
         * </p>
         * 
         * @param knowledgeBaseId
         *        The unique identifier of the knowledge for the data ingestion job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder knowledgeBaseId(String knowledgeBaseId);

        /**
         * <p>
         * The time the data ingestion job started.
         * </p>
         * <p>
         * If you stop a data ingestion job, the <code>startedAt</code> time is the time the job was started before the
         * job was stopped.
         * </p>
         * 
         * @param startedAt
         *        The time the data ingestion job started.</p>
         *        <p>
         *        If you stop a data ingestion job, the <code>startedAt</code> time is the time the job was started
         *        before the job was stopped.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startedAt(Instant startedAt);

        /**
         * <p>
         * Contains statistics about the data ingestion job.
         * </p>
         * 
         * @param statistics
         *        Contains statistics about the data ingestion job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statistics(IngestionJobStatistics statistics);

        /**
         * <p>
         * Contains statistics about the data ingestion job.
         * </p>
         * This is a convenience method that creates an instance of the {@link IngestionJobStatistics.Builder} avoiding
         * the need to create one manually via {@link IngestionJobStatistics#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IngestionJobStatistics.Builder#build()} is called immediately and
         * its result is passed to {@link #statistics(IngestionJobStatistics)}.
         * 
         * @param statistics
         *        a consumer that will call methods on {@link IngestionJobStatistics.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statistics(IngestionJobStatistics)
         */
        default Builder statistics(Consumer<IngestionJobStatistics.Builder> statistics) {
            return statistics(IngestionJobStatistics.builder().applyMutation(statistics).build());
        }

        /**
         * <p>
         * The status of the data ingestion job.
         * </p>
         * 
         * @param status
         *        The status of the data ingestion job.
         * @see IngestionJobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IngestionJobStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the data ingestion job.
         * </p>
         * 
         * @param status
         *        The status of the data ingestion job.
         * @see IngestionJobStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IngestionJobStatus
         */
        Builder status(IngestionJobStatus status);

        /**
         * <p>
         * The time the data ingestion job was last updated.
         * </p>
         * <p>
         * If you stop a data ingestion job, the <code>updatedAt</code> time is the time the job was stopped.
         * </p>
         * 
         * @param updatedAt
         *        The time the data ingestion job was last updated.</p>
         *        <p>
         *        If you stop a data ingestion job, the <code>updatedAt</code> time is the time the job was stopped.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);
    }

    static final class BuilderImpl implements Builder {
        private String dataSourceId;

        private String description;

        private List<String> failureReasons = DefaultSdkAutoConstructList.getInstance();

        private String ingestionJobId;

        private String knowledgeBaseId;

        private Instant startedAt;

        private IngestionJobStatistics statistics;

        private String status;

        private Instant updatedAt;

        private BuilderImpl() {
        }

        private BuilderImpl(IngestionJob model) {
            dataSourceId(model.dataSourceId);
            description(model.description);
            failureReasons(model.failureReasons);
            ingestionJobId(model.ingestionJobId);
            knowledgeBaseId(model.knowledgeBaseId);
            startedAt(model.startedAt);
            statistics(model.statistics);
            status(model.status);
            updatedAt(model.updatedAt);
        }

        public final String getDataSourceId() {
            return dataSourceId;
        }

        public final void setDataSourceId(String dataSourceId) {
            this.dataSourceId = dataSourceId;
        }

        @Override
        public final Builder dataSourceId(String dataSourceId) {
            this.dataSourceId = dataSourceId;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Collection<String> getFailureReasons() {
            if (failureReasons instanceof SdkAutoConstructList) {
                return null;
            }
            return failureReasons;
        }

        public final void setFailureReasons(Collection<String> failureReasons) {
            this.failureReasons = FailureReasonsCopier.copy(failureReasons);
        }

        @Override
        public final Builder failureReasons(Collection<String> failureReasons) {
            this.failureReasons = FailureReasonsCopier.copy(failureReasons);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder failureReasons(String... failureReasons) {
            failureReasons(Arrays.asList(failureReasons));
            return this;
        }

        public final String getIngestionJobId() {
            return ingestionJobId;
        }

        public final void setIngestionJobId(String ingestionJobId) {
            this.ingestionJobId = ingestionJobId;
        }

        @Override
        public final Builder ingestionJobId(String ingestionJobId) {
            this.ingestionJobId = ingestionJobId;
            return this;
        }

        public final String getKnowledgeBaseId() {
            return knowledgeBaseId;
        }

        public final void setKnowledgeBaseId(String knowledgeBaseId) {
            this.knowledgeBaseId = knowledgeBaseId;
        }

        @Override
        public final Builder knowledgeBaseId(String knowledgeBaseId) {
            this.knowledgeBaseId = knowledgeBaseId;
            return this;
        }

        public final Instant getStartedAt() {
            return startedAt;
        }

        public final void setStartedAt(Instant startedAt) {
            this.startedAt = startedAt;
        }

        @Override
        public final Builder startedAt(Instant startedAt) {
            this.startedAt = startedAt;
            return this;
        }

        public final IngestionJobStatistics.Builder getStatistics() {
            return statistics != null ? statistics.toBuilder() : null;
        }

        public final void setStatistics(IngestionJobStatistics.BuilderImpl statistics) {
            this.statistics = statistics != null ? statistics.build() : null;
        }

        @Override
        public final Builder statistics(IngestionJobStatistics statistics) {
            this.statistics = statistics;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(IngestionJobStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        @Override
        public IngestionJob build() {
            return new IngestionJob(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
