/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagent.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration information to connect to Amazon S3 as your data source.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3DataSourceConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<S3DataSourceConfiguration.Builder, S3DataSourceConfiguration> {
    private static final SdkField<String> BUCKET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("bucketArn").getter(getter(S3DataSourceConfiguration::bucketArn)).setter(setter(Builder::bucketArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bucketArn").build()).build();

    private static final SdkField<String> BUCKET_OWNER_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("bucketOwnerAccountId").getter(getter(S3DataSourceConfiguration::bucketOwnerAccountId))
            .setter(setter(Builder::bucketOwnerAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bucketOwnerAccountId").build())
            .build();

    private static final SdkField<List<String>> INCLUSION_PREFIXES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("inclusionPrefixes")
            .getter(getter(S3DataSourceConfiguration::inclusionPrefixes))
            .setter(setter(Builder::inclusionPrefixes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inclusionPrefixes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_ARN_FIELD,
            BUCKET_OWNER_ACCOUNT_ID_FIELD, INCLUSION_PREFIXES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String bucketArn;

    private final String bucketOwnerAccountId;

    private final List<String> inclusionPrefixes;

    private S3DataSourceConfiguration(BuilderImpl builder) {
        this.bucketArn = builder.bucketArn;
        this.bucketOwnerAccountId = builder.bucketOwnerAccountId;
        this.inclusionPrefixes = builder.inclusionPrefixes;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the S3 bucket that contains your data.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the S3 bucket that contains your data.
     */
    public final String bucketArn() {
        return bucketArn;
    }

    /**
     * <p>
     * The account ID for the owner of the S3 bucket.
     * </p>
     * 
     * @return The account ID for the owner of the S3 bucket.
     */
    public final String bucketOwnerAccountId() {
        return bucketOwnerAccountId;
    }

    /**
     * For responses, this returns true if the service returned a value for the InclusionPrefixes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInclusionPrefixes() {
        return inclusionPrefixes != null && !(inclusionPrefixes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of S3 prefixes to include certain files or content. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-prefixes.html">Organizing objects using
     * prefixes</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInclusionPrefixes} method.
     * </p>
     * 
     * @return A list of S3 prefixes to include certain files or content. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-prefixes.html">Organizing objects using
     *         prefixes</a>.
     */
    public final List<String> inclusionPrefixes() {
        return inclusionPrefixes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bucketArn());
        hashCode = 31 * hashCode + Objects.hashCode(bucketOwnerAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(hasInclusionPrefixes() ? inclusionPrefixes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3DataSourceConfiguration)) {
            return false;
        }
        S3DataSourceConfiguration other = (S3DataSourceConfiguration) obj;
        return Objects.equals(bucketArn(), other.bucketArn())
                && Objects.equals(bucketOwnerAccountId(), other.bucketOwnerAccountId())
                && hasInclusionPrefixes() == other.hasInclusionPrefixes()
                && Objects.equals(inclusionPrefixes(), other.inclusionPrefixes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3DataSourceConfiguration").add("BucketArn", bucketArn())
                .add("BucketOwnerAccountId", bucketOwnerAccountId())
                .add("InclusionPrefixes", inclusionPrefixes() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "bucketArn":
            return Optional.ofNullable(clazz.cast(bucketArn()));
        case "bucketOwnerAccountId":
            return Optional.ofNullable(clazz.cast(bucketOwnerAccountId()));
        case "inclusionPrefixes":
            return Optional.ofNullable(clazz.cast(inclusionPrefixes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3DataSourceConfiguration, T> g) {
        return obj -> g.apply((S3DataSourceConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3DataSourceConfiguration> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the S3 bucket that contains your data.
         * </p>
         * 
         * @param bucketArn
         *        The Amazon Resource Name (ARN) of the S3 bucket that contains your data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketArn(String bucketArn);

        /**
         * <p>
         * The account ID for the owner of the S3 bucket.
         * </p>
         * 
         * @param bucketOwnerAccountId
         *        The account ID for the owner of the S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketOwnerAccountId(String bucketOwnerAccountId);

        /**
         * <p>
         * A list of S3 prefixes to include certain files or content. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-prefixes.html">Organizing objects using
         * prefixes</a>.
         * </p>
         * 
         * @param inclusionPrefixes
         *        A list of S3 prefixes to include certain files or content. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-prefixes.html">Organizing objects
         *        using prefixes</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPrefixes(Collection<String> inclusionPrefixes);

        /**
         * <p>
         * A list of S3 prefixes to include certain files or content. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-prefixes.html">Organizing objects using
         * prefixes</a>.
         * </p>
         * 
         * @param inclusionPrefixes
         *        A list of S3 prefixes to include certain files or content. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/using-prefixes.html">Organizing objects
         *        using prefixes</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inclusionPrefixes(String... inclusionPrefixes);
    }

    static final class BuilderImpl implements Builder {
        private String bucketArn;

        private String bucketOwnerAccountId;

        private List<String> inclusionPrefixes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(S3DataSourceConfiguration model) {
            bucketArn(model.bucketArn);
            bucketOwnerAccountId(model.bucketOwnerAccountId);
            inclusionPrefixes(model.inclusionPrefixes);
        }

        public final String getBucketArn() {
            return bucketArn;
        }

        public final void setBucketArn(String bucketArn) {
            this.bucketArn = bucketArn;
        }

        @Override
        public final Builder bucketArn(String bucketArn) {
            this.bucketArn = bucketArn;
            return this;
        }

        public final String getBucketOwnerAccountId() {
            return bucketOwnerAccountId;
        }

        public final void setBucketOwnerAccountId(String bucketOwnerAccountId) {
            this.bucketOwnerAccountId = bucketOwnerAccountId;
        }

        @Override
        public final Builder bucketOwnerAccountId(String bucketOwnerAccountId) {
            this.bucketOwnerAccountId = bucketOwnerAccountId;
            return this;
        }

        public final Collection<String> getInclusionPrefixes() {
            if (inclusionPrefixes instanceof SdkAutoConstructList) {
                return null;
            }
            return inclusionPrefixes;
        }

        public final void setInclusionPrefixes(Collection<String> inclusionPrefixes) {
            this.inclusionPrefixes = S3PrefixesCopier.copy(inclusionPrefixes);
        }

        @Override
        public final Builder inclusionPrefixes(Collection<String> inclusionPrefixes) {
            this.inclusionPrefixes = S3PrefixesCopier.copy(inclusionPrefixes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inclusionPrefixes(String... inclusionPrefixes) {
            inclusionPrefixes(Arrays.asList(inclusionPrefixes));
            return this;
        }

        @Override
        public S3DataSourceConfiguration build() {
            return new S3DataSourceConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
