/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagent.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains configurations for a knowledge base node in a flow. This node takes a query as the input and returns, as the
 * output, the retrieved responses directly (as an array) or a response generated based on the retrieved responses. For
 * more information, see <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/flows-nodes.html">Node types in
 * Amazon Bedrock works</a> in the Amazon Bedrock User Guide.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class KnowledgeBaseFlowNodeConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<KnowledgeBaseFlowNodeConfiguration.Builder, KnowledgeBaseFlowNodeConfiguration> {
    private static final SdkField<GuardrailConfiguration> GUARDRAIL_CONFIGURATION_FIELD = SdkField
            .<GuardrailConfiguration> builder(MarshallingType.SDK_POJO).memberName("guardrailConfiguration")
            .getter(getter(KnowledgeBaseFlowNodeConfiguration::guardrailConfiguration))
            .setter(setter(Builder::guardrailConfiguration)).constructor(GuardrailConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("guardrailConfiguration").build())
            .build();

    private static final SdkField<String> KNOWLEDGE_BASE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("knowledgeBaseId").getter(getter(KnowledgeBaseFlowNodeConfiguration::knowledgeBaseId))
            .setter(setter(Builder::knowledgeBaseId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("knowledgeBaseId").build()).build();

    private static final SdkField<String> MODEL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("modelId").getter(getter(KnowledgeBaseFlowNodeConfiguration::modelId)).setter(setter(Builder::modelId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modelId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(GUARDRAIL_CONFIGURATION_FIELD,
            KNOWLEDGE_BASE_ID_FIELD, MODEL_ID_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final GuardrailConfiguration guardrailConfiguration;

    private final String knowledgeBaseId;

    private final String modelId;

    private KnowledgeBaseFlowNodeConfiguration(BuilderImpl builder) {
        this.guardrailConfiguration = builder.guardrailConfiguration;
        this.knowledgeBaseId = builder.knowledgeBaseId;
        this.modelId = builder.modelId;
    }

    /**
     * <p>
     * Contains configurations for a guardrail to apply during query and response generation for the knowledge base in
     * this configuration.
     * </p>
     * 
     * @return Contains configurations for a guardrail to apply during query and response generation for the knowledge
     *         base in this configuration.
     */
    public final GuardrailConfiguration guardrailConfiguration() {
        return guardrailConfiguration;
    }

    /**
     * <p>
     * The unique identifier of the knowledge base to query.
     * </p>
     * 
     * @return The unique identifier of the knowledge base to query.
     */
    public final String knowledgeBaseId() {
        return knowledgeBaseId;
    }

    /**
     * <p>
     * The unique identifier of the model or <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html">inference profile</a> to
     * use to generate a response from the query results. Omit this field if you want to return the retrieved results as
     * an array.
     * </p>
     * 
     * @return The unique identifier of the model or <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html">inference
     *         profile</a> to use to generate a response from the query results. Omit this field if you want to return
     *         the retrieved results as an array.
     */
    public final String modelId() {
        return modelId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(guardrailConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(knowledgeBaseId());
        hashCode = 31 * hashCode + Objects.hashCode(modelId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KnowledgeBaseFlowNodeConfiguration)) {
            return false;
        }
        KnowledgeBaseFlowNodeConfiguration other = (KnowledgeBaseFlowNodeConfiguration) obj;
        return Objects.equals(guardrailConfiguration(), other.guardrailConfiguration())
                && Objects.equals(knowledgeBaseId(), other.knowledgeBaseId()) && Objects.equals(modelId(), other.modelId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("KnowledgeBaseFlowNodeConfiguration").add("GuardrailConfiguration", guardrailConfiguration())
                .add("KnowledgeBaseId", knowledgeBaseId()).add("ModelId", modelId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "guardrailConfiguration":
            return Optional.ofNullable(clazz.cast(guardrailConfiguration()));
        case "knowledgeBaseId":
            return Optional.ofNullable(clazz.cast(knowledgeBaseId()));
        case "modelId":
            return Optional.ofNullable(clazz.cast(modelId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("guardrailConfiguration", GUARDRAIL_CONFIGURATION_FIELD);
        map.put("knowledgeBaseId", KNOWLEDGE_BASE_ID_FIELD);
        map.put("modelId", MODEL_ID_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<KnowledgeBaseFlowNodeConfiguration, T> g) {
        return obj -> g.apply((KnowledgeBaseFlowNodeConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, KnowledgeBaseFlowNodeConfiguration> {
        /**
         * <p>
         * Contains configurations for a guardrail to apply during query and response generation for the knowledge base
         * in this configuration.
         * </p>
         * 
         * @param guardrailConfiguration
         *        Contains configurations for a guardrail to apply during query and response generation for the
         *        knowledge base in this configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder guardrailConfiguration(GuardrailConfiguration guardrailConfiguration);

        /**
         * <p>
         * Contains configurations for a guardrail to apply during query and response generation for the knowledge base
         * in this configuration.
         * </p>
         * This is a convenience method that creates an instance of the {@link GuardrailConfiguration.Builder} avoiding
         * the need to create one manually via {@link GuardrailConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link GuardrailConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #guardrailConfiguration(GuardrailConfiguration)}.
         * 
         * @param guardrailConfiguration
         *        a consumer that will call methods on {@link GuardrailConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #guardrailConfiguration(GuardrailConfiguration)
         */
        default Builder guardrailConfiguration(Consumer<GuardrailConfiguration.Builder> guardrailConfiguration) {
            return guardrailConfiguration(GuardrailConfiguration.builder().applyMutation(guardrailConfiguration).build());
        }

        /**
         * <p>
         * The unique identifier of the knowledge base to query.
         * </p>
         * 
         * @param knowledgeBaseId
         *        The unique identifier of the knowledge base to query.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder knowledgeBaseId(String knowledgeBaseId);

        /**
         * <p>
         * The unique identifier of the model or <a
         * href="https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html">inference profile</a>
         * to use to generate a response from the query results. Omit this field if you want to return the retrieved
         * results as an array.
         * </p>
         * 
         * @param modelId
         *        The unique identifier of the model or <a
         *        href="https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html">inference
         *        profile</a> to use to generate a response from the query results. Omit this field if you want to
         *        return the retrieved results as an array.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelId(String modelId);
    }

    static final class BuilderImpl implements Builder {
        private GuardrailConfiguration guardrailConfiguration;

        private String knowledgeBaseId;

        private String modelId;

        private BuilderImpl() {
        }

        private BuilderImpl(KnowledgeBaseFlowNodeConfiguration model) {
            guardrailConfiguration(model.guardrailConfiguration);
            knowledgeBaseId(model.knowledgeBaseId);
            modelId(model.modelId);
        }

        public final GuardrailConfiguration.Builder getGuardrailConfiguration() {
            return guardrailConfiguration != null ? guardrailConfiguration.toBuilder() : null;
        }

        public final void setGuardrailConfiguration(GuardrailConfiguration.BuilderImpl guardrailConfiguration) {
            this.guardrailConfiguration = guardrailConfiguration != null ? guardrailConfiguration.build() : null;
        }

        @Override
        public final Builder guardrailConfiguration(GuardrailConfiguration guardrailConfiguration) {
            this.guardrailConfiguration = guardrailConfiguration;
            return this;
        }

        public final String getKnowledgeBaseId() {
            return knowledgeBaseId;
        }

        public final void setKnowledgeBaseId(String knowledgeBaseId) {
            this.knowledgeBaseId = knowledgeBaseId;
        }

        @Override
        public final Builder knowledgeBaseId(String knowledgeBaseId) {
            this.knowledgeBaseId = knowledgeBaseId;
            return this;
        }

        public final String getModelId() {
            return modelId;
        }

        public final void setModelId(String modelId) {
            this.modelId = modelId;
        }

        @Override
        public final Builder modelId(String modelId) {
            this.modelId = modelId;
            return this;
        }

        @Override
        public KnowledgeBaseFlowNodeConfiguration build() {
            return new KnowledgeBaseFlowNodeConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
