/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagent.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details about a version of an agent.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AgentVersion implements SdkPojo, Serializable, ToCopyableBuilder<AgentVersion.Builder, AgentVersion> {
    private static final SdkField<String> AGENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("agentId").getter(getter(AgentVersion::agentId)).setter(setter(Builder::agentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("agentId").build()).build();

    private static final SdkField<String> AGENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("agentName").getter(getter(AgentVersion::agentName)).setter(setter(Builder::agentName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("agentName").build()).build();

    private static final SdkField<String> AGENT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("agentArn").getter(getter(AgentVersion::agentArn)).setter(setter(Builder::agentArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("agentArn").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("version")
            .getter(getter(AgentVersion::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final SdkField<String> INSTRUCTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("instruction").getter(getter(AgentVersion::instruction)).setter(setter(Builder::instruction))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instruction").build()).build();

    private static final SdkField<String> AGENT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("agentStatus").getter(getter(AgentVersion::agentStatusAsString)).setter(setter(Builder::agentStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("agentStatus").build()).build();

    private static final SdkField<String> FOUNDATION_MODEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("foundationModel").getter(getter(AgentVersion::foundationModel)).setter(setter(Builder::foundationModel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("foundationModel").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(AgentVersion::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<Integer> IDLE_SESSION_TTL_IN_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER).memberName("idleSessionTTLInSeconds")
            .getter(getter(AgentVersion::idleSessionTTLInSeconds)).setter(setter(Builder::idleSessionTTLInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("idleSessionTTLInSeconds").build())
            .build();

    private static final SdkField<String> AGENT_RESOURCE_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("agentResourceRoleArn").getter(getter(AgentVersion::agentResourceRoleArn))
            .setter(setter(Builder::agentResourceRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("agentResourceRoleArn").build())
            .build();

    private static final SdkField<String> CUSTOMER_ENCRYPTION_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("customerEncryptionKeyArn").getter(getter(AgentVersion::customerEncryptionKeyArn))
            .setter(setter(Builder::customerEncryptionKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("customerEncryptionKeyArn").build())
            .build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(AgentVersion::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt")
            .getter(getter(AgentVersion::updatedAt))
            .setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<List<String>> FAILURE_REASONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("failureReasons")
            .getter(getter(AgentVersion::failureReasons))
            .setter(setter(Builder::failureReasons))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("failureReasons").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> RECOMMENDED_ACTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("recommendedActions")
            .getter(getter(AgentVersion::recommendedActions))
            .setter(setter(Builder::recommendedActions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recommendedActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<PromptOverrideConfiguration> PROMPT_OVERRIDE_CONFIGURATION_FIELD = SdkField
            .<PromptOverrideConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("promptOverrideConfiguration")
            .getter(getter(AgentVersion::promptOverrideConfiguration))
            .setter(setter(Builder::promptOverrideConfiguration))
            .constructor(PromptOverrideConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("promptOverrideConfiguration")
                    .build()).build();

    private static final SdkField<GuardrailConfiguration> GUARDRAIL_CONFIGURATION_FIELD = SdkField
            .<GuardrailConfiguration> builder(MarshallingType.SDK_POJO).memberName("guardrailConfiguration")
            .getter(getter(AgentVersion::guardrailConfiguration)).setter(setter(Builder::guardrailConfiguration))
            .constructor(GuardrailConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("guardrailConfiguration").build())
            .build();

    private static final SdkField<MemoryConfiguration> MEMORY_CONFIGURATION_FIELD = SdkField
            .<MemoryConfiguration> builder(MarshallingType.SDK_POJO).memberName("memoryConfiguration")
            .getter(getter(AgentVersion::memoryConfiguration)).setter(setter(Builder::memoryConfiguration))
            .constructor(MemoryConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memoryConfiguration").build())
            .build();

    private static final SdkField<String> AGENT_COLLABORATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("agentCollaboration").getter(getter(AgentVersion::agentCollaborationAsString))
            .setter(setter(Builder::agentCollaboration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("agentCollaboration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AGENT_ID_FIELD,
            AGENT_NAME_FIELD, AGENT_ARN_FIELD, VERSION_FIELD, INSTRUCTION_FIELD, AGENT_STATUS_FIELD, FOUNDATION_MODEL_FIELD,
            DESCRIPTION_FIELD, IDLE_SESSION_TTL_IN_SECONDS_FIELD, AGENT_RESOURCE_ROLE_ARN_FIELD,
            CUSTOMER_ENCRYPTION_KEY_ARN_FIELD, CREATED_AT_FIELD, UPDATED_AT_FIELD, FAILURE_REASONS_FIELD,
            RECOMMENDED_ACTIONS_FIELD, PROMPT_OVERRIDE_CONFIGURATION_FIELD, GUARDRAIL_CONFIGURATION_FIELD,
            MEMORY_CONFIGURATION_FIELD, AGENT_COLLABORATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String agentId;

    private final String agentName;

    private final String agentArn;

    private final String version;

    private final String instruction;

    private final String agentStatus;

    private final String foundationModel;

    private final String description;

    private final Integer idleSessionTTLInSeconds;

    private final String agentResourceRoleArn;

    private final String customerEncryptionKeyArn;

    private final Instant createdAt;

    private final Instant updatedAt;

    private final List<String> failureReasons;

    private final List<String> recommendedActions;

    private final PromptOverrideConfiguration promptOverrideConfiguration;

    private final GuardrailConfiguration guardrailConfiguration;

    private final MemoryConfiguration memoryConfiguration;

    private final String agentCollaboration;

    private AgentVersion(BuilderImpl builder) {
        this.agentId = builder.agentId;
        this.agentName = builder.agentName;
        this.agentArn = builder.agentArn;
        this.version = builder.version;
        this.instruction = builder.instruction;
        this.agentStatus = builder.agentStatus;
        this.foundationModel = builder.foundationModel;
        this.description = builder.description;
        this.idleSessionTTLInSeconds = builder.idleSessionTTLInSeconds;
        this.agentResourceRoleArn = builder.agentResourceRoleArn;
        this.customerEncryptionKeyArn = builder.customerEncryptionKeyArn;
        this.createdAt = builder.createdAt;
        this.updatedAt = builder.updatedAt;
        this.failureReasons = builder.failureReasons;
        this.recommendedActions = builder.recommendedActions;
        this.promptOverrideConfiguration = builder.promptOverrideConfiguration;
        this.guardrailConfiguration = builder.guardrailConfiguration;
        this.memoryConfiguration = builder.memoryConfiguration;
        this.agentCollaboration = builder.agentCollaboration;
    }

    /**
     * <p>
     * The unique identifier of the agent that the version belongs to.
     * </p>
     * 
     * @return The unique identifier of the agent that the version belongs to.
     */
    public final String agentId() {
        return agentId;
    }

    /**
     * <p>
     * The name of the agent that the version belongs to.
     * </p>
     * 
     * @return The name of the agent that the version belongs to.
     */
    public final String agentName() {
        return agentName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the agent that the version belongs to.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the agent that the version belongs to.
     */
    public final String agentArn() {
        return agentArn;
    }

    /**
     * <p>
     * The version number.
     * </p>
     * 
     * @return The version number.
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * The instructions provided to the agent.
     * </p>
     * 
     * @return The instructions provided to the agent.
     */
    public final String instruction() {
        return instruction;
    }

    /**
     * <p>
     * The status of the agent that the version belongs to.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #agentStatus} will
     * return {@link AgentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #agentStatusAsString}.
     * </p>
     * 
     * @return The status of the agent that the version belongs to.
     * @see AgentStatus
     */
    public final AgentStatus agentStatus() {
        return AgentStatus.fromValue(agentStatus);
    }

    /**
     * <p>
     * The status of the agent that the version belongs to.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #agentStatus} will
     * return {@link AgentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #agentStatusAsString}.
     * </p>
     * 
     * @return The status of the agent that the version belongs to.
     * @see AgentStatus
     */
    public final String agentStatusAsString() {
        return agentStatus;
    }

    /**
     * <p>
     * The foundation model that the version invokes.
     * </p>
     * 
     * @return The foundation model that the version invokes.
     */
    public final String foundationModel() {
        return foundationModel;
    }

    /**
     * <p>
     * The description of the version.
     * </p>
     * 
     * @return The description of the version.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The number of seconds for which Amazon Bedrock keeps information about a user's conversation with the agent.
     * </p>
     * <p>
     * A user interaction remains active for the amount of time specified. If no conversation occurs during this time,
     * the session expires and Amazon Bedrock deletes any data provided before the timeout.
     * </p>
     * 
     * @return The number of seconds for which Amazon Bedrock keeps information about a user's conversation with the
     *         agent.</p>
     *         <p>
     *         A user interaction remains active for the amount of time specified. If no conversation occurs during this
     *         time, the session expires and Amazon Bedrock deletes any data provided before the timeout.
     */
    public final Integer idleSessionTTLInSeconds() {
        return idleSessionTTLInSeconds;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IAM role with permissions to invoke API operations on the agent.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IAM role with permissions to invoke API operations on the agent.
     */
    public final String agentResourceRoleArn() {
        return agentResourceRoleArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the KMS key that encrypts the agent.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the KMS key that encrypts the agent.
     */
    public final String customerEncryptionKeyArn() {
        return customerEncryptionKeyArn;
    }

    /**
     * <p>
     * The time at which the version was created.
     * </p>
     * 
     * @return The time at which the version was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The time at which the version was last updated.
     * </p>
     * 
     * @return The time at which the version was last updated.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    /**
     * For responses, this returns true if the service returned a value for the FailureReasons property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFailureReasons() {
        return failureReasons != null && !(failureReasons instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of reasons that the API operation on the version failed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFailureReasons} method.
     * </p>
     * 
     * @return A list of reasons that the API operation on the version failed.
     */
    public final List<String> failureReasons() {
        return failureReasons;
    }

    /**
     * For responses, this returns true if the service returned a value for the RecommendedActions property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRecommendedActions() {
        return recommendedActions != null && !(recommendedActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of recommended actions to take for the failed API operation on the version to succeed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecommendedActions} method.
     * </p>
     * 
     * @return A list of recommended actions to take for the failed API operation on the version to succeed.
     */
    public final List<String> recommendedActions() {
        return recommendedActions;
    }

    /**
     * <p>
     * Contains configurations to override prompt templates in different parts of an agent sequence. For more
     * information, see <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html">Advanced
     * prompts</a>.
     * </p>
     * 
     * @return Contains configurations to override prompt templates in different parts of an agent sequence. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html">Advanced prompts</a>.
     */
    public final PromptOverrideConfiguration promptOverrideConfiguration() {
        return promptOverrideConfiguration;
    }

    /**
     * <p>
     * Details about the guardrail associated with the agent.
     * </p>
     * 
     * @return Details about the guardrail associated with the agent.
     */
    public final GuardrailConfiguration guardrailConfiguration() {
        return guardrailConfiguration;
    }

    /**
     * <p>
     * Contains details of the memory configuration on the version of the agent.
     * </p>
     * 
     * @return Contains details of the memory configuration on the version of the agent.
     */
    public final MemoryConfiguration memoryConfiguration() {
        return memoryConfiguration;
    }

    /**
     * <p>
     * The agent's collaboration settings.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #agentCollaboration} will return {@link AgentCollaboration#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #agentCollaborationAsString}.
     * </p>
     * 
     * @return The agent's collaboration settings.
     * @see AgentCollaboration
     */
    public final AgentCollaboration agentCollaboration() {
        return AgentCollaboration.fromValue(agentCollaboration);
    }

    /**
     * <p>
     * The agent's collaboration settings.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #agentCollaboration} will return {@link AgentCollaboration#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #agentCollaborationAsString}.
     * </p>
     * 
     * @return The agent's collaboration settings.
     * @see AgentCollaboration
     */
    public final String agentCollaborationAsString() {
        return agentCollaboration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(agentId());
        hashCode = 31 * hashCode + Objects.hashCode(agentName());
        hashCode = 31 * hashCode + Objects.hashCode(agentArn());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(instruction());
        hashCode = 31 * hashCode + Objects.hashCode(agentStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(foundationModel());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(idleSessionTTLInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(agentResourceRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(customerEncryptionKeyArn());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(hasFailureReasons() ? failureReasons() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasRecommendedActions() ? recommendedActions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(promptOverrideConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(guardrailConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(memoryConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(agentCollaborationAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AgentVersion)) {
            return false;
        }
        AgentVersion other = (AgentVersion) obj;
        return Objects.equals(agentId(), other.agentId()) && Objects.equals(agentName(), other.agentName())
                && Objects.equals(agentArn(), other.agentArn()) && Objects.equals(version(), other.version())
                && Objects.equals(instruction(), other.instruction())
                && Objects.equals(agentStatusAsString(), other.agentStatusAsString())
                && Objects.equals(foundationModel(), other.foundationModel())
                && Objects.equals(description(), other.description())
                && Objects.equals(idleSessionTTLInSeconds(), other.idleSessionTTLInSeconds())
                && Objects.equals(agentResourceRoleArn(), other.agentResourceRoleArn())
                && Objects.equals(customerEncryptionKeyArn(), other.customerEncryptionKeyArn())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(updatedAt(), other.updatedAt())
                && hasFailureReasons() == other.hasFailureReasons() && Objects.equals(failureReasons(), other.failureReasons())
                && hasRecommendedActions() == other.hasRecommendedActions()
                && Objects.equals(recommendedActions(), other.recommendedActions())
                && Objects.equals(promptOverrideConfiguration(), other.promptOverrideConfiguration())
                && Objects.equals(guardrailConfiguration(), other.guardrailConfiguration())
                && Objects.equals(memoryConfiguration(), other.memoryConfiguration())
                && Objects.equals(agentCollaborationAsString(), other.agentCollaborationAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("AgentVersion")
                .add("AgentId", agentId())
                .add("AgentName", agentName())
                .add("AgentArn", agentArn())
                .add("Version", version())
                .add("Instruction", instruction() == null ? null : "*** Sensitive Data Redacted ***")
                .add("AgentStatus", agentStatusAsString())
                .add("FoundationModel", foundationModel())
                .add("Description", description())
                .add("IdleSessionTTLInSeconds", idleSessionTTLInSeconds())
                .add("AgentResourceRoleArn", agentResourceRoleArn())
                .add("CustomerEncryptionKeyArn", customerEncryptionKeyArn())
                .add("CreatedAt", createdAt())
                .add("UpdatedAt", updatedAt())
                .add("FailureReasons", hasFailureReasons() ? failureReasons() : null)
                .add("RecommendedActions", hasRecommendedActions() ? recommendedActions() : null)
                .add("PromptOverrideConfiguration",
                        promptOverrideConfiguration() == null ? null : "*** Sensitive Data Redacted ***")
                .add("GuardrailConfiguration", guardrailConfiguration()).add("MemoryConfiguration", memoryConfiguration())
                .add("AgentCollaboration", agentCollaborationAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "agentId":
            return Optional.ofNullable(clazz.cast(agentId()));
        case "agentName":
            return Optional.ofNullable(clazz.cast(agentName()));
        case "agentArn":
            return Optional.ofNullable(clazz.cast(agentArn()));
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "instruction":
            return Optional.ofNullable(clazz.cast(instruction()));
        case "agentStatus":
            return Optional.ofNullable(clazz.cast(agentStatusAsString()));
        case "foundationModel":
            return Optional.ofNullable(clazz.cast(foundationModel()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "idleSessionTTLInSeconds":
            return Optional.ofNullable(clazz.cast(idleSessionTTLInSeconds()));
        case "agentResourceRoleArn":
            return Optional.ofNullable(clazz.cast(agentResourceRoleArn()));
        case "customerEncryptionKeyArn":
            return Optional.ofNullable(clazz.cast(customerEncryptionKeyArn()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        case "failureReasons":
            return Optional.ofNullable(clazz.cast(failureReasons()));
        case "recommendedActions":
            return Optional.ofNullable(clazz.cast(recommendedActions()));
        case "promptOverrideConfiguration":
            return Optional.ofNullable(clazz.cast(promptOverrideConfiguration()));
        case "guardrailConfiguration":
            return Optional.ofNullable(clazz.cast(guardrailConfiguration()));
        case "memoryConfiguration":
            return Optional.ofNullable(clazz.cast(memoryConfiguration()));
        case "agentCollaboration":
            return Optional.ofNullable(clazz.cast(agentCollaborationAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("agentId", AGENT_ID_FIELD);
        map.put("agentName", AGENT_NAME_FIELD);
        map.put("agentArn", AGENT_ARN_FIELD);
        map.put("version", VERSION_FIELD);
        map.put("instruction", INSTRUCTION_FIELD);
        map.put("agentStatus", AGENT_STATUS_FIELD);
        map.put("foundationModel", FOUNDATION_MODEL_FIELD);
        map.put("description", DESCRIPTION_FIELD);
        map.put("idleSessionTTLInSeconds", IDLE_SESSION_TTL_IN_SECONDS_FIELD);
        map.put("agentResourceRoleArn", AGENT_RESOURCE_ROLE_ARN_FIELD);
        map.put("customerEncryptionKeyArn", CUSTOMER_ENCRYPTION_KEY_ARN_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("updatedAt", UPDATED_AT_FIELD);
        map.put("failureReasons", FAILURE_REASONS_FIELD);
        map.put("recommendedActions", RECOMMENDED_ACTIONS_FIELD);
        map.put("promptOverrideConfiguration", PROMPT_OVERRIDE_CONFIGURATION_FIELD);
        map.put("guardrailConfiguration", GUARDRAIL_CONFIGURATION_FIELD);
        map.put("memoryConfiguration", MEMORY_CONFIGURATION_FIELD);
        map.put("agentCollaboration", AGENT_COLLABORATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AgentVersion, T> g) {
        return obj -> g.apply((AgentVersion) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AgentVersion> {
        /**
         * <p>
         * The unique identifier of the agent that the version belongs to.
         * </p>
         * 
         * @param agentId
         *        The unique identifier of the agent that the version belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentId(String agentId);

        /**
         * <p>
         * The name of the agent that the version belongs to.
         * </p>
         * 
         * @param agentName
         *        The name of the agent that the version belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentName(String agentName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the agent that the version belongs to.
         * </p>
         * 
         * @param agentArn
         *        The Amazon Resource Name (ARN) of the agent that the version belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArn(String agentArn);

        /**
         * <p>
         * The version number.
         * </p>
         * 
         * @param version
         *        The version number.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The instructions provided to the agent.
         * </p>
         * 
         * @param instruction
         *        The instructions provided to the agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instruction(String instruction);

        /**
         * <p>
         * The status of the agent that the version belongs to.
         * </p>
         * 
         * @param agentStatus
         *        The status of the agent that the version belongs to.
         * @see AgentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AgentStatus
         */
        Builder agentStatus(String agentStatus);

        /**
         * <p>
         * The status of the agent that the version belongs to.
         * </p>
         * 
         * @param agentStatus
         *        The status of the agent that the version belongs to.
         * @see AgentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AgentStatus
         */
        Builder agentStatus(AgentStatus agentStatus);

        /**
         * <p>
         * The foundation model that the version invokes.
         * </p>
         * 
         * @param foundationModel
         *        The foundation model that the version invokes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder foundationModel(String foundationModel);

        /**
         * <p>
         * The description of the version.
         * </p>
         * 
         * @param description
         *        The description of the version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The number of seconds for which Amazon Bedrock keeps information about a user's conversation with the agent.
         * </p>
         * <p>
         * A user interaction remains active for the amount of time specified. If no conversation occurs during this
         * time, the session expires and Amazon Bedrock deletes any data provided before the timeout.
         * </p>
         * 
         * @param idleSessionTTLInSeconds
         *        The number of seconds for which Amazon Bedrock keeps information about a user's conversation with the
         *        agent.</p>
         *        <p>
         *        A user interaction remains active for the amount of time specified. If no conversation occurs during
         *        this time, the session expires and Amazon Bedrock deletes any data provided before the timeout.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder idleSessionTTLInSeconds(Integer idleSessionTTLInSeconds);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IAM role with permissions to invoke API operations on the agent.
         * </p>
         * 
         * @param agentResourceRoleArn
         *        The Amazon Resource Name (ARN) of the IAM role with permissions to invoke API operations on the agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentResourceRoleArn(String agentResourceRoleArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the KMS key that encrypts the agent.
         * </p>
         * 
         * @param customerEncryptionKeyArn
         *        The Amazon Resource Name (ARN) of the KMS key that encrypts the agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerEncryptionKeyArn(String customerEncryptionKeyArn);

        /**
         * <p>
         * The time at which the version was created.
         * </p>
         * 
         * @param createdAt
         *        The time at which the version was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The time at which the version was last updated.
         * </p>
         * 
         * @param updatedAt
         *        The time at which the version was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);

        /**
         * <p>
         * A list of reasons that the API operation on the version failed.
         * </p>
         * 
         * @param failureReasons
         *        A list of reasons that the API operation on the version failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReasons(Collection<String> failureReasons);

        /**
         * <p>
         * A list of reasons that the API operation on the version failed.
         * </p>
         * 
         * @param failureReasons
         *        A list of reasons that the API operation on the version failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReasons(String... failureReasons);

        /**
         * <p>
         * A list of recommended actions to take for the failed API operation on the version to succeed.
         * </p>
         * 
         * @param recommendedActions
         *        A list of recommended actions to take for the failed API operation on the version to succeed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendedActions(Collection<String> recommendedActions);

        /**
         * <p>
         * A list of recommended actions to take for the failed API operation on the version to succeed.
         * </p>
         * 
         * @param recommendedActions
         *        A list of recommended actions to take for the failed API operation on the version to succeed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendedActions(String... recommendedActions);

        /**
         * <p>
         * Contains configurations to override prompt templates in different parts of an agent sequence. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html">Advanced prompts</a>.
         * </p>
         * 
         * @param promptOverrideConfiguration
         *        Contains configurations to override prompt templates in different parts of an agent sequence. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html">Advanced
         *        prompts</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder promptOverrideConfiguration(PromptOverrideConfiguration promptOverrideConfiguration);

        /**
         * <p>
         * Contains configurations to override prompt templates in different parts of an agent sequence. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html">Advanced prompts</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link PromptOverrideConfiguration.Builder}
         * avoiding the need to create one manually via {@link PromptOverrideConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PromptOverrideConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #promptOverrideConfiguration(PromptOverrideConfiguration)}.
         * 
         * @param promptOverrideConfiguration
         *        a consumer that will call methods on {@link PromptOverrideConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #promptOverrideConfiguration(PromptOverrideConfiguration)
         */
        default Builder promptOverrideConfiguration(Consumer<PromptOverrideConfiguration.Builder> promptOverrideConfiguration) {
            return promptOverrideConfiguration(PromptOverrideConfiguration.builder().applyMutation(promptOverrideConfiguration)
                    .build());
        }

        /**
         * <p>
         * Details about the guardrail associated with the agent.
         * </p>
         * 
         * @param guardrailConfiguration
         *        Details about the guardrail associated with the agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder guardrailConfiguration(GuardrailConfiguration guardrailConfiguration);

        /**
         * <p>
         * Details about the guardrail associated with the agent.
         * </p>
         * This is a convenience method that creates an instance of the {@link GuardrailConfiguration.Builder} avoiding
         * the need to create one manually via {@link GuardrailConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link GuardrailConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #guardrailConfiguration(GuardrailConfiguration)}.
         * 
         * @param guardrailConfiguration
         *        a consumer that will call methods on {@link GuardrailConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #guardrailConfiguration(GuardrailConfiguration)
         */
        default Builder guardrailConfiguration(Consumer<GuardrailConfiguration.Builder> guardrailConfiguration) {
            return guardrailConfiguration(GuardrailConfiguration.builder().applyMutation(guardrailConfiguration).build());
        }

        /**
         * <p>
         * Contains details of the memory configuration on the version of the agent.
         * </p>
         * 
         * @param memoryConfiguration
         *        Contains details of the memory configuration on the version of the agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memoryConfiguration(MemoryConfiguration memoryConfiguration);

        /**
         * <p>
         * Contains details of the memory configuration on the version of the agent.
         * </p>
         * This is a convenience method that creates an instance of the {@link MemoryConfiguration.Builder} avoiding the
         * need to create one manually via {@link MemoryConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MemoryConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #memoryConfiguration(MemoryConfiguration)}.
         * 
         * @param memoryConfiguration
         *        a consumer that will call methods on {@link MemoryConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #memoryConfiguration(MemoryConfiguration)
         */
        default Builder memoryConfiguration(Consumer<MemoryConfiguration.Builder> memoryConfiguration) {
            return memoryConfiguration(MemoryConfiguration.builder().applyMutation(memoryConfiguration).build());
        }

        /**
         * <p>
         * The agent's collaboration settings.
         * </p>
         * 
         * @param agentCollaboration
         *        The agent's collaboration settings.
         * @see AgentCollaboration
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AgentCollaboration
         */
        Builder agentCollaboration(String agentCollaboration);

        /**
         * <p>
         * The agent's collaboration settings.
         * </p>
         * 
         * @param agentCollaboration
         *        The agent's collaboration settings.
         * @see AgentCollaboration
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AgentCollaboration
         */
        Builder agentCollaboration(AgentCollaboration agentCollaboration);
    }

    static final class BuilderImpl implements Builder {
        private String agentId;

        private String agentName;

        private String agentArn;

        private String version;

        private String instruction;

        private String agentStatus;

        private String foundationModel;

        private String description;

        private Integer idleSessionTTLInSeconds;

        private String agentResourceRoleArn;

        private String customerEncryptionKeyArn;

        private Instant createdAt;

        private Instant updatedAt;

        private List<String> failureReasons = DefaultSdkAutoConstructList.getInstance();

        private List<String> recommendedActions = DefaultSdkAutoConstructList.getInstance();

        private PromptOverrideConfiguration promptOverrideConfiguration;

        private GuardrailConfiguration guardrailConfiguration;

        private MemoryConfiguration memoryConfiguration;

        private String agentCollaboration;

        private BuilderImpl() {
        }

        private BuilderImpl(AgentVersion model) {
            agentId(model.agentId);
            agentName(model.agentName);
            agentArn(model.agentArn);
            version(model.version);
            instruction(model.instruction);
            agentStatus(model.agentStatus);
            foundationModel(model.foundationModel);
            description(model.description);
            idleSessionTTLInSeconds(model.idleSessionTTLInSeconds);
            agentResourceRoleArn(model.agentResourceRoleArn);
            customerEncryptionKeyArn(model.customerEncryptionKeyArn);
            createdAt(model.createdAt);
            updatedAt(model.updatedAt);
            failureReasons(model.failureReasons);
            recommendedActions(model.recommendedActions);
            promptOverrideConfiguration(model.promptOverrideConfiguration);
            guardrailConfiguration(model.guardrailConfiguration);
            memoryConfiguration(model.memoryConfiguration);
            agentCollaboration(model.agentCollaboration);
        }

        public final String getAgentId() {
            return agentId;
        }

        public final void setAgentId(String agentId) {
            this.agentId = agentId;
        }

        @Override
        public final Builder agentId(String agentId) {
            this.agentId = agentId;
            return this;
        }

        public final String getAgentName() {
            return agentName;
        }

        public final void setAgentName(String agentName) {
            this.agentName = agentName;
        }

        @Override
        public final Builder agentName(String agentName) {
            this.agentName = agentName;
            return this;
        }

        public final String getAgentArn() {
            return agentArn;
        }

        public final void setAgentArn(String agentArn) {
            this.agentArn = agentArn;
        }

        @Override
        public final Builder agentArn(String agentArn) {
            this.agentArn = agentArn;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final String getInstruction() {
            return instruction;
        }

        public final void setInstruction(String instruction) {
            this.instruction = instruction;
        }

        @Override
        public final Builder instruction(String instruction) {
            this.instruction = instruction;
            return this;
        }

        public final String getAgentStatus() {
            return agentStatus;
        }

        public final void setAgentStatus(String agentStatus) {
            this.agentStatus = agentStatus;
        }

        @Override
        public final Builder agentStatus(String agentStatus) {
            this.agentStatus = agentStatus;
            return this;
        }

        @Override
        public final Builder agentStatus(AgentStatus agentStatus) {
            this.agentStatus(agentStatus == null ? null : agentStatus.toString());
            return this;
        }

        public final String getFoundationModel() {
            return foundationModel;
        }

        public final void setFoundationModel(String foundationModel) {
            this.foundationModel = foundationModel;
        }

        @Override
        public final Builder foundationModel(String foundationModel) {
            this.foundationModel = foundationModel;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Integer getIdleSessionTTLInSeconds() {
            return idleSessionTTLInSeconds;
        }

        public final void setIdleSessionTTLInSeconds(Integer idleSessionTTLInSeconds) {
            this.idleSessionTTLInSeconds = idleSessionTTLInSeconds;
        }

        @Override
        public final Builder idleSessionTTLInSeconds(Integer idleSessionTTLInSeconds) {
            this.idleSessionTTLInSeconds = idleSessionTTLInSeconds;
            return this;
        }

        public final String getAgentResourceRoleArn() {
            return agentResourceRoleArn;
        }

        public final void setAgentResourceRoleArn(String agentResourceRoleArn) {
            this.agentResourceRoleArn = agentResourceRoleArn;
        }

        @Override
        public final Builder agentResourceRoleArn(String agentResourceRoleArn) {
            this.agentResourceRoleArn = agentResourceRoleArn;
            return this;
        }

        public final String getCustomerEncryptionKeyArn() {
            return customerEncryptionKeyArn;
        }

        public final void setCustomerEncryptionKeyArn(String customerEncryptionKeyArn) {
            this.customerEncryptionKeyArn = customerEncryptionKeyArn;
        }

        @Override
        public final Builder customerEncryptionKeyArn(String customerEncryptionKeyArn) {
            this.customerEncryptionKeyArn = customerEncryptionKeyArn;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public final Collection<String> getFailureReasons() {
            if (failureReasons instanceof SdkAutoConstructList) {
                return null;
            }
            return failureReasons;
        }

        public final void setFailureReasons(Collection<String> failureReasons) {
            this.failureReasons = FailureReasonsCopier.copy(failureReasons);
        }

        @Override
        public final Builder failureReasons(Collection<String> failureReasons) {
            this.failureReasons = FailureReasonsCopier.copy(failureReasons);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder failureReasons(String... failureReasons) {
            failureReasons(Arrays.asList(failureReasons));
            return this;
        }

        public final Collection<String> getRecommendedActions() {
            if (recommendedActions instanceof SdkAutoConstructList) {
                return null;
            }
            return recommendedActions;
        }

        public final void setRecommendedActions(Collection<String> recommendedActions) {
            this.recommendedActions = RecommendedActionsCopier.copy(recommendedActions);
        }

        @Override
        public final Builder recommendedActions(Collection<String> recommendedActions) {
            this.recommendedActions = RecommendedActionsCopier.copy(recommendedActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendedActions(String... recommendedActions) {
            recommendedActions(Arrays.asList(recommendedActions));
            return this;
        }

        public final PromptOverrideConfiguration.Builder getPromptOverrideConfiguration() {
            return promptOverrideConfiguration != null ? promptOverrideConfiguration.toBuilder() : null;
        }

        public final void setPromptOverrideConfiguration(PromptOverrideConfiguration.BuilderImpl promptOverrideConfiguration) {
            this.promptOverrideConfiguration = promptOverrideConfiguration != null ? promptOverrideConfiguration.build() : null;
        }

        @Override
        public final Builder promptOverrideConfiguration(PromptOverrideConfiguration promptOverrideConfiguration) {
            this.promptOverrideConfiguration = promptOverrideConfiguration;
            return this;
        }

        public final GuardrailConfiguration.Builder getGuardrailConfiguration() {
            return guardrailConfiguration != null ? guardrailConfiguration.toBuilder() : null;
        }

        public final void setGuardrailConfiguration(GuardrailConfiguration.BuilderImpl guardrailConfiguration) {
            this.guardrailConfiguration = guardrailConfiguration != null ? guardrailConfiguration.build() : null;
        }

        @Override
        public final Builder guardrailConfiguration(GuardrailConfiguration guardrailConfiguration) {
            this.guardrailConfiguration = guardrailConfiguration;
            return this;
        }

        public final MemoryConfiguration.Builder getMemoryConfiguration() {
            return memoryConfiguration != null ? memoryConfiguration.toBuilder() : null;
        }

        public final void setMemoryConfiguration(MemoryConfiguration.BuilderImpl memoryConfiguration) {
            this.memoryConfiguration = memoryConfiguration != null ? memoryConfiguration.build() : null;
        }

        @Override
        public final Builder memoryConfiguration(MemoryConfiguration memoryConfiguration) {
            this.memoryConfiguration = memoryConfiguration;
            return this;
        }

        public final String getAgentCollaboration() {
            return agentCollaboration;
        }

        public final void setAgentCollaboration(String agentCollaboration) {
            this.agentCollaboration = agentCollaboration;
        }

        @Override
        public final Builder agentCollaboration(String agentCollaboration) {
            this.agentCollaboration = agentCollaboration;
            return this;
        }

        @Override
        public final Builder agentCollaboration(AgentCollaboration agentCollaboration) {
            this.agentCollaboration(agentCollaboration == null ? null : agentCollaboration.toString());
            return this;
        }

        @Override
        public AgentVersion build() {
            return new AgentVersion(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
