/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagent.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details about an action group.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AgentActionGroup implements SdkPojo, Serializable,
        ToCopyableBuilder<AgentActionGroup.Builder, AgentActionGroup> {
    private static final SdkField<String> AGENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("agentId").getter(getter(AgentActionGroup::agentId)).setter(setter(Builder::agentId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("agentId").build()).build();

    private static final SdkField<String> AGENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("agentVersion").getter(getter(AgentActionGroup::agentVersion)).setter(setter(Builder::agentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("agentVersion").build()).build();

    private static final SdkField<String> ACTION_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("actionGroupId").getter(getter(AgentActionGroup::actionGroupId)).setter(setter(Builder::actionGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("actionGroupId").build()).build();

    private static final SdkField<String> ACTION_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("actionGroupName").getter(getter(AgentActionGroup::actionGroupName))
            .setter(setter(Builder::actionGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("actionGroupName").build()).build();

    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("clientToken").getter(getter(AgentActionGroup::clientToken)).setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clientToken").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(AgentActionGroup::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(AgentActionGroup::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt")
            .getter(getter(AgentActionGroup::updatedAt))
            .setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> PARENT_ACTION_SIGNATURE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("parentActionSignature").getter(getter(AgentActionGroup::parentActionSignatureAsString))
            .setter(setter(Builder::parentActionSignature))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("parentActionSignature").build())
            .build();

    private static final SdkField<Map<String, String>> PARENT_ACTION_GROUP_SIGNATURE_PARAMS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("parentActionGroupSignatureParams")
            .getter(getter(AgentActionGroup::parentActionGroupSignatureParams))
            .setter(setter(Builder::parentActionGroupSignatureParams))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("parentActionGroupSignatureParams")
                    .build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<ActionGroupExecutor> ACTION_GROUP_EXECUTOR_FIELD = SdkField
            .<ActionGroupExecutor> builder(MarshallingType.SDK_POJO).memberName("actionGroupExecutor")
            .getter(getter(AgentActionGroup::actionGroupExecutor)).setter(setter(Builder::actionGroupExecutor))
            .constructor(ActionGroupExecutor::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("actionGroupExecutor").build())
            .build();

    private static final SdkField<APISchema> API_SCHEMA_FIELD = SdkField.<APISchema> builder(MarshallingType.SDK_POJO)
            .memberName("apiSchema").getter(getter(AgentActionGroup::apiSchema)).setter(setter(Builder::apiSchema))
            .constructor(APISchema::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("apiSchema").build()).build();

    private static final SdkField<FunctionSchema> FUNCTION_SCHEMA_FIELD = SdkField
            .<FunctionSchema> builder(MarshallingType.SDK_POJO).memberName("functionSchema")
            .getter(getter(AgentActionGroup::functionSchema)).setter(setter(Builder::functionSchema))
            .constructor(FunctionSchema::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("functionSchema").build()).build();

    private static final SdkField<String> ACTION_GROUP_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("actionGroupState").getter(getter(AgentActionGroup::actionGroupStateAsString))
            .setter(setter(Builder::actionGroupState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("actionGroupState").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AGENT_ID_FIELD,
            AGENT_VERSION_FIELD, ACTION_GROUP_ID_FIELD, ACTION_GROUP_NAME_FIELD, CLIENT_TOKEN_FIELD, DESCRIPTION_FIELD,
            CREATED_AT_FIELD, UPDATED_AT_FIELD, PARENT_ACTION_SIGNATURE_FIELD, PARENT_ACTION_GROUP_SIGNATURE_PARAMS_FIELD,
            ACTION_GROUP_EXECUTOR_FIELD, API_SCHEMA_FIELD, FUNCTION_SCHEMA_FIELD, ACTION_GROUP_STATE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String agentId;

    private final String agentVersion;

    private final String actionGroupId;

    private final String actionGroupName;

    private final String clientToken;

    private final String description;

    private final Instant createdAt;

    private final Instant updatedAt;

    private final String parentActionSignature;

    private final Map<String, String> parentActionGroupSignatureParams;

    private final ActionGroupExecutor actionGroupExecutor;

    private final APISchema apiSchema;

    private final FunctionSchema functionSchema;

    private final String actionGroupState;

    private AgentActionGroup(BuilderImpl builder) {
        this.agentId = builder.agentId;
        this.agentVersion = builder.agentVersion;
        this.actionGroupId = builder.actionGroupId;
        this.actionGroupName = builder.actionGroupName;
        this.clientToken = builder.clientToken;
        this.description = builder.description;
        this.createdAt = builder.createdAt;
        this.updatedAt = builder.updatedAt;
        this.parentActionSignature = builder.parentActionSignature;
        this.parentActionGroupSignatureParams = builder.parentActionGroupSignatureParams;
        this.actionGroupExecutor = builder.actionGroupExecutor;
        this.apiSchema = builder.apiSchema;
        this.functionSchema = builder.functionSchema;
        this.actionGroupState = builder.actionGroupState;
    }

    /**
     * <p>
     * The unique identifier of the agent to which the action group belongs.
     * </p>
     * 
     * @return The unique identifier of the agent to which the action group belongs.
     */
    public final String agentId() {
        return agentId;
    }

    /**
     * <p>
     * The version of the agent to which the action group belongs.
     * </p>
     * 
     * @return The version of the agent to which the action group belongs.
     */
    public final String agentVersion() {
        return agentVersion;
    }

    /**
     * <p>
     * The unique identifier of the action group.
     * </p>
     * 
     * @return The unique identifier of the action group.
     */
    public final String actionGroupId() {
        return actionGroupId;
    }

    /**
     * <p>
     * The name of the action group.
     * </p>
     * 
     * @return The name of the action group.
     */
    public final String actionGroupName() {
        return actionGroupName;
    }

    /**
     * <p>
     * A unique, case-sensitive identifier to ensure that the API request completes no more than one time. If this token
     * matches a previous request, Amazon Bedrock ignores the request, but does not return an error. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
     * idempotency</a>.
     * </p>
     * 
     * @return A unique, case-sensitive identifier to ensure that the API request completes no more than one time. If
     *         this token matches a previous request, Amazon Bedrock ignores the request, but does not return an error.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
     *         idempotency</a>.
     */
    public final String clientToken() {
        return clientToken;
    }

    /**
     * <p>
     * The description of the action group.
     * </p>
     * 
     * @return The description of the action group.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The time at which the action group was created.
     * </p>
     * 
     * @return The time at which the action group was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The time at which the action group was last updated.
     * </p>
     * 
     * @return The time at which the action group was last updated.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    /**
     * <p>
     * If this field is set as <code>AMAZON.UserInput</code>, the agent can request the user for additional information
     * when trying to complete a task. The <code>description</code>, <code>apiSchema</code>, and
     * <code>actionGroupExecutor</code> fields must be blank for this action group.
     * </p>
     * <p>
     * During orchestration, if the agent determines that it needs to invoke an API in an action group, but doesn't have
     * enough information to complete the API request, it will invoke this action group instead and return an <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_Observation.html">Observation</a>
     * reprompting the user for more information.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #parentActionSignature} will return {@link ActionGroupSignature#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #parentActionSignatureAsString}.
     * </p>
     * 
     * @return If this field is set as <code>AMAZON.UserInput</code>, the agent can request the user for additional
     *         information when trying to complete a task. The <code>description</code>, <code>apiSchema</code>, and
     *         <code>actionGroupExecutor</code> fields must be blank for this action group.</p>
     *         <p>
     *         During orchestration, if the agent determines that it needs to invoke an API in an action group, but
     *         doesn't have enough information to complete the API request, it will invoke this action group instead and
     *         return an <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_Observation.html"
     *         >Observation</a> reprompting the user for more information.
     * @see ActionGroupSignature
     */
    public final ActionGroupSignature parentActionSignature() {
        return ActionGroupSignature.fromValue(parentActionSignature);
    }

    /**
     * <p>
     * If this field is set as <code>AMAZON.UserInput</code>, the agent can request the user for additional information
     * when trying to complete a task. The <code>description</code>, <code>apiSchema</code>, and
     * <code>actionGroupExecutor</code> fields must be blank for this action group.
     * </p>
     * <p>
     * During orchestration, if the agent determines that it needs to invoke an API in an action group, but doesn't have
     * enough information to complete the API request, it will invoke this action group instead and return an <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_Observation.html">Observation</a>
     * reprompting the user for more information.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #parentActionSignature} will return {@link ActionGroupSignature#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #parentActionSignatureAsString}.
     * </p>
     * 
     * @return If this field is set as <code>AMAZON.UserInput</code>, the agent can request the user for additional
     *         information when trying to complete a task. The <code>description</code>, <code>apiSchema</code>, and
     *         <code>actionGroupExecutor</code> fields must be blank for this action group.</p>
     *         <p>
     *         During orchestration, if the agent determines that it needs to invoke an API in an action group, but
     *         doesn't have enough information to complete the API request, it will invoke this action group instead and
     *         return an <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_Observation.html"
     *         >Observation</a> reprompting the user for more information.
     * @see ActionGroupSignature
     */
    public final String parentActionSignatureAsString() {
        return parentActionSignature;
    }

    /**
     * For responses, this returns true if the service returned a value for the ParentActionGroupSignatureParams
     * property. This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()}
     * method on the property). This is useful because the SDK will never return a null collection or map, but you may
     * need to differentiate between the service returning nothing (or null) and the service returning an empty
     * collection or map. For requests, this returns true if a value for the property was specified in the request
     * builder, and false if a value was not specified.
     */
    public final boolean hasParentActionGroupSignatureParams() {
        return parentActionGroupSignatureParams != null && !(parentActionGroupSignatureParams instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The configuration settings for a computer use action.
     * </p>
     * <important>
     * <p>
     * Computer use is a new Anthropic Claude model capability (in beta) available with Claude 3.7 Sonnet and Claude 3.5
     * Sonnet v2 only. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/agents-computer-use.html">Configure an Amazon Bedrock
     * Agent to complete tasks with computer use tools</a>.
     * </p>
     * </important>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParentActionGroupSignatureParams}
     * method.
     * </p>
     * 
     * @return The configuration settings for a computer use action.</p> <important>
     *         <p>
     *         Computer use is a new Anthropic Claude model capability (in beta) available with Claude 3.7 Sonnet and
     *         Claude 3.5 Sonnet v2 only. For more information, see <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/agents-computer-use.html">Configure an Amazon
     *         Bedrock Agent to complete tasks with computer use tools</a>.
     *         </p>
     */
    public final Map<String, String> parentActionGroupSignatureParams() {
        return parentActionGroupSignatureParams;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Lambda function containing the business logic that is carried out upon
     * invoking the action or the custom control method for handling the information elicited from the user.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Lambda function containing the business logic that is carried out
     *         upon invoking the action or the custom control method for handling the information elicited from the
     *         user.
     */
    public final ActionGroupExecutor actionGroupExecutor() {
        return actionGroupExecutor;
    }

    /**
     * <p>
     * Contains either details about the S3 object containing the OpenAPI schema for the action group or the JSON or
     * YAML-formatted payload defining the schema. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/agents-api-schema.html">Action group OpenAPI
     * schemas</a>.
     * </p>
     * 
     * @return Contains either details about the S3 object containing the OpenAPI schema for the action group or the
     *         JSON or YAML-formatted payload defining the schema. For more information, see <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/agents-api-schema.html">Action group OpenAPI
     *         schemas</a>.
     */
    public final APISchema apiSchema() {
        return apiSchema;
    }

    /**
     * <p>
     * Defines functions that each define parameters that the agent needs to invoke from the user. Each function
     * represents an action in an action group.
     * </p>
     * 
     * @return Defines functions that each define parameters that the agent needs to invoke from the user. Each function
     *         represents an action in an action group.
     */
    public final FunctionSchema functionSchema() {
        return functionSchema;
    }

    /**
     * <p>
     * Specifies whether the action group is available for the agent to invoke or not when sending an <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_InvokeAgent.html">InvokeAgent</a>
     * request.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #actionGroupState}
     * will return {@link ActionGroupState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #actionGroupStateAsString}.
     * </p>
     * 
     * @return Specifies whether the action group is available for the agent to invoke or not when sending an <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_InvokeAgent.html"
     *         >InvokeAgent</a> request.
     * @see ActionGroupState
     */
    public final ActionGroupState actionGroupState() {
        return ActionGroupState.fromValue(actionGroupState);
    }

    /**
     * <p>
     * Specifies whether the action group is available for the agent to invoke or not when sending an <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_InvokeAgent.html">InvokeAgent</a>
     * request.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #actionGroupState}
     * will return {@link ActionGroupState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #actionGroupStateAsString}.
     * </p>
     * 
     * @return Specifies whether the action group is available for the agent to invoke or not when sending an <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_InvokeAgent.html"
     *         >InvokeAgent</a> request.
     * @see ActionGroupState
     */
    public final String actionGroupStateAsString() {
        return actionGroupState;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(agentId());
        hashCode = 31 * hashCode + Objects.hashCode(agentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(actionGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(actionGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(parentActionSignatureAsString());
        hashCode = 31 * hashCode
                + Objects.hashCode(hasParentActionGroupSignatureParams() ? parentActionGroupSignatureParams() : null);
        hashCode = 31 * hashCode + Objects.hashCode(actionGroupExecutor());
        hashCode = 31 * hashCode + Objects.hashCode(apiSchema());
        hashCode = 31 * hashCode + Objects.hashCode(functionSchema());
        hashCode = 31 * hashCode + Objects.hashCode(actionGroupStateAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AgentActionGroup)) {
            return false;
        }
        AgentActionGroup other = (AgentActionGroup) obj;
        return Objects.equals(agentId(), other.agentId()) && Objects.equals(agentVersion(), other.agentVersion())
                && Objects.equals(actionGroupId(), other.actionGroupId())
                && Objects.equals(actionGroupName(), other.actionGroupName())
                && Objects.equals(clientToken(), other.clientToken()) && Objects.equals(description(), other.description())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(updatedAt(), other.updatedAt())
                && Objects.equals(parentActionSignatureAsString(), other.parentActionSignatureAsString())
                && hasParentActionGroupSignatureParams() == other.hasParentActionGroupSignatureParams()
                && Objects.equals(parentActionGroupSignatureParams(), other.parentActionGroupSignatureParams())
                && Objects.equals(actionGroupExecutor(), other.actionGroupExecutor())
                && Objects.equals(apiSchema(), other.apiSchema()) && Objects.equals(functionSchema(), other.functionSchema())
                && Objects.equals(actionGroupStateAsString(), other.actionGroupStateAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("AgentActionGroup")
                .add("AgentId", agentId())
                .add("AgentVersion", agentVersion())
                .add("ActionGroupId", actionGroupId())
                .add("ActionGroupName", actionGroupName())
                .add("ClientToken", clientToken())
                .add("Description", description())
                .add("CreatedAt", createdAt())
                .add("UpdatedAt", updatedAt())
                .add("ParentActionSignature", parentActionSignatureAsString())
                .add("ParentActionGroupSignatureParams",
                        hasParentActionGroupSignatureParams() ? parentActionGroupSignatureParams() : null)
                .add("ActionGroupExecutor", actionGroupExecutor()).add("ApiSchema", apiSchema())
                .add("FunctionSchema", functionSchema()).add("ActionGroupState", actionGroupStateAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "agentId":
            return Optional.ofNullable(clazz.cast(agentId()));
        case "agentVersion":
            return Optional.ofNullable(clazz.cast(agentVersion()));
        case "actionGroupId":
            return Optional.ofNullable(clazz.cast(actionGroupId()));
        case "actionGroupName":
            return Optional.ofNullable(clazz.cast(actionGroupName()));
        case "clientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        case "parentActionSignature":
            return Optional.ofNullable(clazz.cast(parentActionSignatureAsString()));
        case "parentActionGroupSignatureParams":
            return Optional.ofNullable(clazz.cast(parentActionGroupSignatureParams()));
        case "actionGroupExecutor":
            return Optional.ofNullable(clazz.cast(actionGroupExecutor()));
        case "apiSchema":
            return Optional.ofNullable(clazz.cast(apiSchema()));
        case "functionSchema":
            return Optional.ofNullable(clazz.cast(functionSchema()));
        case "actionGroupState":
            return Optional.ofNullable(clazz.cast(actionGroupStateAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("agentId", AGENT_ID_FIELD);
        map.put("agentVersion", AGENT_VERSION_FIELD);
        map.put("actionGroupId", ACTION_GROUP_ID_FIELD);
        map.put("actionGroupName", ACTION_GROUP_NAME_FIELD);
        map.put("clientToken", CLIENT_TOKEN_FIELD);
        map.put("description", DESCRIPTION_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("updatedAt", UPDATED_AT_FIELD);
        map.put("parentActionSignature", PARENT_ACTION_SIGNATURE_FIELD);
        map.put("parentActionGroupSignatureParams", PARENT_ACTION_GROUP_SIGNATURE_PARAMS_FIELD);
        map.put("actionGroupExecutor", ACTION_GROUP_EXECUTOR_FIELD);
        map.put("apiSchema", API_SCHEMA_FIELD);
        map.put("functionSchema", FUNCTION_SCHEMA_FIELD);
        map.put("actionGroupState", ACTION_GROUP_STATE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AgentActionGroup, T> g) {
        return obj -> g.apply((AgentActionGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AgentActionGroup> {
        /**
         * <p>
         * The unique identifier of the agent to which the action group belongs.
         * </p>
         * 
         * @param agentId
         *        The unique identifier of the agent to which the action group belongs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentId(String agentId);

        /**
         * <p>
         * The version of the agent to which the action group belongs.
         * </p>
         * 
         * @param agentVersion
         *        The version of the agent to which the action group belongs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentVersion(String agentVersion);

        /**
         * <p>
         * The unique identifier of the action group.
         * </p>
         * 
         * @param actionGroupId
         *        The unique identifier of the action group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionGroupId(String actionGroupId);

        /**
         * <p>
         * The name of the action group.
         * </p>
         * 
         * @param actionGroupName
         *        The name of the action group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionGroupName(String actionGroupName);

        /**
         * <p>
         * A unique, case-sensitive identifier to ensure that the API request completes no more than one time. If this
         * token matches a previous request, Amazon Bedrock ignores the request, but does not return an error. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
         * idempotency</a>.
         * </p>
         * 
         * @param clientToken
         *        A unique, case-sensitive identifier to ensure that the API request completes no more than one time. If
         *        this token matches a previous request, Amazon Bedrock ignores the request, but does not return an
         *        error. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
         *        idempotency</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        /**
         * <p>
         * The description of the action group.
         * </p>
         * 
         * @param description
         *        The description of the action group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The time at which the action group was created.
         * </p>
         * 
         * @param createdAt
         *        The time at which the action group was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The time at which the action group was last updated.
         * </p>
         * 
         * @param updatedAt
         *        The time at which the action group was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);

        /**
         * <p>
         * If this field is set as <code>AMAZON.UserInput</code>, the agent can request the user for additional
         * information when trying to complete a task. The <code>description</code>, <code>apiSchema</code>, and
         * <code>actionGroupExecutor</code> fields must be blank for this action group.
         * </p>
         * <p>
         * During orchestration, if the agent determines that it needs to invoke an API in an action group, but doesn't
         * have enough information to complete the API request, it will invoke this action group instead and return an
         * <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_Observation.html">
         * Observation</a> reprompting the user for more information.
         * </p>
         * 
         * @param parentActionSignature
         *        If this field is set as <code>AMAZON.UserInput</code>, the agent can request the user for additional
         *        information when trying to complete a task. The <code>description</code>, <code>apiSchema</code>, and
         *        <code>actionGroupExecutor</code> fields must be blank for this action group.</p>
         *        <p>
         *        During orchestration, if the agent determines that it needs to invoke an API in an action group, but
         *        doesn't have enough information to complete the API request, it will invoke this action group instead
         *        and return an <a
         *        href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_Observation.html"
         *        >Observation</a> reprompting the user for more information.
         * @see ActionGroupSignature
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ActionGroupSignature
         */
        Builder parentActionSignature(String parentActionSignature);

        /**
         * <p>
         * If this field is set as <code>AMAZON.UserInput</code>, the agent can request the user for additional
         * information when trying to complete a task. The <code>description</code>, <code>apiSchema</code>, and
         * <code>actionGroupExecutor</code> fields must be blank for this action group.
         * </p>
         * <p>
         * During orchestration, if the agent determines that it needs to invoke an API in an action group, but doesn't
         * have enough information to complete the API request, it will invoke this action group instead and return an
         * <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_Observation.html">
         * Observation</a> reprompting the user for more information.
         * </p>
         * 
         * @param parentActionSignature
         *        If this field is set as <code>AMAZON.UserInput</code>, the agent can request the user for additional
         *        information when trying to complete a task. The <code>description</code>, <code>apiSchema</code>, and
         *        <code>actionGroupExecutor</code> fields must be blank for this action group.</p>
         *        <p>
         *        During orchestration, if the agent determines that it needs to invoke an API in an action group, but
         *        doesn't have enough information to complete the API request, it will invoke this action group instead
         *        and return an <a
         *        href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_Observation.html"
         *        >Observation</a> reprompting the user for more information.
         * @see ActionGroupSignature
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ActionGroupSignature
         */
        Builder parentActionSignature(ActionGroupSignature parentActionSignature);

        /**
         * <p>
         * The configuration settings for a computer use action.
         * </p>
         * <important>
         * <p>
         * Computer use is a new Anthropic Claude model capability (in beta) available with Claude 3.7 Sonnet and Claude
         * 3.5 Sonnet v2 only. For more information, see <a
         * href="https://docs.aws.amazon.com/bedrock/latest/userguide/agents-computer-use.html">Configure an Amazon
         * Bedrock Agent to complete tasks with computer use tools</a>.
         * </p>
         * </important>
         * 
         * @param parentActionGroupSignatureParams
         *        The configuration settings for a computer use action.</p> <important>
         *        <p>
         *        Computer use is a new Anthropic Claude model capability (in beta) available with Claude 3.7 Sonnet and
         *        Claude 3.5 Sonnet v2 only. For more information, see <a
         *        href="https://docs.aws.amazon.com/bedrock/latest/userguide/agents-computer-use.html">Configure an
         *        Amazon Bedrock Agent to complete tasks with computer use tools</a>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parentActionGroupSignatureParams(Map<String, String> parentActionGroupSignatureParams);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Lambda function containing the business logic that is carried out upon
         * invoking the action or the custom control method for handling the information elicited from the user.
         * </p>
         * 
         * @param actionGroupExecutor
         *        The Amazon Resource Name (ARN) of the Lambda function containing the business logic that is carried
         *        out upon invoking the action or the custom control method for handling the information elicited from
         *        the user.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionGroupExecutor(ActionGroupExecutor actionGroupExecutor);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Lambda function containing the business logic that is carried out upon
         * invoking the action or the custom control method for handling the information elicited from the user.
         * </p>
         * This is a convenience method that creates an instance of the {@link ActionGroupExecutor.Builder} avoiding the
         * need to create one manually via {@link ActionGroupExecutor#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ActionGroupExecutor.Builder#build()} is called immediately and
         * its result is passed to {@link #actionGroupExecutor(ActionGroupExecutor)}.
         * 
         * @param actionGroupExecutor
         *        a consumer that will call methods on {@link ActionGroupExecutor.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #actionGroupExecutor(ActionGroupExecutor)
         */
        default Builder actionGroupExecutor(Consumer<ActionGroupExecutor.Builder> actionGroupExecutor) {
            return actionGroupExecutor(ActionGroupExecutor.builder().applyMutation(actionGroupExecutor).build());
        }

        /**
         * <p>
         * Contains either details about the S3 object containing the OpenAPI schema for the action group or the JSON or
         * YAML-formatted payload defining the schema. For more information, see <a
         * href="https://docs.aws.amazon.com/bedrock/latest/userguide/agents-api-schema.html">Action group OpenAPI
         * schemas</a>.
         * </p>
         * 
         * @param apiSchema
         *        Contains either details about the S3 object containing the OpenAPI schema for the action group or the
         *        JSON or YAML-formatted payload defining the schema. For more information, see <a
         *        href="https://docs.aws.amazon.com/bedrock/latest/userguide/agents-api-schema.html">Action group
         *        OpenAPI schemas</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder apiSchema(APISchema apiSchema);

        /**
         * <p>
         * Contains either details about the S3 object containing the OpenAPI schema for the action group or the JSON or
         * YAML-formatted payload defining the schema. For more information, see <a
         * href="https://docs.aws.amazon.com/bedrock/latest/userguide/agents-api-schema.html">Action group OpenAPI
         * schemas</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link APISchema.Builder} avoiding the need to
         * create one manually via {@link APISchema#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link APISchema.Builder#build()} is called immediately and its result
         * is passed to {@link #apiSchema(APISchema)}.
         * 
         * @param apiSchema
         *        a consumer that will call methods on {@link APISchema.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #apiSchema(APISchema)
         */
        default Builder apiSchema(Consumer<APISchema.Builder> apiSchema) {
            return apiSchema(APISchema.builder().applyMutation(apiSchema).build());
        }

        /**
         * <p>
         * Defines functions that each define parameters that the agent needs to invoke from the user. Each function
         * represents an action in an action group.
         * </p>
         * 
         * @param functionSchema
         *        Defines functions that each define parameters that the agent needs to invoke from the user. Each
         *        function represents an action in an action group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionSchema(FunctionSchema functionSchema);

        /**
         * <p>
         * Defines functions that each define parameters that the agent needs to invoke from the user. Each function
         * represents an action in an action group.
         * </p>
         * This is a convenience method that creates an instance of the {@link FunctionSchema.Builder} avoiding the need
         * to create one manually via {@link FunctionSchema#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FunctionSchema.Builder#build()} is called immediately and its
         * result is passed to {@link #functionSchema(FunctionSchema)}.
         * 
         * @param functionSchema
         *        a consumer that will call methods on {@link FunctionSchema.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #functionSchema(FunctionSchema)
         */
        default Builder functionSchema(Consumer<FunctionSchema.Builder> functionSchema) {
            return functionSchema(FunctionSchema.builder().applyMutation(functionSchema).build());
        }

        /**
         * <p>
         * Specifies whether the action group is available for the agent to invoke or not when sending an <a
         * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_InvokeAgent.html"
         * >InvokeAgent</a> request.
         * </p>
         * 
         * @param actionGroupState
         *        Specifies whether the action group is available for the agent to invoke or not when sending an <a
         *        href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_InvokeAgent.html">
         *        InvokeAgent</a> request.
         * @see ActionGroupState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ActionGroupState
         */
        Builder actionGroupState(String actionGroupState);

        /**
         * <p>
         * Specifies whether the action group is available for the agent to invoke or not when sending an <a
         * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_InvokeAgent.html"
         * >InvokeAgent</a> request.
         * </p>
         * 
         * @param actionGroupState
         *        Specifies whether the action group is available for the agent to invoke or not when sending an <a
         *        href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_InvokeAgent.html">
         *        InvokeAgent</a> request.
         * @see ActionGroupState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ActionGroupState
         */
        Builder actionGroupState(ActionGroupState actionGroupState);
    }

    static final class BuilderImpl implements Builder {
        private String agentId;

        private String agentVersion;

        private String actionGroupId;

        private String actionGroupName;

        private String clientToken;

        private String description;

        private Instant createdAt;

        private Instant updatedAt;

        private String parentActionSignature;

        private Map<String, String> parentActionGroupSignatureParams = DefaultSdkAutoConstructMap.getInstance();

        private ActionGroupExecutor actionGroupExecutor;

        private APISchema apiSchema;

        private FunctionSchema functionSchema;

        private String actionGroupState;

        private BuilderImpl() {
        }

        private BuilderImpl(AgentActionGroup model) {
            agentId(model.agentId);
            agentVersion(model.agentVersion);
            actionGroupId(model.actionGroupId);
            actionGroupName(model.actionGroupName);
            clientToken(model.clientToken);
            description(model.description);
            createdAt(model.createdAt);
            updatedAt(model.updatedAt);
            parentActionSignature(model.parentActionSignature);
            parentActionGroupSignatureParams(model.parentActionGroupSignatureParams);
            actionGroupExecutor(model.actionGroupExecutor);
            apiSchema(model.apiSchema);
            functionSchema(model.functionSchema);
            actionGroupState(model.actionGroupState);
        }

        public final String getAgentId() {
            return agentId;
        }

        public final void setAgentId(String agentId) {
            this.agentId = agentId;
        }

        @Override
        public final Builder agentId(String agentId) {
            this.agentId = agentId;
            return this;
        }

        public final String getAgentVersion() {
            return agentVersion;
        }

        public final void setAgentVersion(String agentVersion) {
            this.agentVersion = agentVersion;
        }

        @Override
        public final Builder agentVersion(String agentVersion) {
            this.agentVersion = agentVersion;
            return this;
        }

        public final String getActionGroupId() {
            return actionGroupId;
        }

        public final void setActionGroupId(String actionGroupId) {
            this.actionGroupId = actionGroupId;
        }

        @Override
        public final Builder actionGroupId(String actionGroupId) {
            this.actionGroupId = actionGroupId;
            return this;
        }

        public final String getActionGroupName() {
            return actionGroupName;
        }

        public final void setActionGroupName(String actionGroupName) {
            this.actionGroupName = actionGroupName;
        }

        @Override
        public final Builder actionGroupName(String actionGroupName) {
            this.actionGroupName = actionGroupName;
            return this;
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public final String getParentActionSignature() {
            return parentActionSignature;
        }

        public final void setParentActionSignature(String parentActionSignature) {
            this.parentActionSignature = parentActionSignature;
        }

        @Override
        public final Builder parentActionSignature(String parentActionSignature) {
            this.parentActionSignature = parentActionSignature;
            return this;
        }

        @Override
        public final Builder parentActionSignature(ActionGroupSignature parentActionSignature) {
            this.parentActionSignature(parentActionSignature == null ? null : parentActionSignature.toString());
            return this;
        }

        public final Map<String, String> getParentActionGroupSignatureParams() {
            if (parentActionGroupSignatureParams instanceof SdkAutoConstructMap) {
                return null;
            }
            return parentActionGroupSignatureParams;
        }

        public final void setParentActionGroupSignatureParams(Map<String, String> parentActionGroupSignatureParams) {
            this.parentActionGroupSignatureParams = ActionGroupSignatureParamsCopier.copy(parentActionGroupSignatureParams);
        }

        @Override
        public final Builder parentActionGroupSignatureParams(Map<String, String> parentActionGroupSignatureParams) {
            this.parentActionGroupSignatureParams = ActionGroupSignatureParamsCopier.copy(parentActionGroupSignatureParams);
            return this;
        }

        public final ActionGroupExecutor.Builder getActionGroupExecutor() {
            return actionGroupExecutor != null ? actionGroupExecutor.toBuilder() : null;
        }

        public final void setActionGroupExecutor(ActionGroupExecutor.BuilderImpl actionGroupExecutor) {
            this.actionGroupExecutor = actionGroupExecutor != null ? actionGroupExecutor.build() : null;
        }

        @Override
        public final Builder actionGroupExecutor(ActionGroupExecutor actionGroupExecutor) {
            this.actionGroupExecutor = actionGroupExecutor;
            return this;
        }

        public final APISchema.Builder getApiSchema() {
            return apiSchema != null ? apiSchema.toBuilder() : null;
        }

        public final void setApiSchema(APISchema.BuilderImpl apiSchema) {
            this.apiSchema = apiSchema != null ? apiSchema.build() : null;
        }

        @Override
        public final Builder apiSchema(APISchema apiSchema) {
            this.apiSchema = apiSchema;
            return this;
        }

        public final FunctionSchema.Builder getFunctionSchema() {
            return functionSchema != null ? functionSchema.toBuilder() : null;
        }

        public final void setFunctionSchema(FunctionSchema.BuilderImpl functionSchema) {
            this.functionSchema = functionSchema != null ? functionSchema.build() : null;
        }

        @Override
        public final Builder functionSchema(FunctionSchema functionSchema) {
            this.functionSchema = functionSchema;
            return this;
        }

        public final String getActionGroupState() {
            return actionGroupState;
        }

        public final void setActionGroupState(String actionGroupState) {
            this.actionGroupState = actionGroupState;
        }

        @Override
        public final Builder actionGroupState(String actionGroupState) {
            this.actionGroupState = actionGroupState;
            return this;
        }

        @Override
        public final Builder actionGroupState(ActionGroupState actionGroupState) {
            this.actionGroupState(actionGroupState == null ? null : actionGroupState.toString());
            return this;
        }

        @Override
        public AgentActionGroup build() {
            return new AgentActionGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
