/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagent.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Settings for semantic document chunking for a data source. Semantic chunking splits a document into into smaller
 * documents based on groups of similar content derived from the text with natural language processing.
 * </p>
 * <p>
 * With semantic chunking, each sentence is compared to the next to determine how similar they are. You specify a
 * threshold in the form of a percentile, where adjacent sentences that are less similar than that percentage of
 * sentence pairs are divided into separate chunks. For example, if you set the threshold to 90, then the 10 percent of
 * sentence pairs that are least similar are split. So if you have 101 sentences, 100 sentence pairs are compared, and
 * the 10 with the least similarity are split, creating 11 chunks. These chunks are further split if they exceed the max
 * token size.
 * </p>
 * <p>
 * You must also specify a buffer size, which determines whether sentences are compared in isolation, or within a moving
 * context window that includes the previous and following sentence. For example, if you set the buffer size to
 * <code>1</code>, the embedding for sentence 10 is derived from sentences 9, 10, and 11 combined.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SemanticChunkingConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<SemanticChunkingConfiguration.Builder, SemanticChunkingConfiguration> {
    private static final SdkField<Integer> MAX_TOKENS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxTokens").getter(getter(SemanticChunkingConfiguration::maxTokens)).setter(setter(Builder::maxTokens))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxTokens").build()).build();

    private static final SdkField<Integer> BUFFER_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("bufferSize").getter(getter(SemanticChunkingConfiguration::bufferSize))
            .setter(setter(Builder::bufferSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bufferSize").build()).build();

    private static final SdkField<Integer> BREAKPOINT_PERCENTILE_THRESHOLD_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("breakpointPercentileThreshold")
            .getter(getter(SemanticChunkingConfiguration::breakpointPercentileThreshold))
            .setter(setter(Builder::breakpointPercentileThreshold))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("breakpointPercentileThreshold")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MAX_TOKENS_FIELD,
            BUFFER_SIZE_FIELD, BREAKPOINT_PERCENTILE_THRESHOLD_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer maxTokens;

    private final Integer bufferSize;

    private final Integer breakpointPercentileThreshold;

    private SemanticChunkingConfiguration(BuilderImpl builder) {
        this.maxTokens = builder.maxTokens;
        this.bufferSize = builder.bufferSize;
        this.breakpointPercentileThreshold = builder.breakpointPercentileThreshold;
    }

    /**
     * <p>
     * The maximum number of tokens that a chunk can contain.
     * </p>
     * 
     * @return The maximum number of tokens that a chunk can contain.
     */
    public final Integer maxTokens() {
        return maxTokens;
    }

    /**
     * <p>
     * The buffer size.
     * </p>
     * 
     * @return The buffer size.
     */
    public final Integer bufferSize() {
        return bufferSize;
    }

    /**
     * <p>
     * The dissimilarity threshold for splitting chunks.
     * </p>
     * 
     * @return The dissimilarity threshold for splitting chunks.
     */
    public final Integer breakpointPercentileThreshold() {
        return breakpointPercentileThreshold;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(maxTokens());
        hashCode = 31 * hashCode + Objects.hashCode(bufferSize());
        hashCode = 31 * hashCode + Objects.hashCode(breakpointPercentileThreshold());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SemanticChunkingConfiguration)) {
            return false;
        }
        SemanticChunkingConfiguration other = (SemanticChunkingConfiguration) obj;
        return Objects.equals(maxTokens(), other.maxTokens()) && Objects.equals(bufferSize(), other.bufferSize())
                && Objects.equals(breakpointPercentileThreshold(), other.breakpointPercentileThreshold());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SemanticChunkingConfiguration").add("MaxTokens", maxTokens()).add("BufferSize", bufferSize())
                .add("BreakpointPercentileThreshold", breakpointPercentileThreshold()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "maxTokens":
            return Optional.ofNullable(clazz.cast(maxTokens()));
        case "bufferSize":
            return Optional.ofNullable(clazz.cast(bufferSize()));
        case "breakpointPercentileThreshold":
            return Optional.ofNullable(clazz.cast(breakpointPercentileThreshold()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("maxTokens", MAX_TOKENS_FIELD);
        map.put("bufferSize", BUFFER_SIZE_FIELD);
        map.put("breakpointPercentileThreshold", BREAKPOINT_PERCENTILE_THRESHOLD_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SemanticChunkingConfiguration, T> g) {
        return obj -> g.apply((SemanticChunkingConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SemanticChunkingConfiguration> {
        /**
         * <p>
         * The maximum number of tokens that a chunk can contain.
         * </p>
         * 
         * @param maxTokens
         *        The maximum number of tokens that a chunk can contain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxTokens(Integer maxTokens);

        /**
         * <p>
         * The buffer size.
         * </p>
         * 
         * @param bufferSize
         *        The buffer size.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bufferSize(Integer bufferSize);

        /**
         * <p>
         * The dissimilarity threshold for splitting chunks.
         * </p>
         * 
         * @param breakpointPercentileThreshold
         *        The dissimilarity threshold for splitting chunks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder breakpointPercentileThreshold(Integer breakpointPercentileThreshold);
    }

    static final class BuilderImpl implements Builder {
        private Integer maxTokens;

        private Integer bufferSize;

        private Integer breakpointPercentileThreshold;

        private BuilderImpl() {
        }

        private BuilderImpl(SemanticChunkingConfiguration model) {
            maxTokens(model.maxTokens);
            bufferSize(model.bufferSize);
            breakpointPercentileThreshold(model.breakpointPercentileThreshold);
        }

        public final Integer getMaxTokens() {
            return maxTokens;
        }

        public final void setMaxTokens(Integer maxTokens) {
            this.maxTokens = maxTokens;
        }

        @Override
        public final Builder maxTokens(Integer maxTokens) {
            this.maxTokens = maxTokens;
            return this;
        }

        public final Integer getBufferSize() {
            return bufferSize;
        }

        public final void setBufferSize(Integer bufferSize) {
            this.bufferSize = bufferSize;
        }

        @Override
        public final Builder bufferSize(Integer bufferSize) {
            this.bufferSize = bufferSize;
            return this;
        }

        public final Integer getBreakpointPercentileThreshold() {
            return breakpointPercentileThreshold;
        }

        public final void setBreakpointPercentileThreshold(Integer breakpointPercentileThreshold) {
            this.breakpointPercentileThreshold = breakpointPercentileThreshold;
        }

        @Override
        public final Builder breakpointPercentileThreshold(Integer breakpointPercentileThreshold) {
            this.breakpointPercentileThreshold = breakpointPercentileThreshold;
            return this;
        }

        @Override
        public SemanticChunkingConfiguration build() {
            return new SemanticChunkingConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
