/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagentruntime;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.eventstream.EventStreamAsyncResponseTransformer;
import software.amazon.awssdk.awscore.eventstream.EventStreamTaggedUnionPojoSupplier;
import software.amazon.awssdk.awscore.eventstream.RestEventStreamAsyncResponseTransformer;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkPojoBuilder;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.SdkResponse;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkAdvancedAsyncClientOption;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.AttachHttpMetadataResponseHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.protocol.VoidSdkResponse;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.bedrockagentruntime.internal.BedrockAgentRuntimeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.bedrockagentruntime.model.AccessDeniedException;
import software.amazon.awssdk.services.bedrockagentruntime.model.BadGatewayException;
import software.amazon.awssdk.services.bedrockagentruntime.model.BedrockAgentRuntimeException;
import software.amazon.awssdk.services.bedrockagentruntime.model.ConflictException;
import software.amazon.awssdk.services.bedrockagentruntime.model.DeleteAgentMemoryRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.DeleteAgentMemoryResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.DependencyFailedException;
import software.amazon.awssdk.services.bedrockagentruntime.model.FlowResponseStream;
import software.amazon.awssdk.services.bedrockagentruntime.model.GetAgentMemoryRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.GetAgentMemoryResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.InternalServerException;
import software.amazon.awssdk.services.bedrockagentruntime.model.InvokeAgentRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.InvokeAgentResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.InvokeAgentResponseHandler;
import software.amazon.awssdk.services.bedrockagentruntime.model.InvokeFlowRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.InvokeFlowResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.InvokeFlowResponseHandler;
import software.amazon.awssdk.services.bedrockagentruntime.model.ResourceNotFoundException;
import software.amazon.awssdk.services.bedrockagentruntime.model.ResponseStream;
import software.amazon.awssdk.services.bedrockagentruntime.model.RetrieveAndGenerateRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.RetrieveAndGenerateResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.RetrieveRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.RetrieveResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.bedrockagentruntime.model.ThrottlingException;
import software.amazon.awssdk.services.bedrockagentruntime.model.ValidationException;
import software.amazon.awssdk.services.bedrockagentruntime.transform.DeleteAgentMemoryRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.GetAgentMemoryRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.InvokeAgentRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.InvokeFlowRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.RetrieveAndGenerateRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.RetrieveRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link BedrockAgentRuntimeAsyncClient}.
 *
 * @see BedrockAgentRuntimeAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBedrockAgentRuntimeAsyncClient implements BedrockAgentRuntimeAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultBedrockAgentRuntimeAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final Executor executor;

    protected DefaultBedrockAgentRuntimeAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executor = clientConfiguration.option(SdkAdvancedAsyncClientOption.FUTURE_COMPLETION_EXECUTOR);
    }

    /**
     * <p>
     * Deletes memory from the specified memory identifier.
     * </p>
     *
     * @param deleteAgentMemoryRequest
     * @return A Java Future containing the result of the DeleteAgentMemory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>DependencyFailedException There was an issue with a dependency. Check the resource configurations and
     *         retry the request.</li>
     *         <li>BadGatewayException There was an issue with a dependency due to a server issue. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.DeleteAgentMemory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/DeleteAgentMemory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAgentMemoryResponse> deleteAgentMemory(DeleteAgentMemoryRequest deleteAgentMemoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAgentMemoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAgentMemoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAgentMemory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAgentMemoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAgentMemoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAgentMemoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAgentMemoryRequest, DeleteAgentMemoryResponse>()
                            .withOperationName("DeleteAgentMemory").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAgentMemoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAgentMemoryRequest));
            CompletableFuture<DeleteAgentMemoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the sessions stored in the memory of the agent.
     * </p>
     *
     * @param getAgentMemoryRequest
     * @return A Java Future containing the result of the GetAgentMemory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>DependencyFailedException There was an issue with a dependency. Check the resource configurations and
     *         retry the request.</li>
     *         <li>BadGatewayException There was an issue with a dependency due to a server issue. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.GetAgentMemory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/GetAgentMemory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAgentMemoryResponse> getAgentMemory(GetAgentMemoryRequest getAgentMemoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAgentMemoryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAgentMemoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAgentMemory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAgentMemoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAgentMemoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAgentMemoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAgentMemoryRequest, GetAgentMemoryResponse>()
                            .withOperationName("GetAgentMemory").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAgentMemoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAgentMemoryRequest));
            CompletableFuture<GetAgentMemoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note>
     * <p>
     * The CLI doesn't support streaming operations in Amazon Bedrock, including <code>InvokeAgent</code>.
     * </p>
     * </note>
     * <p>
     * Sends a prompt for the agent to process and respond to. Note the following fields for the request:
     * </p>
     * <ul>
     * <li>
     * <p>
     * To continue the same conversation with an agent, use the same <code>sessionId</code> value in the request.
     * </p>
     * </li>
     * <li>
     * <p>
     * To activate trace enablement, turn <code>enableTrace</code> to <code>true</code>. Trace enablement helps you
     * follow the agent's reasoning process that led it to the information it processed, the actions it took, and the
     * final result it yielded. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/agents-test.html#trace-events">Trace enablement</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * End a conversation by setting <code>endSession</code> to <code>true</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * In the <code>sessionState</code> object, you can include attributes for the session or prompt or, if you
     * configured an action group to return control, results from invocation of the action group.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The response is returned in the <code>bytes</code> field of the <code>chunk</code> object.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The <code>attribution</code> object contains citations for parts of the response.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you set <code>enableTrace</code> to <code>true</code> in the request, you can trace the agent's steps and
     * reasoning process that led it to the response.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the action predicted was configured to return control, the response returns parameters for the action,
     * elicited from the user, in the <code>returnControl</code> field.
     * </p>
     * </li>
     * <li>
     * <p>
     * Errors are also surfaced in the response.
     * </p>
     * </li>
     * </ul>
     *
     * @param invokeAgentRequest
     * @return A Java Future containing the result of the InvokeAgent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>DependencyFailedException There was an issue with a dependency. Check the resource configurations and
     *         retry the request.</li>
     *         <li>BadGatewayException There was an issue with a dependency due to a server issue. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.InvokeAgent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/InvokeAgent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<Void> invokeAgent(InvokeAgentRequest invokeAgentRequest,
            InvokeAgentResponseHandler asyncResponseHandler) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(invokeAgentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, invokeAgentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InvokeAgent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InvokeAgentResponse> responseHandler = new AttachHttpMetadataResponseHandler(
                    protocolFactory.createResponseHandler(operationMetadata, InvokeAgentResponse::builder));

            HttpResponseHandler<SdkResponse> voidResponseHandler = protocolFactory.createResponseHandler(JsonOperationMetadata
                    .builder().isPayloadJson(false).hasStreamingSuccessResponse(true).build(), VoidSdkResponse::builder);

            HttpResponseHandler<? extends ResponseStream> eventResponseHandler = protocolFactory.createResponseHandler(
                    JsonOperationMetadata.builder().isPayloadJson(true).hasStreamingSuccessResponse(false).build(),
                    EventStreamTaggedUnionPojoSupplier.builder().putSdkPojoSupplier("chunk", ResponseStream::chunkBuilder)
                            .putSdkPojoSupplier("files", ResponseStream::filesBuilder)
                            .putSdkPojoSupplier("returnControl", ResponseStream::returnControlBuilder)
                            .putSdkPojoSupplier("trace", ResponseStream::traceBuilder)
                            .defaultSdkPojoSupplier(() -> new SdkPojoBuilder(ResponseStream.UNKNOWN)).build());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            CompletableFuture<Void> future = new CompletableFuture<>();
            EventStreamAsyncResponseTransformer<InvokeAgentResponse, ResponseStream> asyncResponseTransformer = EventStreamAsyncResponseTransformer
                    .<InvokeAgentResponse, ResponseStream> builder().eventStreamResponseHandler(asyncResponseHandler)
                    .eventResponseHandler(eventResponseHandler).initialResponseHandler(responseHandler)
                    .exceptionResponseHandler(errorResponseHandler).future(future).executor(executor).serviceName(serviceName())
                    .build();
            RestEventStreamAsyncResponseTransformer<InvokeAgentResponse, ResponseStream> restAsyncResponseTransformer = RestEventStreamAsyncResponseTransformer
                    .<InvokeAgentResponse, ResponseStream> builder()
                    .eventStreamAsyncResponseTransformer(asyncResponseTransformer)
                    .eventStreamResponseHandler(asyncResponseHandler).build();

            CompletableFuture<Void> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<InvokeAgentRequest, InvokeAgentResponse>().withOperationName("InvokeAgent")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new InvokeAgentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(invokeAgentRequest), restAsyncResponseTransformer);
            CompletableFuture<Void> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    try {
                        asyncResponseHandler.exceptionOccurred(e);
                    } finally {
                        future.completeExceptionally(e);
                    }
                }
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return CompletableFutureUtils.forwardExceptionTo(future, executeFuture);
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseHandler.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Invokes an alias of a flow to run the inputs that you specify and return the output of each node as a stream. If
     * there's an error, the error is returned. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/flows-test.html">Test a flow in Amazon Bedrock</a> in
     * the Amazon Bedrock User Guide.
     * </p>
     *
     * @param invokeFlowRequest
     * @return A Java Future containing the result of the InvokeFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>DependencyFailedException There was an issue with a dependency. Check the resource configurations and
     *         retry the request.</li>
     *         <li>BadGatewayException There was an issue with a dependency due to a server issue. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.InvokeFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/InvokeFlow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<Void> invokeFlow(InvokeFlowRequest invokeFlowRequest, InvokeFlowResponseHandler asyncResponseHandler) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(invokeFlowRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, invokeFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InvokeFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InvokeFlowResponse> responseHandler = new AttachHttpMetadataResponseHandler(
                    protocolFactory.createResponseHandler(operationMetadata, InvokeFlowResponse::builder));

            HttpResponseHandler<SdkResponse> voidResponseHandler = protocolFactory.createResponseHandler(JsonOperationMetadata
                    .builder().isPayloadJson(false).hasStreamingSuccessResponse(true).build(), VoidSdkResponse::builder);

            HttpResponseHandler<? extends FlowResponseStream> eventResponseHandler = protocolFactory.createResponseHandler(
                    JsonOperationMetadata.builder().isPayloadJson(true).hasStreamingSuccessResponse(false).build(),
                    EventStreamTaggedUnionPojoSupplier.builder()
                            .putSdkPojoSupplier("flowCompletionEvent", FlowResponseStream::flowCompletionEventBuilder)
                            .putSdkPojoSupplier("flowOutputEvent", FlowResponseStream::flowOutputEventBuilder)
                            .defaultSdkPojoSupplier(() -> new SdkPojoBuilder(FlowResponseStream.UNKNOWN)).build());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            CompletableFuture<Void> future = new CompletableFuture<>();
            EventStreamAsyncResponseTransformer<InvokeFlowResponse, FlowResponseStream> asyncResponseTransformer = EventStreamAsyncResponseTransformer
                    .<InvokeFlowResponse, FlowResponseStream> builder().eventStreamResponseHandler(asyncResponseHandler)
                    .eventResponseHandler(eventResponseHandler).initialResponseHandler(responseHandler)
                    .exceptionResponseHandler(errorResponseHandler).future(future).executor(executor).serviceName(serviceName())
                    .build();
            RestEventStreamAsyncResponseTransformer<InvokeFlowResponse, FlowResponseStream> restAsyncResponseTransformer = RestEventStreamAsyncResponseTransformer
                    .<InvokeFlowResponse, FlowResponseStream> builder()
                    .eventStreamAsyncResponseTransformer(asyncResponseTransformer)
                    .eventStreamResponseHandler(asyncResponseHandler).build();

            CompletableFuture<Void> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<InvokeFlowRequest, InvokeFlowResponse>().withOperationName("InvokeFlow")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new InvokeFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(invokeFlowRequest), restAsyncResponseTransformer);
            CompletableFuture<Void> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    try {
                        asyncResponseHandler.exceptionOccurred(e);
                    } finally {
                        future.completeExceptionally(e);
                    }
                }
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return CompletableFutureUtils.forwardExceptionTo(future, executeFuture);
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseHandler.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Queries a knowledge base and retrieves information from it.
     * </p>
     *
     * @param retrieveRequest
     * @return A Java Future containing the result of the Retrieve operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>DependencyFailedException There was an issue with a dependency. Check the resource configurations and
     *         retry the request.</li>
     *         <li>BadGatewayException There was an issue with a dependency due to a server issue. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.Retrieve
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/Retrieve"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RetrieveResponse> retrieve(RetrieveRequest retrieveRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(retrieveRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, retrieveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Retrieve");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RetrieveResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RetrieveResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RetrieveResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RetrieveRequest, RetrieveResponse>().withOperationName("Retrieve")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RetrieveRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(retrieveRequest));
            CompletableFuture<RetrieveResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Queries a knowledge base and generates responses based on the retrieved results. The response only cites sources
     * that are relevant to the query.
     * </p>
     *
     * @param retrieveAndGenerateRequest
     * @return A Java Future containing the result of the RetrieveAndGenerate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>DependencyFailedException There was an issue with a dependency. Check the resource configurations and
     *         retry the request.</li>
     *         <li>BadGatewayException There was an issue with a dependency due to a server issue. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.RetrieveAndGenerate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/RetrieveAndGenerate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RetrieveAndGenerateResponse> retrieveAndGenerate(
            RetrieveAndGenerateRequest retrieveAndGenerateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(retrieveAndGenerateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, retrieveAndGenerateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RetrieveAndGenerate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RetrieveAndGenerateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RetrieveAndGenerateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RetrieveAndGenerateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RetrieveAndGenerateRequest, RetrieveAndGenerateResponse>()
                            .withOperationName("RetrieveAndGenerate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RetrieveAndGenerateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(retrieveAndGenerateRequest));
            CompletableFuture<RetrieveAndGenerateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final BedrockAgentRuntimeServiceClientConfiguration serviceClientConfiguration() {
        return new BedrockAgentRuntimeServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BedrockAgentRuntimeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependencyFailedException")
                                .exceptionBuilderSupplier(DependencyFailedException::builder).httpStatusCode(424).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadGatewayException")
                                .exceptionBuilderSupplier(BadGatewayException::builder).httpStatusCode(502).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        BedrockAgentRuntimeServiceClientConfigurationBuilder serviceConfigBuilder = new BedrockAgentRuntimeServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
