/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagentruntime.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the API operation that was called from the action group and the response body that was
 * returned.
 * </p>
 * <p>
 * This data type is used in the following API operations:
 * </p>
 * <ul>
 * <li>
 * <p>
 * In the <code>returnControlInvocationResults</code> of the <a href=
 * "https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_InvokeAgent.html#API_agent-runtime_InvokeAgent_RequestSyntax"
 * >InvokeAgent request</a>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ApiResult implements SdkPojo, Serializable, ToCopyableBuilder<ApiResult.Builder, ApiResult> {
    private static final SdkField<String> ACTION_GROUP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("actionGroup").getter(getter(ApiResult::actionGroup)).setter(setter(Builder::actionGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("actionGroup").build()).build();

    private static final SdkField<String> API_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("apiPath").getter(getter(ApiResult::apiPath)).setter(setter(Builder::apiPath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("apiPath").build()).build();

    private static final SdkField<String> CONFIRMATION_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("confirmationState").getter(getter(ApiResult::confirmationStateAsString))
            .setter(setter(Builder::confirmationState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("confirmationState").build()).build();

    private static final SdkField<String> HTTP_METHOD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("httpMethod").getter(getter(ApiResult::httpMethod)).setter(setter(Builder::httpMethod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("httpMethod").build()).build();

    private static final SdkField<Integer> HTTP_STATUS_CODE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("httpStatusCode").getter(getter(ApiResult::httpStatusCode)).setter(setter(Builder::httpStatusCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("httpStatusCode").build()).build();

    private static final SdkField<Map<String, ContentBody>> RESPONSE_BODY_FIELD = SdkField
            .<Map<String, ContentBody>> builder(MarshallingType.MAP)
            .memberName("responseBody")
            .getter(getter(ApiResult::responseBody))
            .setter(setter(Builder::responseBody))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("responseBody").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<ContentBody> builder(MarshallingType.SDK_POJO)
                                            .constructor(ContentBody::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> RESPONSE_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("responseState").getter(getter(ApiResult::responseStateAsString)).setter(setter(Builder::responseState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("responseState").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACTION_GROUP_FIELD,
            API_PATH_FIELD, CONFIRMATION_STATE_FIELD, HTTP_METHOD_FIELD, HTTP_STATUS_CODE_FIELD, RESPONSE_BODY_FIELD,
            RESPONSE_STATE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String actionGroup;

    private final String apiPath;

    private final String confirmationState;

    private final String httpMethod;

    private final Integer httpStatusCode;

    private final Map<String, ContentBody> responseBody;

    private final String responseState;

    private ApiResult(BuilderImpl builder) {
        this.actionGroup = builder.actionGroup;
        this.apiPath = builder.apiPath;
        this.confirmationState = builder.confirmationState;
        this.httpMethod = builder.httpMethod;
        this.httpStatusCode = builder.httpStatusCode;
        this.responseBody = builder.responseBody;
        this.responseState = builder.responseState;
    }

    /**
     * <p>
     * The action group that the API operation belongs to.
     * </p>
     * 
     * @return The action group that the API operation belongs to.
     */
    public final String actionGroup() {
        return actionGroup;
    }

    /**
     * <p>
     * The path to the API operation.
     * </p>
     * 
     * @return The path to the API operation.
     */
    public final String apiPath() {
        return apiPath;
    }

    /**
     * <p>
     * Controls the API operations or functions to invoke based on the user confirmation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #confirmationState}
     * will return {@link ConfirmationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #confirmationStateAsString}.
     * </p>
     * 
     * @return Controls the API operations or functions to invoke based on the user confirmation.
     * @see ConfirmationState
     */
    public final ConfirmationState confirmationState() {
        return ConfirmationState.fromValue(confirmationState);
    }

    /**
     * <p>
     * Controls the API operations or functions to invoke based on the user confirmation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #confirmationState}
     * will return {@link ConfirmationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #confirmationStateAsString}.
     * </p>
     * 
     * @return Controls the API operations or functions to invoke based on the user confirmation.
     * @see ConfirmationState
     */
    public final String confirmationStateAsString() {
        return confirmationState;
    }

    /**
     * <p>
     * The HTTP method for the API operation.
     * </p>
     * 
     * @return The HTTP method for the API operation.
     */
    public final String httpMethod() {
        return httpMethod;
    }

    /**
     * <p>
     * http status code from API execution response (for example: 200, 400, 500).
     * </p>
     * 
     * @return http status code from API execution response (for example: 200, 400, 500).
     */
    public final Integer httpStatusCode() {
        return httpStatusCode;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResponseBody property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResponseBody() {
        return responseBody != null && !(responseBody instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The response body from the API operation. The key of the object is the content type (currently, only
     * <code>TEXT</code> is supported). The response may be returned directly or from the Lambda function.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResponseBody} method.
     * </p>
     * 
     * @return The response body from the API operation. The key of the object is the content type (currently, only
     *         <code>TEXT</code> is supported). The response may be returned directly or from the Lambda function.
     */
    public final Map<String, ContentBody> responseBody() {
        return responseBody;
    }

    /**
     * <p>
     * Controls the final response state returned to end user when API/Function execution failed. When this state is
     * FAILURE, the request would fail with dependency failure exception. When this state is REPROMPT, the API/function
     * response will be sent to model for re-prompt
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #responseState}
     * will return {@link ResponseState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #responseStateAsString}.
     * </p>
     * 
     * @return Controls the final response state returned to end user when API/Function execution failed. When this
     *         state is FAILURE, the request would fail with dependency failure exception. When this state is REPROMPT,
     *         the API/function response will be sent to model for re-prompt
     * @see ResponseState
     */
    public final ResponseState responseState() {
        return ResponseState.fromValue(responseState);
    }

    /**
     * <p>
     * Controls the final response state returned to end user when API/Function execution failed. When this state is
     * FAILURE, the request would fail with dependency failure exception. When this state is REPROMPT, the API/function
     * response will be sent to model for re-prompt
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #responseState}
     * will return {@link ResponseState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #responseStateAsString}.
     * </p>
     * 
     * @return Controls the final response state returned to end user when API/Function execution failed. When this
     *         state is FAILURE, the request would fail with dependency failure exception. When this state is REPROMPT,
     *         the API/function response will be sent to model for re-prompt
     * @see ResponseState
     */
    public final String responseStateAsString() {
        return responseState;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(actionGroup());
        hashCode = 31 * hashCode + Objects.hashCode(apiPath());
        hashCode = 31 * hashCode + Objects.hashCode(confirmationStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(httpMethod());
        hashCode = 31 * hashCode + Objects.hashCode(httpStatusCode());
        hashCode = 31 * hashCode + Objects.hashCode(hasResponseBody() ? responseBody() : null);
        hashCode = 31 * hashCode + Objects.hashCode(responseStateAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ApiResult)) {
            return false;
        }
        ApiResult other = (ApiResult) obj;
        return Objects.equals(actionGroup(), other.actionGroup()) && Objects.equals(apiPath(), other.apiPath())
                && Objects.equals(confirmationStateAsString(), other.confirmationStateAsString())
                && Objects.equals(httpMethod(), other.httpMethod()) && Objects.equals(httpStatusCode(), other.httpStatusCode())
                && hasResponseBody() == other.hasResponseBody() && Objects.equals(responseBody(), other.responseBody())
                && Objects.equals(responseStateAsString(), other.responseStateAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ApiResult").add("ActionGroup", actionGroup())
                .add("ApiPath", apiPath() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ConfirmationState", confirmationStateAsString()).add("HttpMethod", httpMethod())
                .add("HttpStatusCode", httpStatusCode()).add("ResponseBody", hasResponseBody() ? responseBody() : null)
                .add("ResponseState", responseStateAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "actionGroup":
            return Optional.ofNullable(clazz.cast(actionGroup()));
        case "apiPath":
            return Optional.ofNullable(clazz.cast(apiPath()));
        case "confirmationState":
            return Optional.ofNullable(clazz.cast(confirmationStateAsString()));
        case "httpMethod":
            return Optional.ofNullable(clazz.cast(httpMethod()));
        case "httpStatusCode":
            return Optional.ofNullable(clazz.cast(httpStatusCode()));
        case "responseBody":
            return Optional.ofNullable(clazz.cast(responseBody()));
        case "responseState":
            return Optional.ofNullable(clazz.cast(responseStateAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ApiResult, T> g) {
        return obj -> g.apply((ApiResult) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ApiResult> {
        /**
         * <p>
         * The action group that the API operation belongs to.
         * </p>
         * 
         * @param actionGroup
         *        The action group that the API operation belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actionGroup(String actionGroup);

        /**
         * <p>
         * The path to the API operation.
         * </p>
         * 
         * @param apiPath
         *        The path to the API operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder apiPath(String apiPath);

        /**
         * <p>
         * Controls the API operations or functions to invoke based on the user confirmation.
         * </p>
         * 
         * @param confirmationState
         *        Controls the API operations or functions to invoke based on the user confirmation.
         * @see ConfirmationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfirmationState
         */
        Builder confirmationState(String confirmationState);

        /**
         * <p>
         * Controls the API operations or functions to invoke based on the user confirmation.
         * </p>
         * 
         * @param confirmationState
         *        Controls the API operations or functions to invoke based on the user confirmation.
         * @see ConfirmationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfirmationState
         */
        Builder confirmationState(ConfirmationState confirmationState);

        /**
         * <p>
         * The HTTP method for the API operation.
         * </p>
         * 
         * @param httpMethod
         *        The HTTP method for the API operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder httpMethod(String httpMethod);

        /**
         * <p>
         * http status code from API execution response (for example: 200, 400, 500).
         * </p>
         * 
         * @param httpStatusCode
         *        http status code from API execution response (for example: 200, 400, 500).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder httpStatusCode(Integer httpStatusCode);

        /**
         * <p>
         * The response body from the API operation. The key of the object is the content type (currently, only
         * <code>TEXT</code> is supported). The response may be returned directly or from the Lambda function.
         * </p>
         * 
         * @param responseBody
         *        The response body from the API operation. The key of the object is the content type (currently, only
         *        <code>TEXT</code> is supported). The response may be returned directly or from the Lambda function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseBody(Map<String, ContentBody> responseBody);

        /**
         * <p>
         * Controls the final response state returned to end user when API/Function execution failed. When this state is
         * FAILURE, the request would fail with dependency failure exception. When this state is REPROMPT, the
         * API/function response will be sent to model for re-prompt
         * </p>
         * 
         * @param responseState
         *        Controls the final response state returned to end user when API/Function execution failed. When this
         *        state is FAILURE, the request would fail with dependency failure exception. When this state is
         *        REPROMPT, the API/function response will be sent to model for re-prompt
         * @see ResponseState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResponseState
         */
        Builder responseState(String responseState);

        /**
         * <p>
         * Controls the final response state returned to end user when API/Function execution failed. When this state is
         * FAILURE, the request would fail with dependency failure exception. When this state is REPROMPT, the
         * API/function response will be sent to model for re-prompt
         * </p>
         * 
         * @param responseState
         *        Controls the final response state returned to end user when API/Function execution failed. When this
         *        state is FAILURE, the request would fail with dependency failure exception. When this state is
         *        REPROMPT, the API/function response will be sent to model for re-prompt
         * @see ResponseState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ResponseState
         */
        Builder responseState(ResponseState responseState);
    }

    static final class BuilderImpl implements Builder {
        private String actionGroup;

        private String apiPath;

        private String confirmationState;

        private String httpMethod;

        private Integer httpStatusCode;

        private Map<String, ContentBody> responseBody = DefaultSdkAutoConstructMap.getInstance();

        private String responseState;

        private BuilderImpl() {
        }

        private BuilderImpl(ApiResult model) {
            actionGroup(model.actionGroup);
            apiPath(model.apiPath);
            confirmationState(model.confirmationState);
            httpMethod(model.httpMethod);
            httpStatusCode(model.httpStatusCode);
            responseBody(model.responseBody);
            responseState(model.responseState);
        }

        public final String getActionGroup() {
            return actionGroup;
        }

        public final void setActionGroup(String actionGroup) {
            this.actionGroup = actionGroup;
        }

        @Override
        public final Builder actionGroup(String actionGroup) {
            this.actionGroup = actionGroup;
            return this;
        }

        public final String getApiPath() {
            return apiPath;
        }

        public final void setApiPath(String apiPath) {
            this.apiPath = apiPath;
        }

        @Override
        public final Builder apiPath(String apiPath) {
            this.apiPath = apiPath;
            return this;
        }

        public final String getConfirmationState() {
            return confirmationState;
        }

        public final void setConfirmationState(String confirmationState) {
            this.confirmationState = confirmationState;
        }

        @Override
        public final Builder confirmationState(String confirmationState) {
            this.confirmationState = confirmationState;
            return this;
        }

        @Override
        public final Builder confirmationState(ConfirmationState confirmationState) {
            this.confirmationState(confirmationState == null ? null : confirmationState.toString());
            return this;
        }

        public final String getHttpMethod() {
            return httpMethod;
        }

        public final void setHttpMethod(String httpMethod) {
            this.httpMethod = httpMethod;
        }

        @Override
        public final Builder httpMethod(String httpMethod) {
            this.httpMethod = httpMethod;
            return this;
        }

        public final Integer getHttpStatusCode() {
            return httpStatusCode;
        }

        public final void setHttpStatusCode(Integer httpStatusCode) {
            this.httpStatusCode = httpStatusCode;
        }

        @Override
        public final Builder httpStatusCode(Integer httpStatusCode) {
            this.httpStatusCode = httpStatusCode;
            return this;
        }

        public final Map<String, ContentBody.Builder> getResponseBody() {
            Map<String, ContentBody.Builder> result = ResponseBodyCopier.copyToBuilder(this.responseBody);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setResponseBody(Map<String, ContentBody.BuilderImpl> responseBody) {
            this.responseBody = ResponseBodyCopier.copyFromBuilder(responseBody);
        }

        @Override
        public final Builder responseBody(Map<String, ContentBody> responseBody) {
            this.responseBody = ResponseBodyCopier.copy(responseBody);
            return this;
        }

        public final String getResponseState() {
            return responseState;
        }

        public final void setResponseState(String responseState) {
            this.responseState = responseState;
        }

        @Override
        public final Builder responseState(String responseState) {
            this.responseState = responseState;
            return this;
        }

        @Override
        public final Builder responseState(ResponseState responseState) {
            this.responseState(responseState == null ? null : responseState.toString());
            return this;
        }

        @Override
        public ApiResult build() {
            return new ApiResult(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
