/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagentruntime;

import static software.amazon.awssdk.utils.FunctionalUtils.runAndLogError;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.eventstream.EventStreamAsyncResponseTransformer;
import software.amazon.awssdk.awscore.eventstream.EventStreamTaggedUnionPojoSupplier;
import software.amazon.awssdk.awscore.eventstream.RestEventStreamAsyncResponseTransformer;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkPojoBuilder;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.SdkResponse;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkAdvancedAsyncClientOption;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.AttachHttpMetadataResponseHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.protocol.VoidSdkResponse;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.bedrockagentruntime.internal.BedrockAgentRuntimeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.bedrockagentruntime.model.AccessDeniedException;
import software.amazon.awssdk.services.bedrockagentruntime.model.BadGatewayException;
import software.amazon.awssdk.services.bedrockagentruntime.model.BedrockAgentRuntimeException;
import software.amazon.awssdk.services.bedrockagentruntime.model.ConflictException;
import software.amazon.awssdk.services.bedrockagentruntime.model.CreateInvocationRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.CreateInvocationResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.CreateSessionRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.CreateSessionResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.DeleteAgentMemoryRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.DeleteAgentMemoryResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.DeleteSessionRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.DeleteSessionResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.DependencyFailedException;
import software.amazon.awssdk.services.bedrockagentruntime.model.EndSessionRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.EndSessionResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.FlowResponseStream;
import software.amazon.awssdk.services.bedrockagentruntime.model.GenerateQueryRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.GenerateQueryResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.GetAgentMemoryRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.GetAgentMemoryResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.GetInvocationStepRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.GetInvocationStepResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.GetSessionRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.GetSessionResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.InlineAgentResponseStream;
import software.amazon.awssdk.services.bedrockagentruntime.model.InternalServerException;
import software.amazon.awssdk.services.bedrockagentruntime.model.InvokeAgentRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.InvokeAgentResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.InvokeAgentResponseHandler;
import software.amazon.awssdk.services.bedrockagentruntime.model.InvokeFlowRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.InvokeFlowResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.InvokeFlowResponseHandler;
import software.amazon.awssdk.services.bedrockagentruntime.model.InvokeInlineAgentRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.InvokeInlineAgentResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.InvokeInlineAgentResponseHandler;
import software.amazon.awssdk.services.bedrockagentruntime.model.ListInvocationStepsRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.ListInvocationStepsResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.ListInvocationsRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.ListInvocationsResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.ListSessionsRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.ListSessionsResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.ModelNotReadyException;
import software.amazon.awssdk.services.bedrockagentruntime.model.OptimizePromptRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.OptimizePromptResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.OptimizePromptResponseHandler;
import software.amazon.awssdk.services.bedrockagentruntime.model.OptimizedPromptStream;
import software.amazon.awssdk.services.bedrockagentruntime.model.PutInvocationStepRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.PutInvocationStepResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.RerankRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.RerankResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.ResourceNotFoundException;
import software.amazon.awssdk.services.bedrockagentruntime.model.ResponseStream;
import software.amazon.awssdk.services.bedrockagentruntime.model.RetrieveAndGenerateRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.RetrieveAndGenerateResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.RetrieveAndGenerateStreamRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.RetrieveAndGenerateStreamResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.RetrieveAndGenerateStreamResponseHandler;
import software.amazon.awssdk.services.bedrockagentruntime.model.RetrieveAndGenerateStreamResponseOutput;
import software.amazon.awssdk.services.bedrockagentruntime.model.RetrieveRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.RetrieveResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.bedrockagentruntime.model.TagResourceRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.TagResourceResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.ThrottlingException;
import software.amazon.awssdk.services.bedrockagentruntime.model.UntagResourceRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.UntagResourceResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.UpdateSessionRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.UpdateSessionResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.ValidationException;
import software.amazon.awssdk.services.bedrockagentruntime.transform.CreateInvocationRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.CreateSessionRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.DeleteAgentMemoryRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.DeleteSessionRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.EndSessionRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.GenerateQueryRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.GetAgentMemoryRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.GetInvocationStepRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.GetSessionRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.InvokeAgentRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.InvokeFlowRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.InvokeInlineAgentRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.ListInvocationStepsRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.ListInvocationsRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.ListSessionsRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.OptimizePromptRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.PutInvocationStepRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.RerankRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.RetrieveAndGenerateRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.RetrieveAndGenerateStreamRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.RetrieveRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.UpdateSessionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link BedrockAgentRuntimeAsyncClient}.
 *
 * @see BedrockAgentRuntimeAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBedrockAgentRuntimeAsyncClient implements BedrockAgentRuntimeAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultBedrockAgentRuntimeAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final Executor executor;

    protected DefaultBedrockAgentRuntimeAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        this.executor = clientConfiguration.option(SdkAdvancedAsyncClientOption.FUTURE_COMPLETION_EXECUTOR);
    }

    /**
     * <p>
     * Creates a new invocation within a session. An invocation groups the related invocation steps that store the
     * content from a conversation. For more information about sessions, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history
     * and context with Amazon Bedrock sessions</a>.
     * </p>
     * <p>
     * Related APIs
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_ListInvocations.html">
     * ListInvocations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_ListSessions.html">ListSessions
     * </a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_GetSession.html">GetSession</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createInvocationRequest
     * @return A Java Future containing the result of the CreateInvocation operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.CreateInvocation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/CreateInvocation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateInvocationResponse> createInvocation(CreateInvocationRequest createInvocationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createInvocationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createInvocationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateInvocation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateInvocationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateInvocationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateInvocationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateInvocationRequest, CreateInvocationResponse>()
                            .withOperationName("CreateInvocation").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateInvocationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createInvocationRequest));
            CompletableFuture<CreateInvocationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a session to temporarily store conversations for generative AI (GenAI) applications built with
     * open-source frameworks such as LangGraph and LlamaIndex. Sessions enable you to save the state of conversations
     * at checkpoints, with the added security and infrastructure of Amazon Web Services. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history
     * and context with Amazon Bedrock sessions</a>.
     * </p>
     * <p>
     * By default, Amazon Bedrock uses Amazon Web Services-managed keys for session encryption, including session
     * metadata, or you can use your own KMS key. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/session-encryption.html">Amazon Bedrock session
     * encryption</a>.
     * </p>
     * <note>
     * <p>
     * You use a session to store state and conversation history for generative AI applications built with open-source
     * frameworks. For Amazon Bedrock Agents, the service automatically manages conversation context and associates them
     * with the agent-specific sessionId you specify in the <a
     * href="https://docs.aws.amazon.com/bedrock/latest/API_agent-runtime_InvokeAgent.html">InvokeAgent</a> API
     * operation.
     * </p>
     * </note>
     * <p>
     * Related APIs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_ListSessions.html">ListSessions
     * </a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_GetSession.html">GetSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_EndSession.html">EndSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_DeleteSession.html">
     * DeleteSession</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createSessionRequest
     * @return A Java Future containing the result of the CreateSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.CreateSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/CreateSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSessionResponse> createSession(CreateSessionRequest createSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateSessionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSessionRequest, CreateSessionResponse>()
                            .withOperationName("CreateSession").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSessionRequest));
            CompletableFuture<CreateSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes memory from the specified memory identifier.
     * </p>
     *
     * @param deleteAgentMemoryRequest
     * @return A Java Future containing the result of the DeleteAgentMemory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>DependencyFailedException There was an issue with a dependency. Check the resource configurations and
     *         retry the request.</li>
     *         <li>BadGatewayException There was an issue with a dependency due to a server issue. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.DeleteAgentMemory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/DeleteAgentMemory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAgentMemoryResponse> deleteAgentMemory(DeleteAgentMemoryRequest deleteAgentMemoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAgentMemoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAgentMemoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAgentMemory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAgentMemoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAgentMemoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAgentMemoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAgentMemoryRequest, DeleteAgentMemoryResponse>()
                            .withOperationName("DeleteAgentMemory").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteAgentMemoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteAgentMemoryRequest));
            CompletableFuture<DeleteAgentMemoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a session that you ended. You can't delete a session with an <code>ACTIVE</code> status. To delete an
     * active session, you must first end it with the <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_EndSession.html">EndSession</a>
     * API operation. For more information about sessions, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history
     * and context with Amazon Bedrock sessions</a>.
     * </p>
     *
     * @param deleteSessionRequest
     * @return A Java Future containing the result of the DeleteSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.DeleteSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/DeleteSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSessionResponse> deleteSession(DeleteSessionRequest deleteSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteSessionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSessionRequest, DeleteSessionResponse>()
                            .withOperationName("DeleteSession").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSessionRequest));
            CompletableFuture<DeleteSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Ends the session. After you end a session, you can still access its content but you can’t add to it. To delete
     * the session and it's content, you use the DeleteSession API operation. For more information about sessions, see
     * <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation
     * history and context with Amazon Bedrock sessions</a>.
     * </p>
     *
     * @param endSessionRequest
     * @return A Java Future containing the result of the EndSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.EndSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/EndSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EndSessionResponse> endSession(EndSessionRequest endSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(endSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, endSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EndSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EndSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    EndSessionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EndSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EndSessionRequest, EndSessionResponse>().withOperationName("EndSession")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new EndSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(endSessionRequest));
            CompletableFuture<EndSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates an SQL query from a natural language query. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/knowledge-base-generate-query.html">Generate a query
     * for structured data</a> in the Amazon Bedrock User Guide.
     * </p>
     *
     * @param generateQueryRequest
     * @return A Java Future containing the result of the GenerateQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>DependencyFailedException There was an issue with a dependency. Check the resource configurations and
     *         retry the request.</li>
     *         <li>BadGatewayException There was an issue with a dependency due to a server issue. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.GenerateQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/GenerateQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GenerateQueryResponse> generateQuery(GenerateQueryRequest generateQueryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(generateQueryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, generateQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GenerateQuery");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GenerateQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GenerateQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GenerateQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GenerateQueryRequest, GenerateQueryResponse>()
                            .withOperationName("GenerateQuery").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GenerateQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(generateQueryRequest));
            CompletableFuture<GenerateQueryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the sessions stored in the memory of the agent.
     * </p>
     *
     * @param getAgentMemoryRequest
     * @return A Java Future containing the result of the GetAgentMemory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>DependencyFailedException There was an issue with a dependency. Check the resource configurations and
     *         retry the request.</li>
     *         <li>BadGatewayException There was an issue with a dependency due to a server issue. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.GetAgentMemory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/GetAgentMemory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAgentMemoryResponse> getAgentMemory(GetAgentMemoryRequest getAgentMemoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAgentMemoryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAgentMemoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAgentMemory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAgentMemoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAgentMemoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAgentMemoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAgentMemoryRequest, GetAgentMemoryResponse>()
                            .withOperationName("GetAgentMemory").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAgentMemoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAgentMemoryRequest));
            CompletableFuture<GetAgentMemoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the details of a specific invocation step within an invocation in a session. For more information about
     * sessions, see <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve
     * conversation history and context with Amazon Bedrock sessions</a>.
     * </p>
     *
     * @param getInvocationStepRequest
     * @return A Java Future containing the result of the GetInvocationStep operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.GetInvocationStep
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/GetInvocationStep"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetInvocationStepResponse> getInvocationStep(GetInvocationStepRequest getInvocationStepRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getInvocationStepRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInvocationStepRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInvocationStep");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetInvocationStepResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetInvocationStepResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetInvocationStepResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetInvocationStepRequest, GetInvocationStepResponse>()
                            .withOperationName("GetInvocationStep").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetInvocationStepRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getInvocationStepRequest));
            CompletableFuture<GetInvocationStepResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves details about a specific session. For more information about sessions, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history
     * and context with Amazon Bedrock sessions</a>.
     * </p>
     *
     * @param getSessionRequest
     * @return A Java Future containing the result of the GetSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.GetSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/GetSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSessionResponse> getSession(GetSessionRequest getSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetSessionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSessionRequest, GetSessionResponse>().withOperationName("GetSession")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSessionRequest));
            CompletableFuture<GetSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <note> </note>
     * <p>
     * Sends a prompt for the agent to process and respond to. Note the following fields for the request:
     * </p>
     * <ul>
     * <li>
     * <p>
     * To continue the same conversation with an agent, use the same <code>sessionId</code> value in the request.
     * </p>
     * </li>
     * <li>
     * <p>
     * To activate trace enablement, turn <code>enableTrace</code> to <code>true</code>. Trace enablement helps you
     * follow the agent's reasoning process that led it to the information it processed, the actions it took, and the
     * final result it yielded. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/agents-test.html#trace-events">Trace enablement</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To stream agent responses, make sure that only orchestration prompt is enabled. Agent streaming is not supported
     * for the following steps:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Pre-processing</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Post-processing</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Agent with 1 Knowledge base and <code>User Input</code> not enabled
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * End a conversation by setting <code>endSession</code> to <code>true</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * In the <code>sessionState</code> object, you can include attributes for the session or prompt or, if you
     * configured an action group to return control, results from invocation of the action group.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The response contains both <b>chunk</b> and <b>trace</b> attributes.
     * </p>
     * <p>
     * The final response is returned in the <code>bytes</code> field of the <code>chunk</code> object. The
     * <code>InvokeAgent</code> returns one chunk for the entire interaction.
     * </p>
     * <ul>
     * <li>
     * <p>
     * The <code>attribution</code> object contains citations for parts of the response.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you set <code>enableTrace</code> to <code>true</code> in the request, you can trace the agent's steps and
     * reasoning process that led it to the response.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the action predicted was configured to return control, the response returns parameters for the action,
     * elicited from the user, in the <code>returnControl</code> field.
     * </p>
     * </li>
     * <li>
     * <p>
     * Errors are also surfaced in the response.
     * </p>
     * </li>
     * </ul>
     *
     * @param invokeAgentRequest
     * @return A Java Future containing the result of the InvokeAgent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ModelNotReadyException The model specified in the request is not ready to serve inference requests.
     *         The AWS SDK will automatically retry the operation up to 5 times. For information about configuring
     *         automatic retries, see <a
     *         href="https://docs.aws.amazon.com/sdkref/latest/guide/feature-retry-behavior.html">Retry behavior</a> in
     *         the <i>AWS SDKs and Tools</i> reference guide.</li>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>DependencyFailedException There was an issue with a dependency. Check the resource configurations and
     *         retry the request.</li>
     *         <li>BadGatewayException There was an issue with a dependency due to a server issue. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.InvokeAgent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/InvokeAgent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<Void> invokeAgent(InvokeAgentRequest invokeAgentRequest,
            InvokeAgentResponseHandler asyncResponseHandler) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(invokeAgentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, invokeAgentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InvokeAgent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InvokeAgentResponse> responseHandler = new AttachHttpMetadataResponseHandler(
                    protocolFactory.createResponseHandler(operationMetadata, InvokeAgentResponse::builder));

            HttpResponseHandler<SdkResponse> voidResponseHandler = protocolFactory.createResponseHandler(JsonOperationMetadata
                    .builder().isPayloadJson(false).hasStreamingSuccessResponse(true).build(), VoidSdkResponse::builder);

            HttpResponseHandler<? extends ResponseStream> eventResponseHandler = protocolFactory.createResponseHandler(
                    JsonOperationMetadata.builder().isPayloadJson(true).hasStreamingSuccessResponse(false).build(),
                    EventStreamTaggedUnionPojoSupplier.builder().putSdkPojoSupplier("chunk", ResponseStream::chunkBuilder)
                            .putSdkPojoSupplier("files", ResponseStream::filesBuilder)
                            .putSdkPojoSupplier("returnControl", ResponseStream::returnControlBuilder)
                            .putSdkPojoSupplier("trace", ResponseStream::traceBuilder)
                            .defaultSdkPojoSupplier(() -> new SdkPojoBuilder(ResponseStream.UNKNOWN)).build());

            Function<String, Optional<ExceptionMetadata>> eventstreamExceptionMetadataMapper = errorCode -> {
                switch (errorCode) {
                case "accessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "badGatewayException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadGatewayException").httpStatusCode(502)
                            .exceptionBuilderSupplier(BadGatewayException::builder).build());
                case "conflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "dependencyFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyFailedException").httpStatusCode(424)
                            .exceptionBuilderSupplier(DependencyFailedException::builder).build());
                case "internalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "modelNotReadyException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ModelNotReadyException").httpStatusCode(424)
                            .exceptionBuilderSupplier(ModelNotReadyException::builder).build());
                case "resourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "serviceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "throttlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "validationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };

            HttpResponseHandler<AwsServiceException> errorEventResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, eventstreamExceptionMetadataMapper);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            CompletableFuture<Void> future = new CompletableFuture<>();
            EventStreamAsyncResponseTransformer<InvokeAgentResponse, ResponseStream> asyncResponseTransformer = EventStreamAsyncResponseTransformer
                    .<InvokeAgentResponse, ResponseStream> builder().eventStreamResponseHandler(asyncResponseHandler)
                    .eventResponseHandler(eventResponseHandler).initialResponseHandler(responseHandler)
                    .exceptionResponseHandler(errorEventResponseHandler).future(future).executor(executor)
                    .serviceName(serviceName()).build();
            RestEventStreamAsyncResponseTransformer<InvokeAgentResponse, ResponseStream> restAsyncResponseTransformer = RestEventStreamAsyncResponseTransformer
                    .<InvokeAgentResponse, ResponseStream> builder()
                    .eventStreamAsyncResponseTransformer(asyncResponseTransformer)
                    .eventStreamResponseHandler(asyncResponseHandler).build();

            CompletableFuture<Void> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<InvokeAgentRequest, InvokeAgentResponse>().withOperationName("InvokeAgent")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new InvokeAgentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(invokeAgentRequest), restAsyncResponseTransformer);
            CompletableFuture<Void> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    try {
                        asyncResponseHandler.exceptionOccurred(e);
                    } finally {
                        future.completeExceptionally(e);
                    }
                }
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return CompletableFutureUtils.forwardExceptionTo(future, executeFuture);
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseHandler.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Invokes an alias of a flow to run the inputs that you specify and return the output of each node as a stream. If
     * there's an error, the error is returned. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/flows-test.html">Test a flow in Amazon Bedrock</a> in
     * the <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Amazon Bedrock User
     * Guide</a>.
     * </p>
     * <note>
     * <p>
     * The CLI doesn't support streaming operations in Amazon Bedrock, including <code>InvokeFlow</code>.
     * </p>
     * </note>
     *
     * @param invokeFlowRequest
     * @return A Java Future containing the result of the InvokeFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>DependencyFailedException There was an issue with a dependency. Check the resource configurations and
     *         retry the request.</li>
     *         <li>BadGatewayException There was an issue with a dependency due to a server issue. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.InvokeFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/InvokeFlow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<Void> invokeFlow(InvokeFlowRequest invokeFlowRequest, InvokeFlowResponseHandler asyncResponseHandler) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(invokeFlowRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, invokeFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InvokeFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InvokeFlowResponse> responseHandler = new AttachHttpMetadataResponseHandler(
                    protocolFactory.createResponseHandler(operationMetadata, InvokeFlowResponse::builder));

            HttpResponseHandler<SdkResponse> voidResponseHandler = protocolFactory.createResponseHandler(JsonOperationMetadata
                    .builder().isPayloadJson(false).hasStreamingSuccessResponse(true).build(), VoidSdkResponse::builder);

            HttpResponseHandler<? extends FlowResponseStream> eventResponseHandler = protocolFactory.createResponseHandler(
                    JsonOperationMetadata.builder().isPayloadJson(true).hasStreamingSuccessResponse(false).build(),
                    EventStreamTaggedUnionPojoSupplier
                            .builder()
                            .putSdkPojoSupplier("flowCompletionEvent", FlowResponseStream::flowCompletionEventBuilder)
                            .putSdkPojoSupplier("flowMultiTurnInputRequestEvent",
                                    FlowResponseStream::flowMultiTurnInputRequestEventBuilder)
                            .putSdkPojoSupplier("flowOutputEvent", FlowResponseStream::flowOutputEventBuilder)
                            .putSdkPojoSupplier("flowTraceEvent", FlowResponseStream::flowTraceEventBuilder)
                            .defaultSdkPojoSupplier(() -> new SdkPojoBuilder(FlowResponseStream.UNKNOWN)).build());

            Function<String, Optional<ExceptionMetadata>> eventstreamExceptionMetadataMapper = errorCode -> {
                switch (errorCode) {
                case "accessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "badGatewayException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadGatewayException").httpStatusCode(502)
                            .exceptionBuilderSupplier(BadGatewayException::builder).build());
                case "conflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "dependencyFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyFailedException").httpStatusCode(424)
                            .exceptionBuilderSupplier(DependencyFailedException::builder).build());
                case "internalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "resourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "serviceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "throttlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "validationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };

            HttpResponseHandler<AwsServiceException> errorEventResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, eventstreamExceptionMetadataMapper);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            CompletableFuture<Void> future = new CompletableFuture<>();
            EventStreamAsyncResponseTransformer<InvokeFlowResponse, FlowResponseStream> asyncResponseTransformer = EventStreamAsyncResponseTransformer
                    .<InvokeFlowResponse, FlowResponseStream> builder().eventStreamResponseHandler(asyncResponseHandler)
                    .eventResponseHandler(eventResponseHandler).initialResponseHandler(responseHandler)
                    .exceptionResponseHandler(errorEventResponseHandler).future(future).executor(executor)
                    .serviceName(serviceName()).build();
            RestEventStreamAsyncResponseTransformer<InvokeFlowResponse, FlowResponseStream> restAsyncResponseTransformer = RestEventStreamAsyncResponseTransformer
                    .<InvokeFlowResponse, FlowResponseStream> builder()
                    .eventStreamAsyncResponseTransformer(asyncResponseTransformer)
                    .eventStreamResponseHandler(asyncResponseHandler).build();

            CompletableFuture<Void> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<InvokeFlowRequest, InvokeFlowResponse>().withOperationName("InvokeFlow")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new InvokeFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(invokeFlowRequest), restAsyncResponseTransformer);
            CompletableFuture<Void> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    try {
                        asyncResponseHandler.exceptionOccurred(e);
                    } finally {
                        future.completeExceptionally(e);
                    }
                }
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return CompletableFutureUtils.forwardExceptionTo(future, executeFuture);
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseHandler.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Invokes an inline Amazon Bedrock agent using the configurations you provide with the request.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Specify the following fields for security purposes.
     * </p>
     * <ul>
     * <li>
     * <p>
     * (Optional) <code>customerEncryptionKeyArn</code> – The Amazon Resource Name (ARN) of a KMS key to encrypt the
     * creation of the agent.
     * </p>
     * </li>
     * <li>
     * <p>
     * (Optional) <code>idleSessionTTLinSeconds</code> – Specify the number of seconds for which the agent should
     * maintain session information. After this time expires, the subsequent <code>InvokeInlineAgent</code> request
     * begins a new session.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * To override the default prompt behavior for agent orchestration and to use advanced prompts, include a
     * <code>promptOverrideConfiguration</code> object. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html">Advanced prompts</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * The agent instructions will not be honored if your agent has only one knowledge base, uses default prompts, has
     * no action group, and user input is disabled.
     * </p>
     * </li>
     * </ul>
     * <note> </note>
     *
     * @param invokeInlineAgentRequest
     * @return A Java Future containing the result of the InvokeInlineAgent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>DependencyFailedException There was an issue with a dependency. Check the resource configurations and
     *         retry the request.</li>
     *         <li>BadGatewayException There was an issue with a dependency due to a server issue. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.InvokeInlineAgent
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/InvokeInlineAgent"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<Void> invokeInlineAgent(InvokeInlineAgentRequest invokeInlineAgentRequest,
            InvokeInlineAgentResponseHandler asyncResponseHandler) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(invokeInlineAgentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, invokeInlineAgentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InvokeInlineAgent");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<InvokeInlineAgentResponse> responseHandler = new AttachHttpMetadataResponseHandler(
                    protocolFactory.createResponseHandler(operationMetadata, InvokeInlineAgentResponse::builder));

            HttpResponseHandler<SdkResponse> voidResponseHandler = protocolFactory.createResponseHandler(JsonOperationMetadata
                    .builder().isPayloadJson(false).hasStreamingSuccessResponse(true).build(), VoidSdkResponse::builder);

            HttpResponseHandler<? extends InlineAgentResponseStream> eventResponseHandler = protocolFactory
                    .createResponseHandler(
                            JsonOperationMetadata.builder().isPayloadJson(true).hasStreamingSuccessResponse(false).build(),
                            EventStreamTaggedUnionPojoSupplier.builder()
                                    .putSdkPojoSupplier("chunk", InlineAgentResponseStream::chunkBuilder)
                                    .putSdkPojoSupplier("files", InlineAgentResponseStream::filesBuilder)
                                    .putSdkPojoSupplier("returnControl", InlineAgentResponseStream::returnControlBuilder)
                                    .putSdkPojoSupplier("trace", InlineAgentResponseStream::traceBuilder)
                                    .defaultSdkPojoSupplier(() -> new SdkPojoBuilder(InlineAgentResponseStream.UNKNOWN)).build());

            Function<String, Optional<ExceptionMetadata>> eventstreamExceptionMetadataMapper = errorCode -> {
                switch (errorCode) {
                case "accessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "badGatewayException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadGatewayException").httpStatusCode(502)
                            .exceptionBuilderSupplier(BadGatewayException::builder).build());
                case "conflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "dependencyFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyFailedException").httpStatusCode(424)
                            .exceptionBuilderSupplier(DependencyFailedException::builder).build());
                case "internalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "resourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "serviceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "throttlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "validationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };

            HttpResponseHandler<AwsServiceException> errorEventResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, eventstreamExceptionMetadataMapper);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            CompletableFuture<Void> future = new CompletableFuture<>();
            EventStreamAsyncResponseTransformer<InvokeInlineAgentResponse, InlineAgentResponseStream> asyncResponseTransformer = EventStreamAsyncResponseTransformer
                    .<InvokeInlineAgentResponse, InlineAgentResponseStream> builder()
                    .eventStreamResponseHandler(asyncResponseHandler).eventResponseHandler(eventResponseHandler)
                    .initialResponseHandler(responseHandler).exceptionResponseHandler(errorEventResponseHandler).future(future)
                    .executor(executor).serviceName(serviceName()).build();
            RestEventStreamAsyncResponseTransformer<InvokeInlineAgentResponse, InlineAgentResponseStream> restAsyncResponseTransformer = RestEventStreamAsyncResponseTransformer
                    .<InvokeInlineAgentResponse, InlineAgentResponseStream> builder()
                    .eventStreamAsyncResponseTransformer(asyncResponseTransformer)
                    .eventStreamResponseHandler(asyncResponseHandler).build();

            CompletableFuture<Void> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<InvokeInlineAgentRequest, InvokeInlineAgentResponse>()
                            .withOperationName("InvokeInlineAgent").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new InvokeInlineAgentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(invokeInlineAgentRequest), restAsyncResponseTransformer);
            CompletableFuture<Void> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    try {
                        asyncResponseHandler.exceptionOccurred(e);
                    } finally {
                        future.completeExceptionally(e);
                    }
                }
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return CompletableFutureUtils.forwardExceptionTo(future, executeFuture);
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseHandler.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all invocation steps associated with a session and optionally, an invocation within the session. For more
     * information about sessions, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history
     * and context with Amazon Bedrock sessions</a>.
     * </p>
     *
     * @param listInvocationStepsRequest
     * @return A Java Future containing the result of the ListInvocationSteps operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.ListInvocationSteps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/ListInvocationSteps"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListInvocationStepsResponse> listInvocationSteps(
            ListInvocationStepsRequest listInvocationStepsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInvocationStepsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInvocationStepsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInvocationSteps");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInvocationStepsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListInvocationStepsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInvocationStepsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInvocationStepsRequest, ListInvocationStepsResponse>()
                            .withOperationName("ListInvocationSteps").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListInvocationStepsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listInvocationStepsRequest));
            CompletableFuture<ListInvocationStepsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all invocations associated with a specific session. For more information about sessions, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history
     * and context with Amazon Bedrock sessions</a>.
     * </p>
     *
     * @param listInvocationsRequest
     * @return A Java Future containing the result of the ListInvocations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.ListInvocations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/ListInvocations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListInvocationsResponse> listInvocations(ListInvocationsRequest listInvocationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInvocationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInvocationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInvocations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInvocationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListInvocationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInvocationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInvocationsRequest, ListInvocationsResponse>()
                            .withOperationName("ListInvocations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListInvocationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listInvocationsRequest));
            CompletableFuture<ListInvocationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all sessions in your Amazon Web Services account. For more information about sessions, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history
     * and context with Amazon Bedrock sessions</a>.
     * </p>
     *
     * @param listSessionsRequest
     * @return A Java Future containing the result of the ListSessions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.ListSessions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/ListSessions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSessionsResponse> listSessions(ListSessionsRequest listSessionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSessionsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSessionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSessions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSessionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListSessionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSessionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSessionsRequest, ListSessionsResponse>()
                            .withOperationName("ListSessions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListSessionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listSessionsRequest));
            CompletableFuture<ListSessionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all the tags for the resource you specify.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Optimizes a prompt for the task that you specify. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-management-optimize.html">Optimize a prompt</a>
     * in the <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Amazon Bedrock User
     * Guide</a>.
     * </p>
     *
     * @param optimizePromptRequest
     * @return A Java Future containing the result of the OptimizePrompt operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>DependencyFailedException There was an issue with a dependency. Check the resource configurations and
     *         retry the request.</li>
     *         <li>BadGatewayException There was an issue with a dependency due to a server issue. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.OptimizePrompt
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/OptimizePrompt"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<Void> optimizePrompt(OptimizePromptRequest optimizePromptRequest,
            OptimizePromptResponseHandler asyncResponseHandler) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(optimizePromptRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, optimizePromptRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "OptimizePrompt");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<OptimizePromptResponse> responseHandler = new AttachHttpMetadataResponseHandler(
                    protocolFactory.createResponseHandler(operationMetadata, OptimizePromptResponse::builder));

            HttpResponseHandler<SdkResponse> voidResponseHandler = protocolFactory.createResponseHandler(JsonOperationMetadata
                    .builder().isPayloadJson(false).hasStreamingSuccessResponse(true).build(), VoidSdkResponse::builder);

            HttpResponseHandler<? extends OptimizedPromptStream> eventResponseHandler = protocolFactory.createResponseHandler(
                    JsonOperationMetadata.builder().isPayloadJson(true).hasStreamingSuccessResponse(false).build(),
                    EventStreamTaggedUnionPojoSupplier.builder()
                            .putSdkPojoSupplier("analyzePromptEvent", OptimizedPromptStream::analyzePromptEventBuilder)
                            .putSdkPojoSupplier("optimizedPromptEvent", OptimizedPromptStream::optimizedPromptEventBuilder)
                            .defaultSdkPojoSupplier(() -> new SdkPojoBuilder(OptimizedPromptStream.UNKNOWN)).build());

            Function<String, Optional<ExceptionMetadata>> eventstreamExceptionMetadataMapper = errorCode -> {
                switch (errorCode) {
                case "accessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "badGatewayException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadGatewayException").httpStatusCode(502)
                            .exceptionBuilderSupplier(BadGatewayException::builder).build());
                case "dependencyFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyFailedException").httpStatusCode(424)
                            .exceptionBuilderSupplier(DependencyFailedException::builder).build());
                case "internalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "throttlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "validationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };

            HttpResponseHandler<AwsServiceException> errorEventResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, eventstreamExceptionMetadataMapper);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            CompletableFuture<Void> future = new CompletableFuture<>();
            EventStreamAsyncResponseTransformer<OptimizePromptResponse, OptimizedPromptStream> asyncResponseTransformer = EventStreamAsyncResponseTransformer
                    .<OptimizePromptResponse, OptimizedPromptStream> builder().eventStreamResponseHandler(asyncResponseHandler)
                    .eventResponseHandler(eventResponseHandler).initialResponseHandler(responseHandler)
                    .exceptionResponseHandler(errorEventResponseHandler).future(future).executor(executor)
                    .serviceName(serviceName()).build();
            RestEventStreamAsyncResponseTransformer<OptimizePromptResponse, OptimizedPromptStream> restAsyncResponseTransformer = RestEventStreamAsyncResponseTransformer
                    .<OptimizePromptResponse, OptimizedPromptStream> builder()
                    .eventStreamAsyncResponseTransformer(asyncResponseTransformer)
                    .eventStreamResponseHandler(asyncResponseHandler).build();

            CompletableFuture<Void> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<OptimizePromptRequest, OptimizePromptResponse>()
                            .withOperationName("OptimizePrompt").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new OptimizePromptRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(optimizePromptRequest), restAsyncResponseTransformer);
            CompletableFuture<Void> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    try {
                        asyncResponseHandler.exceptionOccurred(e);
                    } finally {
                        future.completeExceptionally(e);
                    }
                }
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return CompletableFutureUtils.forwardExceptionTo(future, executeFuture);
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseHandler.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Add an invocation step to an invocation in a session. An invocation step stores fine-grained state checkpoints,
     * including text and images, for each interaction. For more information about sessions, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history
     * and context with Amazon Bedrock sessions</a>.
     * </p>
     * <p>
     * Related APIs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_GetInvocationStep.html">
     * GetInvocationStep</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_ListInvocationSteps.html">
     * ListInvocationSteps</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_ListInvocations.html">
     * ListInvocations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_ListInvocations.html">
     * ListSessions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param putInvocationStepRequest
     * @return A Java Future containing the result of the PutInvocationStep operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.PutInvocationStep
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/PutInvocationStep"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutInvocationStepResponse> putInvocationStep(PutInvocationStepRequest putInvocationStepRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putInvocationStepRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putInvocationStepRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutInvocationStep");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutInvocationStepResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutInvocationStepResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutInvocationStepResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutInvocationStepRequest, PutInvocationStepResponse>()
                            .withOperationName("PutInvocationStep").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new PutInvocationStepRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putInvocationStepRequest));
            CompletableFuture<PutInvocationStepResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Reranks the relevance of sources based on queries. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/rerank.html">Improve the relevance of query responses
     * with a reranker model</a>.
     * </p>
     *
     * @param rerankRequest
     * @return A Java Future containing the result of the Rerank operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>DependencyFailedException There was an issue with a dependency. Check the resource configurations and
     *         retry the request.</li>
     *         <li>BadGatewayException There was an issue with a dependency due to a server issue. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.Rerank
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/Rerank" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RerankResponse> rerank(RerankRequest rerankRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(rerankRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rerankRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Rerank");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RerankResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RerankResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RerankResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RerankRequest, RerankResponse>().withOperationName("Rerank")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new RerankRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(rerankRequest));
            CompletableFuture<RerankResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Queries a knowledge base and retrieves information from it.
     * </p>
     *
     * @param retrieveRequest
     * @return A Java Future containing the result of the Retrieve operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>DependencyFailedException There was an issue with a dependency. Check the resource configurations and
     *         retry the request.</li>
     *         <li>BadGatewayException There was an issue with a dependency due to a server issue. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.Retrieve
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/Retrieve"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RetrieveResponse> retrieve(RetrieveRequest retrieveRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(retrieveRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, retrieveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Retrieve");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RetrieveResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RetrieveResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RetrieveResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RetrieveRequest, RetrieveResponse>().withOperationName("Retrieve")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RetrieveRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(retrieveRequest));
            CompletableFuture<RetrieveResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Queries a knowledge base and generates responses based on the retrieved results and using the specified
     * foundation model or <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html">inference profile</a>.
     * The response only cites sources that are relevant to the query.
     * </p>
     *
     * @param retrieveAndGenerateRequest
     * @return A Java Future containing the result of the RetrieveAndGenerate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>DependencyFailedException There was an issue with a dependency. Check the resource configurations and
     *         retry the request.</li>
     *         <li>BadGatewayException There was an issue with a dependency due to a server issue. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.RetrieveAndGenerate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/RetrieveAndGenerate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RetrieveAndGenerateResponse> retrieveAndGenerate(
            RetrieveAndGenerateRequest retrieveAndGenerateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(retrieveAndGenerateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, retrieveAndGenerateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RetrieveAndGenerate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RetrieveAndGenerateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RetrieveAndGenerateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RetrieveAndGenerateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RetrieveAndGenerateRequest, RetrieveAndGenerateResponse>()
                            .withOperationName("RetrieveAndGenerate").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RetrieveAndGenerateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(retrieveAndGenerateRequest));
            CompletableFuture<RetrieveAndGenerateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Queries a knowledge base and generates responses based on the retrieved results, with output in streaming format.
     * </p>
     * <note>
     * <p>
     * The CLI doesn't support streaming operations in Amazon Bedrock, including
     * <code>InvokeModelWithResponseStream</code>.
     * </p>
     * </note>
     * <p>
     * This operation requires permission for the <code> bedrock:RetrieveAndGenerate</code> action.
     * </p>
     *
     * @param retrieveAndGenerateStreamRequest
     * @return A Java Future containing the result of the RetrieveAndGenerateStream operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>DependencyFailedException There was an issue with a dependency. Check the resource configurations and
     *         retry the request.</li>
     *         <li>BadGatewayException There was an issue with a dependency due to a server issue. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.RetrieveAndGenerateStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/RetrieveAndGenerateStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<Void> retrieveAndGenerateStream(RetrieveAndGenerateStreamRequest retrieveAndGenerateStreamRequest,
            RetrieveAndGenerateStreamResponseHandler asyncResponseHandler) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(retrieveAndGenerateStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, retrieveAndGenerateStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RetrieveAndGenerateStream");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RetrieveAndGenerateStreamResponse> responseHandler = new AttachHttpMetadataResponseHandler(
                    protocolFactory.createResponseHandler(operationMetadata, RetrieveAndGenerateStreamResponse::builder));

            HttpResponseHandler<SdkResponse> voidResponseHandler = protocolFactory.createResponseHandler(JsonOperationMetadata
                    .builder().isPayloadJson(false).hasStreamingSuccessResponse(true).build(), VoidSdkResponse::builder);

            HttpResponseHandler<? extends RetrieveAndGenerateStreamResponseOutput> eventResponseHandler = protocolFactory
                    .createResponseHandler(
                            JsonOperationMetadata.builder().isPayloadJson(true).hasStreamingSuccessResponse(false).build(),
                            EventStreamTaggedUnionPojoSupplier
                                    .builder()
                                    .putSdkPojoSupplier("citation", RetrieveAndGenerateStreamResponseOutput::citationBuilder)
                                    .putSdkPojoSupplier("guardrail", RetrieveAndGenerateStreamResponseOutput::guardrailBuilder)
                                    .putSdkPojoSupplier("output", RetrieveAndGenerateStreamResponseOutput::outputBuilder)
                                    .defaultSdkPojoSupplier(
                                            () -> new SdkPojoBuilder(RetrieveAndGenerateStreamResponseOutput.UNKNOWN)).build());

            Function<String, Optional<ExceptionMetadata>> eventstreamExceptionMetadataMapper = errorCode -> {
                switch (errorCode) {
                case "accessDeniedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(AccessDeniedException::builder).build());
                case "badGatewayException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("BadGatewayException").httpStatusCode(502)
                            .exceptionBuilderSupplier(BadGatewayException::builder).build());
                case "conflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "dependencyFailedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DependencyFailedException").httpStatusCode(424)
                            .exceptionBuilderSupplier(DependencyFailedException::builder).build());
                case "internalServerException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServerException::builder).build());
                case "resourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "serviceQuotaExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
                case "throttlingException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ThrottlingException::builder).build());
                case "validationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                            .exceptionBuilderSupplier(ValidationException::builder).build());
                default:
                    return Optional.empty();
                }
            };

            HttpResponseHandler<AwsServiceException> errorEventResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, eventstreamExceptionMetadataMapper);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            CompletableFuture<Void> future = new CompletableFuture<>();
            EventStreamAsyncResponseTransformer<RetrieveAndGenerateStreamResponse, RetrieveAndGenerateStreamResponseOutput> asyncResponseTransformer = EventStreamAsyncResponseTransformer
                    .<RetrieveAndGenerateStreamResponse, RetrieveAndGenerateStreamResponseOutput> builder()
                    .eventStreamResponseHandler(asyncResponseHandler).eventResponseHandler(eventResponseHandler)
                    .initialResponseHandler(responseHandler).exceptionResponseHandler(errorEventResponseHandler).future(future)
                    .executor(executor).serviceName(serviceName()).build();
            RestEventStreamAsyncResponseTransformer<RetrieveAndGenerateStreamResponse, RetrieveAndGenerateStreamResponseOutput> restAsyncResponseTransformer = RestEventStreamAsyncResponseTransformer
                    .<RetrieveAndGenerateStreamResponse, RetrieveAndGenerateStreamResponseOutput> builder()
                    .eventStreamAsyncResponseTransformer(asyncResponseTransformer)
                    .eventStreamResponseHandler(asyncResponseHandler).build();

            CompletableFuture<Void> executeFuture = clientHandler.execute(
                    new ClientExecutionParams<RetrieveAndGenerateStreamRequest, RetrieveAndGenerateStreamResponse>()
                            .withOperationName("RetrieveAndGenerateStream").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new RetrieveAndGenerateStreamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(retrieveAndGenerateStreamRequest), restAsyncResponseTransformer);
            CompletableFuture<Void> whenCompleted = executeFuture.whenComplete((r, e) -> {
                if (e != null) {
                    try {
                        asyncResponseHandler.exceptionOccurred(e);
                    } finally {
                        future.completeExceptionally(e);
                    }
                }
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return CompletableFutureUtils.forwardExceptionTo(future, executeFuture);
        } catch (Throwable t) {
            runAndLogError(log, "Exception thrown in exceptionOccurred callback, ignoring",
                    () -> asyncResponseHandler.exceptionOccurred(t));
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Associate tags with a resource. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Tagging resources</a> in the
     * Amazon Bedrock User Guide.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the metadata or encryption settings of a session. For more information about sessions, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history
     * and context with Amazon Bedrock sessions</a>.
     * </p>
     *
     * @param updateSessionRequest
     * @return A Java Future containing the result of the UpdateSession operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException There was a conflict performing an operation. Resolve the conflict and retry your
     *         request.</li>
     *         <li>ResourceNotFoundException The specified resource Amazon Resource Name (ARN) was not found. Check the
     *         Amazon Resource Name (ARN) and try your request again.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>AccessDeniedException The request is denied because of missing access permissions. Check your
     *         permissions and retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockAgentRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample BedrockAgentRuntimeAsyncClient.UpdateSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/UpdateSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSessionResponse> updateSession(UpdateSessionRequest updateSessionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSession");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateSessionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSessionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSessionRequest, UpdateSessionResponse>()
                            .withOperationName("UpdateSession").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateSessionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateSessionRequest));
            CompletableFuture<UpdateSessionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final BedrockAgentRuntimeServiceClientConfiguration serviceClientConfiguration() {
        return new BedrockAgentRuntimeServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BedrockAgentRuntimeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ModelNotReadyException")
                                .exceptionBuilderSupplier(ModelNotReadyException::builder).httpStatusCode(424).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependencyFailedException")
                                .exceptionBuilderSupplier(DependencyFailedException::builder).httpStatusCode(424).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadGatewayException")
                                .exceptionBuilderSupplier(BadGatewayException::builder).httpStatusCode(502).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        BedrockAgentRuntimeServiceClientConfigurationBuilder serviceConfigBuilder = new BedrockAgentRuntimeServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
