/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagentruntime;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.bedrockagentruntime.internal.BedrockAgentRuntimeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.bedrockagentruntime.model.AccessDeniedException;
import software.amazon.awssdk.services.bedrockagentruntime.model.BadGatewayException;
import software.amazon.awssdk.services.bedrockagentruntime.model.BedrockAgentRuntimeException;
import software.amazon.awssdk.services.bedrockagentruntime.model.ConflictException;
import software.amazon.awssdk.services.bedrockagentruntime.model.CreateInvocationRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.CreateInvocationResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.CreateSessionRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.CreateSessionResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.DeleteAgentMemoryRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.DeleteAgentMemoryResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.DeleteSessionRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.DeleteSessionResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.DependencyFailedException;
import software.amazon.awssdk.services.bedrockagentruntime.model.EndSessionRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.EndSessionResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.GenerateQueryRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.GenerateQueryResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.GetAgentMemoryRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.GetAgentMemoryResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.GetInvocationStepRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.GetInvocationStepResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.GetSessionRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.GetSessionResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.InternalServerException;
import software.amazon.awssdk.services.bedrockagentruntime.model.ListInvocationStepsRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.ListInvocationStepsResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.ListInvocationsRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.ListInvocationsResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.ListSessionsRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.ListSessionsResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.ModelNotReadyException;
import software.amazon.awssdk.services.bedrockagentruntime.model.PutInvocationStepRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.PutInvocationStepResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.RerankRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.RerankResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.ResourceNotFoundException;
import software.amazon.awssdk.services.bedrockagentruntime.model.RetrieveAndGenerateRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.RetrieveAndGenerateResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.RetrieveRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.RetrieveResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.bedrockagentruntime.model.TagResourceRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.TagResourceResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.ThrottlingException;
import software.amazon.awssdk.services.bedrockagentruntime.model.UntagResourceRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.UntagResourceResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.UpdateSessionRequest;
import software.amazon.awssdk.services.bedrockagentruntime.model.UpdateSessionResponse;
import software.amazon.awssdk.services.bedrockagentruntime.model.ValidationException;
import software.amazon.awssdk.services.bedrockagentruntime.transform.CreateInvocationRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.CreateSessionRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.DeleteAgentMemoryRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.DeleteSessionRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.EndSessionRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.GenerateQueryRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.GetAgentMemoryRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.GetInvocationStepRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.GetSessionRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.ListInvocationStepsRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.ListInvocationsRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.ListSessionsRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.PutInvocationStepRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.RerankRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.RetrieveAndGenerateRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.RetrieveRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.bedrockagentruntime.transform.UpdateSessionRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link BedrockAgentRuntimeClient}.
 *
 * @see BedrockAgentRuntimeClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBedrockAgentRuntimeClient implements BedrockAgentRuntimeClient {
    private static final Logger log = Logger.loggerFor(DefaultBedrockAgentRuntimeClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBedrockAgentRuntimeClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a new invocation within a session. An invocation groups the related invocation steps that store the
     * content from a conversation. For more information about sessions, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history
     * and context with Amazon Bedrock sessions</a>.
     * </p>
     * <p>
     * Related APIs
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_ListInvocations.html">
     * ListInvocations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_ListSessions.html">ListSessions
     * </a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_GetSession.html">GetSession</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createInvocationRequest
     * @return Result of the CreateInvocation operation returned by the service.
     * @throws ConflictException
     *         There was a conflict performing an operation. Resolve the conflict and retry your request.
     * @throws ResourceNotFoundException
     *         The specified resource Amazon Resource Name (ARN) was not found. Check the Amazon Resource Name (ARN) and
     *         try your request again.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws ServiceQuotaExceededException
     *         The number of requests exceeds the service quota. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.CreateInvocation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/CreateInvocation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateInvocationResponse createInvocation(CreateInvocationRequest createInvocationRequest) throws ConflictException,
            ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException, AccessDeniedException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateInvocationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateInvocationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createInvocationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createInvocationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateInvocation");

            return clientHandler.execute(new ClientExecutionParams<CreateInvocationRequest, CreateInvocationResponse>()
                    .withOperationName("CreateInvocation").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createInvocationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateInvocationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a session to temporarily store conversations for generative AI (GenAI) applications built with
     * open-source frameworks such as LangGraph and LlamaIndex. Sessions enable you to save the state of conversations
     * at checkpoints, with the added security and infrastructure of Amazon Web Services. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history
     * and context with Amazon Bedrock sessions</a>.
     * </p>
     * <p>
     * By default, Amazon Bedrock uses Amazon Web Services-managed keys for session encryption, including session
     * metadata, or you can use your own KMS key. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/session-encryption.html">Amazon Bedrock session
     * encryption</a>.
     * </p>
     * <note>
     * <p>
     * You use a session to store state and conversation history for generative AI applications built with open-source
     * frameworks. For Amazon Bedrock Agents, the service automatically manages conversation context and associates them
     * with the agent-specific sessionId you specify in the <a
     * href="https://docs.aws.amazon.com/bedrock/latest/API_agent-runtime_InvokeAgent.html">InvokeAgent</a> API
     * operation.
     * </p>
     * </note>
     * <p>
     * Related APIs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_ListSessions.html">ListSessions
     * </a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_GetSession.html">GetSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_EndSession.html">EndSession</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_DeleteSession.html">
     * DeleteSession</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createSessionRequest
     * @return Result of the CreateSession operation returned by the service.
     * @throws ConflictException
     *         There was a conflict performing an operation. Resolve the conflict and retry your request.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws ServiceQuotaExceededException
     *         The number of requests exceeds the service quota. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.CreateSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/CreateSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateSessionResponse createSession(CreateSessionRequest createSessionRequest) throws ConflictException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateSessionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSession");

            return clientHandler.execute(new ClientExecutionParams<CreateSessionRequest, CreateSessionResponse>()
                    .withOperationName("CreateSession").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createSessionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateSessionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes memory from the specified memory identifier.
     * </p>
     *
     * @param deleteAgentMemoryRequest
     * @return Result of the DeleteAgentMemory operation returned by the service.
     * @throws ConflictException
     *         There was a conflict performing an operation. Resolve the conflict and retry your request.
     * @throws ResourceNotFoundException
     *         The specified resource Amazon Resource Name (ARN) was not found. Check the Amazon Resource Name (ARN) and
     *         try your request again.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws DependencyFailedException
     *         There was an issue with a dependency. Check the resource configurations and retry the request.
     * @throws BadGatewayException
     *         There was an issue with a dependency due to a server issue. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws ServiceQuotaExceededException
     *         The number of requests exceeds the service quota. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.DeleteAgentMemory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/DeleteAgentMemory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAgentMemoryResponse deleteAgentMemory(DeleteAgentMemoryRequest deleteAgentMemoryRequest)
            throws ConflictException, ResourceNotFoundException, ValidationException, InternalServerException,
            DependencyFailedException, BadGatewayException, ThrottlingException, AccessDeniedException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAgentMemoryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAgentMemoryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAgentMemoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAgentMemoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAgentMemory");

            return clientHandler.execute(new ClientExecutionParams<DeleteAgentMemoryRequest, DeleteAgentMemoryResponse>()
                    .withOperationName("DeleteAgentMemory").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteAgentMemoryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAgentMemoryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a session that you ended. You can't delete a session with an <code>ACTIVE</code> status. To delete an
     * active session, you must first end it with the <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_EndSession.html">EndSession</a>
     * API operation. For more information about sessions, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history
     * and context with Amazon Bedrock sessions</a>.
     * </p>
     *
     * @param deleteSessionRequest
     * @return Result of the DeleteSession operation returned by the service.
     * @throws ConflictException
     *         There was a conflict performing an operation. Resolve the conflict and retry your request.
     * @throws ResourceNotFoundException
     *         The specified resource Amazon Resource Name (ARN) was not found. Check the Amazon Resource Name (ARN) and
     *         try your request again.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.DeleteSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/DeleteSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteSessionResponse deleteSession(DeleteSessionRequest deleteSessionRequest) throws ConflictException,
            ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteSessionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSession");

            return clientHandler.execute(new ClientExecutionParams<DeleteSessionRequest, DeleteSessionResponse>()
                    .withOperationName("DeleteSession").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteSessionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSessionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Ends the session. After you end a session, you can still access its content but you can’t add to it. To delete
     * the session and it's content, you use the DeleteSession API operation. For more information about sessions, see
     * <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation
     * history and context with Amazon Bedrock sessions</a>.
     * </p>
     *
     * @param endSessionRequest
     * @return Result of the EndSession operation returned by the service.
     * @throws ConflictException
     *         There was a conflict performing an operation. Resolve the conflict and retry your request.
     * @throws ResourceNotFoundException
     *         The specified resource Amazon Resource Name (ARN) was not found. Check the Amazon Resource Name (ARN) and
     *         try your request again.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.EndSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/EndSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public EndSessionResponse endSession(EndSessionRequest endSessionRequest) throws ConflictException,
            ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EndSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                EndSessionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(endSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, endSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EndSession");

            return clientHandler.execute(new ClientExecutionParams<EndSessionRequest, EndSessionResponse>()
                    .withOperationName("EndSession").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(endSessionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new EndSessionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Generates an SQL query from a natural language query. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/knowledge-base-generate-query.html">Generate a query
     * for structured data</a> in the Amazon Bedrock User Guide.
     * </p>
     *
     * @param generateQueryRequest
     * @return Result of the GenerateQuery operation returned by the service.
     * @throws ConflictException
     *         There was a conflict performing an operation. Resolve the conflict and retry your request.
     * @throws ResourceNotFoundException
     *         The specified resource Amazon Resource Name (ARN) was not found. Check the Amazon Resource Name (ARN) and
     *         try your request again.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws DependencyFailedException
     *         There was an issue with a dependency. Check the resource configurations and retry the request.
     * @throws BadGatewayException
     *         There was an issue with a dependency due to a server issue. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws ServiceQuotaExceededException
     *         The number of requests exceeds the service quota. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.GenerateQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/GenerateQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GenerateQueryResponse generateQuery(GenerateQueryRequest generateQueryRequest) throws ConflictException,
            ResourceNotFoundException, ValidationException, InternalServerException, DependencyFailedException,
            BadGatewayException, ThrottlingException, AccessDeniedException, ServiceQuotaExceededException, AwsServiceException,
            SdkClientException, BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GenerateQueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GenerateQueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(generateQueryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, generateQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GenerateQuery");

            return clientHandler.execute(new ClientExecutionParams<GenerateQueryRequest, GenerateQueryResponse>()
                    .withOperationName("GenerateQuery").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(generateQueryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GenerateQueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the sessions stored in the memory of the agent.
     * </p>
     *
     * @param getAgentMemoryRequest
     * @return Result of the GetAgentMemory operation returned by the service.
     * @throws ConflictException
     *         There was a conflict performing an operation. Resolve the conflict and retry your request.
     * @throws ResourceNotFoundException
     *         The specified resource Amazon Resource Name (ARN) was not found. Check the Amazon Resource Name (ARN) and
     *         try your request again.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws DependencyFailedException
     *         There was an issue with a dependency. Check the resource configurations and retry the request.
     * @throws BadGatewayException
     *         There was an issue with a dependency due to a server issue. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws ServiceQuotaExceededException
     *         The number of requests exceeds the service quota. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.GetAgentMemory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/GetAgentMemory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAgentMemoryResponse getAgentMemory(GetAgentMemoryRequest getAgentMemoryRequest) throws ConflictException,
            ResourceNotFoundException, ValidationException, InternalServerException, DependencyFailedException,
            BadGatewayException, ThrottlingException, AccessDeniedException, ServiceQuotaExceededException, AwsServiceException,
            SdkClientException, BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAgentMemoryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAgentMemoryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAgentMemoryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAgentMemoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAgentMemory");

            return clientHandler.execute(new ClientExecutionParams<GetAgentMemoryRequest, GetAgentMemoryResponse>()
                    .withOperationName("GetAgentMemory").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getAgentMemoryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAgentMemoryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the details of a specific invocation step within an invocation in a session. For more information about
     * sessions, see <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve
     * conversation history and context with Amazon Bedrock sessions</a>.
     * </p>
     *
     * @param getInvocationStepRequest
     * @return Result of the GetInvocationStep operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource Amazon Resource Name (ARN) was not found. Check the Amazon Resource Name (ARN) and
     *         try your request again.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.GetInvocationStep
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/GetInvocationStep"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetInvocationStepResponse getInvocationStep(GetInvocationStepRequest getInvocationStepRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetInvocationStepResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetInvocationStepResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getInvocationStepRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInvocationStepRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInvocationStep");

            return clientHandler.execute(new ClientExecutionParams<GetInvocationStepRequest, GetInvocationStepResponse>()
                    .withOperationName("GetInvocationStep").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getInvocationStepRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetInvocationStepRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves details about a specific session. For more information about sessions, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history
     * and context with Amazon Bedrock sessions</a>.
     * </p>
     *
     * @param getSessionRequest
     * @return Result of the GetSession operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource Amazon Resource Name (ARN) was not found. Check the Amazon Resource Name (ARN) and
     *         try your request again.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.GetSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/GetSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSessionResponse getSession(GetSessionRequest getSessionRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSessionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSession");

            return clientHandler.execute(new ClientExecutionParams<GetSessionRequest, GetSessionResponse>()
                    .withOperationName("GetSession").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getSessionRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetSessionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all invocation steps associated with a session and optionally, an invocation within the session. For more
     * information about sessions, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history
     * and context with Amazon Bedrock sessions</a>.
     * </p>
     *
     * @param listInvocationStepsRequest
     * @return Result of the ListInvocationSteps operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource Amazon Resource Name (ARN) was not found. Check the Amazon Resource Name (ARN) and
     *         try your request again.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.ListInvocationSteps
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/ListInvocationSteps"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListInvocationStepsResponse listInvocationSteps(ListInvocationStepsRequest listInvocationStepsRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInvocationStepsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListInvocationStepsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInvocationStepsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInvocationStepsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInvocationSteps");

            return clientHandler.execute(new ClientExecutionParams<ListInvocationStepsRequest, ListInvocationStepsResponse>()
                    .withOperationName("ListInvocationSteps").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listInvocationStepsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListInvocationStepsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all invocations associated with a specific session. For more information about sessions, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history
     * and context with Amazon Bedrock sessions</a>.
     * </p>
     *
     * @param listInvocationsRequest
     * @return Result of the ListInvocations operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource Amazon Resource Name (ARN) was not found. Check the Amazon Resource Name (ARN) and
     *         try your request again.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.ListInvocations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/ListInvocations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListInvocationsResponse listInvocations(ListInvocationsRequest listInvocationsRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListInvocationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListInvocationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listInvocationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInvocationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInvocations");

            return clientHandler.execute(new ClientExecutionParams<ListInvocationsRequest, ListInvocationsResponse>()
                    .withOperationName("ListInvocations").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listInvocationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListInvocationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all sessions in your Amazon Web Services account. For more information about sessions, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history
     * and context with Amazon Bedrock sessions</a>.
     * </p>
     *
     * @param listSessionsRequest
     * @return Result of the ListSessions operation returned by the service.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.ListSessions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/ListSessions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSessionsResponse listSessions(ListSessionsRequest listSessionsRequest) throws ValidationException,
            InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException,
            BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSessionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListSessionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSessionsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSessionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSessions");

            return clientHandler.execute(new ClientExecutionParams<ListSessionsRequest, ListSessionsResponse>()
                    .withOperationName("ListSessions").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listSessionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListSessionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List all the tags for the resource you specify.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource Amazon Resource Name (ARN) was not found. Check the Amazon Resource Name (ARN) and
     *         try your request again.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, AwsServiceException, SdkClientException, BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Add an invocation step to an invocation in a session. An invocation step stores fine-grained state checkpoints,
     * including text and images, for each interaction. For more information about sessions, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history
     * and context with Amazon Bedrock sessions</a>.
     * </p>
     * <p>
     * Related APIs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_GetInvocationStep.html">
     * GetInvocationStep</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_ListInvocationSteps.html">
     * ListInvocationSteps</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_ListInvocations.html">
     * ListInvocations</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent-runtime_ListInvocations.html">
     * ListSessions</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param putInvocationStepRequest
     * @return Result of the PutInvocationStep operation returned by the service.
     * @throws ConflictException
     *         There was a conflict performing an operation. Resolve the conflict and retry your request.
     * @throws ResourceNotFoundException
     *         The specified resource Amazon Resource Name (ARN) was not found. Check the Amazon Resource Name (ARN) and
     *         try your request again.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws ServiceQuotaExceededException
     *         The number of requests exceeds the service quota. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.PutInvocationStep
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/PutInvocationStep"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutInvocationStepResponse putInvocationStep(PutInvocationStepRequest putInvocationStepRequest)
            throws ConflictException, ResourceNotFoundException, ValidationException, InternalServerException,
            ThrottlingException, AccessDeniedException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutInvocationStepResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutInvocationStepResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putInvocationStepRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putInvocationStepRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutInvocationStep");

            return clientHandler.execute(new ClientExecutionParams<PutInvocationStepRequest, PutInvocationStepResponse>()
                    .withOperationName("PutInvocationStep").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putInvocationStepRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutInvocationStepRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Reranks the relevance of sources based on queries. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/rerank.html">Improve the relevance of query responses
     * with a reranker model</a>.
     * </p>
     *
     * @param rerankRequest
     * @return Result of the Rerank operation returned by the service.
     * @throws ConflictException
     *         There was a conflict performing an operation. Resolve the conflict and retry your request.
     * @throws ResourceNotFoundException
     *         The specified resource Amazon Resource Name (ARN) was not found. Check the Amazon Resource Name (ARN) and
     *         try your request again.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws DependencyFailedException
     *         There was an issue with a dependency. Check the resource configurations and retry the request.
     * @throws BadGatewayException
     *         There was an issue with a dependency due to a server issue. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws ServiceQuotaExceededException
     *         The number of requests exceeds the service quota. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.Rerank
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/Rerank" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RerankResponse rerank(RerankRequest rerankRequest) throws ConflictException, ResourceNotFoundException,
            ValidationException, InternalServerException, DependencyFailedException, BadGatewayException, ThrottlingException,
            AccessDeniedException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RerankResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RerankResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(rerankRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rerankRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Rerank");

            return clientHandler.execute(new ClientExecutionParams<RerankRequest, RerankResponse>().withOperationName("Rerank")
                    .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(rerankRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RerankRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Queries a knowledge base and retrieves information from it.
     * </p>
     *
     * @param retrieveRequest
     * @return Result of the Retrieve operation returned by the service.
     * @throws ConflictException
     *         There was a conflict performing an operation. Resolve the conflict and retry your request.
     * @throws ResourceNotFoundException
     *         The specified resource Amazon Resource Name (ARN) was not found. Check the Amazon Resource Name (ARN) and
     *         try your request again.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws DependencyFailedException
     *         There was an issue with a dependency. Check the resource configurations and retry the request.
     * @throws BadGatewayException
     *         There was an issue with a dependency due to a server issue. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws ServiceQuotaExceededException
     *         The number of requests exceeds the service quota. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.Retrieve
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/Retrieve"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RetrieveResponse retrieve(RetrieveRequest retrieveRequest) throws ConflictException, ResourceNotFoundException,
            ValidationException, InternalServerException, DependencyFailedException, BadGatewayException, ThrottlingException,
            AccessDeniedException, ServiceQuotaExceededException, AwsServiceException, SdkClientException,
            BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RetrieveResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RetrieveResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(retrieveRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, retrieveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Retrieve");

            return clientHandler.execute(new ClientExecutionParams<RetrieveRequest, RetrieveResponse>()
                    .withOperationName("Retrieve").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(retrieveRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RetrieveRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Queries a knowledge base and generates responses based on the retrieved results and using the specified
     * foundation model or <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html">inference profile</a>.
     * The response only cites sources that are relevant to the query.
     * </p>
     *
     * @param retrieveAndGenerateRequest
     * @return Result of the RetrieveAndGenerate operation returned by the service.
     * @throws ConflictException
     *         There was a conflict performing an operation. Resolve the conflict and retry your request.
     * @throws ResourceNotFoundException
     *         The specified resource Amazon Resource Name (ARN) was not found. Check the Amazon Resource Name (ARN) and
     *         try your request again.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws DependencyFailedException
     *         There was an issue with a dependency. Check the resource configurations and retry the request.
     * @throws BadGatewayException
     *         There was an issue with a dependency due to a server issue. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws ServiceQuotaExceededException
     *         The number of requests exceeds the service quota. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.RetrieveAndGenerate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/RetrieveAndGenerate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RetrieveAndGenerateResponse retrieveAndGenerate(RetrieveAndGenerateRequest retrieveAndGenerateRequest)
            throws ConflictException, ResourceNotFoundException, ValidationException, InternalServerException,
            DependencyFailedException, BadGatewayException, ThrottlingException, AccessDeniedException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RetrieveAndGenerateResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RetrieveAndGenerateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(retrieveAndGenerateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, retrieveAndGenerateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RetrieveAndGenerate");

            return clientHandler.execute(new ClientExecutionParams<RetrieveAndGenerateRequest, RetrieveAndGenerateResponse>()
                    .withOperationName("RetrieveAndGenerate").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(retrieveAndGenerateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RetrieveAndGenerateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associate tags with a resource. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Tagging resources</a> in the
     * Amazon Bedrock User Guide.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource Amazon Resource Name (ARN) was not found. Check the Amazon Resource Name (ARN) and
     *         try your request again.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws ServiceQuotaExceededException
     *         The number of requests exceeds the service quota. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Remove tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource Amazon Resource Name (ARN) was not found. Check the Amazon Resource Name (ARN) and
     *         try your request again.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, AwsServiceException,
            SdkClientException, BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the metadata or encryption settings of a session. For more information about sessions, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history
     * and context with Amazon Bedrock sessions</a>.
     * </p>
     *
     * @param updateSessionRequest
     * @return Result of the UpdateSession operation returned by the service.
     * @throws ConflictException
     *         There was a conflict performing an operation. Resolve the conflict and retry your request.
     * @throws ResourceNotFoundException
     *         The specified resource Amazon Resource Name (ARN) was not found. Check the Amazon Resource Name (ARN) and
     *         try your request again.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions. Check your permissions and retry your
     *         request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockAgentRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockAgentRuntimeClient.UpdateSession
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-runtime-2023-07-26/UpdateSession"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateSessionResponse updateSession(UpdateSessionRequest updateSessionRequest) throws ConflictException,
            ResourceNotFoundException, ValidationException, InternalServerException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, BedrockAgentRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSessionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateSessionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSessionRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSessionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Agent Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSession");

            return clientHandler.execute(new ClientExecutionParams<UpdateSessionRequest, UpdateSessionResponse>()
                    .withOperationName("UpdateSession").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateSessionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateSessionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        BedrockAgentRuntimeServiceClientConfigurationBuilder serviceConfigBuilder = new BedrockAgentRuntimeServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BedrockAgentRuntimeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ModelNotReadyException")
                                .exceptionBuilderSupplier(ModelNotReadyException::builder).httpStatusCode(424).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DependencyFailedException")
                                .exceptionBuilderSupplier(DependencyFailedException::builder).httpStatusCode(424).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadGatewayException")
                                .exceptionBuilderSupplier(BadGatewayException::builder).httpStatusCode(502).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build());
    }

    @Override
    public final BedrockAgentRuntimeServiceClientConfiguration serviceClientConfiguration() {
        return new BedrockAgentRuntimeServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
