/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagentruntime.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Stores fine-grained state checkpoints, including text and images, for each interaction in an invocation in a session.
 * For more information about sessions, see <a
 * href="https://docs.aws.amazon.com/bedrock/latest/userguide/sessions.html">Store and retrieve conversation history and
 * context with Amazon Bedrock sessions</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InvocationStep implements SdkPojo, Serializable, ToCopyableBuilder<InvocationStep.Builder, InvocationStep> {
    private static final SdkField<String> INVOCATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("invocationId").getter(getter(InvocationStep::invocationId)).setter(setter(Builder::invocationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("invocationId").build()).build();

    private static final SdkField<String> INVOCATION_STEP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("invocationStepId").getter(getter(InvocationStep::invocationStepId))
            .setter(setter(Builder::invocationStepId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("invocationStepId").build()).build();

    private static final SdkField<Instant> INVOCATION_STEP_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("invocationStepTime")
            .getter(getter(InvocationStep::invocationStepTime))
            .setter(setter(Builder::invocationStepTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("invocationStepTime").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<InvocationStepPayload> PAYLOAD_FIELD = SdkField
            .<InvocationStepPayload> builder(MarshallingType.SDK_POJO).memberName("payload")
            .getter(getter(InvocationStep::payload)).setter(setter(Builder::payload)).constructor(InvocationStepPayload::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("payload").build()).build();

    private static final SdkField<String> SESSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("sessionId").getter(getter(InvocationStep::sessionId)).setter(setter(Builder::sessionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sessionId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INVOCATION_ID_FIELD,
            INVOCATION_STEP_ID_FIELD, INVOCATION_STEP_TIME_FIELD, PAYLOAD_FIELD, SESSION_ID_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String invocationId;

    private final String invocationStepId;

    private final Instant invocationStepTime;

    private final InvocationStepPayload payload;

    private final String sessionId;

    private InvocationStep(BuilderImpl builder) {
        this.invocationId = builder.invocationId;
        this.invocationStepId = builder.invocationStepId;
        this.invocationStepTime = builder.invocationStepTime;
        this.payload = builder.payload;
        this.sessionId = builder.sessionId;
    }

    /**
     * <p>
     * The unique identifier (in UUID format) for the invocation that includes the invocation step.
     * </p>
     * 
     * @return The unique identifier (in UUID format) for the invocation that includes the invocation step.
     */
    public final String invocationId() {
        return invocationId;
    }

    /**
     * <p>
     * The unique identifier (in UUID format) for the invocation step.
     * </p>
     * 
     * @return The unique identifier (in UUID format) for the invocation step.
     */
    public final String invocationStepId() {
        return invocationStepId;
    }

    /**
     * <p>
     * The timestamp for when the invocation step was created.
     * </p>
     * 
     * @return The timestamp for when the invocation step was created.
     */
    public final Instant invocationStepTime() {
        return invocationStepTime;
    }

    /**
     * <p>
     * Payload content, such as text and images, for the invocation step.
     * </p>
     * 
     * @return Payload content, such as text and images, for the invocation step.
     */
    public final InvocationStepPayload payload() {
        return payload;
    }

    /**
     * <p>
     * The unique identifier of the session containing the invocation step.
     * </p>
     * 
     * @return The unique identifier of the session containing the invocation step.
     */
    public final String sessionId() {
        return sessionId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(invocationId());
        hashCode = 31 * hashCode + Objects.hashCode(invocationStepId());
        hashCode = 31 * hashCode + Objects.hashCode(invocationStepTime());
        hashCode = 31 * hashCode + Objects.hashCode(payload());
        hashCode = 31 * hashCode + Objects.hashCode(sessionId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InvocationStep)) {
            return false;
        }
        InvocationStep other = (InvocationStep) obj;
        return Objects.equals(invocationId(), other.invocationId())
                && Objects.equals(invocationStepId(), other.invocationStepId())
                && Objects.equals(invocationStepTime(), other.invocationStepTime()) && Objects.equals(payload(), other.payload())
                && Objects.equals(sessionId(), other.sessionId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InvocationStep").add("InvocationId", invocationId()).add("InvocationStepId", invocationStepId())
                .add("InvocationStepTime", invocationStepTime()).add("Payload", payload()).add("SessionId", sessionId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "invocationId":
            return Optional.ofNullable(clazz.cast(invocationId()));
        case "invocationStepId":
            return Optional.ofNullable(clazz.cast(invocationStepId()));
        case "invocationStepTime":
            return Optional.ofNullable(clazz.cast(invocationStepTime()));
        case "payload":
            return Optional.ofNullable(clazz.cast(payload()));
        case "sessionId":
            return Optional.ofNullable(clazz.cast(sessionId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("invocationId", INVOCATION_ID_FIELD);
        map.put("invocationStepId", INVOCATION_STEP_ID_FIELD);
        map.put("invocationStepTime", INVOCATION_STEP_TIME_FIELD);
        map.put("payload", PAYLOAD_FIELD);
        map.put("sessionId", SESSION_ID_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InvocationStep, T> g) {
        return obj -> g.apply((InvocationStep) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InvocationStep> {
        /**
         * <p>
         * The unique identifier (in UUID format) for the invocation that includes the invocation step.
         * </p>
         * 
         * @param invocationId
         *        The unique identifier (in UUID format) for the invocation that includes the invocation step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder invocationId(String invocationId);

        /**
         * <p>
         * The unique identifier (in UUID format) for the invocation step.
         * </p>
         * 
         * @param invocationStepId
         *        The unique identifier (in UUID format) for the invocation step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder invocationStepId(String invocationStepId);

        /**
         * <p>
         * The timestamp for when the invocation step was created.
         * </p>
         * 
         * @param invocationStepTime
         *        The timestamp for when the invocation step was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder invocationStepTime(Instant invocationStepTime);

        /**
         * <p>
         * Payload content, such as text and images, for the invocation step.
         * </p>
         * 
         * @param payload
         *        Payload content, such as text and images, for the invocation step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder payload(InvocationStepPayload payload);

        /**
         * <p>
         * Payload content, such as text and images, for the invocation step.
         * </p>
         * This is a convenience method that creates an instance of the {@link InvocationStepPayload.Builder} avoiding
         * the need to create one manually via {@link InvocationStepPayload#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InvocationStepPayload.Builder#build()} is called immediately and
         * its result is passed to {@link #payload(InvocationStepPayload)}.
         * 
         * @param payload
         *        a consumer that will call methods on {@link InvocationStepPayload.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #payload(InvocationStepPayload)
         */
        default Builder payload(Consumer<InvocationStepPayload.Builder> payload) {
            return payload(InvocationStepPayload.builder().applyMutation(payload).build());
        }

        /**
         * <p>
         * The unique identifier of the session containing the invocation step.
         * </p>
         * 
         * @param sessionId
         *        The unique identifier of the session containing the invocation step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sessionId(String sessionId);
    }

    static final class BuilderImpl implements Builder {
        private String invocationId;

        private String invocationStepId;

        private Instant invocationStepTime;

        private InvocationStepPayload payload;

        private String sessionId;

        private BuilderImpl() {
        }

        private BuilderImpl(InvocationStep model) {
            invocationId(model.invocationId);
            invocationStepId(model.invocationStepId);
            invocationStepTime(model.invocationStepTime);
            payload(model.payload);
            sessionId(model.sessionId);
        }

        public final String getInvocationId() {
            return invocationId;
        }

        public final void setInvocationId(String invocationId) {
            this.invocationId = invocationId;
        }

        @Override
        public final Builder invocationId(String invocationId) {
            this.invocationId = invocationId;
            return this;
        }

        public final String getInvocationStepId() {
            return invocationStepId;
        }

        public final void setInvocationStepId(String invocationStepId) {
            this.invocationStepId = invocationStepId;
        }

        @Override
        public final Builder invocationStepId(String invocationStepId) {
            this.invocationStepId = invocationStepId;
            return this;
        }

        public final Instant getInvocationStepTime() {
            return invocationStepTime;
        }

        public final void setInvocationStepTime(Instant invocationStepTime) {
            this.invocationStepTime = invocationStepTime;
        }

        @Override
        public final Builder invocationStepTime(Instant invocationStepTime) {
            this.invocationStepTime = invocationStepTime;
            return this;
        }

        public final InvocationStepPayload.Builder getPayload() {
            return payload != null ? payload.toBuilder() : null;
        }

        public final void setPayload(InvocationStepPayload.BuilderImpl payload) {
            this.payload = payload != null ? payload.build() : null;
        }

        @Override
        public final Builder payload(InvocationStepPayload payload) {
            this.payload = payload;
            return this;
        }

        public final String getSessionId() {
            return sessionId;
        }

        public final void setSessionId(String sessionId) {
            this.sessionId = sessionId;
        }

        @Override
        public final Builder sessionId(String sessionId) {
            this.sessionId = sessionId;
            return this;
        }

        @Override
        public InvocationStep build() {
            return new InvocationStep(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
