/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagentruntime.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.document.Document;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains configurations to override a prompt template in one part of an agent sequence. For more information, see <a
 * href="https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts.html">Advanced prompts</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PromptConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<PromptConfiguration.Builder, PromptConfiguration> {
    private static final SdkField<Document> ADDITIONAL_MODEL_REQUEST_FIELDS_FIELD = SdkField
            .<Document> builder(MarshallingType.DOCUMENT)
            .memberName("additionalModelRequestFields")
            .getter(getter(PromptConfiguration::additionalModelRequestFields))
            .setter(setter(Builder::additionalModelRequestFields))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("additionalModelRequestFields")
                    .build()).build();

    private static final SdkField<String> BASE_PROMPT_TEMPLATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("basePromptTemplate").getter(getter(PromptConfiguration::basePromptTemplate))
            .setter(setter(Builder::basePromptTemplate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("basePromptTemplate").build())
            .build();

    private static final SdkField<String> FOUNDATION_MODEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("foundationModel").getter(getter(PromptConfiguration::foundationModel))
            .setter(setter(Builder::foundationModel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("foundationModel").build()).build();

    private static final SdkField<InferenceConfiguration> INFERENCE_CONFIGURATION_FIELD = SdkField
            .<InferenceConfiguration> builder(MarshallingType.SDK_POJO).memberName("inferenceConfiguration")
            .getter(getter(PromptConfiguration::inferenceConfiguration)).setter(setter(Builder::inferenceConfiguration))
            .constructor(InferenceConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inferenceConfiguration").build())
            .build();

    private static final SdkField<String> PARSER_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("parserMode").getter(getter(PromptConfiguration::parserModeAsString)).setter(setter(Builder::parserMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("parserMode").build()).build();

    private static final SdkField<String> PROMPT_CREATION_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("promptCreationMode").getter(getter(PromptConfiguration::promptCreationModeAsString))
            .setter(setter(Builder::promptCreationMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("promptCreationMode").build())
            .build();

    private static final SdkField<String> PROMPT_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("promptState").getter(getter(PromptConfiguration::promptStateAsString))
            .setter(setter(Builder::promptState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("promptState").build()).build();

    private static final SdkField<String> PROMPT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("promptType").getter(getter(PromptConfiguration::promptTypeAsString)).setter(setter(Builder::promptType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("promptType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            ADDITIONAL_MODEL_REQUEST_FIELDS_FIELD, BASE_PROMPT_TEMPLATE_FIELD, FOUNDATION_MODEL_FIELD,
            INFERENCE_CONFIGURATION_FIELD, PARSER_MODE_FIELD, PROMPT_CREATION_MODE_FIELD, PROMPT_STATE_FIELD, PROMPT_TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Document additionalModelRequestFields;

    private final String basePromptTemplate;

    private final String foundationModel;

    private final InferenceConfiguration inferenceConfiguration;

    private final String parserMode;

    private final String promptCreationMode;

    private final String promptState;

    private final String promptType;

    private PromptConfiguration(BuilderImpl builder) {
        this.additionalModelRequestFields = builder.additionalModelRequestFields;
        this.basePromptTemplate = builder.basePromptTemplate;
        this.foundationModel = builder.foundationModel;
        this.inferenceConfiguration = builder.inferenceConfiguration;
        this.parserMode = builder.parserMode;
        this.promptCreationMode = builder.promptCreationMode;
        this.promptState = builder.promptState;
        this.promptType = builder.promptType;
    }

    /**
     * <p>
     * If the Converse or ConverseStream operations support the model, <code>additionalModelRequestFields</code>
     * contains additional inference parameters, beyond the base set of inference parameters in the
     * <code>inferenceConfiguration</code> field.
     * </p>
     * <p>
     * For more information, see <i>Inference request parameters and response fields for foundation models</i> in the
     * Amazon Bedrock user guide.
     * </p>
     * 
     * @return If the Converse or ConverseStream operations support the model, <code>additionalModelRequestFields</code>
     *         contains additional inference parameters, beyond the base set of inference parameters in the
     *         <code>inferenceConfiguration</code> field. </p>
     *         <p>
     *         For more information, see <i>Inference request parameters and response fields for foundation models</i>
     *         in the Amazon Bedrock user guide.
     */
    public final Document additionalModelRequestFields() {
        return additionalModelRequestFields;
    }

    /**
     * <p>
     * Defines the prompt template with which to replace the default prompt template. You can use placeholder variables
     * in the base prompt template to customize the prompt. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-placeholders.html">Prompt template placeholder
     * variables</a>. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts-configure.html">Configure the prompt
     * templates</a>.
     * </p>
     * 
     * @return Defines the prompt template with which to replace the default prompt template. You can use placeholder
     *         variables in the base prompt template to customize the prompt. For more information, see <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-placeholders.html">Prompt template
     *         placeholder variables</a>. For more information, see <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts-configure.html">Configure the
     *         prompt templates</a>.
     */
    public final String basePromptTemplate() {
        return basePromptTemplate;
    }

    /**
     * <p>
     * The foundation model to use.
     * </p>
     * 
     * @return The foundation model to use.
     */
    public final String foundationModel() {
        return foundationModel;
    }

    /**
     * <p>
     * Contains inference parameters to use when the agent invokes a foundation model in the part of the agent sequence
     * defined by the <code>promptType</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html">Inference parameters for
     * foundation models</a>.
     * </p>
     * 
     * @return Contains inference parameters to use when the agent invokes a foundation model in the part of the agent
     *         sequence defined by the <code>promptType</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html">Inference parameters
     *         for foundation models</a>.
     */
    public final InferenceConfiguration inferenceConfiguration() {
        return inferenceConfiguration;
    }

    /**
     * <p>
     * Specifies whether to override the default parser Lambda function when parsing the raw foundation model output in
     * the part of the agent sequence defined by the <code>promptType</code>. If you set the field as
     * <code>OVERRIDDEN</code>, the <code>overrideLambda</code> field in the <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_PromptOverrideConfiguration.html"
     * >PromptOverrideConfiguration</a> must be specified with the ARN of a Lambda function.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #parserMode} will
     * return {@link CreationMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #parserModeAsString}.
     * </p>
     * 
     * @return Specifies whether to override the default parser Lambda function when parsing the raw foundation model
     *         output in the part of the agent sequence defined by the <code>promptType</code>. If you set the field as
     *         <code>OVERRIDDEN</code>, the <code>overrideLambda</code> field in the <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_PromptOverrideConfiguration.html"
     *         >PromptOverrideConfiguration</a> must be specified with the ARN of a Lambda function.
     * @see CreationMode
     */
    public final CreationMode parserMode() {
        return CreationMode.fromValue(parserMode);
    }

    /**
     * <p>
     * Specifies whether to override the default parser Lambda function when parsing the raw foundation model output in
     * the part of the agent sequence defined by the <code>promptType</code>. If you set the field as
     * <code>OVERRIDDEN</code>, the <code>overrideLambda</code> field in the <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_PromptOverrideConfiguration.html"
     * >PromptOverrideConfiguration</a> must be specified with the ARN of a Lambda function.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #parserMode} will
     * return {@link CreationMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #parserModeAsString}.
     * </p>
     * 
     * @return Specifies whether to override the default parser Lambda function when parsing the raw foundation model
     *         output in the part of the agent sequence defined by the <code>promptType</code>. If you set the field as
     *         <code>OVERRIDDEN</code>, the <code>overrideLambda</code> field in the <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_PromptOverrideConfiguration.html"
     *         >PromptOverrideConfiguration</a> must be specified with the ARN of a Lambda function.
     * @see CreationMode
     */
    public final String parserModeAsString() {
        return parserMode;
    }

    /**
     * <p>
     * Specifies whether to override the default prompt template for this <code>promptType</code>. Set this value to
     * <code>OVERRIDDEN</code> to use the prompt that you provide in the <code>basePromptTemplate</code>. If you leave
     * it as <code>DEFAULT</code>, the agent uses a default prompt template.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #promptCreationMode} will return {@link CreationMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by
     * the service is available from {@link #promptCreationModeAsString}.
     * </p>
     * 
     * @return Specifies whether to override the default prompt template for this <code>promptType</code>. Set this
     *         value to <code>OVERRIDDEN</code> to use the prompt that you provide in the
     *         <code>basePromptTemplate</code>. If you leave it as <code>DEFAULT</code>, the agent uses a default prompt
     *         template.
     * @see CreationMode
     */
    public final CreationMode promptCreationMode() {
        return CreationMode.fromValue(promptCreationMode);
    }

    /**
     * <p>
     * Specifies whether to override the default prompt template for this <code>promptType</code>. Set this value to
     * <code>OVERRIDDEN</code> to use the prompt that you provide in the <code>basePromptTemplate</code>. If you leave
     * it as <code>DEFAULT</code>, the agent uses a default prompt template.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #promptCreationMode} will return {@link CreationMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by
     * the service is available from {@link #promptCreationModeAsString}.
     * </p>
     * 
     * @return Specifies whether to override the default prompt template for this <code>promptType</code>. Set this
     *         value to <code>OVERRIDDEN</code> to use the prompt that you provide in the
     *         <code>basePromptTemplate</code>. If you leave it as <code>DEFAULT</code>, the agent uses a default prompt
     *         template.
     * @see CreationMode
     */
    public final String promptCreationModeAsString() {
        return promptCreationMode;
    }

    /**
     * <p>
     * Specifies whether to allow the inline agent to carry out the step specified in the <code>promptType</code>. If
     * you set this value to <code>DISABLED</code>, the agent skips that step. The default state for each
     * <code>promptType</code> is as follows.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PRE_PROCESSING</code> – <code>ENABLED</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ORCHESTRATION</code> – <code>ENABLED</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KNOWLEDGE_BASE_RESPONSE_GENERATION</code> – <code>ENABLED</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>POST_PROCESSING</code> – <code>DISABLED</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #promptState} will
     * return {@link PromptState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #promptStateAsString}.
     * </p>
     * 
     * @return Specifies whether to allow the inline agent to carry out the step specified in the
     *         <code>promptType</code>. If you set this value to <code>DISABLED</code>, the agent skips that step. The
     *         default state for each <code>promptType</code> is as follows.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PRE_PROCESSING</code> – <code>ENABLED</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ORCHESTRATION</code> – <code>ENABLED</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KNOWLEDGE_BASE_RESPONSE_GENERATION</code> – <code>ENABLED</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>POST_PROCESSING</code> – <code>DISABLED</code>
     *         </p>
     *         </li>
     * @see PromptState
     */
    public final PromptState promptState() {
        return PromptState.fromValue(promptState);
    }

    /**
     * <p>
     * Specifies whether to allow the inline agent to carry out the step specified in the <code>promptType</code>. If
     * you set this value to <code>DISABLED</code>, the agent skips that step. The default state for each
     * <code>promptType</code> is as follows.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PRE_PROCESSING</code> – <code>ENABLED</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ORCHESTRATION</code> – <code>ENABLED</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KNOWLEDGE_BASE_RESPONSE_GENERATION</code> – <code>ENABLED</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>POST_PROCESSING</code> – <code>DISABLED</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #promptState} will
     * return {@link PromptState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #promptStateAsString}.
     * </p>
     * 
     * @return Specifies whether to allow the inline agent to carry out the step specified in the
     *         <code>promptType</code>. If you set this value to <code>DISABLED</code>, the agent skips that step. The
     *         default state for each <code>promptType</code> is as follows.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PRE_PROCESSING</code> – <code>ENABLED</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ORCHESTRATION</code> – <code>ENABLED</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KNOWLEDGE_BASE_RESPONSE_GENERATION</code> – <code>ENABLED</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>POST_PROCESSING</code> – <code>DISABLED</code>
     *         </p>
     *         </li>
     * @see PromptState
     */
    public final String promptStateAsString() {
        return promptState;
    }

    /**
     * <p>
     * The step in the agent sequence that this prompt configuration applies to.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #promptType} will
     * return {@link PromptType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #promptTypeAsString}.
     * </p>
     * 
     * @return The step in the agent sequence that this prompt configuration applies to.
     * @see PromptType
     */
    public final PromptType promptType() {
        return PromptType.fromValue(promptType);
    }

    /**
     * <p>
     * The step in the agent sequence that this prompt configuration applies to.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #promptType} will
     * return {@link PromptType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #promptTypeAsString}.
     * </p>
     * 
     * @return The step in the agent sequence that this prompt configuration applies to.
     * @see PromptType
     */
    public final String promptTypeAsString() {
        return promptType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(additionalModelRequestFields());
        hashCode = 31 * hashCode + Objects.hashCode(basePromptTemplate());
        hashCode = 31 * hashCode + Objects.hashCode(foundationModel());
        hashCode = 31 * hashCode + Objects.hashCode(inferenceConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(parserModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(promptCreationModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(promptStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(promptTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PromptConfiguration)) {
            return false;
        }
        PromptConfiguration other = (PromptConfiguration) obj;
        return Objects.equals(additionalModelRequestFields(), other.additionalModelRequestFields())
                && Objects.equals(basePromptTemplate(), other.basePromptTemplate())
                && Objects.equals(foundationModel(), other.foundationModel())
                && Objects.equals(inferenceConfiguration(), other.inferenceConfiguration())
                && Objects.equals(parserModeAsString(), other.parserModeAsString())
                && Objects.equals(promptCreationModeAsString(), other.promptCreationModeAsString())
                && Objects.equals(promptStateAsString(), other.promptStateAsString())
                && Objects.equals(promptTypeAsString(), other.promptTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PromptConfiguration").add("AdditionalModelRequestFields", additionalModelRequestFields())
                .add("BasePromptTemplate", basePromptTemplate() == null ? null : "*** Sensitive Data Redacted ***")
                .add("FoundationModel", foundationModel()).add("InferenceConfiguration", inferenceConfiguration())
                .add("ParserMode", parserModeAsString()).add("PromptCreationMode", promptCreationModeAsString())
                .add("PromptState", promptStateAsString()).add("PromptType", promptTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "additionalModelRequestFields":
            return Optional.ofNullable(clazz.cast(additionalModelRequestFields()));
        case "basePromptTemplate":
            return Optional.ofNullable(clazz.cast(basePromptTemplate()));
        case "foundationModel":
            return Optional.ofNullable(clazz.cast(foundationModel()));
        case "inferenceConfiguration":
            return Optional.ofNullable(clazz.cast(inferenceConfiguration()));
        case "parserMode":
            return Optional.ofNullable(clazz.cast(parserModeAsString()));
        case "promptCreationMode":
            return Optional.ofNullable(clazz.cast(promptCreationModeAsString()));
        case "promptState":
            return Optional.ofNullable(clazz.cast(promptStateAsString()));
        case "promptType":
            return Optional.ofNullable(clazz.cast(promptTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("additionalModelRequestFields", ADDITIONAL_MODEL_REQUEST_FIELDS_FIELD);
        map.put("basePromptTemplate", BASE_PROMPT_TEMPLATE_FIELD);
        map.put("foundationModel", FOUNDATION_MODEL_FIELD);
        map.put("inferenceConfiguration", INFERENCE_CONFIGURATION_FIELD);
        map.put("parserMode", PARSER_MODE_FIELD);
        map.put("promptCreationMode", PROMPT_CREATION_MODE_FIELD);
        map.put("promptState", PROMPT_STATE_FIELD);
        map.put("promptType", PROMPT_TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<PromptConfiguration, T> g) {
        return obj -> g.apply((PromptConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PromptConfiguration> {
        /**
         * <p>
         * If the Converse or ConverseStream operations support the model, <code>additionalModelRequestFields</code>
         * contains additional inference parameters, beyond the base set of inference parameters in the
         * <code>inferenceConfiguration</code> field.
         * </p>
         * <p>
         * For more information, see <i>Inference request parameters and response fields for foundation models</i> in
         * the Amazon Bedrock user guide.
         * </p>
         * 
         * @param additionalModelRequestFields
         *        If the Converse or ConverseStream operations support the model,
         *        <code>additionalModelRequestFields</code> contains additional inference parameters, beyond the base
         *        set of inference parameters in the <code>inferenceConfiguration</code> field. </p>
         *        <p>
         *        For more information, see <i>Inference request parameters and response fields for foundation
         *        models</i> in the Amazon Bedrock user guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalModelRequestFields(Document additionalModelRequestFields);

        /**
         * <p>
         * Defines the prompt template with which to replace the default prompt template. You can use placeholder
         * variables in the base prompt template to customize the prompt. For more information, see <a
         * href="https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-placeholders.html">Prompt template
         * placeholder variables</a>. For more information, see <a
         * href="https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts-configure.html">Configure the
         * prompt templates</a>.
         * </p>
         * 
         * @param basePromptTemplate
         *        Defines the prompt template with which to replace the default prompt template. You can use placeholder
         *        variables in the base prompt template to customize the prompt. For more information, see <a
         *        href="https://docs.aws.amazon.com/bedrock/latest/userguide/prompt-placeholders.html">Prompt template
         *        placeholder variables</a>. For more information, see <a
         *        href="https://docs.aws.amazon.com/bedrock/latest/userguide/advanced-prompts-configure.html">Configure
         *        the prompt templates</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder basePromptTemplate(String basePromptTemplate);

        /**
         * <p>
         * The foundation model to use.
         * </p>
         * 
         * @param foundationModel
         *        The foundation model to use.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder foundationModel(String foundationModel);

        /**
         * <p>
         * Contains inference parameters to use when the agent invokes a foundation model in the part of the agent
         * sequence defined by the <code>promptType</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html">Inference parameters for
         * foundation models</a>.
         * </p>
         * 
         * @param inferenceConfiguration
         *        Contains inference parameters to use when the agent invokes a foundation model in the part of the
         *        agent sequence defined by the <code>promptType</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html">Inference parameters
         *        for foundation models</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferenceConfiguration(InferenceConfiguration inferenceConfiguration);

        /**
         * <p>
         * Contains inference parameters to use when the agent invokes a foundation model in the part of the agent
         * sequence defined by the <code>promptType</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html">Inference parameters for
         * foundation models</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link InferenceConfiguration.Builder} avoiding
         * the need to create one manually via {@link InferenceConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InferenceConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #inferenceConfiguration(InferenceConfiguration)}.
         * 
         * @param inferenceConfiguration
         *        a consumer that will call methods on {@link InferenceConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inferenceConfiguration(InferenceConfiguration)
         */
        default Builder inferenceConfiguration(Consumer<InferenceConfiguration.Builder> inferenceConfiguration) {
            return inferenceConfiguration(InferenceConfiguration.builder().applyMutation(inferenceConfiguration).build());
        }

        /**
         * <p>
         * Specifies whether to override the default parser Lambda function when parsing the raw foundation model output
         * in the part of the agent sequence defined by the <code>promptType</code>. If you set the field as
         * <code>OVERRIDDEN</code>, the <code>overrideLambda</code> field in the <a
         * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_PromptOverrideConfiguration.html"
         * >PromptOverrideConfiguration</a> must be specified with the ARN of a Lambda function.
         * </p>
         * 
         * @param parserMode
         *        Specifies whether to override the default parser Lambda function when parsing the raw foundation model
         *        output in the part of the agent sequence defined by the <code>promptType</code>. If you set the field
         *        as <code>OVERRIDDEN</code>, the <code>overrideLambda</code> field in the <a
         *        href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_PromptOverrideConfiguration.html"
         *        >PromptOverrideConfiguration</a> must be specified with the ARN of a Lambda function.
         * @see CreationMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CreationMode
         */
        Builder parserMode(String parserMode);

        /**
         * <p>
         * Specifies whether to override the default parser Lambda function when parsing the raw foundation model output
         * in the part of the agent sequence defined by the <code>promptType</code>. If you set the field as
         * <code>OVERRIDDEN</code>, the <code>overrideLambda</code> field in the <a
         * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_PromptOverrideConfiguration.html"
         * >PromptOverrideConfiguration</a> must be specified with the ARN of a Lambda function.
         * </p>
         * 
         * @param parserMode
         *        Specifies whether to override the default parser Lambda function when parsing the raw foundation model
         *        output in the part of the agent sequence defined by the <code>promptType</code>. If you set the field
         *        as <code>OVERRIDDEN</code>, the <code>overrideLambda</code> field in the <a
         *        href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_agent_PromptOverrideConfiguration.html"
         *        >PromptOverrideConfiguration</a> must be specified with the ARN of a Lambda function.
         * @see CreationMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CreationMode
         */
        Builder parserMode(CreationMode parserMode);

        /**
         * <p>
         * Specifies whether to override the default prompt template for this <code>promptType</code>. Set this value to
         * <code>OVERRIDDEN</code> to use the prompt that you provide in the <code>basePromptTemplate</code>. If you
         * leave it as <code>DEFAULT</code>, the agent uses a default prompt template.
         * </p>
         * 
         * @param promptCreationMode
         *        Specifies whether to override the default prompt template for this <code>promptType</code>. Set this
         *        value to <code>OVERRIDDEN</code> to use the prompt that you provide in the
         *        <code>basePromptTemplate</code>. If you leave it as <code>DEFAULT</code>, the agent uses a default
         *        prompt template.
         * @see CreationMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CreationMode
         */
        Builder promptCreationMode(String promptCreationMode);

        /**
         * <p>
         * Specifies whether to override the default prompt template for this <code>promptType</code>. Set this value to
         * <code>OVERRIDDEN</code> to use the prompt that you provide in the <code>basePromptTemplate</code>. If you
         * leave it as <code>DEFAULT</code>, the agent uses a default prompt template.
         * </p>
         * 
         * @param promptCreationMode
         *        Specifies whether to override the default prompt template for this <code>promptType</code>. Set this
         *        value to <code>OVERRIDDEN</code> to use the prompt that you provide in the
         *        <code>basePromptTemplate</code>. If you leave it as <code>DEFAULT</code>, the agent uses a default
         *        prompt template.
         * @see CreationMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CreationMode
         */
        Builder promptCreationMode(CreationMode promptCreationMode);

        /**
         * <p>
         * Specifies whether to allow the inline agent to carry out the step specified in the <code>promptType</code>.
         * If you set this value to <code>DISABLED</code>, the agent skips that step. The default state for each
         * <code>promptType</code> is as follows.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PRE_PROCESSING</code> – <code>ENABLED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ORCHESTRATION</code> – <code>ENABLED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KNOWLEDGE_BASE_RESPONSE_GENERATION</code> – <code>ENABLED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>POST_PROCESSING</code> – <code>DISABLED</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param promptState
         *        Specifies whether to allow the inline agent to carry out the step specified in the
         *        <code>promptType</code>. If you set this value to <code>DISABLED</code>, the agent skips that step.
         *        The default state for each <code>promptType</code> is as follows.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PRE_PROCESSING</code> – <code>ENABLED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ORCHESTRATION</code> – <code>ENABLED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KNOWLEDGE_BASE_RESPONSE_GENERATION</code> – <code>ENABLED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>POST_PROCESSING</code> – <code>DISABLED</code>
         *        </p>
         *        </li>
         * @see PromptState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PromptState
         */
        Builder promptState(String promptState);

        /**
         * <p>
         * Specifies whether to allow the inline agent to carry out the step specified in the <code>promptType</code>.
         * If you set this value to <code>DISABLED</code>, the agent skips that step. The default state for each
         * <code>promptType</code> is as follows.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PRE_PROCESSING</code> – <code>ENABLED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ORCHESTRATION</code> – <code>ENABLED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KNOWLEDGE_BASE_RESPONSE_GENERATION</code> – <code>ENABLED</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>POST_PROCESSING</code> – <code>DISABLED</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param promptState
         *        Specifies whether to allow the inline agent to carry out the step specified in the
         *        <code>promptType</code>. If you set this value to <code>DISABLED</code>, the agent skips that step.
         *        The default state for each <code>promptType</code> is as follows.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PRE_PROCESSING</code> – <code>ENABLED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ORCHESTRATION</code> – <code>ENABLED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KNOWLEDGE_BASE_RESPONSE_GENERATION</code> – <code>ENABLED</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>POST_PROCESSING</code> – <code>DISABLED</code>
         *        </p>
         *        </li>
         * @see PromptState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PromptState
         */
        Builder promptState(PromptState promptState);

        /**
         * <p>
         * The step in the agent sequence that this prompt configuration applies to.
         * </p>
         * 
         * @param promptType
         *        The step in the agent sequence that this prompt configuration applies to.
         * @see PromptType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PromptType
         */
        Builder promptType(String promptType);

        /**
         * <p>
         * The step in the agent sequence that this prompt configuration applies to.
         * </p>
         * 
         * @param promptType
         *        The step in the agent sequence that this prompt configuration applies to.
         * @see PromptType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PromptType
         */
        Builder promptType(PromptType promptType);
    }

    static final class BuilderImpl implements Builder {
        private Document additionalModelRequestFields;

        private String basePromptTemplate;

        private String foundationModel;

        private InferenceConfiguration inferenceConfiguration;

        private String parserMode;

        private String promptCreationMode;

        private String promptState;

        private String promptType;

        private BuilderImpl() {
        }

        private BuilderImpl(PromptConfiguration model) {
            additionalModelRequestFields(model.additionalModelRequestFields);
            basePromptTemplate(model.basePromptTemplate);
            foundationModel(model.foundationModel);
            inferenceConfiguration(model.inferenceConfiguration);
            parserMode(model.parserMode);
            promptCreationMode(model.promptCreationMode);
            promptState(model.promptState);
            promptType(model.promptType);
        }

        public final Document getAdditionalModelRequestFields() {
            return additionalModelRequestFields;
        }

        public final void setAdditionalModelRequestFields(Document additionalModelRequestFields) {
            this.additionalModelRequestFields = additionalModelRequestFields;
        }

        @Override
        public final Builder additionalModelRequestFields(Document additionalModelRequestFields) {
            this.additionalModelRequestFields = additionalModelRequestFields;
            return this;
        }

        public final String getBasePromptTemplate() {
            return basePromptTemplate;
        }

        public final void setBasePromptTemplate(String basePromptTemplate) {
            this.basePromptTemplate = basePromptTemplate;
        }

        @Override
        public final Builder basePromptTemplate(String basePromptTemplate) {
            this.basePromptTemplate = basePromptTemplate;
            return this;
        }

        public final String getFoundationModel() {
            return foundationModel;
        }

        public final void setFoundationModel(String foundationModel) {
            this.foundationModel = foundationModel;
        }

        @Override
        public final Builder foundationModel(String foundationModel) {
            this.foundationModel = foundationModel;
            return this;
        }

        public final InferenceConfiguration.Builder getInferenceConfiguration() {
            return inferenceConfiguration != null ? inferenceConfiguration.toBuilder() : null;
        }

        public final void setInferenceConfiguration(InferenceConfiguration.BuilderImpl inferenceConfiguration) {
            this.inferenceConfiguration = inferenceConfiguration != null ? inferenceConfiguration.build() : null;
        }

        @Override
        public final Builder inferenceConfiguration(InferenceConfiguration inferenceConfiguration) {
            this.inferenceConfiguration = inferenceConfiguration;
            return this;
        }

        public final String getParserMode() {
            return parserMode;
        }

        public final void setParserMode(String parserMode) {
            this.parserMode = parserMode;
        }

        @Override
        public final Builder parserMode(String parserMode) {
            this.parserMode = parserMode;
            return this;
        }

        @Override
        public final Builder parserMode(CreationMode parserMode) {
            this.parserMode(parserMode == null ? null : parserMode.toString());
            return this;
        }

        public final String getPromptCreationMode() {
            return promptCreationMode;
        }

        public final void setPromptCreationMode(String promptCreationMode) {
            this.promptCreationMode = promptCreationMode;
        }

        @Override
        public final Builder promptCreationMode(String promptCreationMode) {
            this.promptCreationMode = promptCreationMode;
            return this;
        }

        @Override
        public final Builder promptCreationMode(CreationMode promptCreationMode) {
            this.promptCreationMode(promptCreationMode == null ? null : promptCreationMode.toString());
            return this;
        }

        public final String getPromptState() {
            return promptState;
        }

        public final void setPromptState(String promptState) {
            this.promptState = promptState;
        }

        @Override
        public final Builder promptState(String promptState) {
            this.promptState = promptState;
            return this;
        }

        @Override
        public final Builder promptState(PromptState promptState) {
            this.promptState(promptState == null ? null : promptState.toString());
            return this;
        }

        public final String getPromptType() {
            return promptType;
        }

        public final void setPromptType(String promptType) {
            this.promptType = promptType;
        }

        @Override
        public final Builder promptType(String promptType) {
            this.promptType = promptType;
            return this;
        }

        @Override
        public final Builder promptType(PromptType promptType) {
            this.promptType(promptType == null ? null : promptType.toString());
            return this;
        }

        @Override
        public PromptConfiguration build() {
            return new PromptConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
