/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockdataautomation;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.bedrockdataautomation.internal.BedrockDataAutomationServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.bedrockdataautomation.model.AccessDeniedException;
import software.amazon.awssdk.services.bedrockdataautomation.model.BedrockDataAutomationException;
import software.amazon.awssdk.services.bedrockdataautomation.model.ConflictException;
import software.amazon.awssdk.services.bedrockdataautomation.model.CreateBlueprintRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.CreateBlueprintResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.CreateBlueprintVersionRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.CreateBlueprintVersionResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.CreateDataAutomationProjectRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.CreateDataAutomationProjectResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.DeleteBlueprintRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.DeleteBlueprintResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.DeleteDataAutomationProjectRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.DeleteDataAutomationProjectResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.GetBlueprintRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.GetBlueprintResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.GetDataAutomationProjectRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.GetDataAutomationProjectResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.InternalServerException;
import software.amazon.awssdk.services.bedrockdataautomation.model.ListBlueprintsRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.ListBlueprintsResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.ListDataAutomationProjectsRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.ListDataAutomationProjectsResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.ResourceNotFoundException;
import software.amazon.awssdk.services.bedrockdataautomation.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.bedrockdataautomation.model.ThrottlingException;
import software.amazon.awssdk.services.bedrockdataautomation.model.UpdateBlueprintRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.UpdateBlueprintResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.UpdateDataAutomationProjectRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.UpdateDataAutomationProjectResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.ValidationException;
import software.amazon.awssdk.services.bedrockdataautomation.transform.CreateBlueprintRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.CreateBlueprintVersionRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.CreateDataAutomationProjectRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.DeleteBlueprintRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.DeleteDataAutomationProjectRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.GetBlueprintRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.GetDataAutomationProjectRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.ListBlueprintsRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.ListDataAutomationProjectsRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.UpdateBlueprintRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.UpdateDataAutomationProjectRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link BedrockDataAutomationAsyncClient}.
 *
 * @see BedrockDataAutomationAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBedrockDataAutomationAsyncClient implements BedrockDataAutomationAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultBedrockDataAutomationAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBedrockDataAutomationAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an Amazon Bedrock Data Automation Blueprint
     * </p>
     *
     * @param createBlueprintRequest
     *        Create Blueprint Request
     * @return A Java Future containing the result of the CreateBlueprint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException This exception is thrown when a request is made beyond the service
     *         quota</li>
     *         <li>ValidationException This exception is thrown when the request's input validation fails</li>
     *         <li>ConflictException This exception is thrown when there is a conflict performing an operation</li>
     *         <li>InternalServerException This exception is thrown if there was an unexpected error during processing
     *         of request</li>
     *         <li>ThrottlingException This exception is thrown when the number of requests exceeds the limit</li>
     *         <li>AccessDeniedException This exception is thrown when a request is denied per access permissions</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockDataAutomationException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample BedrockDataAutomationAsyncClient.CreateBlueprint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/CreateBlueprint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBlueprintResponse> createBlueprint(CreateBlueprintRequest createBlueprintRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBlueprintRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBlueprintRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBlueprint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBlueprintResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateBlueprintResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBlueprintResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBlueprintRequest, CreateBlueprintResponse>()
                            .withOperationName("CreateBlueprint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateBlueprintRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createBlueprintRequest));
            CompletableFuture<CreateBlueprintResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new version of an existing Amazon Bedrock Data Automation Blueprint
     * </p>
     *
     * @param createBlueprintVersionRequest
     *        Create Blueprint Version Request
     * @return A Java Future containing the result of the CreateBlueprintVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException This exception is thrown when a request is made beyond the service
     *         quota</li>
     *         <li>ValidationException This exception is thrown when the request's input validation fails</li>
     *         <li>InternalServerException This exception is thrown if there was an unexpected error during processing
     *         of request</li>
     *         <li>ThrottlingException This exception is thrown when the number of requests exceeds the limit</li>
     *         <li>AccessDeniedException This exception is thrown when a request is denied per access permissions</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource referenced by the operation does
     *         not exist</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockDataAutomationException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample BedrockDataAutomationAsyncClient.CreateBlueprintVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/CreateBlueprintVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBlueprintVersionResponse> createBlueprintVersion(
            CreateBlueprintVersionRequest createBlueprintVersionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBlueprintVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBlueprintVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBlueprintVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBlueprintVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateBlueprintVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBlueprintVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBlueprintVersionRequest, CreateBlueprintVersionResponse>()
                            .withOperationName("CreateBlueprintVersion").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateBlueprintVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createBlueprintVersionRequest));
            CompletableFuture<CreateBlueprintVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon Bedrock Data Automation Project
     * </p>
     *
     * @param createDataAutomationProjectRequest
     *        Create DataAutomationProject Request
     * @return A Java Future containing the result of the CreateDataAutomationProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceQuotaExceededException This exception is thrown when a request is made beyond the service
     *         quota</li>
     *         <li>ValidationException This exception is thrown when the request's input validation fails</li>
     *         <li>ConflictException This exception is thrown when there is a conflict performing an operation</li>
     *         <li>InternalServerException This exception is thrown if there was an unexpected error during processing
     *         of request</li>
     *         <li>ThrottlingException This exception is thrown when the number of requests exceeds the limit</li>
     *         <li>AccessDeniedException This exception is thrown when a request is denied per access permissions</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockDataAutomationException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample BedrockDataAutomationAsyncClient.CreateDataAutomationProject
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/CreateDataAutomationProject"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDataAutomationProjectResponse> createDataAutomationProject(
            CreateDataAutomationProjectRequest createDataAutomationProjectRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDataAutomationProjectRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataAutomationProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataAutomationProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDataAutomationProjectResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDataAutomationProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDataAutomationProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDataAutomationProjectRequest, CreateDataAutomationProjectResponse>()
                            .withOperationName("CreateDataAutomationProject").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateDataAutomationProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createDataAutomationProjectRequest));
            CompletableFuture<CreateDataAutomationProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing Amazon Bedrock Data Automation Blueprint
     * </p>
     *
     * @param deleteBlueprintRequest
     *        Delete Blueprint Request
     * @return A Java Future containing the result of the DeleteBlueprint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException This exception is thrown when the request's input validation fails</li>
     *         <li>InternalServerException This exception is thrown if there was an unexpected error during processing
     *         of request</li>
     *         <li>ThrottlingException This exception is thrown when the number of requests exceeds the limit</li>
     *         <li>AccessDeniedException This exception is thrown when a request is denied per access permissions</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource referenced by the operation does
     *         not exist</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockDataAutomationException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample BedrockDataAutomationAsyncClient.DeleteBlueprint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/DeleteBlueprint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBlueprintResponse> deleteBlueprint(DeleteBlueprintRequest deleteBlueprintRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBlueprintRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBlueprintRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBlueprint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBlueprintResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteBlueprintResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteBlueprintResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBlueprintRequest, DeleteBlueprintResponse>()
                            .withOperationName("DeleteBlueprint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteBlueprintRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteBlueprintRequest));
            CompletableFuture<DeleteBlueprintResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing Amazon Bedrock Data Automation Project
     * </p>
     *
     * @param deleteDataAutomationProjectRequest
     *        Delete DataAutomationProject Request
     * @return A Java Future containing the result of the DeleteDataAutomationProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException This exception is thrown when the request's input validation fails</li>
     *         <li>InternalServerException This exception is thrown if there was an unexpected error during processing
     *         of request</li>
     *         <li>ThrottlingException This exception is thrown when the number of requests exceeds the limit</li>
     *         <li>AccessDeniedException This exception is thrown when a request is denied per access permissions</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource referenced by the operation does
     *         not exist</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockDataAutomationException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample BedrockDataAutomationAsyncClient.DeleteDataAutomationProject
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/DeleteDataAutomationProject"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDataAutomationProjectResponse> deleteDataAutomationProject(
            DeleteDataAutomationProjectRequest deleteDataAutomationProjectRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDataAutomationProjectRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDataAutomationProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataAutomationProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDataAutomationProjectResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDataAutomationProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDataAutomationProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDataAutomationProjectRequest, DeleteDataAutomationProjectResponse>()
                            .withOperationName("DeleteDataAutomationProject").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteDataAutomationProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteDataAutomationProjectRequest));
            CompletableFuture<DeleteDataAutomationProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets an existing Amazon Bedrock Data Automation Blueprint
     * </p>
     *
     * @param getBlueprintRequest
     *        Get Blueprint Request
     * @return A Java Future containing the result of the GetBlueprint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException This exception is thrown when the request's input validation fails</li>
     *         <li>InternalServerException This exception is thrown if there was an unexpected error during processing
     *         of request</li>
     *         <li>ThrottlingException This exception is thrown when the number of requests exceeds the limit</li>
     *         <li>AccessDeniedException This exception is thrown when a request is denied per access permissions</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource referenced by the operation does
     *         not exist</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockDataAutomationException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample BedrockDataAutomationAsyncClient.GetBlueprint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/GetBlueprint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetBlueprintResponse> getBlueprint(GetBlueprintRequest getBlueprintRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBlueprintRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBlueprintRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBlueprint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBlueprintResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetBlueprintResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBlueprintResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBlueprintRequest, GetBlueprintResponse>()
                            .withOperationName("GetBlueprint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetBlueprintRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getBlueprintRequest));
            CompletableFuture<GetBlueprintResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets an existing Amazon Bedrock Data Automation Project
     * </p>
     *
     * @param getDataAutomationProjectRequest
     *        Get DataAutomationProject Request
     * @return A Java Future containing the result of the GetDataAutomationProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException This exception is thrown when the request's input validation fails</li>
     *         <li>InternalServerException This exception is thrown if there was an unexpected error during processing
     *         of request</li>
     *         <li>ThrottlingException This exception is thrown when the number of requests exceeds the limit</li>
     *         <li>AccessDeniedException This exception is thrown when a request is denied per access permissions</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource referenced by the operation does
     *         not exist</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockDataAutomationException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample BedrockDataAutomationAsyncClient.GetDataAutomationProject
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/GetDataAutomationProject"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDataAutomationProjectResponse> getDataAutomationProject(
            GetDataAutomationProjectRequest getDataAutomationProjectRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDataAutomationProjectRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataAutomationProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataAutomationProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDataAutomationProjectResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDataAutomationProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDataAutomationProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDataAutomationProjectRequest, GetDataAutomationProjectResponse>()
                            .withOperationName("GetDataAutomationProject").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDataAutomationProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getDataAutomationProjectRequest));
            CompletableFuture<GetDataAutomationProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all existing Amazon Bedrock Data Automation Blueprints
     * </p>
     *
     * @param listBlueprintsRequest
     *        List Blueprint Request
     * @return A Java Future containing the result of the ListBlueprints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException This exception is thrown when the request's input validation fails</li>
     *         <li>InternalServerException This exception is thrown if there was an unexpected error during processing
     *         of request</li>
     *         <li>ThrottlingException This exception is thrown when the number of requests exceeds the limit</li>
     *         <li>AccessDeniedException This exception is thrown when a request is denied per access permissions</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource referenced by the operation does
     *         not exist</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockDataAutomationException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample BedrockDataAutomationAsyncClient.ListBlueprints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/ListBlueprints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListBlueprintsResponse> listBlueprints(ListBlueprintsRequest listBlueprintsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listBlueprintsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBlueprintsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBlueprints");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListBlueprintsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListBlueprintsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListBlueprintsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListBlueprintsRequest, ListBlueprintsResponse>()
                            .withOperationName("ListBlueprints").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListBlueprintsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listBlueprintsRequest));
            CompletableFuture<ListBlueprintsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all existing Amazon Bedrock Data Automation Projects
     * </p>
     *
     * @param listDataAutomationProjectsRequest
     *        List DataAutomationProject Request
     * @return A Java Future containing the result of the ListDataAutomationProjects operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ValidationException This exception is thrown when the request's input validation fails</li>
     *         <li>InternalServerException This exception is thrown if there was an unexpected error during processing
     *         of request</li>
     *         <li>ThrottlingException This exception is thrown when the number of requests exceeds the limit</li>
     *         <li>AccessDeniedException This exception is thrown when a request is denied per access permissions</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource referenced by the operation does
     *         not exist</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockDataAutomationException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample BedrockDataAutomationAsyncClient.ListDataAutomationProjects
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/ListDataAutomationProjects"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDataAutomationProjectsResponse> listDataAutomationProjects(
            ListDataAutomationProjectsRequest listDataAutomationProjectsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDataAutomationProjectsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataAutomationProjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataAutomationProjects");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDataAutomationProjectsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDataAutomationProjectsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDataAutomationProjectsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDataAutomationProjectsRequest, ListDataAutomationProjectsResponse>()
                            .withOperationName("ListDataAutomationProjects").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListDataAutomationProjectsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDataAutomationProjectsRequest));
            CompletableFuture<ListDataAutomationProjectsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing Amazon Bedrock Data Automation Blueprint
     * </p>
     *
     * @param updateBlueprintRequest
     *        Update Blueprint Request
     * @return A Java Future containing the result of the UpdateBlueprint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException This exception is thrown when there is a conflict performing an operation</li>
     *         <li>ValidationException This exception is thrown when the request's input validation fails</li>
     *         <li>InternalServerException This exception is thrown if there was an unexpected error during processing
     *         of request</li>
     *         <li>ThrottlingException This exception is thrown when the number of requests exceeds the limit</li>
     *         <li>AccessDeniedException This exception is thrown when a request is denied per access permissions</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource referenced by the operation does
     *         not exist</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockDataAutomationException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample BedrockDataAutomationAsyncClient.UpdateBlueprint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/UpdateBlueprint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateBlueprintResponse> updateBlueprint(UpdateBlueprintRequest updateBlueprintRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateBlueprintRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBlueprintRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBlueprint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateBlueprintResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateBlueprintResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateBlueprintResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateBlueprintRequest, UpdateBlueprintResponse>()
                            .withOperationName("UpdateBlueprint").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateBlueprintRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateBlueprintRequest));
            CompletableFuture<UpdateBlueprintResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing Amazon Bedrock Data Automation Project
     * </p>
     *
     * @param updateDataAutomationProjectRequest
     *        Update DataAutomationProject Request
     * @return A Java Future containing the result of the UpdateDataAutomationProject operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException This exception is thrown when there is a conflict performing an operation</li>
     *         <li>ValidationException This exception is thrown when the request's input validation fails</li>
     *         <li>InternalServerException This exception is thrown if there was an unexpected error during processing
     *         of request</li>
     *         <li>ThrottlingException This exception is thrown when the number of requests exceeds the limit</li>
     *         <li>AccessDeniedException This exception is thrown when a request is denied per access permissions</li>
     *         <li>ResourceNotFoundException This exception is thrown when a resource referenced by the operation does
     *         not exist</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockDataAutomationException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample BedrockDataAutomationAsyncClient.UpdateDataAutomationProject
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/UpdateDataAutomationProject"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDataAutomationProjectResponse> updateDataAutomationProject(
            UpdateDataAutomationProjectRequest updateDataAutomationProjectRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDataAutomationProjectRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDataAutomationProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataAutomationProject");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDataAutomationProjectResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDataAutomationProjectResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDataAutomationProjectResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDataAutomationProjectRequest, UpdateDataAutomationProjectResponse>()
                            .withOperationName("UpdateDataAutomationProject").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateDataAutomationProjectRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateDataAutomationProjectRequest));
            CompletableFuture<UpdateDataAutomationProjectResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final BedrockDataAutomationServiceClientConfiguration serviceClientConfiguration() {
        return new BedrockDataAutomationServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BedrockDataAutomationException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        BedrockDataAutomationServiceClientConfigurationBuilder serviceConfigBuilder = new BedrockDataAutomationServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
