/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockdataautomation;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.bedrockdataautomation.internal.BedrockDataAutomationServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.bedrockdataautomation.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.bedrockdataautomation.model.AccessDeniedException;
import software.amazon.awssdk.services.bedrockdataautomation.model.BedrockDataAutomationException;
import software.amazon.awssdk.services.bedrockdataautomation.model.ConflictException;
import software.amazon.awssdk.services.bedrockdataautomation.model.CreateBlueprintRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.CreateBlueprintResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.CreateBlueprintVersionRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.CreateBlueprintVersionResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.CreateDataAutomationProjectRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.CreateDataAutomationProjectResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.DeleteBlueprintRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.DeleteBlueprintResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.DeleteDataAutomationProjectRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.DeleteDataAutomationProjectResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.GetBlueprintRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.GetBlueprintResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.GetDataAutomationProjectRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.GetDataAutomationProjectResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.InternalServerException;
import software.amazon.awssdk.services.bedrockdataautomation.model.ListBlueprintsRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.ListBlueprintsResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.ListDataAutomationProjectsRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.ListDataAutomationProjectsResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.ResourceNotFoundException;
import software.amazon.awssdk.services.bedrockdataautomation.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.bedrockdataautomation.model.TagResourceRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.TagResourceResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.ThrottlingException;
import software.amazon.awssdk.services.bedrockdataautomation.model.UntagResourceRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.UntagResourceResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.UpdateBlueprintRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.UpdateBlueprintResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.UpdateDataAutomationProjectRequest;
import software.amazon.awssdk.services.bedrockdataautomation.model.UpdateDataAutomationProjectResponse;
import software.amazon.awssdk.services.bedrockdataautomation.model.ValidationException;
import software.amazon.awssdk.services.bedrockdataautomation.transform.CreateBlueprintRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.CreateBlueprintVersionRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.CreateDataAutomationProjectRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.DeleteBlueprintRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.DeleteDataAutomationProjectRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.GetBlueprintRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.GetDataAutomationProjectRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.ListBlueprintsRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.ListDataAutomationProjectsRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.UpdateBlueprintRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomation.transform.UpdateDataAutomationProjectRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link BedrockDataAutomationClient}.
 *
 * @see BedrockDataAutomationClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBedrockDataAutomationClient implements BedrockDataAutomationClient {
    private static final Logger log = Logger.loggerFor(DefaultBedrockDataAutomationClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBedrockDataAutomationClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Bedrock_Data_Automation" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an Amazon Bedrock Data Automation Blueprint
     * </p>
     *
     * @param createBlueprintRequest
     *        Create Blueprint Request
     * @return Result of the CreateBlueprint operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         This exception is thrown when a request is made beyond the service quota
     * @throws ValidationException
     *         This exception is thrown when the request's input validation fails
     * @throws ConflictException
     *         This exception is thrown when there is a conflict performing an operation
     * @throws InternalServerException
     *         This exception is thrown if there was an unexpected error during processing of request
     * @throws ThrottlingException
     *         This exception is thrown when the number of requests exceeds the limit
     * @throws AccessDeniedException
     *         This exception is thrown when a request is denied per access permissions
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockDataAutomationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockDataAutomationClient.CreateBlueprint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/CreateBlueprint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateBlueprintResponse createBlueprint(CreateBlueprintRequest createBlueprintRequest)
            throws ServiceQuotaExceededException, ValidationException, ConflictException, InternalServerException,
            ThrottlingException, AccessDeniedException, AwsServiceException, SdkClientException, BedrockDataAutomationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBlueprintResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateBlueprintResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBlueprintRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBlueprintRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBlueprint");

            return clientHandler.execute(new ClientExecutionParams<CreateBlueprintRequest, CreateBlueprintResponse>()
                    .withOperationName("CreateBlueprint").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createBlueprintRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateBlueprintRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new version of an existing Amazon Bedrock Data Automation Blueprint
     * </p>
     *
     * @param createBlueprintVersionRequest
     *        Create Blueprint Version Request
     * @return Result of the CreateBlueprintVersion operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         This exception is thrown when a request is made beyond the service quota
     * @throws ValidationException
     *         This exception is thrown when the request's input validation fails
     * @throws InternalServerException
     *         This exception is thrown if there was an unexpected error during processing of request
     * @throws ThrottlingException
     *         This exception is thrown when the number of requests exceeds the limit
     * @throws AccessDeniedException
     *         This exception is thrown when a request is denied per access permissions
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource referenced by the operation does not exist
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockDataAutomationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockDataAutomationClient.CreateBlueprintVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/CreateBlueprintVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateBlueprintVersionResponse createBlueprintVersion(CreateBlueprintVersionRequest createBlueprintVersionRequest)
            throws ServiceQuotaExceededException, ValidationException, InternalServerException, ThrottlingException,
            AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            BedrockDataAutomationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBlueprintVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateBlueprintVersionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBlueprintVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBlueprintVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBlueprintVersion");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateBlueprintVersionRequest, CreateBlueprintVersionResponse>()
                            .withOperationName("CreateBlueprintVersion").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createBlueprintVersionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateBlueprintVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Amazon Bedrock Data Automation Project
     * </p>
     *
     * @param createDataAutomationProjectRequest
     *        Create DataAutomationProject Request
     * @return Result of the CreateDataAutomationProject operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         This exception is thrown when a request is made beyond the service quota
     * @throws ValidationException
     *         This exception is thrown when the request's input validation fails
     * @throws ConflictException
     *         This exception is thrown when there is a conflict performing an operation
     * @throws InternalServerException
     *         This exception is thrown if there was an unexpected error during processing of request
     * @throws ThrottlingException
     *         This exception is thrown when the number of requests exceeds the limit
     * @throws AccessDeniedException
     *         This exception is thrown when a request is denied per access permissions
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockDataAutomationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockDataAutomationClient.CreateDataAutomationProject
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/CreateDataAutomationProject"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDataAutomationProjectResponse createDataAutomationProject(
            CreateDataAutomationProjectRequest createDataAutomationProjectRequest) throws ServiceQuotaExceededException,
            ValidationException, ConflictException, InternalServerException, ThrottlingException, AccessDeniedException,
            AwsServiceException, SdkClientException, BedrockDataAutomationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDataAutomationProjectResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDataAutomationProjectResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDataAutomationProjectRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataAutomationProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataAutomationProject");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateDataAutomationProjectRequest, CreateDataAutomationProjectResponse>()
                            .withOperationName("CreateDataAutomationProject").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createDataAutomationProjectRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateDataAutomationProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing Amazon Bedrock Data Automation Blueprint
     * </p>
     *
     * @param deleteBlueprintRequest
     *        Delete Blueprint Request
     * @return Result of the DeleteBlueprint operation returned by the service.
     * @throws ValidationException
     *         This exception is thrown when the request's input validation fails
     * @throws InternalServerException
     *         This exception is thrown if there was an unexpected error during processing of request
     * @throws ThrottlingException
     *         This exception is thrown when the number of requests exceeds the limit
     * @throws AccessDeniedException
     *         This exception is thrown when a request is denied per access permissions
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource referenced by the operation does not exist
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockDataAutomationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockDataAutomationClient.DeleteBlueprint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/DeleteBlueprint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteBlueprintResponse deleteBlueprint(DeleteBlueprintRequest deleteBlueprintRequest) throws ValidationException,
            InternalServerException, ThrottlingException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, BedrockDataAutomationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBlueprintResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteBlueprintResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBlueprintRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBlueprintRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBlueprint");

            return clientHandler.execute(new ClientExecutionParams<DeleteBlueprintRequest, DeleteBlueprintResponse>()
                    .withOperationName("DeleteBlueprint").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteBlueprintRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteBlueprintRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing Amazon Bedrock Data Automation Project
     * </p>
     *
     * @param deleteDataAutomationProjectRequest
     *        Delete DataAutomationProject Request
     * @return Result of the DeleteDataAutomationProject operation returned by the service.
     * @throws ValidationException
     *         This exception is thrown when the request's input validation fails
     * @throws InternalServerException
     *         This exception is thrown if there was an unexpected error during processing of request
     * @throws ThrottlingException
     *         This exception is thrown when the number of requests exceeds the limit
     * @throws AccessDeniedException
     *         This exception is thrown when a request is denied per access permissions
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource referenced by the operation does not exist
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockDataAutomationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockDataAutomationClient.DeleteDataAutomationProject
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/DeleteDataAutomationProject"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDataAutomationProjectResponse deleteDataAutomationProject(
            DeleteDataAutomationProjectRequest deleteDataAutomationProjectRequest) throws ValidationException,
            InternalServerException, ThrottlingException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, BedrockDataAutomationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDataAutomationProjectResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteDataAutomationProjectResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDataAutomationProjectRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDataAutomationProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataAutomationProject");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteDataAutomationProjectRequest, DeleteDataAutomationProjectResponse>()
                            .withOperationName("DeleteDataAutomationProject").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteDataAutomationProjectRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteDataAutomationProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets an existing Amazon Bedrock Data Automation Blueprint
     * </p>
     *
     * @param getBlueprintRequest
     *        Get Blueprint Request
     * @return Result of the GetBlueprint operation returned by the service.
     * @throws ValidationException
     *         This exception is thrown when the request's input validation fails
     * @throws InternalServerException
     *         This exception is thrown if there was an unexpected error during processing of request
     * @throws ThrottlingException
     *         This exception is thrown when the number of requests exceeds the limit
     * @throws AccessDeniedException
     *         This exception is thrown when a request is denied per access permissions
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource referenced by the operation does not exist
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockDataAutomationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockDataAutomationClient.GetBlueprint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/GetBlueprint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetBlueprintResponse getBlueprint(GetBlueprintRequest getBlueprintRequest) throws ValidationException,
            InternalServerException, ThrottlingException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, BedrockDataAutomationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBlueprintResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetBlueprintResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBlueprintRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBlueprintRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBlueprint");

            return clientHandler.execute(new ClientExecutionParams<GetBlueprintRequest, GetBlueprintResponse>()
                    .withOperationName("GetBlueprint").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getBlueprintRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetBlueprintRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets an existing Amazon Bedrock Data Automation Project
     * </p>
     *
     * @param getDataAutomationProjectRequest
     *        Get DataAutomationProject Request
     * @return Result of the GetDataAutomationProject operation returned by the service.
     * @throws ValidationException
     *         This exception is thrown when the request's input validation fails
     * @throws InternalServerException
     *         This exception is thrown if there was an unexpected error during processing of request
     * @throws ThrottlingException
     *         This exception is thrown when the number of requests exceeds the limit
     * @throws AccessDeniedException
     *         This exception is thrown when a request is denied per access permissions
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource referenced by the operation does not exist
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockDataAutomationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockDataAutomationClient.GetDataAutomationProject
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/GetDataAutomationProject"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDataAutomationProjectResponse getDataAutomationProject(
            GetDataAutomationProjectRequest getDataAutomationProjectRequest) throws ValidationException, InternalServerException,
            ThrottlingException, AccessDeniedException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            BedrockDataAutomationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDataAutomationProjectResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDataAutomationProjectResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDataAutomationProjectRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataAutomationProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataAutomationProject");

            return clientHandler
                    .execute(new ClientExecutionParams<GetDataAutomationProjectRequest, GetDataAutomationProjectResponse>()
                            .withOperationName("GetDataAutomationProject").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getDataAutomationProjectRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetDataAutomationProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all existing Amazon Bedrock Data Automation Blueprints
     * </p>
     *
     * @param listBlueprintsRequest
     *        List Blueprint Request
     * @return Result of the ListBlueprints operation returned by the service.
     * @throws ValidationException
     *         This exception is thrown when the request's input validation fails
     * @throws InternalServerException
     *         This exception is thrown if there was an unexpected error during processing of request
     * @throws ThrottlingException
     *         This exception is thrown when the number of requests exceeds the limit
     * @throws AccessDeniedException
     *         This exception is thrown when a request is denied per access permissions
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource referenced by the operation does not exist
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockDataAutomationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockDataAutomationClient.ListBlueprints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/ListBlueprints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListBlueprintsResponse listBlueprints(ListBlueprintsRequest listBlueprintsRequest) throws ValidationException,
            InternalServerException, ThrottlingException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, BedrockDataAutomationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBlueprintsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListBlueprintsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listBlueprintsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBlueprintsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBlueprints");

            return clientHandler.execute(new ClientExecutionParams<ListBlueprintsRequest, ListBlueprintsResponse>()
                    .withOperationName("ListBlueprints").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listBlueprintsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListBlueprintsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all existing Amazon Bedrock Data Automation Projects
     * </p>
     *
     * @param listDataAutomationProjectsRequest
     *        List DataAutomationProject Request
     * @return Result of the ListDataAutomationProjects operation returned by the service.
     * @throws ValidationException
     *         This exception is thrown when the request's input validation fails
     * @throws InternalServerException
     *         This exception is thrown if there was an unexpected error during processing of request
     * @throws ThrottlingException
     *         This exception is thrown when the number of requests exceeds the limit
     * @throws AccessDeniedException
     *         This exception is thrown when a request is denied per access permissions
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource referenced by the operation does not exist
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockDataAutomationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockDataAutomationClient.ListDataAutomationProjects
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/ListDataAutomationProjects"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDataAutomationProjectsResponse listDataAutomationProjects(
            ListDataAutomationProjectsRequest listDataAutomationProjectsRequest) throws ValidationException,
            InternalServerException, ThrottlingException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, BedrockDataAutomationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDataAutomationProjectsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDataAutomationProjectsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDataAutomationProjectsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDataAutomationProjectsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDataAutomationProjects");

            return clientHandler
                    .execute(new ClientExecutionParams<ListDataAutomationProjectsRequest, ListDataAutomationProjectsResponse>()
                            .withOperationName("ListDataAutomationProjects").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listDataAutomationProjectsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListDataAutomationProjectsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List tags for an Amazon Bedrock Data Automation resource
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ValidationException
     *         This exception is thrown when the request's input validation fails
     * @throws InternalServerException
     *         This exception is thrown if there was an unexpected error during processing of request
     * @throws ThrottlingException
     *         This exception is thrown when the number of requests exceeds the limit
     * @throws AccessDeniedException
     *         This exception is thrown when a request is denied per access permissions
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource referenced by the operation does not exist
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockDataAutomationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockDataAutomationClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ValidationException, InternalServerException, ThrottlingException, AccessDeniedException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, BedrockDataAutomationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tag an Amazon Bedrock Data Automation resource
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         This exception is thrown when a request is made beyond the service quota
     * @throws ValidationException
     *         This exception is thrown when the request's input validation fails
     * @throws InternalServerException
     *         This exception is thrown if there was an unexpected error during processing of request
     * @throws ThrottlingException
     *         This exception is thrown when the number of requests exceeds the limit
     * @throws AccessDeniedException
     *         This exception is thrown when a request is denied per access permissions
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource referenced by the operation does not exist
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockDataAutomationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockDataAutomationClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ServiceQuotaExceededException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, BedrockDataAutomationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Untag an Amazon Bedrock Data Automation resource
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ValidationException
     *         This exception is thrown when the request's input validation fails
     * @throws InternalServerException
     *         This exception is thrown if there was an unexpected error during processing of request
     * @throws ThrottlingException
     *         This exception is thrown when the number of requests exceeds the limit
     * @throws AccessDeniedException
     *         This exception is thrown when a request is denied per access permissions
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource referenced by the operation does not exist
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockDataAutomationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockDataAutomationClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ValidationException,
            InternalServerException, ThrottlingException, AccessDeniedException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, BedrockDataAutomationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing Amazon Bedrock Data Automation Blueprint
     * </p>
     *
     * @param updateBlueprintRequest
     *        Update Blueprint Request
     * @return Result of the UpdateBlueprint operation returned by the service.
     * @throws ConflictException
     *         This exception is thrown when there is a conflict performing an operation
     * @throws ValidationException
     *         This exception is thrown when the request's input validation fails
     * @throws InternalServerException
     *         This exception is thrown if there was an unexpected error during processing of request
     * @throws ThrottlingException
     *         This exception is thrown when the number of requests exceeds the limit
     * @throws AccessDeniedException
     *         This exception is thrown when a request is denied per access permissions
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource referenced by the operation does not exist
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockDataAutomationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockDataAutomationClient.UpdateBlueprint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/UpdateBlueprint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateBlueprintResponse updateBlueprint(UpdateBlueprintRequest updateBlueprintRequest) throws ConflictException,
            ValidationException, InternalServerException, ThrottlingException, AccessDeniedException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, BedrockDataAutomationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateBlueprintResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateBlueprintResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateBlueprintRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBlueprintRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBlueprint");

            return clientHandler.execute(new ClientExecutionParams<UpdateBlueprintRequest, UpdateBlueprintResponse>()
                    .withOperationName("UpdateBlueprint").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateBlueprintRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateBlueprintRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing Amazon Bedrock Data Automation Project
     * </p>
     *
     * @param updateDataAutomationProjectRequest
     *        Update DataAutomationProject Request
     * @return Result of the UpdateDataAutomationProject operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         This exception is thrown when a request is made beyond the service quota
     * @throws ConflictException
     *         This exception is thrown when there is a conflict performing an operation
     * @throws ValidationException
     *         This exception is thrown when the request's input validation fails
     * @throws InternalServerException
     *         This exception is thrown if there was an unexpected error during processing of request
     * @throws ThrottlingException
     *         This exception is thrown when the number of requests exceeds the limit
     * @throws AccessDeniedException
     *         This exception is thrown when a request is denied per access permissions
     * @throws ResourceNotFoundException
     *         This exception is thrown when a resource referenced by the operation does not exist
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockDataAutomationException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockDataAutomationClient.UpdateDataAutomationProject
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-2023-07-26/UpdateDataAutomationProject"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDataAutomationProjectResponse updateDataAutomationProject(
            UpdateDataAutomationProjectRequest updateDataAutomationProjectRequest) throws ServiceQuotaExceededException,
            ConflictException, ValidationException, InternalServerException, ThrottlingException, AccessDeniedException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, BedrockDataAutomationException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDataAutomationProjectResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDataAutomationProjectResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "ValidationException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ValidationException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ValidationException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(402)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "InternalServerException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerException").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDataAutomationProjectRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDataAutomationProjectRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataAutomationProject");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateDataAutomationProjectRequest, UpdateDataAutomationProjectResponse>()
                            .withOperationName("UpdateDataAutomationProject").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateDataAutomationProjectRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateDataAutomationProjectRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        BedrockDataAutomationServiceClientConfigurationBuilder serviceConfigBuilder = new BedrockDataAutomationServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BedrockDataAutomationException::builder).protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1");
    }

    @Override
    public final BedrockDataAutomationServiceClientConfiguration serviceClientConfiguration() {
        return new BedrockDataAutomationServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
