/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockdataautomationruntime;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.bedrockdataautomationruntime.internal.BedrockDataAutomationRuntimeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.bedrockdataautomationruntime.model.AccessDeniedException;
import software.amazon.awssdk.services.bedrockdataautomationruntime.model.BedrockDataAutomationRuntimeException;
import software.amazon.awssdk.services.bedrockdataautomationruntime.model.GetDataAutomationStatusRequest;
import software.amazon.awssdk.services.bedrockdataautomationruntime.model.GetDataAutomationStatusResponse;
import software.amazon.awssdk.services.bedrockdataautomationruntime.model.InternalServerException;
import software.amazon.awssdk.services.bedrockdataautomationruntime.model.InvokeDataAutomationAsyncRequest;
import software.amazon.awssdk.services.bedrockdataautomationruntime.model.InvokeDataAutomationAsyncResponse;
import software.amazon.awssdk.services.bedrockdataautomationruntime.model.ResourceNotFoundException;
import software.amazon.awssdk.services.bedrockdataautomationruntime.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.bedrockdataautomationruntime.model.ThrottlingException;
import software.amazon.awssdk.services.bedrockdataautomationruntime.model.ValidationException;
import software.amazon.awssdk.services.bedrockdataautomationruntime.transform.GetDataAutomationStatusRequestMarshaller;
import software.amazon.awssdk.services.bedrockdataautomationruntime.transform.InvokeDataAutomationAsyncRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link BedrockDataAutomationRuntimeClient}.
 *
 * @see BedrockDataAutomationRuntimeClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBedrockDataAutomationRuntimeClient implements BedrockDataAutomationRuntimeClient {
    private static final Logger log = Logger.loggerFor(DefaultBedrockDataAutomationRuntimeClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBedrockDataAutomationRuntimeClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * API used to get data automation status.
     * </p>
     *
     * @param getDataAutomationStatusRequest
     *        Structure for request of GetDataAutomationStatus API.
     * @return Result of the GetDataAutomationStatus operation returned by the service.
     * @throws AccessDeniedException
     *         This exception will be thrown when customer does not have access to API.
     * @throws ValidationException
     *         This exception will be thrown when customer provided invalid parameters.
     * @throws InternalServerException
     *         This exception is for any internal un-expected service errors.
     * @throws ThrottlingException
     *         This exception will be thrown when customer reached API TPS limit.
     * @throws ResourceNotFoundException
     *         This exception will be thrown when resource provided from customer not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockDataAutomationRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockDataAutomationRuntimeClient.GetDataAutomationStatus
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-runtime-2024-06-13/GetDataAutomationStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDataAutomationStatusResponse getDataAutomationStatus(GetDataAutomationStatusRequest getDataAutomationStatusRequest)
            throws AccessDeniedException, ValidationException, InternalServerException, ThrottlingException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, BedrockDataAutomationRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDataAutomationStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDataAutomationStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDataAutomationStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDataAutomationStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDataAutomationStatus");

            return clientHandler
                    .execute(new ClientExecutionParams<GetDataAutomationStatusRequest, GetDataAutomationStatusResponse>()
                            .withOperationName("GetDataAutomationStatus").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getDataAutomationStatusRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetDataAutomationStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Async API: Invoke data automation.
     * </p>
     *
     * @param invokeDataAutomationAsyncRequest
     *        Invoke Data Automation Async Request
     * @return Result of the InvokeDataAutomationAsync operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         This exception will be thrown when service quota is exceeded.
     * @throws AccessDeniedException
     *         This exception will be thrown when customer does not have access to API.
     * @throws ValidationException
     *         This exception will be thrown when customer provided invalid parameters.
     * @throws InternalServerException
     *         This exception is for any internal un-expected service errors.
     * @throws ThrottlingException
     *         This exception will be thrown when customer reached API TPS limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockDataAutomationRuntimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockDataAutomationRuntimeClient.InvokeDataAutomationAsync
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-data-automation-runtime-2024-06-13/InvokeDataAutomationAsync"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public InvokeDataAutomationAsyncResponse invokeDataAutomationAsync(
            InvokeDataAutomationAsyncRequest invokeDataAutomationAsyncRequest) throws ServiceQuotaExceededException,
            AccessDeniedException, ValidationException, InternalServerException, ThrottlingException, AwsServiceException,
            SdkClientException, BedrockDataAutomationRuntimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InvokeDataAutomationAsyncResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, InvokeDataAutomationAsyncResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(invokeDataAutomationAsyncRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, invokeDataAutomationAsyncRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock Data Automation Runtime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InvokeDataAutomationAsync");

            return clientHandler
                    .execute(new ClientExecutionParams<InvokeDataAutomationAsyncRequest, InvokeDataAutomationAsyncResponse>()
                            .withOperationName("InvokeDataAutomationAsync").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(invokeDataAutomationAsyncRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new InvokeDataAutomationAsyncRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        BedrockDataAutomationRuntimeServiceClientConfigurationBuilder serviceConfigBuilder = new BedrockDataAutomationRuntimeServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BedrockDataAutomationRuntimeException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build());
    }

    @Override
    public final BedrockDataAutomationRuntimeServiceClientConfiguration serviceClientConfiguration() {
        return new BedrockDataAutomationRuntimeServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
