/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockruntime;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.services.bedrockruntime.model.InvokeModelRequest;
import software.amazon.awssdk.services.bedrockruntime.model.InvokeModelResponse;
import software.amazon.awssdk.services.bedrockruntime.model.InvokeModelWithResponseStreamRequest;
import software.amazon.awssdk.services.bedrockruntime.model.InvokeModelWithResponseStreamResponseHandler;

/**
 * Service client for accessing Amazon Bedrock Runtime asynchronously. This can be created using the static
 * {@link #builder()} method.The asynchronous client performs non-blocking I/O when configured with any
 * {@code SdkAsyncHttpClient} supported in the SDK. However, full non-blocking is not guaranteed as the async client may
 * perform blocking calls in some cases such as credentials retrieval and endpoint discovery as part of the async API
 * call.
 *
 * <p>
 * Describes the API operations for running inference using Amazon Bedrock models.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface BedrockRuntimeAsyncClient extends AwsClient {
    String SERVICE_NAME = "bedrock";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "bedrock-runtime";

    /**
     * <p>
     * Invokes the specified Amazon Bedrock model to run inference using the prompt and inference parameters provided in
     * the request body. You use model inference to generate text, images, and embeddings.
     * </p>
     * <p>
     * For example code, see <i>Invoke model code examples</i> in the <i>Amazon Bedrock User Guide</i>.
     * </p>
     * <p>
     * This operation requires permission for the <code>bedrock:InvokeModel</code> action.
     * </p>
     *
     * @param invokeModelRequest
     * @return A Java Future containing the result of the InvokeModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>ModelTimeoutException The request took too long to process. Processing time exceeded the model
     *         timeout length.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ModelNotReadyException The model specified in the request is not ready to serve inference requests.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>ModelErrorException The request failed due to an error while processing the model.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockRuntimeAsyncClient.InvokeModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/InvokeModel" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<InvokeModelResponse> invokeModel(InvokeModelRequest invokeModelRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Invokes the specified Amazon Bedrock model to run inference using the prompt and inference parameters provided in
     * the request body. You use model inference to generate text, images, and embeddings.
     * </p>
     * <p>
     * For example code, see <i>Invoke model code examples</i> in the <i>Amazon Bedrock User Guide</i>.
     * </p>
     * <p>
     * This operation requires permission for the <code>bedrock:InvokeModel</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link InvokeModelRequest.Builder} avoiding the need to
     * create one manually via {@link InvokeModelRequest#builder()}
     * </p>
     *
     * @param invokeModelRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bedrockruntime.model.InvokeModelRequest.Builder} to create a
     *        request.
     * @return A Java Future containing the result of the InvokeModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>ModelTimeoutException The request took too long to process. Processing time exceeded the model
     *         timeout length.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ModelNotReadyException The model specified in the request is not ready to serve inference requests.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>ModelErrorException The request failed due to an error while processing the model.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockRuntimeAsyncClient.InvokeModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/InvokeModel" target="_top">AWS
     *      API Documentation</a>
     */
    default CompletableFuture<InvokeModelResponse> invokeModel(Consumer<InvokeModelRequest.Builder> invokeModelRequest) {
        return invokeModel(InvokeModelRequest.builder().applyMutation(invokeModelRequest).build());
    }

    /**
     * <p>
     * Invoke the specified Amazon Bedrock model to run inference using the prompt and inference parameters provided in
     * the request body. The response is returned in a stream.
     * </p>
     * <p>
     * To see if a model supports streaming, call <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetFoundationModel.html">GetFoundationModel</a>
     * and check the <code>responseStreamingSupported</code> field in the response.
     * </p>
     * <note>
     * <p>
     * The CLI doesn't support <code>InvokeModelWithResponseStream</code>.
     * </p>
     * </note>
     * <p>
     * For example code, see <i>Invoke model with streaming code example</i> in the <i>Amazon Bedrock User Guide</i>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>bedrock:InvokeModelWithResponseStream</code> action.
     * </p>
     *
     * @param invokeModelWithResponseStreamRequest
     * @return A Java Future containing the result of the InvokeModelWithResponseStream operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>ModelTimeoutException The request took too long to process. Processing time exceeded the model
     *         timeout length.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ModelStreamErrorException An error occurred while streaming the response. Retry your request.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ModelNotReadyException The model specified in the request is not ready to serve inference requests.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>ModelErrorException The request failed due to an error while processing the model.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockRuntimeAsyncClient.InvokeModelWithResponseStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/InvokeModelWithResponseStream"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> invokeModelWithResponseStream(
            InvokeModelWithResponseStreamRequest invokeModelWithResponseStreamRequest,
            InvokeModelWithResponseStreamResponseHandler asyncResponseHandler) {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Invoke the specified Amazon Bedrock model to run inference using the prompt and inference parameters provided in
     * the request body. The response is returned in a stream.
     * </p>
     * <p>
     * To see if a model supports streaming, call <a
     * href="https://docs.aws.amazon.com/bedrock/latest/APIReference/API_GetFoundationModel.html">GetFoundationModel</a>
     * and check the <code>responseStreamingSupported</code> field in the response.
     * </p>
     * <note>
     * <p>
     * The CLI doesn't support <code>InvokeModelWithResponseStream</code>.
     * </p>
     * </note>
     * <p>
     * For example code, see <i>Invoke model with streaming code example</i> in the <i>Amazon Bedrock User Guide</i>.
     * </p>
     * <p>
     * This operation requires permissions to perform the <code>bedrock:InvokeModelWithResponseStream</code> action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link InvokeModelWithResponseStreamRequest.Builder}
     * avoiding the need to create one manually via {@link InvokeModelWithResponseStreamRequest#builder()}
     * </p>
     *
     * @param invokeModelWithResponseStreamRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bedrockruntime.model.InvokeModelWithResponseStreamRequest.Builder}
     *        to create a request.
     * @return A Java Future containing the result of the InvokeModelWithResponseStream operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException The request is denied because of missing access permissions.</li>
     *         <li>ResourceNotFoundException The specified resource ARN was not found. Check the ARN and try your
     *         request again.</li>
     *         <li>ThrottlingException The number of requests exceeds the limit. Resubmit your request later.</li>
     *         <li>ModelTimeoutException The request took too long to process. Processing time exceeded the model
     *         timeout length.</li>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ModelStreamErrorException An error occurred while streaming the response. Retry your request.</li>
     *         <li>ValidationException Input validation failed. Check your request parameters and retry the request.</li>
     *         <li>ModelNotReadyException The model specified in the request is not ready to serve inference requests.</li>
     *         <li>ServiceQuotaExceededException The number of requests exceeds the service quota. Resubmit your request
     *         later.</li>
     *         <li>ModelErrorException The request failed due to an error while processing the model.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BedrockRuntimeException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample BedrockRuntimeAsyncClient.InvokeModelWithResponseStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-runtime-2023-09-30/InvokeModelWithResponseStream"
     *      target="_top">AWS API Documentation</a>
     */
    default CompletableFuture<Void> invokeModelWithResponseStream(
            Consumer<InvokeModelWithResponseStreamRequest.Builder> invokeModelWithResponseStreamRequest,
            InvokeModelWithResponseStreamResponseHandler asyncResponseHandler) {
        return invokeModelWithResponseStream(
                InvokeModelWithResponseStreamRequest.builder().applyMutation(invokeModelWithResponseStreamRequest).build(),
                asyncResponseHandler);
    }

    @Override
    default BedrockRuntimeServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }

    /**
     * Create a {@link BedrockRuntimeAsyncClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static BedrockRuntimeAsyncClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link BedrockRuntimeAsyncClient}.
     */
    static BedrockRuntimeAsyncClientBuilder builder() {
        return new DefaultBedrockRuntimeAsyncClientBuilder();
    }
}
