/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockruntime.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the content of a document.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DocumentSource implements SdkPojo, Serializable, ToCopyableBuilder<DocumentSource.Builder, DocumentSource> {
    private static final SdkField<SdkBytes> BYTES_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("bytes").getter(getter(DocumentSource::bytes)).setter(setter(Builder::bytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bytes").build()).build();

    private static final SdkField<S3Location> S3_LOCATION_FIELD = SdkField.<S3Location> builder(MarshallingType.SDK_POJO)
            .memberName("s3Location").getter(getter(DocumentSource::s3Location)).setter(setter(Builder::s3Location))
            .constructor(S3Location::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("s3Location").build()).build();

    private static final SdkField<String> TEXT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("text")
            .getter(getter(DocumentSource::text)).setter(setter(Builder::text))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("text").build()).build();

    private static final SdkField<List<DocumentContentBlock>> CONTENT_FIELD = SdkField
            .<List<DocumentContentBlock>> builder(MarshallingType.LIST)
            .memberName("content")
            .getter(getter(DocumentSource::content))
            .setter(setter(Builder::content))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("content").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DocumentContentBlock> builder(MarshallingType.SDK_POJO)
                                            .constructor(DocumentContentBlock::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BYTES_FIELD,
            S3_LOCATION_FIELD, TEXT_FIELD, CONTENT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final SdkBytes bytes;

    private final S3Location s3Location;

    private final String text;

    private final List<DocumentContentBlock> content;

    private final Type type;

    private DocumentSource(BuilderImpl builder) {
        this.bytes = builder.bytes;
        this.s3Location = builder.s3Location;
        this.text = builder.text;
        this.content = builder.content;
        this.type = builder.type;
    }

    /**
     * <p>
     * The raw bytes for the document. If you use an Amazon Web Services SDK, you don't need to encode the bytes in
     * base64.
     * </p>
     * 
     * @return The raw bytes for the document. If you use an Amazon Web Services SDK, you don't need to encode the bytes
     *         in base64.
     */
    public final SdkBytes bytes() {
        return bytes;
    }

    /**
     * <p>
     * The location of a document object in an Amazon S3 bucket. To see which models support S3 uploads, see <a href=
     * "https://docs.aws.amazon.com/bedrock/latest/userguide/conversation-inference-supported-models-features.html"
     * >Supported models and features for Converse</a>.
     * </p>
     * 
     * @return The location of a document object in an Amazon S3 bucket. To see which models support S3 uploads, see <a
     *         href=
     *         "https://docs.aws.amazon.com/bedrock/latest/userguide/conversation-inference-supported-models-features.html"
     *         >Supported models and features for Converse</a>.
     */
    public final S3Location s3Location() {
        return s3Location;
    }

    /**
     * <p>
     * The text content of the document source.
     * </p>
     * 
     * @return The text content of the document source.
     */
    public final String text() {
        return text;
    }

    /**
     * For responses, this returns true if the service returned a value for the Content property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasContent() {
        return content != null && !(content instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The structured content of the document source, which may include various content blocks such as text, images, or
     * other document elements.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasContent} method.
     * </p>
     * 
     * @return The structured content of the document source, which may include various content blocks such as text,
     *         images, or other document elements.
     */
    public final List<DocumentContentBlock> content() {
        return content;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bytes());
        hashCode = 31 * hashCode + Objects.hashCode(s3Location());
        hashCode = 31 * hashCode + Objects.hashCode(text());
        hashCode = 31 * hashCode + Objects.hashCode(hasContent() ? content() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DocumentSource)) {
            return false;
        }
        DocumentSource other = (DocumentSource) obj;
        return Objects.equals(bytes(), other.bytes()) && Objects.equals(s3Location(), other.s3Location())
                && Objects.equals(text(), other.text()) && hasContent() == other.hasContent()
                && Objects.equals(content(), other.content());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DocumentSource").add("Bytes", bytes()).add("S3Location", s3Location()).add("Text", text())
                .add("Content", hasContent() ? content() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "bytes":
            return Optional.ofNullable(clazz.cast(bytes()));
        case "s3Location":
            return Optional.ofNullable(clazz.cast(s3Location()));
        case "text":
            return Optional.ofNullable(clazz.cast(text()));
        case "content":
            return Optional.ofNullable(clazz.cast(content()));
        default:
            return Optional.empty();
        }
    }

    /**
     * Create an instance of this class with {@link #bytes()} initialized to the given value.
     *
     * <p>
     * The raw bytes for the document. If you use an Amazon Web Services SDK, you don't need to encode the bytes in
     * base64.
     * </p>
     * 
     * @param bytes
     *        The raw bytes for the document. If you use an Amazon Web Services SDK, you don't need to encode the bytes
     *        in base64.
     */
    public static DocumentSource fromBytes(SdkBytes bytes) {
        return builder().bytes(bytes).build();
    }

    /**
     * Create an instance of this class with {@link #s3Location()} initialized to the given value.
     *
     * <p>
     * The location of a document object in an Amazon S3 bucket. To see which models support S3 uploads, see <a href=
     * "https://docs.aws.amazon.com/bedrock/latest/userguide/conversation-inference-supported-models-features.html"
     * >Supported models and features for Converse</a>.
     * </p>
     * 
     * @param s3Location
     *        The location of a document object in an Amazon S3 bucket. To see which models support S3 uploads, see <a
     *        href=
     *        "https://docs.aws.amazon.com/bedrock/latest/userguide/conversation-inference-supported-models-features.html"
     *        >Supported models and features for Converse</a>.
     */
    public static DocumentSource fromS3Location(S3Location s3Location) {
        return builder().s3Location(s3Location).build();
    }

    /**
     * Create an instance of this class with {@link #s3Location()} initialized to the given value.
     *
     * <p>
     * The location of a document object in an Amazon S3 bucket. To see which models support S3 uploads, see <a href=
     * "https://docs.aws.amazon.com/bedrock/latest/userguide/conversation-inference-supported-models-features.html"
     * >Supported models and features for Converse</a>.
     * </p>
     * 
     * @param s3Location
     *        The location of a document object in an Amazon S3 bucket. To see which models support S3 uploads, see <a
     *        href=
     *        "https://docs.aws.amazon.com/bedrock/latest/userguide/conversation-inference-supported-models-features.html"
     *        >Supported models and features for Converse</a>.
     */
    public static DocumentSource fromS3Location(Consumer<S3Location.Builder> s3Location) {
        S3Location.Builder builder = S3Location.builder();
        s3Location.accept(builder);
        return fromS3Location(builder.build());
    }

    /**
     * Create an instance of this class with {@link #text()} initialized to the given value.
     *
     * <p>
     * The text content of the document source.
     * </p>
     * 
     * @param text
     *        The text content of the document source.
     */
    public static DocumentSource fromText(String text) {
        return builder().text(text).build();
    }

    /**
     * Create an instance of this class with {@link #content()} initialized to the given value.
     *
     * <p>
     * The structured content of the document source, which may include various content blocks such as text, images, or
     * other document elements.
     * </p>
     * 
     * @param content
     *        The structured content of the document source, which may include various content blocks such as text,
     *        images, or other document elements.
     */
    public static DocumentSource fromContent(List<DocumentContentBlock> content) {
        return builder().content(content).build();
    }

    /**
     * Retrieve an enum value representing which member of this object is populated.
     *
     * When this class is returned in a service response, this will be {@link Type#UNKNOWN_TO_SDK_VERSION} if the
     * service returned a member that is only known to a newer SDK version.
     *
     * When this class is created directly in your code, this will be {@link Type#UNKNOWN_TO_SDK_VERSION} if zero
     * members are set, and {@code null} if more than one member is set.
     */
    public Type type() {
        return type;
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("bytes", BYTES_FIELD);
        map.put("s3Location", S3_LOCATION_FIELD);
        map.put("text", TEXT_FIELD);
        map.put("content", CONTENT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DocumentSource, T> g) {
        return obj -> g.apply((DocumentSource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DocumentSource> {
        /**
         * <p>
         * The raw bytes for the document. If you use an Amazon Web Services SDK, you don't need to encode the bytes in
         * base64.
         * </p>
         * 
         * @param bytes
         *        The raw bytes for the document. If you use an Amazon Web Services SDK, you don't need to encode the
         *        bytes in base64.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bytes(SdkBytes bytes);

        /**
         * <p>
         * The location of a document object in an Amazon S3 bucket. To see which models support S3 uploads, see <a
         * href=
         * "https://docs.aws.amazon.com/bedrock/latest/userguide/conversation-inference-supported-models-features.html"
         * >Supported models and features for Converse</a>.
         * </p>
         * 
         * @param s3Location
         *        The location of a document object in an Amazon S3 bucket. To see which models support S3 uploads, see
         *        <a href=
         *        "https://docs.aws.amazon.com/bedrock/latest/userguide/conversation-inference-supported-models-features.html"
         *        >Supported models and features for Converse</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Location(S3Location s3Location);

        /**
         * <p>
         * The location of a document object in an Amazon S3 bucket. To see which models support S3 uploads, see <a
         * href=
         * "https://docs.aws.amazon.com/bedrock/latest/userguide/conversation-inference-supported-models-features.html"
         * >Supported models and features for Converse</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link S3Location.Builder} avoiding the need to
         * create one manually via {@link S3Location#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link S3Location.Builder#build()} is called immediately and its result
         * is passed to {@link #s3Location(S3Location)}.
         * 
         * @param s3Location
         *        a consumer that will call methods on {@link S3Location.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3Location(S3Location)
         */
        default Builder s3Location(Consumer<S3Location.Builder> s3Location) {
            return s3Location(S3Location.builder().applyMutation(s3Location).build());
        }

        /**
         * <p>
         * The text content of the document source.
         * </p>
         * 
         * @param text
         *        The text content of the document source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder text(String text);

        /**
         * <p>
         * The structured content of the document source, which may include various content blocks such as text, images,
         * or other document elements.
         * </p>
         * 
         * @param content
         *        The structured content of the document source, which may include various content blocks such as text,
         *        images, or other document elements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder content(Collection<DocumentContentBlock> content);

        /**
         * <p>
         * The structured content of the document source, which may include various content blocks such as text, images,
         * or other document elements.
         * </p>
         * 
         * @param content
         *        The structured content of the document source, which may include various content blocks such as text,
         *        images, or other document elements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder content(DocumentContentBlock... content);

        /**
         * <p>
         * The structured content of the document source, which may include various content blocks such as text, images,
         * or other document elements.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.bedrockruntime.model.DocumentContentBlock.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.bedrockruntime.model.DocumentContentBlock#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.bedrockruntime.model.DocumentContentBlock.Builder#build()} is called
         * immediately and its result is passed to {@link #content(List<DocumentContentBlock>)}.
         * 
         * @param content
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.bedrockruntime.model.DocumentContentBlock.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #content(java.util.Collection<DocumentContentBlock>)
         */
        Builder content(Consumer<DocumentContentBlock.Builder>... content);
    }

    static final class BuilderImpl implements Builder {
        private SdkBytes bytes;

        private S3Location s3Location;

        private String text;

        private List<DocumentContentBlock> content = DefaultSdkAutoConstructList.getInstance();

        private Type type = Type.UNKNOWN_TO_SDK_VERSION;

        private Set<Type> setTypes = EnumSet.noneOf(Type.class);

        private BuilderImpl() {
        }

        private BuilderImpl(DocumentSource model) {
            bytes(model.bytes);
            s3Location(model.s3Location);
            text(model.text);
            content(model.content);
        }

        public final ByteBuffer getBytes() {
            return bytes == null ? null : bytes.asByteBuffer();
        }

        public final void setBytes(ByteBuffer bytes) {
            bytes(bytes == null ? null : SdkBytes.fromByteBuffer(bytes));
        }

        @Override
        public final Builder bytes(SdkBytes bytes) {
            Object oldValue = this.bytes;
            this.bytes = bytes;
            handleUnionValueChange(Type.BYTES, oldValue, this.bytes);
            return this;
        }

        public final S3Location.Builder getS3Location() {
            return s3Location != null ? s3Location.toBuilder() : null;
        }

        public final void setS3Location(S3Location.BuilderImpl s3Location) {
            Object oldValue = this.s3Location;
            this.s3Location = s3Location != null ? s3Location.build() : null;
            handleUnionValueChange(Type.S3_LOCATION, oldValue, this.s3Location);
        }

        @Override
        public final Builder s3Location(S3Location s3Location) {
            Object oldValue = this.s3Location;
            this.s3Location = s3Location;
            handleUnionValueChange(Type.S3_LOCATION, oldValue, this.s3Location);
            return this;
        }

        public final String getText() {
            return text;
        }

        public final void setText(String text) {
            Object oldValue = this.text;
            this.text = text;
            handleUnionValueChange(Type.TEXT, oldValue, this.text);
        }

        @Override
        public final Builder text(String text) {
            Object oldValue = this.text;
            this.text = text;
            handleUnionValueChange(Type.TEXT, oldValue, this.text);
            return this;
        }

        public final List<DocumentContentBlock.Builder> getContent() {
            List<DocumentContentBlock.Builder> result = DocumentContentBlocksCopier.copyToBuilder(this.content);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setContent(Collection<DocumentContentBlock.BuilderImpl> content) {
            Object oldValue = this.content;
            this.content = DocumentContentBlocksCopier.copyFromBuilder(content);
            handleUnionValueChange(Type.CONTENT, oldValue, this.content);
        }

        @Override
        public final Builder content(Collection<DocumentContentBlock> content) {
            Object oldValue = this.content;
            this.content = DocumentContentBlocksCopier.copy(content);
            handleUnionValueChange(Type.CONTENT, oldValue, this.content);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder content(DocumentContentBlock... content) {
            content(Arrays.asList(content));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder content(Consumer<DocumentContentBlock.Builder>... content) {
            content(Stream.of(content).map(c -> DocumentContentBlock.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public DocumentSource build() {
            return new DocumentSource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }

        private final void handleUnionValueChange(Type type, Object oldValue, Object newValue) {
            if (this.type == type || oldValue == newValue) {
                return;
            }
            if (newValue == null || newValue instanceof SdkAutoConstructList || newValue instanceof SdkAutoConstructMap) {
                setTypes.remove(type);
            } else if (oldValue == null || oldValue instanceof SdkAutoConstructList || oldValue instanceof SdkAutoConstructMap) {
                setTypes.add(type);
            }
            if (setTypes.size() == 1) {
                this.type = setTypes.iterator().next();
            } else if (setTypes.isEmpty()) {
                this.type = Type.UNKNOWN_TO_SDK_VERSION;
            } else {
                this.type = null;
            }
        }
    }

    /**
     * @see DocumentSource#type()
     */
    public enum Type {
        BYTES,

        S3_LOCATION,

        TEXT,

        CONTENT,

        UNKNOWN_TO_SDK_VERSION
    }
}
