/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.billing;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.billing.internal.BillingServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.billing.model.AccessDeniedException;
import software.amazon.awssdk.services.billing.model.BillingException;
import software.amazon.awssdk.services.billing.model.ConflictException;
import software.amazon.awssdk.services.billing.model.CreateBillingViewRequest;
import software.amazon.awssdk.services.billing.model.CreateBillingViewResponse;
import software.amazon.awssdk.services.billing.model.DeleteBillingViewRequest;
import software.amazon.awssdk.services.billing.model.DeleteBillingViewResponse;
import software.amazon.awssdk.services.billing.model.GetBillingViewRequest;
import software.amazon.awssdk.services.billing.model.GetBillingViewResponse;
import software.amazon.awssdk.services.billing.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.billing.model.GetResourcePolicyResponse;
import software.amazon.awssdk.services.billing.model.InternalServerException;
import software.amazon.awssdk.services.billing.model.ListBillingViewsRequest;
import software.amazon.awssdk.services.billing.model.ListBillingViewsResponse;
import software.amazon.awssdk.services.billing.model.ListSourceViewsForBillingViewRequest;
import software.amazon.awssdk.services.billing.model.ListSourceViewsForBillingViewResponse;
import software.amazon.awssdk.services.billing.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.billing.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.billing.model.ResourceNotFoundException;
import software.amazon.awssdk.services.billing.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.billing.model.TagResourceRequest;
import software.amazon.awssdk.services.billing.model.TagResourceResponse;
import software.amazon.awssdk.services.billing.model.ThrottlingException;
import software.amazon.awssdk.services.billing.model.UntagResourceRequest;
import software.amazon.awssdk.services.billing.model.UntagResourceResponse;
import software.amazon.awssdk.services.billing.model.UpdateBillingViewRequest;
import software.amazon.awssdk.services.billing.model.UpdateBillingViewResponse;
import software.amazon.awssdk.services.billing.model.ValidationException;
import software.amazon.awssdk.services.billing.transform.CreateBillingViewRequestMarshaller;
import software.amazon.awssdk.services.billing.transform.DeleteBillingViewRequestMarshaller;
import software.amazon.awssdk.services.billing.transform.GetBillingViewRequestMarshaller;
import software.amazon.awssdk.services.billing.transform.GetResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.billing.transform.ListBillingViewsRequestMarshaller;
import software.amazon.awssdk.services.billing.transform.ListSourceViewsForBillingViewRequestMarshaller;
import software.amazon.awssdk.services.billing.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.billing.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.billing.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.billing.transform.UpdateBillingViewRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link BillingClient}.
 *
 * @see BillingClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBillingClient implements BillingClient {
    private static final Logger log = Logger.loggerFor(DefaultBillingClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBillingClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a billing view with the specified billing view attributes.
     * </p>
     *
     * @param createBillingViewRequest
     * @return Result of the CreateBillingView operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         You've reached the limit of resources you can create, or exceeded the size of an individual resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ConflictException
     *         The requested operation would cause a conflict with the current state of a service resource associated
     *         with the request. Resolve the conflict before retrying this request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BillingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BillingClient.CreateBillingView
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/billing-2023-09-07/CreateBillingView" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateBillingViewResponse createBillingView(CreateBillingViewRequest createBillingViewRequest)
            throws ServiceQuotaExceededException, ThrottlingException, AccessDeniedException, ConflictException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException, BillingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBillingViewResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateBillingViewResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBillingViewRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBillingViewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Billing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBillingView");

            return clientHandler.execute(new ClientExecutionParams<CreateBillingViewRequest, CreateBillingViewResponse>()
                    .withOperationName("CreateBillingView").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createBillingViewRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateBillingViewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified billing view.
     * </p>
     *
     * @param deleteBillingViewRequest
     * @return Result of the DeleteBillingView operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ConflictException
     *         The requested operation would cause a conflict with the current state of a service resource associated
     *         with the request. Resolve the conflict before retrying this request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BillingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BillingClient.DeleteBillingView
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/billing-2023-09-07/DeleteBillingView" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteBillingViewResponse deleteBillingView(DeleteBillingViewRequest deleteBillingViewRequest)
            throws ThrottlingException, AccessDeniedException, ConflictException, ValidationException, InternalServerException,
            AwsServiceException, SdkClientException, BillingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBillingViewResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteBillingViewResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBillingViewRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBillingViewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Billing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBillingView");

            return clientHandler.execute(new ClientExecutionParams<DeleteBillingViewRequest, DeleteBillingViewResponse>()
                    .withOperationName("DeleteBillingView").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteBillingViewRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteBillingViewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the metadata associated to the specified billing view ARN.
     * </p>
     *
     * @param getBillingViewRequest
     * @return Result of the GetBillingView operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The specified ARN in the request doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BillingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BillingClient.GetBillingView
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/billing-2023-09-07/GetBillingView" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetBillingViewResponse getBillingView(GetBillingViewRequest getBillingViewRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, BillingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBillingViewResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetBillingViewResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBillingViewRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBillingViewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Billing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBillingView");

            return clientHandler.execute(new ClientExecutionParams<GetBillingViewRequest, GetBillingViewResponse>()
                    .withOperationName("GetBillingView").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getBillingViewRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetBillingViewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the resource-based policy document attached to the resource in <code>JSON</code> format.
     * </p>
     *
     * @param getResourcePolicyRequest
     * @return Result of the GetResourcePolicy operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The specified ARN in the request doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BillingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BillingClient.GetResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/billing-2023-09-07/GetResourcePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetResourcePolicyResponse getResourcePolicy(GetResourcePolicyRequest getResourcePolicyRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, BillingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResourcePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Billing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<GetResourcePolicyRequest, GetResourcePolicyResponse>()
                    .withOperationName("GetResourcePolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getResourcePolicyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the billing views available for a given time period.
     * </p>
     * <p>
     * Every Amazon Web Services account has a unique <code>PRIMARY</code> billing view that represents the billing data
     * available by default. Accounts that use Billing Conductor also have <code>BILLING_GROUP</code> billing views
     * representing pro forma costs associated with each created billing group.
     * </p>
     *
     * @param listBillingViewsRequest
     * @return Result of the ListBillingViews operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BillingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BillingClient.ListBillingViews
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/billing-2023-09-07/ListBillingViews" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListBillingViewsResponse listBillingViews(ListBillingViewsRequest listBillingViewsRequest) throws ThrottlingException,
            AccessDeniedException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            BillingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBillingViewsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListBillingViewsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listBillingViewsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBillingViewsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Billing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBillingViews");

            return clientHandler.execute(new ClientExecutionParams<ListBillingViewsRequest, ListBillingViewsResponse>()
                    .withOperationName("ListBillingViews").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listBillingViewsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListBillingViewsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the source views (managed Amazon Web Services billing views) associated with the billing view.
     * </p>
     *
     * @param listSourceViewsForBillingViewRequest
     * @return Result of the ListSourceViewsForBillingView operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The specified ARN in the request doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BillingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BillingClient.ListSourceViewsForBillingView
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/billing-2023-09-07/ListSourceViewsForBillingView"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSourceViewsForBillingViewResponse listSourceViewsForBillingView(
            ListSourceViewsForBillingViewRequest listSourceViewsForBillingViewRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, BillingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSourceViewsForBillingViewResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSourceViewsForBillingViewResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSourceViewsForBillingViewRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listSourceViewsForBillingViewRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Billing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSourceViewsForBillingView");

            return clientHandler
                    .execute(new ClientExecutionParams<ListSourceViewsForBillingViewRequest, ListSourceViewsForBillingViewResponse>()
                            .withOperationName("ListSourceViewsForBillingView").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listSourceViewsForBillingViewRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListSourceViewsForBillingViewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists tags associated with the billing view resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The specified ARN in the request doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BillingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BillingClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/billing-2023-09-07/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ThrottlingException, ResourceNotFoundException, AccessDeniedException, ValidationException,
            InternalServerException, AwsServiceException, SdkClientException, BillingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Billing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * An API operation for adding one or more tags (key-value pairs) to a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The specified ARN in the request doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BillingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BillingClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/billing-2023-09-07/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, BillingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Billing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from a resource. Specify only tag keys in your request. Don't specify the value.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The specified ARN in the request doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BillingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BillingClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/billing-2023-09-07/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ThrottlingException,
            ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, BillingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Billing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * An API to update the attributes of the billing view.
     * </p>
     *
     * @param updateBillingViewRequest
     * @return Result of the UpdateBillingView operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         You've reached the limit of resources you can create, or exceeded the size of an individual resource.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ResourceNotFoundException
     *         The specified ARN in the request doesn't exist.
     * @throws AccessDeniedException
     *         You don't have sufficient access to perform this action.
     * @throws ConflictException
     *         The requested operation would cause a conflict with the current state of a service resource associated
     *         with the request. Resolve the conflict before retrying this request.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         The request processing failed because of an unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BillingException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BillingClient.UpdateBillingView
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/billing-2023-09-07/UpdateBillingView" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateBillingViewResponse updateBillingView(UpdateBillingViewRequest updateBillingViewRequest)
            throws ServiceQuotaExceededException, ThrottlingException, ResourceNotFoundException, AccessDeniedException,
            ConflictException, ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            BillingException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateBillingViewResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateBillingViewResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateBillingViewRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBillingViewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Billing");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBillingView");

            return clientHandler.execute(new ClientExecutionParams<UpdateBillingViewRequest, UpdateBillingViewResponse>()
                    .withOperationName("UpdateBillingView").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateBillingViewRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateBillingViewRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        BillingServiceClientConfigurationBuilder serviceConfigBuilder = new BillingServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BillingException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build());
    }

    @Override
    public final BillingServiceClientConfiguration serviceClientConfiguration() {
        return new BillingServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
