/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.billing.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateBillingViewRequest extends BillingRequest implements
        ToCopyableBuilder<CreateBillingViewRequest.Builder, CreateBillingViewRequest> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(CreateBillingViewRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(CreateBillingViewRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<List<String>> SOURCE_VIEWS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("sourceViews")
            .getter(getter(CreateBillingViewRequest::sourceViews))
            .setter(setter(Builder::sourceViews))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceViews").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Expression> DATA_FILTER_EXPRESSION_FIELD = SdkField
            .<Expression> builder(MarshallingType.SDK_POJO).memberName("dataFilterExpression")
            .getter(getter(CreateBillingViewRequest::dataFilterExpression)).setter(setter(Builder::dataFilterExpression))
            .constructor(Expression::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dataFilterExpression").build())
            .build();

    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("clientToken")
            .getter(getter(CreateBillingViewRequest::clientToken))
            .setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clientToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<List<ResourceTag>> RESOURCE_TAGS_FIELD = SdkField
            .<List<ResourceTag>> builder(MarshallingType.LIST)
            .memberName("resourceTags")
            .getter(getter(CreateBillingViewRequest::resourceTags))
            .setter(setter(Builder::resourceTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceTags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceTag> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceTag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD,
            SOURCE_VIEWS_FIELD, DATA_FILTER_EXPRESSION_FIELD, CLIENT_TOKEN_FIELD, RESOURCE_TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String name;

    private final String description;

    private final List<String> sourceViews;

    private final Expression dataFilterExpression;

    private final String clientToken;

    private final List<ResourceTag> resourceTags;

    private CreateBillingViewRequest(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.description = builder.description;
        this.sourceViews = builder.sourceViews;
        this.dataFilterExpression = builder.dataFilterExpression;
        this.clientToken = builder.clientToken;
        this.resourceTags = builder.resourceTags;
    }

    /**
     * <p>
     * The name of the billing view.
     * </p>
     * 
     * @return The name of the billing view.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The description of the billing view.
     * </p>
     * 
     * @return The description of the billing view.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the SourceViews property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSourceViews() {
        return sourceViews != null && !(sourceViews instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of billing views used as the data source for the custom billing view.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSourceViews} method.
     * </p>
     * 
     * @return A list of billing views used as the data source for the custom billing view.
     */
    public final List<String> sourceViews() {
        return sourceViews;
    }

    /**
     * <p>
     * See <a href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_billing_Expression.html">
     * Expression</a>. Billing view only supports <code>LINKED_ACCOUNT</code> and <code>Tags</code>.
     * </p>
     * 
     * @return See <a
     *         href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_billing_Expression.html"
     *         >Expression</a>. Billing view only supports <code>LINKED_ACCOUNT</code> and <code>Tags</code>.
     */
    public final Expression dataFilterExpression() {
        return dataFilterExpression;
    }

    /**
     * <p>
     * A unique, case-sensitive identifier you specify to ensure idempotency of the request. Idempotency ensures that an
     * API request completes no more than one time. If the original request completes successfully, any subsequent
     * retries complete successfully without performing any further actions with an idempotent request.
     * </p>
     * 
     * @return A unique, case-sensitive identifier you specify to ensure idempotency of the request. Idempotency ensures
     *         that an API request completes no more than one time. If the original request completes successfully, any
     *         subsequent retries complete successfully without performing any further actions with an idempotent
     *         request.
     */
    public final String clientToken() {
        return clientToken;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceTags property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceTags() {
        return resourceTags != null && !(resourceTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of key value map specifying tags associated to the billing view being created.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTags} method.
     * </p>
     * 
     * @return A list of key value map specifying tags associated to the billing view being created.
     */
    public final List<ResourceTag> resourceTags() {
        return resourceTags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasSourceViews() ? sourceViews() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dataFilterExpression());
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTags() ? resourceTags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateBillingViewRequest)) {
            return false;
        }
        CreateBillingViewRequest other = (CreateBillingViewRequest) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && hasSourceViews() == other.hasSourceViews() && Objects.equals(sourceViews(), other.sourceViews())
                && Objects.equals(dataFilterExpression(), other.dataFilterExpression())
                && Objects.equals(clientToken(), other.clientToken()) && hasResourceTags() == other.hasResourceTags()
                && Objects.equals(resourceTags(), other.resourceTags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateBillingViewRequest")
                .add("Name", name() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Description", description() == null ? null : "*** Sensitive Data Redacted ***")
                .add("SourceViews", hasSourceViews() ? sourceViews() : null).add("DataFilterExpression", dataFilterExpression())
                .add("ClientToken", clientToken()).add("ResourceTags", hasResourceTags() ? resourceTags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "sourceViews":
            return Optional.ofNullable(clazz.cast(sourceViews()));
        case "dataFilterExpression":
            return Optional.ofNullable(clazz.cast(dataFilterExpression()));
        case "clientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        case "resourceTags":
            return Optional.ofNullable(clazz.cast(resourceTags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("description", DESCRIPTION_FIELD);
        map.put("sourceViews", SOURCE_VIEWS_FIELD);
        map.put("dataFilterExpression", DATA_FILTER_EXPRESSION_FIELD);
        map.put("clientToken", CLIENT_TOKEN_FIELD);
        map.put("resourceTags", RESOURCE_TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateBillingViewRequest, T> g) {
        return obj -> g.apply((CreateBillingViewRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends BillingRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateBillingViewRequest> {
        /**
         * <p>
         * The name of the billing view.
         * </p>
         * 
         * @param name
         *        The name of the billing view.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The description of the billing view.
         * </p>
         * 
         * @param description
         *        The description of the billing view.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A list of billing views used as the data source for the custom billing view.
         * </p>
         * 
         * @param sourceViews
         *        A list of billing views used as the data source for the custom billing view.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceViews(Collection<String> sourceViews);

        /**
         * <p>
         * A list of billing views used as the data source for the custom billing view.
         * </p>
         * 
         * @param sourceViews
         *        A list of billing views used as the data source for the custom billing view.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceViews(String... sourceViews);

        /**
         * <p>
         * See <a
         * href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_billing_Expression.html">
         * Expression</a>. Billing view only supports <code>LINKED_ACCOUNT</code> and <code>Tags</code>.
         * </p>
         * 
         * @param dataFilterExpression
         *        See <a href=
         *        "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_billing_Expression.html"
         *        >Expression</a>. Billing view only supports <code>LINKED_ACCOUNT</code> and <code>Tags</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataFilterExpression(Expression dataFilterExpression);

        /**
         * <p>
         * See <a
         * href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_billing_Expression.html">
         * Expression</a>. Billing view only supports <code>LINKED_ACCOUNT</code> and <code>Tags</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link Expression.Builder} avoiding the need to
         * create one manually via {@link Expression#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Expression.Builder#build()} is called immediately and its result
         * is passed to {@link #dataFilterExpression(Expression)}.
         * 
         * @param dataFilterExpression
         *        a consumer that will call methods on {@link Expression.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataFilterExpression(Expression)
         */
        default Builder dataFilterExpression(Consumer<Expression.Builder> dataFilterExpression) {
            return dataFilterExpression(Expression.builder().applyMutation(dataFilterExpression).build());
        }

        /**
         * <p>
         * A unique, case-sensitive identifier you specify to ensure idempotency of the request. Idempotency ensures
         * that an API request completes no more than one time. If the original request completes successfully, any
         * subsequent retries complete successfully without performing any further actions with an idempotent request.
         * </p>
         * 
         * @param clientToken
         *        A unique, case-sensitive identifier you specify to ensure idempotency of the request. Idempotency
         *        ensures that an API request completes no more than one time. If the original request completes
         *        successfully, any subsequent retries complete successfully without performing any further actions with
         *        an idempotent request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        /**
         * <p>
         * A list of key value map specifying tags associated to the billing view being created.
         * </p>
         * 
         * @param resourceTags
         *        A list of key value map specifying tags associated to the billing view being created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTags(Collection<ResourceTag> resourceTags);

        /**
         * <p>
         * A list of key value map specifying tags associated to the billing view being created.
         * </p>
         * 
         * @param resourceTags
         *        A list of key value map specifying tags associated to the billing view being created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTags(ResourceTag... resourceTags);

        /**
         * <p>
         * A list of key value map specifying tags associated to the billing view being created.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.billing.model.ResourceTag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.billing.model.ResourceTag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.billing.model.ResourceTag.Builder#build()} is called immediately and
         * its result is passed to {@link #resourceTags(List<ResourceTag>)}.
         * 
         * @param resourceTags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.billing.model.ResourceTag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceTags(java.util.Collection<ResourceTag>)
         */
        Builder resourceTags(Consumer<ResourceTag.Builder>... resourceTags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends BillingRequest.BuilderImpl implements Builder {
        private String name;

        private String description;

        private List<String> sourceViews = DefaultSdkAutoConstructList.getInstance();

        private Expression dataFilterExpression;

        private String clientToken;

        private List<ResourceTag> resourceTags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateBillingViewRequest model) {
            super(model);
            name(model.name);
            description(model.description);
            sourceViews(model.sourceViews);
            dataFilterExpression(model.dataFilterExpression);
            clientToken(model.clientToken);
            resourceTags(model.resourceTags);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Collection<String> getSourceViews() {
            if (sourceViews instanceof SdkAutoConstructList) {
                return null;
            }
            return sourceViews;
        }

        public final void setSourceViews(Collection<String> sourceViews) {
            this.sourceViews = BillingViewSourceViewsListCopier.copy(sourceViews);
        }

        @Override
        public final Builder sourceViews(Collection<String> sourceViews) {
            this.sourceViews = BillingViewSourceViewsListCopier.copy(sourceViews);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sourceViews(String... sourceViews) {
            sourceViews(Arrays.asList(sourceViews));
            return this;
        }

        public final Expression.Builder getDataFilterExpression() {
            return dataFilterExpression != null ? dataFilterExpression.toBuilder() : null;
        }

        public final void setDataFilterExpression(Expression.BuilderImpl dataFilterExpression) {
            this.dataFilterExpression = dataFilterExpression != null ? dataFilterExpression.build() : null;
        }

        @Override
        public final Builder dataFilterExpression(Expression dataFilterExpression) {
            this.dataFilterExpression = dataFilterExpression;
            return this;
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        public final List<ResourceTag.Builder> getResourceTags() {
            List<ResourceTag.Builder> result = ResourceTagListCopier.copyToBuilder(this.resourceTags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setResourceTags(Collection<ResourceTag.BuilderImpl> resourceTags) {
            this.resourceTags = ResourceTagListCopier.copyFromBuilder(resourceTags);
        }

        @Override
        public final Builder resourceTags(Collection<ResourceTag> resourceTags) {
            this.resourceTags = ResourceTagListCopier.copy(resourceTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTags(ResourceTag... resourceTags) {
            resourceTags(Arrays.asList(resourceTags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTags(Consumer<ResourceTag.Builder>... resourceTags) {
            resourceTags(Stream.of(resourceTags).map(c -> ResourceTag.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateBillingViewRequest build() {
            return new CreateBillingViewRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
