/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.billing.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the health status of a billing view, including a status code and optional reasons for the status.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BillingViewHealthStatus implements SdkPojo, Serializable,
        ToCopyableBuilder<BillingViewHealthStatus.Builder, BillingViewHealthStatus> {
    private static final SdkField<String> STATUS_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusCode").getter(getter(BillingViewHealthStatus::statusCodeAsString))
            .setter(setter(Builder::statusCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusCode").build()).build();

    private static final SdkField<List<String>> STATUS_REASONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("statusReasons")
            .getter(getter(BillingViewHealthStatus::statusReasonsAsStrings))
            .setter(setter(Builder::statusReasonsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusReasons").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATUS_CODE_FIELD,
            STATUS_REASONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String statusCodeValue;

    private final List<String> statusReasons;

    private BillingViewHealthStatus(BuilderImpl builder) {
        this.statusCodeValue = builder.statusCodeValue;
        this.statusReasons = builder.statusReasons;
    }

    /**
     * <p>
     * The current health status code of the billing view.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link BillingViewStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusCodeAsString}.
     * </p>
     * 
     * @return The current health status code of the billing view.
     * @see BillingViewStatus
     */
    public final BillingViewStatus statusCode() {
        return BillingViewStatus.fromValue(statusCodeValue);
    }

    /**
     * <p>
     * The current health status code of the billing view.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link BillingViewStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusCodeAsString}.
     * </p>
     * 
     * @return The current health status code of the billing view.
     * @see BillingViewStatus
     */
    public final String statusCodeAsString() {
        return statusCodeValue;
    }

    /**
     * <p>
     * A list of reasons explaining the current health status, if applicable.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStatusReasons} method.
     * </p>
     * 
     * @return A list of reasons explaining the current health status, if applicable.
     */
    public final List<BillingViewStatusReason> statusReasons() {
        return BillingViewStatusReasonsCopier.copyStringToEnum(statusReasons);
    }

    /**
     * For responses, this returns true if the service returned a value for the StatusReasons property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStatusReasons() {
        return statusReasons != null && !(statusReasons instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of reasons explaining the current health status, if applicable.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStatusReasons} method.
     * </p>
     * 
     * @return A list of reasons explaining the current health status, if applicable.
     */
    public final List<String> statusReasonsAsStrings() {
        return statusReasons;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statusCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasStatusReasons() ? statusReasonsAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BillingViewHealthStatus)) {
            return false;
        }
        BillingViewHealthStatus other = (BillingViewHealthStatus) obj;
        return Objects.equals(statusCodeAsString(), other.statusCodeAsString()) && hasStatusReasons() == other.hasStatusReasons()
                && Objects.equals(statusReasonsAsStrings(), other.statusReasonsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BillingViewHealthStatus").add("StatusCode", statusCodeAsString())
                .add("StatusReasons", hasStatusReasons() ? statusReasonsAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "statusCode":
            return Optional.ofNullable(clazz.cast(statusCodeAsString()));
        case "statusReasons":
            return Optional.ofNullable(clazz.cast(statusReasonsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("statusCode", STATUS_CODE_FIELD);
        map.put("statusReasons", STATUS_REASONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<BillingViewHealthStatus, T> g) {
        return obj -> g.apply((BillingViewHealthStatus) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BillingViewHealthStatus> {
        /**
         * <p>
         * The current health status code of the billing view.
         * </p>
         * 
         * @param statusCodeValue
         *        The current health status code of the billing view.
         * @see BillingViewStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BillingViewStatus
         */
        Builder statusCode(String statusCodeValue);

        /**
         * <p>
         * The current health status code of the billing view.
         * </p>
         * 
         * @param statusCodeValue
         *        The current health status code of the billing view.
         * @see BillingViewStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BillingViewStatus
         */
        Builder statusCode(BillingViewStatus statusCodeValue);

        /**
         * <p>
         * A list of reasons explaining the current health status, if applicable.
         * </p>
         * 
         * @param statusReasons
         *        A list of reasons explaining the current health status, if applicable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReasonsWithStrings(Collection<String> statusReasons);

        /**
         * <p>
         * A list of reasons explaining the current health status, if applicable.
         * </p>
         * 
         * @param statusReasons
         *        A list of reasons explaining the current health status, if applicable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReasonsWithStrings(String... statusReasons);

        /**
         * <p>
         * A list of reasons explaining the current health status, if applicable.
         * </p>
         * 
         * @param statusReasons
         *        A list of reasons explaining the current health status, if applicable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReasons(Collection<BillingViewStatusReason> statusReasons);

        /**
         * <p>
         * A list of reasons explaining the current health status, if applicable.
         * </p>
         * 
         * @param statusReasons
         *        A list of reasons explaining the current health status, if applicable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReasons(BillingViewStatusReason... statusReasons);
    }

    static final class BuilderImpl implements Builder {
        private String statusCodeValue;

        private List<String> statusReasons = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(BillingViewHealthStatus model) {
            statusCode(model.statusCodeValue);
            statusReasonsWithStrings(model.statusReasons);
        }

        public final String getStatusCode() {
            return statusCodeValue;
        }

        public final void setStatusCode(String statusCodeValue) {
            this.statusCodeValue = statusCodeValue;
        }

        @Override
        public final Builder statusCode(String statusCodeValue) {
            this.statusCodeValue = statusCodeValue;
            return this;
        }

        @Override
        public final Builder statusCode(BillingViewStatus statusCodeValue) {
            this.statusCode(statusCodeValue == null ? null : statusCodeValue.toString());
            return this;
        }

        public final Collection<String> getStatusReasons() {
            if (statusReasons instanceof SdkAutoConstructList) {
                return null;
            }
            return statusReasons;
        }

        public final void setStatusReasons(Collection<String> statusReasons) {
            this.statusReasons = BillingViewStatusReasonsCopier.copy(statusReasons);
        }

        @Override
        public final Builder statusReasonsWithStrings(Collection<String> statusReasons) {
            this.statusReasons = BillingViewStatusReasonsCopier.copy(statusReasons);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statusReasonsWithStrings(String... statusReasons) {
            statusReasonsWithStrings(Arrays.asList(statusReasons));
            return this;
        }

        @Override
        public final Builder statusReasons(Collection<BillingViewStatusReason> statusReasons) {
            this.statusReasons = BillingViewStatusReasonsCopier.copyEnumToString(statusReasons);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statusReasons(BillingViewStatusReason... statusReasons) {
            statusReasons(Arrays.asList(statusReasons));
            return this;
        }

        @Override
        public BillingViewHealthStatus build() {
            return new BillingViewHealthStatus(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
