/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.braket;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.braket.internal.BraketServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.braket.model.AccessDeniedException;
import software.amazon.awssdk.services.braket.model.BraketException;
import software.amazon.awssdk.services.braket.model.CancelJobRequest;
import software.amazon.awssdk.services.braket.model.CancelJobResponse;
import software.amazon.awssdk.services.braket.model.CancelQuantumTaskRequest;
import software.amazon.awssdk.services.braket.model.CancelQuantumTaskResponse;
import software.amazon.awssdk.services.braket.model.ConflictException;
import software.amazon.awssdk.services.braket.model.CreateJobRequest;
import software.amazon.awssdk.services.braket.model.CreateJobResponse;
import software.amazon.awssdk.services.braket.model.CreateQuantumTaskRequest;
import software.amazon.awssdk.services.braket.model.CreateQuantumTaskResponse;
import software.amazon.awssdk.services.braket.model.DeviceOfflineException;
import software.amazon.awssdk.services.braket.model.DeviceRetiredException;
import software.amazon.awssdk.services.braket.model.GetDeviceRequest;
import software.amazon.awssdk.services.braket.model.GetDeviceResponse;
import software.amazon.awssdk.services.braket.model.GetJobRequest;
import software.amazon.awssdk.services.braket.model.GetJobResponse;
import software.amazon.awssdk.services.braket.model.GetQuantumTaskRequest;
import software.amazon.awssdk.services.braket.model.GetQuantumTaskResponse;
import software.amazon.awssdk.services.braket.model.InternalServiceException;
import software.amazon.awssdk.services.braket.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.braket.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.braket.model.ResourceNotFoundException;
import software.amazon.awssdk.services.braket.model.SearchDevicesRequest;
import software.amazon.awssdk.services.braket.model.SearchDevicesResponse;
import software.amazon.awssdk.services.braket.model.SearchJobsRequest;
import software.amazon.awssdk.services.braket.model.SearchJobsResponse;
import software.amazon.awssdk.services.braket.model.SearchQuantumTasksRequest;
import software.amazon.awssdk.services.braket.model.SearchQuantumTasksResponse;
import software.amazon.awssdk.services.braket.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.braket.model.TagResourceRequest;
import software.amazon.awssdk.services.braket.model.TagResourceResponse;
import software.amazon.awssdk.services.braket.model.ThrottlingException;
import software.amazon.awssdk.services.braket.model.UntagResourceRequest;
import software.amazon.awssdk.services.braket.model.UntagResourceResponse;
import software.amazon.awssdk.services.braket.model.ValidationException;
import software.amazon.awssdk.services.braket.transform.CancelJobRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.CancelQuantumTaskRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.CreateJobRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.CreateQuantumTaskRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.GetDeviceRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.GetJobRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.GetQuantumTaskRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.SearchDevicesRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.SearchJobsRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.SearchQuantumTasksRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.braket.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link BraketAsyncClient}.
 *
 * @see BraketAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBraketAsyncClient implements BraketAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultBraketAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBraketAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Cancels an Amazon Braket job.
     * </p>
     *
     * @param cancelJobRequest
     * @return A Java Future containing the result of the CancelJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException An error occurred due to a conflict.</li>
     *         <li>ThrottlingException The throttling rate limit is met.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BraketException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BraketAsyncClient.CancelJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/CancelJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CancelJobResponse> cancelJob(CancelJobRequest cancelJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CancelJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelJobRequest, CancelJobResponse>().withOperationName("CancelJob")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CancelJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelJobRequest));
            CompletableFuture<CancelJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Cancels the specified task.
     * </p>
     *
     * @param cancelQuantumTaskRequest
     * @return A Java Future containing the result of the CancelQuantumTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException An error occurred due to a conflict.</li>
     *         <li>ThrottlingException The throttling rate limit is met.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BraketException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BraketAsyncClient.CancelQuantumTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/CancelQuantumTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CancelQuantumTaskResponse> cancelQuantumTask(CancelQuantumTaskRequest cancelQuantumTaskRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelQuantumTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelQuantumTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelQuantumTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelQuantumTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CancelQuantumTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelQuantumTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelQuantumTaskRequest, CancelQuantumTaskResponse>()
                            .withOperationName("CancelQuantumTask").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CancelQuantumTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(cancelQuantumTaskRequest));
            CompletableFuture<CancelQuantumTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an Amazon Braket job.
     * </p>
     *
     * @param createJobRequest
     * @return A Java Future containing the result of the CreateJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ConflictException An error occurred due to a conflict.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The throttling rate limit is met.</li>
     *         <li>DeviceOfflineException The specified device is currently offline.</li>
     *         <li>DeviceRetiredException The specified device has been retired.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ServiceQuotaExceededException The request failed because a service quota is exceeded.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BraketException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BraketAsyncClient.CreateJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/CreateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateJobResponse> createJob(CreateJobRequest createJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateJobRequest, CreateJobResponse>().withOperationName("CreateJob")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(createJobRequest));
            CompletableFuture<CreateJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a quantum task.
     * </p>
     *
     * @param createQuantumTaskRequest
     * @return A Java Future containing the result of the CreateQuantumTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The throttling rate limit is met.</li>
     *         <li>DeviceOfflineException The specified device is currently offline.</li>
     *         <li>DeviceRetiredException The specified device has been retired.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ServiceQuotaExceededException The request failed because a service quota is exceeded.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BraketException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BraketAsyncClient.CreateQuantumTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/CreateQuantumTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateQuantumTaskResponse> createQuantumTask(CreateQuantumTaskRequest createQuantumTaskRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createQuantumTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createQuantumTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateQuantumTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateQuantumTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateQuantumTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateQuantumTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateQuantumTaskRequest, CreateQuantumTaskResponse>()
                            .withOperationName("CreateQuantumTask").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateQuantumTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createQuantumTaskRequest));
            CompletableFuture<CreateQuantumTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the devices available in Amazon Braket.
     * </p>
     * <note>
     * <p>
     * For backwards compatibility with older versions of BraketSchemas, OpenQASM information is omitted from GetDevice
     * API calls. To get this information the user-agent needs to present a recent version of the BraketSchemas (1.8.0
     * or later). The Braket SDK automatically reports this for you. If you do not see OpenQASM results in the GetDevice
     * response when using a Braket SDK, you may need to set AWS_EXECUTION_ENV environment variable to configure
     * user-agent. See the code examples provided below for how to do this for the AWS CLI, Boto3, and the Go, Java, and
     * JavaScript/TypeScript SDKs.
     * </p>
     * </note>
     *
     * @param getDeviceRequest
     * @return A Java Future containing the result of the GetDevice operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The throttling rate limit is met.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BraketException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BraketAsyncClient.GetDevice
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/GetDevice" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDeviceResponse> getDevice(GetDeviceRequest getDeviceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getDeviceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDeviceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDevice");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDeviceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetDeviceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDeviceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDeviceRequest, GetDeviceResponse>().withOperationName("GetDevice")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetDeviceRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getDeviceRequest));
            CompletableFuture<GetDeviceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the specified Amazon Braket job.
     * </p>
     *
     * @param getJobRequest
     * @return A Java Future containing the result of the GetJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The throttling rate limit is met.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BraketException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BraketAsyncClient.GetJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/GetJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetJobResponse> getJob(GetJobRequest getJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetJobRequest, GetJobResponse>().withOperationName("GetJob")
                            .withProtocolMetadata(protocolMetadata).withMarshaller(new GetJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getJobRequest));
            CompletableFuture<GetJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the specified quantum task.
     * </p>
     *
     * @param getQuantumTaskRequest
     * @return A Java Future containing the result of the GetQuantumTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The throttling rate limit is met.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BraketException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BraketAsyncClient.GetQuantumTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/GetQuantumTask" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetQuantumTaskResponse> getQuantumTask(GetQuantumTaskRequest getQuantumTaskRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getQuantumTaskRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQuantumTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQuantumTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQuantumTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetQuantumTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetQuantumTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQuantumTaskRequest, GetQuantumTaskResponse>()
                            .withOperationName("GetQuantumTask").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetQuantumTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getQuantumTaskRequest));
            CompletableFuture<GetQuantumTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Shows the tags associated with this resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BraketException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BraketAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Searches for devices using the specified filters.
     * </p>
     *
     * @param searchDevicesRequest
     * @return A Java Future containing the result of the SearchDevices operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The throttling rate limit is met.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BraketException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BraketAsyncClient.SearchDevices
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/SearchDevices" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SearchDevicesResponse> searchDevices(SearchDevicesRequest searchDevicesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(searchDevicesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchDevicesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchDevices");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchDevicesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SearchDevicesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchDevicesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchDevicesRequest, SearchDevicesResponse>()
                            .withOperationName("SearchDevices").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SearchDevicesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(searchDevicesRequest));
            CompletableFuture<SearchDevicesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Searches for Amazon Braket jobs that match the specified filter values.
     * </p>
     *
     * @param searchJobsRequest
     * @return A Java Future containing the result of the SearchJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The throttling rate limit is met.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BraketException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BraketAsyncClient.SearchJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/SearchJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SearchJobsResponse> searchJobs(SearchJobsRequest searchJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(searchJobsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    SearchJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchJobsRequest, SearchJobsResponse>().withOperationName("SearchJobs")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SearchJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(searchJobsRequest));
            CompletableFuture<SearchJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Searches for tasks that match the specified filter values.
     * </p>
     *
     * @param searchQuantumTasksRequest
     * @return A Java Future containing the result of the SearchQuantumTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The throttling rate limit is met.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BraketException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BraketAsyncClient.SearchQuantumTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/SearchQuantumTasks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SearchQuantumTasksResponse> searchQuantumTasks(SearchQuantumTasksRequest searchQuantumTasksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(searchQuantumTasksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchQuantumTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchQuantumTasks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchQuantumTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchQuantumTasksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchQuantumTasksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchQuantumTasksRequest, SearchQuantumTasksResponse>()
                            .withOperationName("SearchQuantumTasks").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new SearchQuantumTasksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(searchQuantumTasksRequest));
            CompletableFuture<SearchQuantumTasksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Add a tag to the specified resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BraketException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BraketAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ResourceNotFoundException The specified resource was not found.</li>
     *         <li>InternalServiceException The request processing has failed because of an unknown error, exception, or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BraketException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BraketAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/braket-2019-09-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Braket");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final BraketServiceClientConfiguration serviceClientConfiguration() {
        return new BraketServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BraketException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DeviceRetiredException")
                                .exceptionBuilderSupplier(DeviceRetiredException::builder).httpStatusCode(410).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceException")
                                .exceptionBuilderSupplier(InternalServiceException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DeviceOfflineException")
                                .exceptionBuilderSupplier(DeviceOfflineException::builder).httpStatusCode(424).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        BraketServiceClientConfigurationBuilder serviceConfigBuilder = new BraketServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
