/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.budgets;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.budgets.model.BudgetsException;
import software.amazon.awssdk.services.budgets.model.CreateBudgetRequest;
import software.amazon.awssdk.services.budgets.model.CreateBudgetResponse;
import software.amazon.awssdk.services.budgets.model.CreateNotificationRequest;
import software.amazon.awssdk.services.budgets.model.CreateNotificationResponse;
import software.amazon.awssdk.services.budgets.model.CreateSubscriberRequest;
import software.amazon.awssdk.services.budgets.model.CreateSubscriberResponse;
import software.amazon.awssdk.services.budgets.model.CreationLimitExceededException;
import software.amazon.awssdk.services.budgets.model.DeleteBudgetRequest;
import software.amazon.awssdk.services.budgets.model.DeleteBudgetResponse;
import software.amazon.awssdk.services.budgets.model.DeleteNotificationRequest;
import software.amazon.awssdk.services.budgets.model.DeleteNotificationResponse;
import software.amazon.awssdk.services.budgets.model.DeleteSubscriberRequest;
import software.amazon.awssdk.services.budgets.model.DeleteSubscriberResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetsRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetsResponse;
import software.amazon.awssdk.services.budgets.model.DescribeNotificationsForBudgetRequest;
import software.amazon.awssdk.services.budgets.model.DescribeNotificationsForBudgetResponse;
import software.amazon.awssdk.services.budgets.model.DescribeSubscribersForNotificationRequest;
import software.amazon.awssdk.services.budgets.model.DescribeSubscribersForNotificationResponse;
import software.amazon.awssdk.services.budgets.model.DuplicateRecordException;
import software.amazon.awssdk.services.budgets.model.ExpiredNextTokenException;
import software.amazon.awssdk.services.budgets.model.InternalErrorException;
import software.amazon.awssdk.services.budgets.model.InvalidNextTokenException;
import software.amazon.awssdk.services.budgets.model.InvalidParameterException;
import software.amazon.awssdk.services.budgets.model.NotFoundException;
import software.amazon.awssdk.services.budgets.model.UpdateBudgetRequest;
import software.amazon.awssdk.services.budgets.model.UpdateBudgetResponse;
import software.amazon.awssdk.services.budgets.model.UpdateNotificationRequest;
import software.amazon.awssdk.services.budgets.model.UpdateNotificationResponse;
import software.amazon.awssdk.services.budgets.model.UpdateSubscriberRequest;
import software.amazon.awssdk.services.budgets.model.UpdateSubscriberResponse;

/**
 * Service client for accessing AWSBudgets. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * Budgets enable you to plan your service usage, service costs, and your RI utilization. You can also track how close
 * your plan is to your budgeted amount or to the free tier limits. Budgets provide you with a quick way to see your
 * usage-to-date and current estimated charges from AWS and to see how much your predicted usage accrues in charges by
 * the end of the month. Budgets also compare current estimates and charges to the amount that you indicated you want to
 * use or spend and lets you see how much of your budget has been used. AWS updates your budget status several times a
 * day. Budgets track your unblended costs, subscriptions, and refunds. You can create the following types of budgets:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Cost budgets allow you to say how much you want to spend on a service.
 * </p>
 * </li>
 * <li>
 * <p>
 * Usage budgets allow you to say how many hours you want to use for one or more services.
 * </p>
 * </li>
 * <li>
 * <p>
 * RI utilization budgets allow you to define a utilization threshold and receive alerts when RIs are tracking below
 * that threshold.
 * </p>
 * </li>
 * </ul>
 * <p>
 * You can create up to 20,000 budgets per AWS master account. Your first two budgets are free of charge. Each
 * additional budget costs $0.02 per day. You can set up optional notifications that warn you if you exceed, or are
 * forecasted to exceed, your budgeted amount. You can have notifications sent to an Amazon SNS topic, to an email
 * address, or to both. For more information, see <a
 * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/budgets-sns-policy.html">Creating an Amazon SNS
 * Topic for Budget Notifications</a>. AWS Free Tier usage alerts via AWS Budgets are provided for you, and do not count
 * toward your budget limits.
 * </p>
 * <p>
 * Service Endpoint
 * </p>
 * <p>
 * The AWS Budgets API provides the following endpoint:
 * </p>
 * <ul>
 * <li>
 * <p>
 * https://budgets.amazonaws.com
 * </p>
 * </li>
 * </ul>
 * <p>
 * For information about costs associated with the AWS Budgets API, see <a
 * href="https://aws.amazon.com/aws-cost-management/pricing/">AWS Cost Management Pricing</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public interface BudgetsClient extends SdkClient {
    String SERVICE_NAME = "budgets";

    /**
     * Create a {@link BudgetsClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static BudgetsClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link BudgetsClient}.
     */
    static BudgetsClientBuilder builder() {
        return new DefaultBudgetsClientBuilder();
    }

    /**
     * <p>
     * Creates a budget and, if included, notifications and subscribers.
     * </p>
     *
     * @param createBudgetRequest
     *        Request of CreateBudget
     * @return Result of the CreateBudget operation returned by the service.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws CreationLimitExceededException
     *         You've exceeded the notification or subscriber limit.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.CreateBudget
     */
    default CreateBudgetResponse createBudget(CreateBudgetRequest createBudgetRequest) throws InvalidParameterException,
            InternalErrorException, CreationLimitExceededException, DuplicateRecordException, AwsServiceException,
            SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a budget and, if included, notifications and subscribers.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateBudgetRequest.Builder} avoiding the need to
     * create one manually via {@link CreateBudgetRequest#builder()}
     * </p>
     *
     * @param createBudgetRequest
     *        A {@link Consumer} that will call methods on {@link CreateBudgetRequest.Builder} to create a request.
     *        Request of CreateBudget
     * @return Result of the CreateBudget operation returned by the service.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws CreationLimitExceededException
     *         You've exceeded the notification or subscriber limit.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.CreateBudget
     */
    default CreateBudgetResponse createBudget(Consumer<CreateBudgetRequest.Builder> createBudgetRequest)
            throws InvalidParameterException, InternalErrorException, CreationLimitExceededException, DuplicateRecordException,
            AwsServiceException, SdkClientException, BudgetsException {
        return createBudget(CreateBudgetRequest.builder().applyMutation(createBudgetRequest).build());
    }

    /**
     * <p>
     * Creates a notification. You must create the budget before you create the associated notification.
     * </p>
     *
     * @param createNotificationRequest
     *        Request of CreateNotification
     * @return Result of the CreateNotification operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws CreationLimitExceededException
     *         You've exceeded the notification or subscriber limit.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.CreateNotification
     */
    default CreateNotificationResponse createNotification(CreateNotificationRequest createNotificationRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, CreationLimitExceededException,
            DuplicateRecordException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a notification. You must create the budget before you create the associated notification.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateNotificationRequest.Builder} avoiding the
     * need to create one manually via {@link CreateNotificationRequest#builder()}
     * </p>
     *
     * @param createNotificationRequest
     *        A {@link Consumer} that will call methods on {@link CreateNotificationRequest.Builder} to create a
     *        request. Request of CreateNotification
     * @return Result of the CreateNotification operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws CreationLimitExceededException
     *         You've exceeded the notification or subscriber limit.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.CreateNotification
     */
    default CreateNotificationResponse createNotification(Consumer<CreateNotificationRequest.Builder> createNotificationRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, CreationLimitExceededException,
            DuplicateRecordException, AwsServiceException, SdkClientException, BudgetsException {
        return createNotification(CreateNotificationRequest.builder().applyMutation(createNotificationRequest).build());
    }

    /**
     * <p>
     * Creates a subscriber. You must create the associated budget and notification before you create the subscriber.
     * </p>
     *
     * @param createSubscriberRequest
     *        Request of CreateSubscriber
     * @return Result of the CreateSubscriber operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws CreationLimitExceededException
     *         You've exceeded the notification or subscriber limit.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.CreateSubscriber
     */
    default CreateSubscriberResponse createSubscriber(CreateSubscriberRequest createSubscriberRequest)
            throws InternalErrorException, InvalidParameterException, CreationLimitExceededException, DuplicateRecordException,
            NotFoundException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a subscriber. You must create the associated budget and notification before you create the subscriber.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateSubscriberRequest.Builder} avoiding the need
     * to create one manually via {@link CreateSubscriberRequest#builder()}
     * </p>
     *
     * @param createSubscriberRequest
     *        A {@link Consumer} that will call methods on {@link CreateSubscriberRequest.Builder} to create a request.
     *        Request of CreateSubscriber
     * @return Result of the CreateSubscriber operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws CreationLimitExceededException
     *         You've exceeded the notification or subscriber limit.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.CreateSubscriber
     */
    default CreateSubscriberResponse createSubscriber(Consumer<CreateSubscriberRequest.Builder> createSubscriberRequest)
            throws InternalErrorException, InvalidParameterException, CreationLimitExceededException, DuplicateRecordException,
            NotFoundException, AwsServiceException, SdkClientException, BudgetsException {
        return createSubscriber(CreateSubscriberRequest.builder().applyMutation(createSubscriberRequest).build());
    }

    /**
     * <p>
     * Deletes a budget. You can delete your budget at any time.
     * </p>
     * <p>
     * <b>Deleting a budget also deletes the notifications and subscribers associated with that budget.</b>
     * </p>
     *
     * @param deleteBudgetRequest
     *        Request of DeleteBudget
     * @return Result of the DeleteBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DeleteBudget
     */
    default DeleteBudgetResponse deleteBudget(DeleteBudgetRequest deleteBudgetRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a budget. You can delete your budget at any time.
     * </p>
     * <p>
     * <b>Deleting a budget also deletes the notifications and subscribers associated with that budget.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteBudgetRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteBudgetRequest#builder()}
     * </p>
     *
     * @param deleteBudgetRequest
     *        A {@link Consumer} that will call methods on {@link DeleteBudgetRequest.Builder} to create a request.
     *        Request of DeleteBudget
     * @return Result of the DeleteBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DeleteBudget
     */
    default DeleteBudgetResponse deleteBudget(Consumer<DeleteBudgetRequest.Builder> deleteBudgetRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AwsServiceException, SdkClientException,
            BudgetsException {
        return deleteBudget(DeleteBudgetRequest.builder().applyMutation(deleteBudgetRequest).build());
    }

    /**
     * <p>
     * Deletes a notification.
     * </p>
     * <p>
     * <b>Deleting a notification also deletes the subscribers associated with the notification.</b>
     * </p>
     *
     * @param deleteNotificationRequest
     *        Request of DeleteNotification
     * @return Result of the DeleteNotification operation returned by the service.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DeleteNotification
     */
    default DeleteNotificationResponse deleteNotification(DeleteNotificationRequest deleteNotificationRequest)
            throws InvalidParameterException, InternalErrorException, NotFoundException, AwsServiceException, SdkClientException,
            BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a notification.
     * </p>
     * <p>
     * <b>Deleting a notification also deletes the subscribers associated with the notification.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteNotificationRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteNotificationRequest#builder()}
     * </p>
     *
     * @param deleteNotificationRequest
     *        A {@link Consumer} that will call methods on {@link DeleteNotificationRequest.Builder} to create a
     *        request. Request of DeleteNotification
     * @return Result of the DeleteNotification operation returned by the service.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DeleteNotification
     */
    default DeleteNotificationResponse deleteNotification(Consumer<DeleteNotificationRequest.Builder> deleteNotificationRequest)
            throws InvalidParameterException, InternalErrorException, NotFoundException, AwsServiceException, SdkClientException,
            BudgetsException {
        return deleteNotification(DeleteNotificationRequest.builder().applyMutation(deleteNotificationRequest).build());
    }

    /**
     * <p>
     * Deletes a subscriber.
     * </p>
     * <p>
     * <b>Deleting the last subscriber to a notification also deletes the notification.</b>
     * </p>
     *
     * @param deleteSubscriberRequest
     *        Request of DeleteSubscriber
     * @return Result of the DeleteSubscriber operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DeleteSubscriber
     */
    default DeleteSubscriberResponse deleteSubscriber(DeleteSubscriberRequest deleteSubscriberRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AwsServiceException, SdkClientException,
            BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a subscriber.
     * </p>
     * <p>
     * <b>Deleting the last subscriber to a notification also deletes the notification.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteSubscriberRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteSubscriberRequest#builder()}
     * </p>
     *
     * @param deleteSubscriberRequest
     *        A {@link Consumer} that will call methods on {@link DeleteSubscriberRequest.Builder} to create a request.
     *        Request of DeleteSubscriber
     * @return Result of the DeleteSubscriber operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DeleteSubscriber
     */
    default DeleteSubscriberResponse deleteSubscriber(Consumer<DeleteSubscriberRequest.Builder> deleteSubscriberRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AwsServiceException, SdkClientException,
            BudgetsException {
        return deleteSubscriber(DeleteSubscriberRequest.builder().applyMutation(deleteSubscriberRequest).build());
    }

    /**
     * <p>
     * Describes a budget.
     * </p>
     *
     * @param describeBudgetRequest
     *        Request of DescribeBudget
     * @return Result of the DescribeBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudget
     */
    default DescribeBudgetResponse describeBudget(DescribeBudgetRequest describeBudgetRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes a budget.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeBudgetRequest.Builder} avoiding the need to
     * create one manually via {@link DescribeBudgetRequest#builder()}
     * </p>
     *
     * @param describeBudgetRequest
     *        A {@link Consumer} that will call methods on {@link DescribeBudgetRequest.Builder} to create a request.
     *        Request of DescribeBudget
     * @return Result of the DescribeBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudget
     */
    default DescribeBudgetResponse describeBudget(Consumer<DescribeBudgetRequest.Builder> describeBudgetRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AwsServiceException, SdkClientException,
            BudgetsException {
        return describeBudget(DescribeBudgetRequest.builder().applyMutation(describeBudgetRequest).build());
    }

    /**
     * <p>
     * Lists the budgets associated with an account.
     * </p>
     *
     * @param describeBudgetsRequest
     *        Request of DescribeBudgets
     * @return Result of the DescribeBudgets operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgets
     */
    default DescribeBudgetsResponse describeBudgets(DescribeBudgetsRequest describeBudgetsRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, InvalidNextTokenException, ExpiredNextTokenException,
            AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the budgets associated with an account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeBudgetsRequest.Builder} avoiding the need
     * to create one manually via {@link DescribeBudgetsRequest#builder()}
     * </p>
     *
     * @param describeBudgetsRequest
     *        A {@link Consumer} that will call methods on {@link DescribeBudgetsRequest.Builder} to create a request.
     *        Request of DescribeBudgets
     * @return Result of the DescribeBudgets operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgets
     */
    default DescribeBudgetsResponse describeBudgets(Consumer<DescribeBudgetsRequest.Builder> describeBudgetsRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AwsServiceException, SdkClientException, BudgetsException {
        return describeBudgets(DescribeBudgetsRequest.builder().applyMutation(describeBudgetsRequest).build());
    }

    /**
     * <p>
     * Lists the notifications associated with a budget.
     * </p>
     *
     * @param describeNotificationsForBudgetRequest
     *        Request of DescribeNotificationsForBudget
     * @return Result of the DescribeNotificationsForBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeNotificationsForBudget
     */
    default DescribeNotificationsForBudgetResponse describeNotificationsForBudget(
            DescribeNotificationsForBudgetRequest describeNotificationsForBudgetRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, InvalidNextTokenException, ExpiredNextTokenException,
            AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the notifications associated with a budget.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeNotificationsForBudgetRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeNotificationsForBudgetRequest#builder()}
     * </p>
     *
     * @param describeNotificationsForBudgetRequest
     *        A {@link Consumer} that will call methods on {@link DescribeNotificationsForBudgetRequest.Builder} to
     *        create a request. Request of DescribeNotificationsForBudget
     * @return Result of the DescribeNotificationsForBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeNotificationsForBudget
     */
    default DescribeNotificationsForBudgetResponse describeNotificationsForBudget(
            Consumer<DescribeNotificationsForBudgetRequest.Builder> describeNotificationsForBudgetRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AwsServiceException, SdkClientException, BudgetsException {
        return describeNotificationsForBudget(DescribeNotificationsForBudgetRequest.builder()
                .applyMutation(describeNotificationsForBudgetRequest).build());
    }

    /**
     * <p>
     * Lists the subscribers associated with a notification.
     * </p>
     *
     * @param describeSubscribersForNotificationRequest
     *        Request of DescribeSubscribersForNotification
     * @return Result of the DescribeSubscribersForNotification operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeSubscribersForNotification
     */
    default DescribeSubscribersForNotificationResponse describeSubscribersForNotification(
            DescribeSubscribersForNotificationRequest describeSubscribersForNotificationRequest) throws InternalErrorException,
            NotFoundException, InvalidParameterException, InvalidNextTokenException, ExpiredNextTokenException,
            AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the subscribers associated with a notification.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeSubscribersForNotificationRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeSubscribersForNotificationRequest#builder()}
     * </p>
     *
     * @param describeSubscribersForNotificationRequest
     *        A {@link Consumer} that will call methods on {@link DescribeSubscribersForNotificationRequest.Builder} to
     *        create a request. Request of DescribeSubscribersForNotification
     * @return Result of the DescribeSubscribersForNotification operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeSubscribersForNotification
     */
    default DescribeSubscribersForNotificationResponse describeSubscribersForNotification(
            Consumer<DescribeSubscribersForNotificationRequest.Builder> describeSubscribersForNotificationRequest)
            throws InternalErrorException, NotFoundException, InvalidParameterException, InvalidNextTokenException,
            ExpiredNextTokenException, AwsServiceException, SdkClientException, BudgetsException {
        return describeSubscribersForNotification(DescribeSubscribersForNotificationRequest.builder()
                .applyMutation(describeSubscribersForNotificationRequest).build());
    }

    /**
     * <p>
     * Updates a budget. You can change every part of a budget except for the <code>budgetName</code> and the
     * <code>calculatedSpend</code>. When a budget is modified, the <code>calculatedSpend</code> drops to zero until AWS
     * has new usage data to use for forecasting.
     * </p>
     *
     * @param updateBudgetRequest
     *        Request of UpdateBudget
     * @return Result of the UpdateBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UpdateBudget
     */
    default UpdateBudgetResponse updateBudget(UpdateBudgetRequest updateBudgetRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a budget. You can change every part of a budget except for the <code>budgetName</code> and the
     * <code>calculatedSpend</code>. When a budget is modified, the <code>calculatedSpend</code> drops to zero until AWS
     * has new usage data to use for forecasting.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateBudgetRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateBudgetRequest#builder()}
     * </p>
     *
     * @param updateBudgetRequest
     *        A {@link Consumer} that will call methods on {@link UpdateBudgetRequest.Builder} to create a request.
     *        Request of UpdateBudget
     * @return Result of the UpdateBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UpdateBudget
     */
    default UpdateBudgetResponse updateBudget(Consumer<UpdateBudgetRequest.Builder> updateBudgetRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AwsServiceException, SdkClientException,
            BudgetsException {
        return updateBudget(UpdateBudgetRequest.builder().applyMutation(updateBudgetRequest).build());
    }

    /**
     * <p>
     * Updates a notification.
     * </p>
     *
     * @param updateNotificationRequest
     *        Request of UpdateNotification
     * @return Result of the UpdateNotification operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UpdateNotification
     */
    default UpdateNotificationResponse updateNotification(UpdateNotificationRequest updateNotificationRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, DuplicateRecordException,
            AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a notification.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateNotificationRequest.Builder} avoiding the
     * need to create one manually via {@link UpdateNotificationRequest#builder()}
     * </p>
     *
     * @param updateNotificationRequest
     *        A {@link Consumer} that will call methods on {@link UpdateNotificationRequest.Builder} to create a
     *        request. Request of UpdateNotification
     * @return Result of the UpdateNotification operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UpdateNotification
     */
    default UpdateNotificationResponse updateNotification(Consumer<UpdateNotificationRequest.Builder> updateNotificationRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, DuplicateRecordException,
            AwsServiceException, SdkClientException, BudgetsException {
        return updateNotification(UpdateNotificationRequest.builder().applyMutation(updateNotificationRequest).build());
    }

    /**
     * <p>
     * Updates a subscriber.
     * </p>
     *
     * @param updateSubscriberRequest
     *        Request of UpdateSubscriber
     * @return Result of the UpdateSubscriber operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UpdateSubscriber
     */
    default UpdateSubscriberResponse updateSubscriber(UpdateSubscriberRequest updateSubscriberRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, DuplicateRecordException,
            AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a subscriber.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateSubscriberRequest.Builder} avoiding the need
     * to create one manually via {@link UpdateSubscriberRequest#builder()}
     * </p>
     *
     * @param updateSubscriberRequest
     *        A {@link Consumer} that will call methods on {@link UpdateSubscriberRequest.Builder} to create a request.
     *        Request of UpdateSubscriber
     * @return Result of the UpdateSubscriber operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UpdateSubscriber
     */
    default UpdateSubscriberResponse updateSubscriber(Consumer<UpdateSubscriberRequest.Builder> updateSubscriberRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, DuplicateRecordException,
            AwsServiceException, SdkClientException, BudgetsException {
        return updateSubscriber(UpdateSubscriberRequest.builder().applyMutation(updateSubscriberRequest).build());
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of("budgets");
    }
}
