/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.budgets;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.protocol.json.AwsJsonProtocol;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolMetadata;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.internal.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.protocol.json.JsonClientMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorResponseMetadata;
import software.amazon.awssdk.core.protocol.json.JsonErrorShapeMetadata;
import software.amazon.awssdk.core.protocol.json.JsonOperationMetadata;
import software.amazon.awssdk.services.budgets.model.CreateBudgetRequest;
import software.amazon.awssdk.services.budgets.model.CreateBudgetResponse;
import software.amazon.awssdk.services.budgets.model.CreateNotificationRequest;
import software.amazon.awssdk.services.budgets.model.CreateNotificationResponse;
import software.amazon.awssdk.services.budgets.model.CreateSubscriberRequest;
import software.amazon.awssdk.services.budgets.model.CreateSubscriberResponse;
import software.amazon.awssdk.services.budgets.model.CreationLimitExceededException;
import software.amazon.awssdk.services.budgets.model.DeleteBudgetRequest;
import software.amazon.awssdk.services.budgets.model.DeleteBudgetResponse;
import software.amazon.awssdk.services.budgets.model.DeleteNotificationRequest;
import software.amazon.awssdk.services.budgets.model.DeleteNotificationResponse;
import software.amazon.awssdk.services.budgets.model.DeleteSubscriberRequest;
import software.amazon.awssdk.services.budgets.model.DeleteSubscriberResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetsRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetsResponse;
import software.amazon.awssdk.services.budgets.model.DescribeNotificationsForBudgetRequest;
import software.amazon.awssdk.services.budgets.model.DescribeNotificationsForBudgetResponse;
import software.amazon.awssdk.services.budgets.model.DescribeSubscribersForNotificationRequest;
import software.amazon.awssdk.services.budgets.model.DescribeSubscribersForNotificationResponse;
import software.amazon.awssdk.services.budgets.model.DuplicateRecordException;
import software.amazon.awssdk.services.budgets.model.ExpiredNextTokenException;
import software.amazon.awssdk.services.budgets.model.InternalErrorException;
import software.amazon.awssdk.services.budgets.model.InvalidNextTokenException;
import software.amazon.awssdk.services.budgets.model.InvalidParameterException;
import software.amazon.awssdk.services.budgets.model.NotFoundException;
import software.amazon.awssdk.services.budgets.model.UpdateBudgetRequest;
import software.amazon.awssdk.services.budgets.model.UpdateBudgetResponse;
import software.amazon.awssdk.services.budgets.model.UpdateNotificationRequest;
import software.amazon.awssdk.services.budgets.model.UpdateNotificationResponse;
import software.amazon.awssdk.services.budgets.model.UpdateSubscriberRequest;
import software.amazon.awssdk.services.budgets.model.UpdateSubscriberResponse;
import software.amazon.awssdk.services.budgets.transform.CreateBudgetRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.CreateBudgetResponseUnmarshaller;
import software.amazon.awssdk.services.budgets.transform.CreateNotificationRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.CreateNotificationResponseUnmarshaller;
import software.amazon.awssdk.services.budgets.transform.CreateSubscriberRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.CreateSubscriberResponseUnmarshaller;
import software.amazon.awssdk.services.budgets.transform.DeleteBudgetRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DeleteBudgetResponseUnmarshaller;
import software.amazon.awssdk.services.budgets.transform.DeleteNotificationRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DeleteNotificationResponseUnmarshaller;
import software.amazon.awssdk.services.budgets.transform.DeleteSubscriberRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DeleteSubscriberResponseUnmarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetResponseUnmarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetsRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetsResponseUnmarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeNotificationsForBudgetRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeNotificationsForBudgetResponseUnmarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeSubscribersForNotificationRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeSubscribersForNotificationResponseUnmarshaller;
import software.amazon.awssdk.services.budgets.transform.UpdateBudgetRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.UpdateBudgetResponseUnmarshaller;
import software.amazon.awssdk.services.budgets.transform.UpdateNotificationRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.UpdateNotificationResponseUnmarshaller;
import software.amazon.awssdk.services.budgets.transform.UpdateSubscriberRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.UpdateSubscriberResponseUnmarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link BudgetsAsyncClient}.
 *
 * @see BudgetsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBudgetsAsyncClient implements BudgetsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultBudgetsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    protected DefaultBudgetsAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.protocolFactory = init(false);
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a budget and, if included, notifications and subscribers.
     * </p>
     *
     * @param createBudgetRequest
     *        Request of CreateBudget
     * @return A Java Future containing the result of the CreateBudget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>CreationLimitExceededException You've exceeded the notification or subscriber limit.</li>
     *         <li>DuplicateRecordException The budget name already exists. Budget names must be unique within an
     *         account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.CreateBudget
     */
    @Override
    public CompletableFuture<CreateBudgetResponse> createBudget(CreateBudgetRequest createBudgetRequest) {
        try {

            HttpResponseHandler<CreateBudgetResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new CreateBudgetResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<CreateBudgetRequest, CreateBudgetResponse>()
                    .withMarshaller(new CreateBudgetRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createBudgetRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a notification. You must create the budget before you create the associated notification.
     * </p>
     *
     * @param createNotificationRequest
     *        Request of CreateNotification
     * @return A Java Future containing the result of the CreateNotification operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>CreationLimitExceededException You've exceeded the notification or subscriber limit.</li>
     *         <li>DuplicateRecordException The budget name already exists. Budget names must be unique within an
     *         account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.CreateNotification
     */
    @Override
    public CompletableFuture<CreateNotificationResponse> createNotification(CreateNotificationRequest createNotificationRequest) {
        try {

            HttpResponseHandler<CreateNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new CreateNotificationResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<CreateNotificationRequest, CreateNotificationResponse>()
                    .withMarshaller(new CreateNotificationRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(createNotificationRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a subscriber. You must create the associated budget and notification before you create the subscriber.
     * </p>
     *
     * @param createSubscriberRequest
     *        Request of CreateSubscriber
     * @return A Java Future containing the result of the CreateSubscriber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>CreationLimitExceededException You've exceeded the notification or subscriber limit.</li>
     *         <li>DuplicateRecordException The budget name already exists. Budget names must be unique within an
     *         account.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.CreateSubscriber
     */
    @Override
    public CompletableFuture<CreateSubscriberResponse> createSubscriber(CreateSubscriberRequest createSubscriberRequest) {
        try {

            HttpResponseHandler<CreateSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new CreateSubscriberResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<CreateSubscriberRequest, CreateSubscriberResponse>()
                    .withMarshaller(new CreateSubscriberRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createSubscriberRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a budget. You can delete your budget at any time.
     * </p>
     * <p>
     * <b>Deleting a budget also deletes the notifications and subscribers associated with that budget.</b>
     * </p>
     *
     * @param deleteBudgetRequest
     *        Request of DeleteBudget
     * @return A Java Future containing the result of the DeleteBudget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DeleteBudget
     */
    @Override
    public CompletableFuture<DeleteBudgetResponse> deleteBudget(DeleteBudgetRequest deleteBudgetRequest) {
        try {

            HttpResponseHandler<DeleteBudgetResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new DeleteBudgetResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<DeleteBudgetRequest, DeleteBudgetResponse>()
                    .withMarshaller(new DeleteBudgetRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteBudgetRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a notification.
     * </p>
     * <p>
     * <b>Deleting a notification also deletes the subscribers associated with the notification.</b>
     * </p>
     *
     * @param deleteNotificationRequest
     *        Request of DeleteNotification
     * @return A Java Future containing the result of the DeleteNotification operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DeleteNotification
     */
    @Override
    public CompletableFuture<DeleteNotificationResponse> deleteNotification(DeleteNotificationRequest deleteNotificationRequest) {
        try {

            HttpResponseHandler<DeleteNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new DeleteNotificationResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<DeleteNotificationRequest, DeleteNotificationResponse>()
                    .withMarshaller(new DeleteNotificationRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(deleteNotificationRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a subscriber.
     * </p>
     * <p>
     * <b>Deleting the last subscriber to a notification also deletes the notification.</b>
     * </p>
     *
     * @param deleteSubscriberRequest
     *        Request of DeleteSubscriber
     * @return A Java Future containing the result of the DeleteSubscriber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DeleteSubscriber
     */
    @Override
    public CompletableFuture<DeleteSubscriberResponse> deleteSubscriber(DeleteSubscriberRequest deleteSubscriberRequest) {
        try {

            HttpResponseHandler<DeleteSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new DeleteSubscriberResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<DeleteSubscriberRequest, DeleteSubscriberResponse>()
                    .withMarshaller(new DeleteSubscriberRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteSubscriberRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes a budget.
     * </p>
     *
     * @param describeBudgetRequest
     *        Request of DescribeBudget
     * @return A Java Future containing the result of the DescribeBudget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DescribeBudget
     */
    @Override
    public CompletableFuture<DescribeBudgetResponse> describeBudget(DescribeBudgetRequest describeBudgetRequest) {
        try {

            HttpResponseHandler<DescribeBudgetResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new DescribeBudgetResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<DescribeBudgetRequest, DescribeBudgetResponse>()
                    .withMarshaller(new DescribeBudgetRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeBudgetRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the budgets associated with an account.
     * </p>
     *
     * @param describeBudgetsRequest
     *        Request of DescribeBudgets
     * @return A Java Future containing the result of the DescribeBudgets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid.</li>
     *         <li>ExpiredNextTokenException The pagination token expired.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DescribeBudgets
     */
    @Override
    public CompletableFuture<DescribeBudgetsResponse> describeBudgets(DescribeBudgetsRequest describeBudgetsRequest) {
        try {

            HttpResponseHandler<DescribeBudgetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new DescribeBudgetsResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<DescribeBudgetsRequest, DescribeBudgetsResponse>()
                    .withMarshaller(new DescribeBudgetsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeBudgetsRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the notifications associated with a budget.
     * </p>
     *
     * @param describeNotificationsForBudgetRequest
     *        Request of DescribeNotificationsForBudget
     * @return A Java Future containing the result of the DescribeNotificationsForBudget operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid.</li>
     *         <li>ExpiredNextTokenException The pagination token expired.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DescribeNotificationsForBudget
     */
    @Override
    public CompletableFuture<DescribeNotificationsForBudgetResponse> describeNotificationsForBudget(
            DescribeNotificationsForBudgetRequest describeNotificationsForBudgetRequest) {
        try {

            HttpResponseHandler<DescribeNotificationsForBudgetResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new DescribeNotificationsForBudgetResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeNotificationsForBudgetRequest, DescribeNotificationsForBudgetResponse>()
                            .withMarshaller(new DescribeNotificationsForBudgetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeNotificationsForBudgetRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the subscribers associated with a notification.
     * </p>
     *
     * @param describeSubscribersForNotificationRequest
     *        Request of DescribeSubscribersForNotification
     * @return A Java Future containing the result of the DescribeSubscribersForNotification operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid.</li>
     *         <li>ExpiredNextTokenException The pagination token expired.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.DescribeSubscribersForNotification
     */
    @Override
    public CompletableFuture<DescribeSubscribersForNotificationResponse> describeSubscribersForNotification(
            DescribeSubscribersForNotificationRequest describeSubscribersForNotificationRequest) {
        try {

            HttpResponseHandler<DescribeSubscribersForNotificationResponse> responseHandler = protocolFactory
                    .createResponseHandler(
                            new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                            new DescribeSubscribersForNotificationResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeSubscribersForNotificationRequest, DescribeSubscribersForNotificationResponse>()
                            .withMarshaller(new DescribeSubscribersForNotificationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeSubscribersForNotificationRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a budget. You can change every part of a budget except for the <code>budgetName</code> and the
     * <code>calculatedSpend</code>. When a budget is modified, the <code>calculatedSpend</code> drops to zero until AWS
     * has new usage data to use for forecasting.
     * </p>
     *
     * @param updateBudgetRequest
     *        Request of UpdateBudget
     * @return A Java Future containing the result of the UpdateBudget operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.UpdateBudget
     */
    @Override
    public CompletableFuture<UpdateBudgetResponse> updateBudget(UpdateBudgetRequest updateBudgetRequest) {
        try {

            HttpResponseHandler<UpdateBudgetResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new UpdateBudgetResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<UpdateBudgetRequest, UpdateBudgetResponse>()
                    .withMarshaller(new UpdateBudgetRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateBudgetRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a notification.
     * </p>
     *
     * @param updateNotificationRequest
     *        Request of UpdateNotification
     * @return A Java Future containing the result of the UpdateNotification operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>DuplicateRecordException The budget name already exists. Budget names must be unique within an
     *         account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.UpdateNotification
     */
    @Override
    public CompletableFuture<UpdateNotificationResponse> updateNotification(UpdateNotificationRequest updateNotificationRequest) {
        try {

            HttpResponseHandler<UpdateNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new UpdateNotificationResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<UpdateNotificationRequest, UpdateNotificationResponse>()
                    .withMarshaller(new UpdateNotificationRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(updateNotificationRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a subscriber.
     * </p>
     *
     * @param updateSubscriberRequest
     *        Request of UpdateSubscriber
     * @return A Java Future containing the result of the UpdateSubscriber operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>InvalidParameterException An error on the client occurred. Typically, the cause is an invalid input
     *         value.</li>
     *         <li>NotFoundException We can’t locate the resource that you specified.</li>
     *         <li>DuplicateRecordException The budget name already exists. Budget names must be unique within an
     *         account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>BudgetsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample BudgetsAsyncClient.UpdateSubscriber
     */
    @Override
    public CompletableFuture<UpdateSubscriberResponse> updateSubscriber(UpdateSubscriberRequest updateSubscriberRequest) {
        try {

            HttpResponseHandler<UpdateSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(
                    new JsonOperationMetadata().withPayloadJson(true).withHasStreamingSuccessResponse(false),
                    new UpdateSubscriberResponseUnmarshaller());

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory);

            return clientHandler.execute(new ClientExecutionParams<UpdateSubscriberRequest, UpdateSubscriberResponse>()
                    .withMarshaller(new UpdateSubscriberRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateSubscriberRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private software.amazon.awssdk.awscore.protocol.json.AwsJsonProtocolFactory init(boolean supportsCbor) {
        return new AwsJsonProtocolFactory(
                new JsonClientMetadata()
                        .withSupportsCbor(supportsCbor)
                        .withSupportsIon(false)
                        .withBaseServiceExceptionClass(software.amazon.awssdk.services.budgets.model.BudgetsException.class)
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("NotFoundException").withModeledClass(
                                        NotFoundException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("InvalidParameterException").withModeledClass(
                                        InvalidParameterException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("DuplicateRecordException").withModeledClass(
                                        DuplicateRecordException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("InternalErrorException").withModeledClass(
                                        InternalErrorException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("CreationLimitExceededException").withModeledClass(
                                        CreationLimitExceededException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("InvalidNextTokenException").withModeledClass(
                                        InvalidNextTokenException.class))
                        .addErrorMetadata(
                                new JsonErrorShapeMetadata().withErrorCode("ExpiredNextTokenException").withModeledClass(
                                        ExpiredNextTokenException.class)), AwsJsonProtocolMetadata.builder()
                        .protocolVersion("1.1").protocol(AwsJsonProtocol.AWS_JSON).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(AwsJsonProtocolFactory protocolFactory) {
        return protocolFactory.createErrorResponseHandler(new JsonErrorResponseMetadata());
    }
}
