/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.budgets.model;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.services.budgets.transform.BudgetMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the output of the <code>CreateBudget</code> operation. The content consists of the detailed metadata and
 * data file information, and the current status of the <code>budget</code>.
 * </p>
 * <p>
 * The ARN pattern for a budget is: <code>arn:aws:budgetservice::AccountId:budget/budgetName</code>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Budget implements StructuredPojo, ToCopyableBuilder<Budget.Builder, Budget> {
    private final String budgetName;

    private final Spend budgetLimit;

    private final Map<String, List<String>> costFilters;

    private final CostTypes costTypes;

    private final String timeUnit;

    private final TimePeriod timePeriod;

    private final CalculatedSpend calculatedSpend;

    private final String budgetType;

    private Budget(BuilderImpl builder) {
        this.budgetName = builder.budgetName;
        this.budgetLimit = builder.budgetLimit;
        this.costFilters = builder.costFilters;
        this.costTypes = builder.costTypes;
        this.timeUnit = builder.timeUnit;
        this.timePeriod = builder.timePeriod;
        this.calculatedSpend = builder.calculatedSpend;
        this.budgetType = builder.budgetType;
    }

    /**
     * <p>
     * The name of a budget. Unique within accounts. <code>:</code> and <code>\</code> characters are not allowed in the
     * <code>BudgetName</code>.
     * </p>
     * 
     * @return The name of a budget. Unique within accounts. <code>:</code> and <code>\</code> characters are not
     *         allowed in the <code>BudgetName</code>.
     */
    public String budgetName() {
        return budgetName;
    }

    /**
     * <p>
     * The total amount of cost, usage, or RI utilization that you want to track with your budget.
     * </p>
     * <p>
     * <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI utilization budgets. RI
     * utilization budgets default to the only valid value for RI utilization budgets, which is <code>100</code>.
     * </p>
     * 
     * @return The total amount of cost, usage, or RI utilization that you want to track with your budget.</p>
     *         <p>
     *         <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI utilization budgets.
     *         RI utilization budgets default to the only valid value for RI utilization budgets, which is
     *         <code>100</code>.
     */
    public Spend budgetLimit() {
        return budgetLimit;
    }

    /**
     * <p>
     * The cost filters applied to a budget, such as service or region.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The cost filters applied to a budget, such as service or region.
     */
    public Map<String, List<String>> costFilters() {
        return costFilters;
    }

    /**
     * <p>
     * The types of costs included in this budget.
     * </p>
     * 
     * @return The types of costs included in this budget.
     */
    public CostTypes costTypes() {
        return costTypes;
    }

    /**
     * <p>
     * The length of time until a budget resets the actual and forecasted spend.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #timeUnit} will
     * return {@link TimeUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #timeUnitAsString}.
     * </p>
     * 
     * @return The length of time until a budget resets the actual and forecasted spend.
     * @see TimeUnit
     */
    public TimeUnit timeUnit() {
        return TimeUnit.fromValue(timeUnit);
    }

    /**
     * <p>
     * The length of time until a budget resets the actual and forecasted spend.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #timeUnit} will
     * return {@link TimeUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #timeUnitAsString}.
     * </p>
     * 
     * @return The length of time until a budget resets the actual and forecasted spend.
     * @see TimeUnit
     */
    public String timeUnitAsString() {
        return timeUnit;
    }

    /**
     * <p>
     * The period of time covered by a budget. Has a start date and an end date. The start date must come before the end
     * date. There are no restrictions on the end date.
     * </p>
     * <p>
     * If you created your budget and didn't specify a start date, AWS defaults to the start of your chosen time period
     * (i.e. DAILY, MONTHLY, QUARTERLY, ANNUALLY). For example, if you created your budget on January 24th 2018, chose
     * <code>DAILY</code>, and didn't set a start date, AWS set your start date to <code>01/24/18 00:00 UTC</code>. If
     * you chose <code>MONTHLY</code>, AWS set your start date to <code>01/01/18 00:00 UTC</code>. If you didn't specify
     * an end date, AWS set your end date to <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS
     * Billing and Cost Management console and the API.
     * </p>
     * <p>
     * You can change either date with the <code>UpdateBudget</code> operation.
     * </p>
     * <p>
     * After the end date, AWS deletes the budget and all associated notifications and subscribers.
     * </p>
     * 
     * @return The period of time covered by a budget. Has a start date and an end date. The start date must come before
     *         the end date. There are no restrictions on the end date. </p>
     *         <p>
     *         If you created your budget and didn't specify a start date, AWS defaults to the start of your chosen time
     *         period (i.e. DAILY, MONTHLY, QUARTERLY, ANNUALLY). For example, if you created your budget on January
     *         24th 2018, chose <code>DAILY</code>, and didn't set a start date, AWS set your start date to
     *         <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, AWS set your start date to
     *         <code>01/01/18 00:00 UTC</code>. If you didn't specify an end date, AWS set your end date to
     *         <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost Management
     *         console and the API.
     *         </p>
     *         <p>
     *         You can change either date with the <code>UpdateBudget</code> operation.
     *         </p>
     *         <p>
     *         After the end date, AWS deletes the budget and all associated notifications and subscribers.
     */
    public TimePeriod timePeriod() {
        return timePeriod;
    }

    /**
     * <p>
     * The actual and forecasted cost or usage being tracked by a budget.
     * </p>
     * 
     * @return The actual and forecasted cost or usage being tracked by a budget.
     */
    public CalculatedSpend calculatedSpend() {
        return calculatedSpend;
    }

    /**
     * <p>
     * Whether this budget tracks monetary costs, usage, or RI utilization.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #budgetType} will
     * return {@link BudgetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #budgetTypeAsString}.
     * </p>
     * 
     * @return Whether this budget tracks monetary costs, usage, or RI utilization.
     * @see BudgetType
     */
    public BudgetType budgetType() {
        return BudgetType.fromValue(budgetType);
    }

    /**
     * <p>
     * Whether this budget tracks monetary costs, usage, or RI utilization.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #budgetType} will
     * return {@link BudgetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #budgetTypeAsString}.
     * </p>
     * 
     * @return Whether this budget tracks monetary costs, usage, or RI utilization.
     * @see BudgetType
     */
    public String budgetTypeAsString() {
        return budgetType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(budgetName());
        hashCode = 31 * hashCode + Objects.hashCode(budgetLimit());
        hashCode = 31 * hashCode + Objects.hashCode(costFilters());
        hashCode = 31 * hashCode + Objects.hashCode(costTypes());
        hashCode = 31 * hashCode + Objects.hashCode(timeUnitAsString());
        hashCode = 31 * hashCode + Objects.hashCode(timePeriod());
        hashCode = 31 * hashCode + Objects.hashCode(calculatedSpend());
        hashCode = 31 * hashCode + Objects.hashCode(budgetTypeAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Budget)) {
            return false;
        }
        Budget other = (Budget) obj;
        return Objects.equals(budgetName(), other.budgetName()) && Objects.equals(budgetLimit(), other.budgetLimit())
                && Objects.equals(costFilters(), other.costFilters()) && Objects.equals(costTypes(), other.costTypes())
                && Objects.equals(timeUnitAsString(), other.timeUnitAsString())
                && Objects.equals(timePeriod(), other.timePeriod()) && Objects.equals(calculatedSpend(), other.calculatedSpend())
                && Objects.equals(budgetTypeAsString(), other.budgetTypeAsString());
    }

    @Override
    public String toString() {
        return ToString.builder("Budget").add("BudgetName", budgetName()).add("BudgetLimit", budgetLimit())
                .add("CostFilters", costFilters()).add("CostTypes", costTypes()).add("TimeUnit", timeUnitAsString())
                .add("TimePeriod", timePeriod()).add("CalculatedSpend", calculatedSpend())
                .add("BudgetType", budgetTypeAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BudgetName":
            return Optional.ofNullable(clazz.cast(budgetName()));
        case "BudgetLimit":
            return Optional.ofNullable(clazz.cast(budgetLimit()));
        case "CostFilters":
            return Optional.ofNullable(clazz.cast(costFilters()));
        case "CostTypes":
            return Optional.ofNullable(clazz.cast(costTypes()));
        case "TimeUnit":
            return Optional.ofNullable(clazz.cast(timeUnitAsString()));
        case "TimePeriod":
            return Optional.ofNullable(clazz.cast(timePeriod()));
        case "CalculatedSpend":
            return Optional.ofNullable(clazz.cast(calculatedSpend()));
        case "BudgetType":
            return Optional.ofNullable(clazz.cast(budgetTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        BudgetMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Budget> {
        /**
         * <p>
         * The name of a budget. Unique within accounts. <code>:</code> and <code>\</code> characters are not allowed in
         * the <code>BudgetName</code>.
         * </p>
         * 
         * @param budgetName
         *        The name of a budget. Unique within accounts. <code>:</code> and <code>\</code> characters are not
         *        allowed in the <code>BudgetName</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder budgetName(String budgetName);

        /**
         * <p>
         * The total amount of cost, usage, or RI utilization that you want to track with your budget.
         * </p>
         * <p>
         * <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI utilization budgets. RI
         * utilization budgets default to the only valid value for RI utilization budgets, which is <code>100</code>.
         * </p>
         * 
         * @param budgetLimit
         *        The total amount of cost, usage, or RI utilization that you want to track with your budget.</p>
         *        <p>
         *        <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI utilization
         *        budgets. RI utilization budgets default to the only valid value for RI utilization budgets, which is
         *        <code>100</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder budgetLimit(Spend budgetLimit);

        /**
         * <p>
         * The total amount of cost, usage, or RI utilization that you want to track with your budget.
         * </p>
         * <p>
         * <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI utilization budgets. RI
         * utilization budgets default to the only valid value for RI utilization budgets, which is <code>100</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link Spend.Builder} avoiding the need to create one
         * manually via {@link Spend#builder()}.
         *
         * When the {@link Consumer} completes, {@link Spend.Builder#build()} is called immediately and its result is
         * passed to {@link #budgetLimit(Spend)}.
         * 
         * @param budgetLimit
         *        a consumer that will call methods on {@link Spend.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #budgetLimit(Spend)
         */
        default Builder budgetLimit(Consumer<Spend.Builder> budgetLimit) {
            return budgetLimit(Spend.builder().applyMutation(budgetLimit).build());
        }

        /**
         * <p>
         * The cost filters applied to a budget, such as service or region.
         * </p>
         * 
         * @param costFilters
         *        The cost filters applied to a budget, such as service or region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder costFilters(Map<String, ? extends Collection<String>> costFilters);

        /**
         * <p>
         * The types of costs included in this budget.
         * </p>
         * 
         * @param costTypes
         *        The types of costs included in this budget.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder costTypes(CostTypes costTypes);

        /**
         * <p>
         * The types of costs included in this budget.
         * </p>
         * This is a convenience that creates an instance of the {@link CostTypes.Builder} avoiding the need to create
         * one manually via {@link CostTypes#builder()}.
         *
         * When the {@link Consumer} completes, {@link CostTypes.Builder#build()} is called immediately and its result
         * is passed to {@link #costTypes(CostTypes)}.
         * 
         * @param costTypes
         *        a consumer that will call methods on {@link CostTypes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #costTypes(CostTypes)
         */
        default Builder costTypes(Consumer<CostTypes.Builder> costTypes) {
            return costTypes(CostTypes.builder().applyMutation(costTypes).build());
        }

        /**
         * <p>
         * The length of time until a budget resets the actual and forecasted spend.
         * </p>
         * 
         * @param timeUnit
         *        The length of time until a budget resets the actual and forecasted spend.
         * @see TimeUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TimeUnit
         */
        Builder timeUnit(String timeUnit);

        /**
         * <p>
         * The length of time until a budget resets the actual and forecasted spend.
         * </p>
         * 
         * @param timeUnit
         *        The length of time until a budget resets the actual and forecasted spend.
         * @see TimeUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TimeUnit
         */
        Builder timeUnit(TimeUnit timeUnit);

        /**
         * <p>
         * The period of time covered by a budget. Has a start date and an end date. The start date must come before the
         * end date. There are no restrictions on the end date.
         * </p>
         * <p>
         * If you created your budget and didn't specify a start date, AWS defaults to the start of your chosen time
         * period (i.e. DAILY, MONTHLY, QUARTERLY, ANNUALLY). For example, if you created your budget on January 24th
         * 2018, chose <code>DAILY</code>, and didn't set a start date, AWS set your start date to
         * <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, AWS set your start date to
         * <code>01/01/18 00:00 UTC</code>. If you didn't specify an end date, AWS set your end date to
         * <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost Management console
         * and the API.
         * </p>
         * <p>
         * You can change either date with the <code>UpdateBudget</code> operation.
         * </p>
         * <p>
         * After the end date, AWS deletes the budget and all associated notifications and subscribers.
         * </p>
         * 
         * @param timePeriod
         *        The period of time covered by a budget. Has a start date and an end date. The start date must come
         *        before the end date. There are no restrictions on the end date. </p>
         *        <p>
         *        If you created your budget and didn't specify a start date, AWS defaults to the start of your chosen
         *        time period (i.e. DAILY, MONTHLY, QUARTERLY, ANNUALLY). For example, if you created your budget on
         *        January 24th 2018, chose <code>DAILY</code>, and didn't set a start date, AWS set your start date to
         *        <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, AWS set your start date to
         *        <code>01/01/18 00:00 UTC</code>. If you didn't specify an end date, AWS set your end date to
         *        <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost Management
         *        console and the API.
         *        </p>
         *        <p>
         *        You can change either date with the <code>UpdateBudget</code> operation.
         *        </p>
         *        <p>
         *        After the end date, AWS deletes the budget and all associated notifications and subscribers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timePeriod(TimePeriod timePeriod);

        /**
         * <p>
         * The period of time covered by a budget. Has a start date and an end date. The start date must come before the
         * end date. There are no restrictions on the end date.
         * </p>
         * <p>
         * If you created your budget and didn't specify a start date, AWS defaults to the start of your chosen time
         * period (i.e. DAILY, MONTHLY, QUARTERLY, ANNUALLY). For example, if you created your budget on January 24th
         * 2018, chose <code>DAILY</code>, and didn't set a start date, AWS set your start date to
         * <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, AWS set your start date to
         * <code>01/01/18 00:00 UTC</code>. If you didn't specify an end date, AWS set your end date to
         * <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost Management console
         * and the API.
         * </p>
         * <p>
         * You can change either date with the <code>UpdateBudget</code> operation.
         * </p>
         * <p>
         * After the end date, AWS deletes the budget and all associated notifications and subscribers.
         * </p>
         * This is a convenience that creates an instance of the {@link TimePeriod.Builder} avoiding the need to create
         * one manually via {@link TimePeriod#builder()}.
         *
         * When the {@link Consumer} completes, {@link TimePeriod.Builder#build()} is called immediately and its result
         * is passed to {@link #timePeriod(TimePeriod)}.
         * 
         * @param timePeriod
         *        a consumer that will call methods on {@link TimePeriod.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #timePeriod(TimePeriod)
         */
        default Builder timePeriod(Consumer<TimePeriod.Builder> timePeriod) {
            return timePeriod(TimePeriod.builder().applyMutation(timePeriod).build());
        }

        /**
         * <p>
         * The actual and forecasted cost or usage being tracked by a budget.
         * </p>
         * 
         * @param calculatedSpend
         *        The actual and forecasted cost or usage being tracked by a budget.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder calculatedSpend(CalculatedSpend calculatedSpend);

        /**
         * <p>
         * The actual and forecasted cost or usage being tracked by a budget.
         * </p>
         * This is a convenience that creates an instance of the {@link CalculatedSpend.Builder} avoiding the need to
         * create one manually via {@link CalculatedSpend#builder()}.
         *
         * When the {@link Consumer} completes, {@link CalculatedSpend.Builder#build()} is called immediately and its
         * result is passed to {@link #calculatedSpend(CalculatedSpend)}.
         * 
         * @param calculatedSpend
         *        a consumer that will call methods on {@link CalculatedSpend.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #calculatedSpend(CalculatedSpend)
         */
        default Builder calculatedSpend(Consumer<CalculatedSpend.Builder> calculatedSpend) {
            return calculatedSpend(CalculatedSpend.builder().applyMutation(calculatedSpend).build());
        }

        /**
         * <p>
         * Whether this budget tracks monetary costs, usage, or RI utilization.
         * </p>
         * 
         * @param budgetType
         *        Whether this budget tracks monetary costs, usage, or RI utilization.
         * @see BudgetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BudgetType
         */
        Builder budgetType(String budgetType);

        /**
         * <p>
         * Whether this budget tracks monetary costs, usage, or RI utilization.
         * </p>
         * 
         * @param budgetType
         *        Whether this budget tracks monetary costs, usage, or RI utilization.
         * @see BudgetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BudgetType
         */
        Builder budgetType(BudgetType budgetType);
    }

    static final class BuilderImpl implements Builder {
        private String budgetName;

        private Spend budgetLimit;

        private Map<String, List<String>> costFilters = DefaultSdkAutoConstructMap.getInstance();

        private CostTypes costTypes;

        private String timeUnit;

        private TimePeriod timePeriod;

        private CalculatedSpend calculatedSpend;

        private String budgetType;

        private BuilderImpl() {
        }

        private BuilderImpl(Budget model) {
            budgetName(model.budgetName);
            budgetLimit(model.budgetLimit);
            costFilters(model.costFilters);
            costTypes(model.costTypes);
            timeUnit(model.timeUnit);
            timePeriod(model.timePeriod);
            calculatedSpend(model.calculatedSpend);
            budgetType(model.budgetType);
        }

        public final String getBudgetName() {
            return budgetName;
        }

        @Override
        public final Builder budgetName(String budgetName) {
            this.budgetName = budgetName;
            return this;
        }

        public final void setBudgetName(String budgetName) {
            this.budgetName = budgetName;
        }

        public final Spend.Builder getBudgetLimit() {
            return budgetLimit != null ? budgetLimit.toBuilder() : null;
        }

        @Override
        public final Builder budgetLimit(Spend budgetLimit) {
            this.budgetLimit = budgetLimit;
            return this;
        }

        public final void setBudgetLimit(Spend.BuilderImpl budgetLimit) {
            this.budgetLimit = budgetLimit != null ? budgetLimit.build() : null;
        }

        public final Map<String, ? extends Collection<String>> getCostFilters() {
            return costFilters;
        }

        @Override
        public final Builder costFilters(Map<String, ? extends Collection<String>> costFilters) {
            this.costFilters = CostFiltersCopier.copy(costFilters);
            return this;
        }

        public final void setCostFilters(Map<String, ? extends Collection<String>> costFilters) {
            this.costFilters = CostFiltersCopier.copy(costFilters);
        }

        public final CostTypes.Builder getCostTypes() {
            return costTypes != null ? costTypes.toBuilder() : null;
        }

        @Override
        public final Builder costTypes(CostTypes costTypes) {
            this.costTypes = costTypes;
            return this;
        }

        public final void setCostTypes(CostTypes.BuilderImpl costTypes) {
            this.costTypes = costTypes != null ? costTypes.build() : null;
        }

        public final String getTimeUnit() {
            return timeUnit;
        }

        @Override
        public final Builder timeUnit(String timeUnit) {
            this.timeUnit = timeUnit;
            return this;
        }

        @Override
        public final Builder timeUnit(TimeUnit timeUnit) {
            this.timeUnit(timeUnit.toString());
            return this;
        }

        public final void setTimeUnit(String timeUnit) {
            this.timeUnit = timeUnit;
        }

        public final TimePeriod.Builder getTimePeriod() {
            return timePeriod != null ? timePeriod.toBuilder() : null;
        }

        @Override
        public final Builder timePeriod(TimePeriod timePeriod) {
            this.timePeriod = timePeriod;
            return this;
        }

        public final void setTimePeriod(TimePeriod.BuilderImpl timePeriod) {
            this.timePeriod = timePeriod != null ? timePeriod.build() : null;
        }

        public final CalculatedSpend.Builder getCalculatedSpend() {
            return calculatedSpend != null ? calculatedSpend.toBuilder() : null;
        }

        @Override
        public final Builder calculatedSpend(CalculatedSpend calculatedSpend) {
            this.calculatedSpend = calculatedSpend;
            return this;
        }

        public final void setCalculatedSpend(CalculatedSpend.BuilderImpl calculatedSpend) {
            this.calculatedSpend = calculatedSpend != null ? calculatedSpend.build() : null;
        }

        public final String getBudgetType() {
            return budgetType;
        }

        @Override
        public final Builder budgetType(String budgetType) {
            this.budgetType = budgetType;
            return this;
        }

        @Override
        public final Builder budgetType(BudgetType budgetType) {
            this.budgetType(budgetType.toString());
            return this;
        }

        public final void setBudgetType(String budgetType) {
            this.budgetType = budgetType;
        }

        @Override
        public Budget build() {
            return new Budget(this);
        }
    }
}
