/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.budgets.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.budgets.transform.NotificationWithSubscribersMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A notification with subscribers. A notification can have one SNS subscriber and up to ten email subscribers, for a
 * total of 11 subscribers.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NotificationWithSubscribers implements StructuredPojo,
        ToCopyableBuilder<NotificationWithSubscribers.Builder, NotificationWithSubscribers> {
    private final Notification notification;

    private final List<Subscriber> subscribers;

    private NotificationWithSubscribers(BuilderImpl builder) {
        this.notification = builder.notification;
        this.subscribers = builder.subscribers;
    }

    /**
     * <p>
     * The notification associated with a budget.
     * </p>
     * 
     * @return The notification associated with a budget.
     */
    public Notification notification() {
        return notification;
    }

    /**
     * <p>
     * A list of subscribers who are subscribed to this notification.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return A list of subscribers who are subscribed to this notification.
     */
    public List<Subscriber> subscribers() {
        return subscribers;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(notification());
        hashCode = 31 * hashCode + Objects.hashCode(subscribers());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NotificationWithSubscribers)) {
            return false;
        }
        NotificationWithSubscribers other = (NotificationWithSubscribers) obj;
        return Objects.equals(notification(), other.notification()) && Objects.equals(subscribers(), other.subscribers());
    }

    @Override
    public String toString() {
        return ToString.builder("NotificationWithSubscribers").add("Notification", notification())
                .add("Subscribers", subscribers()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Notification":
            return Optional.ofNullable(clazz.cast(notification()));
        case "Subscribers":
            return Optional.ofNullable(clazz.cast(subscribers()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        NotificationWithSubscribersMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, NotificationWithSubscribers> {
        /**
         * <p>
         * The notification associated with a budget.
         * </p>
         * 
         * @param notification
         *        The notification associated with a budget.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notification(Notification notification);

        /**
         * <p>
         * The notification associated with a budget.
         * </p>
         * This is a convenience that creates an instance of the {@link Notification.Builder} avoiding the need to
         * create one manually via {@link Notification#builder()}.
         *
         * When the {@link Consumer} completes, {@link Notification.Builder#build()} is called immediately and its
         * result is passed to {@link #notification(Notification)}.
         * 
         * @param notification
         *        a consumer that will call methods on {@link Notification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notification(Notification)
         */
        default Builder notification(Consumer<Notification.Builder> notification) {
            return notification(Notification.builder().applyMutation(notification).build());
        }

        /**
         * <p>
         * A list of subscribers who are subscribed to this notification.
         * </p>
         * 
         * @param subscribers
         *        A list of subscribers who are subscribed to this notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subscribers(Collection<Subscriber> subscribers);

        /**
         * <p>
         * A list of subscribers who are subscribed to this notification.
         * </p>
         * 
         * @param subscribers
         *        A list of subscribers who are subscribed to this notification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subscribers(Subscriber... subscribers);

        /**
         * <p>
         * A list of subscribers who are subscribed to this notification.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Subscriber>.Builder} avoiding the need to
         * create one manually via {@link List<Subscriber>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Subscriber>.Builder#build()} is called immediately and its
         * result is passed to {@link #subscribers(List<Subscriber>)}.
         * 
         * @param subscribers
         *        a consumer that will call methods on {@link List<Subscriber>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #subscribers(List<Subscriber>)
         */
        Builder subscribers(Consumer<Subscriber.Builder>... subscribers);
    }

    static final class BuilderImpl implements Builder {
        private Notification notification;

        private List<Subscriber> subscribers = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(NotificationWithSubscribers model) {
            notification(model.notification);
            subscribers(model.subscribers);
        }

        public final Notification.Builder getNotification() {
            return notification != null ? notification.toBuilder() : null;
        }

        @Override
        public final Builder notification(Notification notification) {
            this.notification = notification;
            return this;
        }

        public final void setNotification(Notification.BuilderImpl notification) {
            this.notification = notification != null ? notification.build() : null;
        }

        public final Collection<Subscriber.Builder> getSubscribers() {
            return subscribers != null ? subscribers.stream().map(Subscriber::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder subscribers(Collection<Subscriber> subscribers) {
            this.subscribers = SubscribersCopier.copy(subscribers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subscribers(Subscriber... subscribers) {
            subscribers(Arrays.asList(subscribers));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subscribers(Consumer<Subscriber.Builder>... subscribers) {
            subscribers(Stream.of(subscribers).map(c -> Subscriber.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setSubscribers(Collection<Subscriber.BuilderImpl> subscribers) {
            this.subscribers = SubscribersCopier.copyFromBuilder(subscribers);
        }

        @Override
        public NotificationWithSubscribers build() {
            return new NotificationWithSubscribers(this);
        }
    }
}
