/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.budgets.model;

import java.math.BigDecimal;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.budgets.transform.SpendMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The amount of cost or usage being measured for a budget.
 * </p>
 * <p>
 * For example, a <code>Spend</code> for <code>3 GB</code> of S3 usage would have the following parameters:
 * </p>
 * <ul>
 * <li>
 * <p>
 * An <code>Amount</code> of <code>3</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * A <code>unit</code> of <code>GB</code>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Spend implements StructuredPojo, ToCopyableBuilder<Spend.Builder, Spend> {
    private final BigDecimal amount;

    private final String unit;

    private Spend(BuilderImpl builder) {
        this.amount = builder.amount;
        this.unit = builder.unit;
    }

    /**
     * <p>
     * The cost or usage amount associated with a budget forecast, actual spend, or budget threshold.
     * </p>
     * 
     * @return The cost or usage amount associated with a budget forecast, actual spend, or budget threshold.
     */
    public BigDecimal amount() {
        return amount;
    }

    /**
     * <p>
     * The unit of measurement used for the budget forecast, actual spend, or budget threshold, such as dollars or GB.
     * </p>
     * 
     * @return The unit of measurement used for the budget forecast, actual spend, or budget threshold, such as dollars
     *         or GB.
     */
    public String unit() {
        return unit;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(amount());
        hashCode = 31 * hashCode + Objects.hashCode(unit());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Spend)) {
            return false;
        }
        Spend other = (Spend) obj;
        return Objects.equals(amount(), other.amount()) && Objects.equals(unit(), other.unit());
    }

    @Override
    public String toString() {
        return ToString.builder("Spend").add("Amount", amount()).add("Unit", unit()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Amount":
            return Optional.ofNullable(clazz.cast(amount()));
        case "Unit":
            return Optional.ofNullable(clazz.cast(unit()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        SpendMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Spend> {
        /**
         * <p>
         * The cost or usage amount associated with a budget forecast, actual spend, or budget threshold.
         * </p>
         * 
         * @param amount
         *        The cost or usage amount associated with a budget forecast, actual spend, or budget threshold.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder amount(BigDecimal amount);

        /**
         * <p>
         * The unit of measurement used for the budget forecast, actual spend, or budget threshold, such as dollars or
         * GB.
         * </p>
         * 
         * @param unit
         *        The unit of measurement used for the budget forecast, actual spend, or budget threshold, such as
         *        dollars or GB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unit(String unit);
    }

    static final class BuilderImpl implements Builder {
        private BigDecimal amount;

        private String unit;

        private BuilderImpl() {
        }

        private BuilderImpl(Spend model) {
            amount(model.amount);
            unit(model.unit);
        }

        public final BigDecimal getAmount() {
            return amount;
        }

        @Override
        public final Builder amount(BigDecimal amount) {
            this.amount = amount;
            return this;
        }

        public final void setAmount(BigDecimal amount) {
            this.amount = amount;
        }

        public final String getUnit() {
            return unit;
        }

        @Override
        public final Builder unit(String unit) {
            this.unit = unit;
            return this;
        }

        public final void setUnit(String unit) {
            this.unit = unit;
        }

        @Override
        public Spend build() {
            return new Spend(this);
        }
    }
}
