/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.budgets.model;

import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.budgets.transform.SubscriberMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The subscriber to a budget notification. The subscriber consists of a subscription type and either an Amazon Simple
 * Notification Service topic or an email address.
 * </p>
 * <p>
 * For example, an email subscriber would have the following parameters:
 * </p>
 * <ul>
 * <li>
 * <p>
 * A <code>subscriptionType</code> of <code>EMAIL</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * An <code>address</code> of <code>example@example.com</code>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Subscriber implements StructuredPojo, ToCopyableBuilder<Subscriber.Builder, Subscriber> {
    private final String subscriptionType;

    private final String address;

    private Subscriber(BuilderImpl builder) {
        this.subscriptionType = builder.subscriptionType;
        this.address = builder.address;
    }

    /**
     * <p>
     * The type of notification that AWS sends to a subscriber.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #subscriptionType}
     * will return {@link SubscriptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #subscriptionTypeAsString}.
     * </p>
     * 
     * @return The type of notification that AWS sends to a subscriber.
     * @see SubscriptionType
     */
    public SubscriptionType subscriptionType() {
        return SubscriptionType.fromValue(subscriptionType);
    }

    /**
     * <p>
     * The type of notification that AWS sends to a subscriber.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #subscriptionType}
     * will return {@link SubscriptionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #subscriptionTypeAsString}.
     * </p>
     * 
     * @return The type of notification that AWS sends to a subscriber.
     * @see SubscriptionType
     */
    public String subscriptionTypeAsString() {
        return subscriptionType;
    }

    /**
     * <p>
     * The address that AWS sends budget notifications to, either an SNS topic or an email.
     * </p>
     * 
     * @return The address that AWS sends budget notifications to, either an SNS topic or an email.
     */
    public String address() {
        return address;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(subscriptionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(address());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Subscriber)) {
            return false;
        }
        Subscriber other = (Subscriber) obj;
        return Objects.equals(subscriptionTypeAsString(), other.subscriptionTypeAsString())
                && Objects.equals(address(), other.address());
    }

    @Override
    public String toString() {
        return ToString.builder("Subscriber").add("SubscriptionType", subscriptionTypeAsString()).add("Address", address())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SubscriptionType":
            return Optional.ofNullable(clazz.cast(subscriptionTypeAsString()));
        case "Address":
            return Optional.ofNullable(clazz.cast(address()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        SubscriberMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, Subscriber> {
        /**
         * <p>
         * The type of notification that AWS sends to a subscriber.
         * </p>
         * 
         * @param subscriptionType
         *        The type of notification that AWS sends to a subscriber.
         * @see SubscriptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SubscriptionType
         */
        Builder subscriptionType(String subscriptionType);

        /**
         * <p>
         * The type of notification that AWS sends to a subscriber.
         * </p>
         * 
         * @param subscriptionType
         *        The type of notification that AWS sends to a subscriber.
         * @see SubscriptionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SubscriptionType
         */
        Builder subscriptionType(SubscriptionType subscriptionType);

        /**
         * <p>
         * The address that AWS sends budget notifications to, either an SNS topic or an email.
         * </p>
         * 
         * @param address
         *        The address that AWS sends budget notifications to, either an SNS topic or an email.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder address(String address);
    }

    static final class BuilderImpl implements Builder {
        private String subscriptionType;

        private String address;

        private BuilderImpl() {
        }

        private BuilderImpl(Subscriber model) {
            subscriptionType(model.subscriptionType);
            address(model.address);
        }

        public final String getSubscriptionType() {
            return subscriptionType;
        }

        @Override
        public final Builder subscriptionType(String subscriptionType) {
            this.subscriptionType = subscriptionType;
            return this;
        }

        @Override
        public final Builder subscriptionType(SubscriptionType subscriptionType) {
            this.subscriptionType(subscriptionType.toString());
            return this;
        }

        public final void setSubscriptionType(String subscriptionType) {
            this.subscriptionType = subscriptionType;
        }

        public final String getAddress() {
            return address;
        }

        @Override
        public final Builder address(String address) {
            this.address = address;
            return this;
        }

        public final void setAddress(String address) {
            this.address = address;
        }

        @Override
        public Subscriber build() {
            return new Subscriber(this);
        }
    }
}
