/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.budgets.model;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.budgets.transform.TimePeriodMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The period of time covered by a budget. Has a start date and an end date. The start date must come before the end
 * date. There are no restrictions on the end date.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TimePeriod implements StructuredPojo, ToCopyableBuilder<TimePeriod.Builder, TimePeriod> {
    private final Instant start;

    private final Instant end;

    private TimePeriod(BuilderImpl builder) {
        this.start = builder.start;
        this.end = builder.end;
    }

    /**
     * <p>
     * The start date for a budget. If you created your budget and didn't specify a start date, AWS defaults to the
     * start of your chosen time period (i.e. DAILY, MONTHLY, QUARTERLY, ANNUALLY). For example, if you created your
     * budget on January 24th 2018, chose <code>DAILY</code>, and didn't set a start date, AWS set your start date to
     * <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, AWS set your start date to
     * <code>01/01/18 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost Management console and
     * the API.
     * </p>
     * <p>
     * You can change your start date with the <code>UpdateBudget</code> operation.
     * </p>
     * 
     * @return The start date for a budget. If you created your budget and didn't specify a start date, AWS defaults to
     *         the start of your chosen time period (i.e. DAILY, MONTHLY, QUARTERLY, ANNUALLY). For example, if you
     *         created your budget on January 24th 2018, chose <code>DAILY</code>, and didn't set a start date, AWS set
     *         your start date to <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, AWS set your start
     *         date to <code>01/01/18 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost
     *         Management console and the API.</p>
     *         <p>
     *         You can change your start date with the <code>UpdateBudget</code> operation.
     */
    public Instant start() {
        return start;
    }

    /**
     * <p>
     * The end date for a budget. If you didn't specify an end date, AWS set your end date to
     * <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost Management console and
     * the API.
     * </p>
     * <p>
     * After the end date, AWS deletes the budget and all associated notifications and subscribers. You can change your
     * end date with the <code>UpdateBudget</code> operation.
     * </p>
     * 
     * @return The end date for a budget. If you didn't specify an end date, AWS set your end date to
     *         <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost Management
     *         console and the API.</p>
     *         <p>
     *         After the end date, AWS deletes the budget and all associated notifications and subscribers. You can
     *         change your end date with the <code>UpdateBudget</code> operation.
     */
    public Instant end() {
        return end;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(start());
        hashCode = 31 * hashCode + Objects.hashCode(end());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TimePeriod)) {
            return false;
        }
        TimePeriod other = (TimePeriod) obj;
        return Objects.equals(start(), other.start()) && Objects.equals(end(), other.end());
    }

    @Override
    public String toString() {
        return ToString.builder("TimePeriod").add("Start", start()).add("End", end()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Start":
            return Optional.ofNullable(clazz.cast(start()));
        case "End":
            return Optional.ofNullable(clazz.cast(end()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        TimePeriodMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, TimePeriod> {
        /**
         * <p>
         * The start date for a budget. If you created your budget and didn't specify a start date, AWS defaults to the
         * start of your chosen time period (i.e. DAILY, MONTHLY, QUARTERLY, ANNUALLY). For example, if you created your
         * budget on January 24th 2018, chose <code>DAILY</code>, and didn't set a start date, AWS set your start date
         * to <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, AWS set your start date to
         * <code>01/01/18 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost Management console
         * and the API.
         * </p>
         * <p>
         * You can change your start date with the <code>UpdateBudget</code> operation.
         * </p>
         * 
         * @param start
         *        The start date for a budget. If you created your budget and didn't specify a start date, AWS defaults
         *        to the start of your chosen time period (i.e. DAILY, MONTHLY, QUARTERLY, ANNUALLY). For example, if
         *        you created your budget on January 24th 2018, chose <code>DAILY</code>, and didn't set a start date,
         *        AWS set your start date to <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, AWS set
         *        your start date to <code>01/01/18 00:00 UTC</code>. The defaults are the same for the AWS Billing and
         *        Cost Management console and the API.</p>
         *        <p>
         *        You can change your start date with the <code>UpdateBudget</code> operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder start(Instant start);

        /**
         * <p>
         * The end date for a budget. If you didn't specify an end date, AWS set your end date to
         * <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost Management console
         * and the API.
         * </p>
         * <p>
         * After the end date, AWS deletes the budget and all associated notifications and subscribers. You can change
         * your end date with the <code>UpdateBudget</code> operation.
         * </p>
         * 
         * @param end
         *        The end date for a budget. If you didn't specify an end date, AWS set your end date to
         *        <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost Management
         *        console and the API.</p>
         *        <p>
         *        After the end date, AWS deletes the budget and all associated notifications and subscribers. You can
         *        change your end date with the <code>UpdateBudget</code> operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder end(Instant end);
    }

    static final class BuilderImpl implements Builder {
        private Instant start;

        private Instant end;

        private BuilderImpl() {
        }

        private BuilderImpl(TimePeriod model) {
            start(model.start);
            end(model.end);
        }

        public final Instant getStart() {
            return start;
        }

        @Override
        public final Builder start(Instant start) {
            this.start = start;
            return this;
        }

        public final void setStart(Instant start) {
            this.start = start;
        }

        public final Instant getEnd() {
            return end;
        }

        @Override
        public final Builder end(Instant end) {
            this.end = end;
            return this;
        }

        public final void setEnd(Instant end) {
            this.end = end;
        }

        @Override
        public TimePeriod build() {
            return new TimePeriod(this);
        }
    }
}
