/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.budgets.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the output of the <code>CreateBudget</code> operation. The content consists of the detailed metadata and
 * data file information, and the current status of the <code>budget</code> object.
 * </p>
 * <p>
 * This is the ARN pattern for a budget:
 * </p>
 * <p>
 * <code>arn:aws:budgets::AccountId:budget/budgetName</code>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Budget implements SdkPojo, Serializable, ToCopyableBuilder<Budget.Builder, Budget> {
    private static final SdkField<String> BUDGET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BudgetName").getter(getter(Budget::budgetName)).setter(setter(Builder::budgetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BudgetName").build()).build();

    private static final SdkField<Spend> BUDGET_LIMIT_FIELD = SdkField.<Spend> builder(MarshallingType.SDK_POJO)
            .memberName("BudgetLimit").getter(getter(Budget::budgetLimit)).setter(setter(Builder::budgetLimit))
            .constructor(Spend::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BudgetLimit").build()).build();

    private static final SdkField<Map<String, Spend>> PLANNED_BUDGET_LIMITS_FIELD = SdkField
            .<Map<String, Spend>> builder(MarshallingType.MAP)
            .memberName("PlannedBudgetLimits")
            .getter(getter(Budget::plannedBudgetLimits))
            .setter(setter(Builder::plannedBudgetLimits))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlannedBudgetLimits").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Spend> builder(MarshallingType.SDK_POJO)
                                            .constructor(Spend::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, List<String>>> COST_FILTERS_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .memberName("CostFilters")
            .getter(getter(Budget::costFilters))
            .setter(setter(Builder::costFilters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CostFilters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<CostTypes> COST_TYPES_FIELD = SdkField.<CostTypes> builder(MarshallingType.SDK_POJO)
            .memberName("CostTypes").getter(getter(Budget::costTypes)).setter(setter(Builder::costTypes))
            .constructor(CostTypes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CostTypes").build()).build();

    private static final SdkField<String> TIME_UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TimeUnit").getter(getter(Budget::timeUnitAsString)).setter(setter(Builder::timeUnit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeUnit").build()).build();

    private static final SdkField<TimePeriod> TIME_PERIOD_FIELD = SdkField.<TimePeriod> builder(MarshallingType.SDK_POJO)
            .memberName("TimePeriod").getter(getter(Budget::timePeriod)).setter(setter(Builder::timePeriod))
            .constructor(TimePeriod::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimePeriod").build()).build();

    private static final SdkField<CalculatedSpend> CALCULATED_SPEND_FIELD = SdkField
            .<CalculatedSpend> builder(MarshallingType.SDK_POJO).memberName("CalculatedSpend")
            .getter(getter(Budget::calculatedSpend)).setter(setter(Builder::calculatedSpend))
            .constructor(CalculatedSpend::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CalculatedSpend").build()).build();

    private static final SdkField<String> BUDGET_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BudgetType").getter(getter(Budget::budgetTypeAsString)).setter(setter(Builder::budgetType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BudgetType").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedTime").getter(getter(Budget::lastUpdatedTime)).setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUDGET_NAME_FIELD,
            BUDGET_LIMIT_FIELD, PLANNED_BUDGET_LIMITS_FIELD, COST_FILTERS_FIELD, COST_TYPES_FIELD, TIME_UNIT_FIELD,
            TIME_PERIOD_FIELD, CALCULATED_SPEND_FIELD, BUDGET_TYPE_FIELD, LAST_UPDATED_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String budgetName;

    private final Spend budgetLimit;

    private final Map<String, Spend> plannedBudgetLimits;

    private final Map<String, List<String>> costFilters;

    private final CostTypes costTypes;

    private final String timeUnit;

    private final TimePeriod timePeriod;

    private final CalculatedSpend calculatedSpend;

    private final String budgetType;

    private final Instant lastUpdatedTime;

    private Budget(BuilderImpl builder) {
        this.budgetName = builder.budgetName;
        this.budgetLimit = builder.budgetLimit;
        this.plannedBudgetLimits = builder.plannedBudgetLimits;
        this.costFilters = builder.costFilters;
        this.costTypes = builder.costTypes;
        this.timeUnit = builder.timeUnit;
        this.timePeriod = builder.timePeriod;
        this.calculatedSpend = builder.calculatedSpend;
        this.budgetType = builder.budgetType;
        this.lastUpdatedTime = builder.lastUpdatedTime;
    }

    /**
     * <p>
     * The name of a budget. The name must be unique within an account. The <code>:</code> and <code>\</code> characters
     * aren't allowed in <code>BudgetName</code>.
     * </p>
     * 
     * @return The name of a budget. The name must be unique within an account. The <code>:</code> and <code>\</code>
     *         characters aren't allowed in <code>BudgetName</code>.
     */
    public final String budgetName() {
        return budgetName;
    }

    /**
     * <p>
     * The total amount of cost, usage, RI utilization, RI coverage, Savings Plans utilization, or Savings Plans
     * coverage that you want to track with your budget.
     * </p>
     * <p>
     * <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI or Savings Plans utilization
     * or coverage budgets. RI and Savings Plans utilization or coverage budgets default to <code>100</code>, which is
     * the only valid value for RI or Savings Plans utilization or coverage budgets. You can't use
     * <code>BudgetLimit</code> with <code>PlannedBudgetLimits</code> for <code>CreateBudget</code> and
     * <code>UpdateBudget</code> actions.
     * </p>
     * 
     * @return The total amount of cost, usage, RI utilization, RI coverage, Savings Plans utilization, or Savings Plans
     *         coverage that you want to track with your budget.</p>
     *         <p>
     *         <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI or Savings Plans
     *         utilization or coverage budgets. RI and Savings Plans utilization or coverage budgets default to
     *         <code>100</code>, which is the only valid value for RI or Savings Plans utilization or coverage budgets.
     *         You can't use <code>BudgetLimit</code> with <code>PlannedBudgetLimits</code> for
     *         <code>CreateBudget</code> and <code>UpdateBudget</code> actions.
     */
    public final Spend budgetLimit() {
        return budgetLimit;
    }

    /**
     * Returns true if the PlannedBudgetLimits property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasPlannedBudgetLimits() {
        return plannedBudgetLimits != null && !(plannedBudgetLimits instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map containing multiple <code>BudgetLimit</code>, including current or future limits.
     * </p>
     * <p>
     * <code>PlannedBudgetLimits</code> is available for cost or usage budget and supports monthly and quarterly
     * <code>TimeUnit</code>.
     * </p>
     * <p>
     * For monthly budgets, provide 12 months of <code>PlannedBudgetLimits</code> values. This must start from the
     * current month and include the next 11 months. The <code>key</code> is the start of the month, <code>UTC</code> in
     * epoch seconds.
     * </p>
     * <p>
     * For quarterly budgets, provide 4 quarters of <code>PlannedBudgetLimits</code> value entries in standard calendar
     * quarter increments. This must start from the current quarter and include the next 3 quarters. The
     * <code>key</code> is the start of the quarter, <code>UTC</code> in epoch seconds.
     * </p>
     * <p>
     * If the planned budget expires before 12 months for monthly or 4 quarters for quarterly, provide the
     * <code>PlannedBudgetLimits</code> values only for the remaining periods.
     * </p>
     * <p>
     * If the budget begins at a date in the future, provide <code>PlannedBudgetLimits</code> values from the start date
     * of the budget.
     * </p>
     * <p>
     * After all of the <code>BudgetLimit</code> values in <code>PlannedBudgetLimits</code> are used, the budget
     * continues to use the last limit as the <code>BudgetLimit</code>. At that point, the planned budget provides the
     * same experience as a fixed budget.
     * </p>
     * <p>
     * <code>DescribeBudget</code> and <code>DescribeBudgets</code> response along with <code>PlannedBudgetLimits</code>
     * will also contain <code>BudgetLimit</code> representing the current month or quarter limit present in
     * <code>PlannedBudgetLimits</code>. This only applies to budgets created with <code>PlannedBudgetLimits</code>.
     * Budgets created without <code>PlannedBudgetLimits</code> will only contain <code>BudgetLimit</code>, and no
     * <code>PlannedBudgetLimits</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPlannedBudgetLimits()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A map containing multiple <code>BudgetLimit</code>, including current or future limits.</p>
     *         <p>
     *         <code>PlannedBudgetLimits</code> is available for cost or usage budget and supports monthly and quarterly
     *         <code>TimeUnit</code>.
     *         </p>
     *         <p>
     *         For monthly budgets, provide 12 months of <code>PlannedBudgetLimits</code> values. This must start from
     *         the current month and include the next 11 months. The <code>key</code> is the start of the month,
     *         <code>UTC</code> in epoch seconds.
     *         </p>
     *         <p>
     *         For quarterly budgets, provide 4 quarters of <code>PlannedBudgetLimits</code> value entries in standard
     *         calendar quarter increments. This must start from the current quarter and include the next 3 quarters.
     *         The <code>key</code> is the start of the quarter, <code>UTC</code> in epoch seconds.
     *         </p>
     *         <p>
     *         If the planned budget expires before 12 months for monthly or 4 quarters for quarterly, provide the
     *         <code>PlannedBudgetLimits</code> values only for the remaining periods.
     *         </p>
     *         <p>
     *         If the budget begins at a date in the future, provide <code>PlannedBudgetLimits</code> values from the
     *         start date of the budget.
     *         </p>
     *         <p>
     *         After all of the <code>BudgetLimit</code> values in <code>PlannedBudgetLimits</code> are used, the budget
     *         continues to use the last limit as the <code>BudgetLimit</code>. At that point, the planned budget
     *         provides the same experience as a fixed budget.
     *         </p>
     *         <p>
     *         <code>DescribeBudget</code> and <code>DescribeBudgets</code> response along with
     *         <code>PlannedBudgetLimits</code> will also contain <code>BudgetLimit</code> representing the current
     *         month or quarter limit present in <code>PlannedBudgetLimits</code>. This only applies to budgets created
     *         with <code>PlannedBudgetLimits</code>. Budgets created without <code>PlannedBudgetLimits</code> will only
     *         contain <code>BudgetLimit</code>, and no <code>PlannedBudgetLimits</code>.
     */
    public final Map<String, Spend> plannedBudgetLimits() {
        return plannedBudgetLimits;
    }

    /**
     * Returns true if the CostFilters property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasCostFilters() {
        return costFilters != null && !(costFilters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The cost filters, such as service or tag, that are applied to a budget.
     * </p>
     * <p>
     * AWS Budgets supports the following services as a filter for RI budgets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon Elastic Compute Cloud - Compute
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Redshift
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Relational Database Service
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon ElastiCache
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Elasticsearch Service
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasCostFilters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The cost filters, such as service or tag, that are applied to a budget.</p>
     *         <p>
     *         AWS Budgets supports the following services as a filter for RI budgets:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Amazon Elastic Compute Cloud - Compute
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Redshift
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Relational Database Service
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon ElastiCache
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Elasticsearch Service
     *         </p>
     *         </li>
     */
    public final Map<String, List<String>> costFilters() {
        return costFilters;
    }

    /**
     * <p>
     * The types of costs that are included in this <code>COST</code> budget.
     * </p>
     * <p>
     * <code>USAGE</code>, <code>RI_UTILIZATION</code>, <code>RI_COVERAGE</code>, <code>SAVINGS_PLANS_UTILIZATION</code>
     * , and <code>SAVINGS_PLANS_COVERAGE</code> budgets do not have <code>CostTypes</code>.
     * </p>
     * 
     * @return The types of costs that are included in this <code>COST</code> budget.</p>
     *         <p>
     *         <code>USAGE</code>, <code>RI_UTILIZATION</code>, <code>RI_COVERAGE</code>,
     *         <code>SAVINGS_PLANS_UTILIZATION</code>, and <code>SAVINGS_PLANS_COVERAGE</code> budgets do not have
     *         <code>CostTypes</code>.
     */
    public final CostTypes costTypes() {
        return costTypes;
    }

    /**
     * <p>
     * The length of time until a budget resets the actual and forecasted spend.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #timeUnit} will
     * return {@link TimeUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #timeUnitAsString}.
     * </p>
     * 
     * @return The length of time until a budget resets the actual and forecasted spend.
     * @see TimeUnit
     */
    public final TimeUnit timeUnit() {
        return TimeUnit.fromValue(timeUnit);
    }

    /**
     * <p>
     * The length of time until a budget resets the actual and forecasted spend.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #timeUnit} will
     * return {@link TimeUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #timeUnitAsString}.
     * </p>
     * 
     * @return The length of time until a budget resets the actual and forecasted spend.
     * @see TimeUnit
     */
    public final String timeUnitAsString() {
        return timeUnit;
    }

    /**
     * <p>
     * The period of time that is covered by a budget. The period has a start date and an end date. The start date must
     * come before the end date. The end date must come before <code>06/15/87 00:00 UTC</code>.
     * </p>
     * <p>
     * If you create your budget and don't specify a start date, AWS defaults to the start of your chosen time period
     * (DAILY, MONTHLY, QUARTERLY, or ANNUALLY). For example, if you created your budget on January 24, 2018, chose
     * <code>DAILY</code>, and didn't set a start date, AWS set your start date to <code>01/24/18 00:00 UTC</code>. If
     * you chose <code>MONTHLY</code>, AWS set your start date to <code>01/01/18 00:00 UTC</code>. If you didn't specify
     * an end date, AWS set your end date to <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS
     * Billing and Cost Management console and the API.
     * </p>
     * <p>
     * You can change either date with the <code>UpdateBudget</code> operation.
     * </p>
     * <p>
     * After the end date, AWS deletes the budget and all associated notifications and subscribers.
     * </p>
     * 
     * @return The period of time that is covered by a budget. The period has a start date and an end date. The start
     *         date must come before the end date. The end date must come before <code>06/15/87 00:00 UTC</code>. </p>
     *         <p>
     *         If you create your budget and don't specify a start date, AWS defaults to the start of your chosen time
     *         period (DAILY, MONTHLY, QUARTERLY, or ANNUALLY). For example, if you created your budget on January 24,
     *         2018, chose <code>DAILY</code>, and didn't set a start date, AWS set your start date to
     *         <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, AWS set your start date to
     *         <code>01/01/18 00:00 UTC</code>. If you didn't specify an end date, AWS set your end date to
     *         <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost Management
     *         console and the API.
     *         </p>
     *         <p>
     *         You can change either date with the <code>UpdateBudget</code> operation.
     *         </p>
     *         <p>
     *         After the end date, AWS deletes the budget and all associated notifications and subscribers.
     */
    public final TimePeriod timePeriod() {
        return timePeriod;
    }

    /**
     * <p>
     * The actual and forecasted cost or usage that the budget tracks.
     * </p>
     * 
     * @return The actual and forecasted cost or usage that the budget tracks.
     */
    public final CalculatedSpend calculatedSpend() {
        return calculatedSpend;
    }

    /**
     * <p>
     * Whether this budget tracks costs, usage, RI utilization, RI coverage, Savings Plans utilization, or Savings Plans
     * coverage.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #budgetType} will
     * return {@link BudgetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #budgetTypeAsString}.
     * </p>
     * 
     * @return Whether this budget tracks costs, usage, RI utilization, RI coverage, Savings Plans utilization, or
     *         Savings Plans coverage.
     * @see BudgetType
     */
    public final BudgetType budgetType() {
        return BudgetType.fromValue(budgetType);
    }

    /**
     * <p>
     * Whether this budget tracks costs, usage, RI utilization, RI coverage, Savings Plans utilization, or Savings Plans
     * coverage.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #budgetType} will
     * return {@link BudgetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #budgetTypeAsString}.
     * </p>
     * 
     * @return Whether this budget tracks costs, usage, RI utilization, RI coverage, Savings Plans utilization, or
     *         Savings Plans coverage.
     * @see BudgetType
     */
    public final String budgetTypeAsString() {
        return budgetType;
    }

    /**
     * <p>
     * The last time that you updated this budget.
     * </p>
     * 
     * @return The last time that you updated this budget.
     */
    public final Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(budgetName());
        hashCode = 31 * hashCode + Objects.hashCode(budgetLimit());
        hashCode = 31 * hashCode + Objects.hashCode(hasPlannedBudgetLimits() ? plannedBudgetLimits() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCostFilters() ? costFilters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(costTypes());
        hashCode = 31 * hashCode + Objects.hashCode(timeUnitAsString());
        hashCode = 31 * hashCode + Objects.hashCode(timePeriod());
        hashCode = 31 * hashCode + Objects.hashCode(calculatedSpend());
        hashCode = 31 * hashCode + Objects.hashCode(budgetTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Budget)) {
            return false;
        }
        Budget other = (Budget) obj;
        return Objects.equals(budgetName(), other.budgetName()) && Objects.equals(budgetLimit(), other.budgetLimit())
                && hasPlannedBudgetLimits() == other.hasPlannedBudgetLimits()
                && Objects.equals(plannedBudgetLimits(), other.plannedBudgetLimits())
                && hasCostFilters() == other.hasCostFilters() && Objects.equals(costFilters(), other.costFilters())
                && Objects.equals(costTypes(), other.costTypes()) && Objects.equals(timeUnitAsString(), other.timeUnitAsString())
                && Objects.equals(timePeriod(), other.timePeriod()) && Objects.equals(calculatedSpend(), other.calculatedSpend())
                && Objects.equals(budgetTypeAsString(), other.budgetTypeAsString())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Budget").add("BudgetName", budgetName()).add("BudgetLimit", budgetLimit())
                .add("PlannedBudgetLimits", hasPlannedBudgetLimits() ? plannedBudgetLimits() : null)
                .add("CostFilters", hasCostFilters() ? costFilters() : null).add("CostTypes", costTypes())
                .add("TimeUnit", timeUnitAsString()).add("TimePeriod", timePeriod()).add("CalculatedSpend", calculatedSpend())
                .add("BudgetType", budgetTypeAsString()).add("LastUpdatedTime", lastUpdatedTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BudgetName":
            return Optional.ofNullable(clazz.cast(budgetName()));
        case "BudgetLimit":
            return Optional.ofNullable(clazz.cast(budgetLimit()));
        case "PlannedBudgetLimits":
            return Optional.ofNullable(clazz.cast(plannedBudgetLimits()));
        case "CostFilters":
            return Optional.ofNullable(clazz.cast(costFilters()));
        case "CostTypes":
            return Optional.ofNullable(clazz.cast(costTypes()));
        case "TimeUnit":
            return Optional.ofNullable(clazz.cast(timeUnitAsString()));
        case "TimePeriod":
            return Optional.ofNullable(clazz.cast(timePeriod()));
        case "CalculatedSpend":
            return Optional.ofNullable(clazz.cast(calculatedSpend()));
        case "BudgetType":
            return Optional.ofNullable(clazz.cast(budgetTypeAsString()));
        case "LastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Budget, T> g) {
        return obj -> g.apply((Budget) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Budget> {
        /**
         * <p>
         * The name of a budget. The name must be unique within an account. The <code>:</code> and <code>\</code>
         * characters aren't allowed in <code>BudgetName</code>.
         * </p>
         * 
         * @param budgetName
         *        The name of a budget. The name must be unique within an account. The <code>:</code> and <code>\</code>
         *        characters aren't allowed in <code>BudgetName</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder budgetName(String budgetName);

        /**
         * <p>
         * The total amount of cost, usage, RI utilization, RI coverage, Savings Plans utilization, or Savings Plans
         * coverage that you want to track with your budget.
         * </p>
         * <p>
         * <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI or Savings Plans
         * utilization or coverage budgets. RI and Savings Plans utilization or coverage budgets default to
         * <code>100</code>, which is the only valid value for RI or Savings Plans utilization or coverage budgets. You
         * can't use <code>BudgetLimit</code> with <code>PlannedBudgetLimits</code> for <code>CreateBudget</code> and
         * <code>UpdateBudget</code> actions.
         * </p>
         * 
         * @param budgetLimit
         *        The total amount of cost, usage, RI utilization, RI coverage, Savings Plans utilization, or Savings
         *        Plans coverage that you want to track with your budget.</p>
         *        <p>
         *        <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI or Savings Plans
         *        utilization or coverage budgets. RI and Savings Plans utilization or coverage budgets default to
         *        <code>100</code>, which is the only valid value for RI or Savings Plans utilization or coverage
         *        budgets. You can't use <code>BudgetLimit</code> with <code>PlannedBudgetLimits</code> for
         *        <code>CreateBudget</code> and <code>UpdateBudget</code> actions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder budgetLimit(Spend budgetLimit);

        /**
         * <p>
         * The total amount of cost, usage, RI utilization, RI coverage, Savings Plans utilization, or Savings Plans
         * coverage that you want to track with your budget.
         * </p>
         * <p>
         * <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI or Savings Plans
         * utilization or coverage budgets. RI and Savings Plans utilization or coverage budgets default to
         * <code>100</code>, which is the only valid value for RI or Savings Plans utilization or coverage budgets. You
         * can't use <code>BudgetLimit</code> with <code>PlannedBudgetLimits</code> for <code>CreateBudget</code> and
         * <code>UpdateBudget</code> actions.
         * </p>
         * This is a convenience that creates an instance of the {@link Spend.Builder} avoiding the need to create one
         * manually via {@link Spend#builder()}.
         *
         * When the {@link Consumer} completes, {@link Spend.Builder#build()} is called immediately and its result is
         * passed to {@link #budgetLimit(Spend)}.
         * 
         * @param budgetLimit
         *        a consumer that will call methods on {@link Spend.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #budgetLimit(Spend)
         */
        default Builder budgetLimit(Consumer<Spend.Builder> budgetLimit) {
            return budgetLimit(Spend.builder().applyMutation(budgetLimit).build());
        }

        /**
         * <p>
         * A map containing multiple <code>BudgetLimit</code>, including current or future limits.
         * </p>
         * <p>
         * <code>PlannedBudgetLimits</code> is available for cost or usage budget and supports monthly and quarterly
         * <code>TimeUnit</code>.
         * </p>
         * <p>
         * For monthly budgets, provide 12 months of <code>PlannedBudgetLimits</code> values. This must start from the
         * current month and include the next 11 months. The <code>key</code> is the start of the month,
         * <code>UTC</code> in epoch seconds.
         * </p>
         * <p>
         * For quarterly budgets, provide 4 quarters of <code>PlannedBudgetLimits</code> value entries in standard
         * calendar quarter increments. This must start from the current quarter and include the next 3 quarters. The
         * <code>key</code> is the start of the quarter, <code>UTC</code> in epoch seconds.
         * </p>
         * <p>
         * If the planned budget expires before 12 months for monthly or 4 quarters for quarterly, provide the
         * <code>PlannedBudgetLimits</code> values only for the remaining periods.
         * </p>
         * <p>
         * If the budget begins at a date in the future, provide <code>PlannedBudgetLimits</code> values from the start
         * date of the budget.
         * </p>
         * <p>
         * After all of the <code>BudgetLimit</code> values in <code>PlannedBudgetLimits</code> are used, the budget
         * continues to use the last limit as the <code>BudgetLimit</code>. At that point, the planned budget provides
         * the same experience as a fixed budget.
         * </p>
         * <p>
         * <code>DescribeBudget</code> and <code>DescribeBudgets</code> response along with
         * <code>PlannedBudgetLimits</code> will also contain <code>BudgetLimit</code> representing the current month or
         * quarter limit present in <code>PlannedBudgetLimits</code>. This only applies to budgets created with
         * <code>PlannedBudgetLimits</code>. Budgets created without <code>PlannedBudgetLimits</code> will only contain
         * <code>BudgetLimit</code>, and no <code>PlannedBudgetLimits</code>.
         * </p>
         * 
         * @param plannedBudgetLimits
         *        A map containing multiple <code>BudgetLimit</code>, including current or future limits.</p>
         *        <p>
         *        <code>PlannedBudgetLimits</code> is available for cost or usage budget and supports monthly and
         *        quarterly <code>TimeUnit</code>.
         *        </p>
         *        <p>
         *        For monthly budgets, provide 12 months of <code>PlannedBudgetLimits</code> values. This must start
         *        from the current month and include the next 11 months. The <code>key</code> is the start of the month,
         *        <code>UTC</code> in epoch seconds.
         *        </p>
         *        <p>
         *        For quarterly budgets, provide 4 quarters of <code>PlannedBudgetLimits</code> value entries in
         *        standard calendar quarter increments. This must start from the current quarter and include the next 3
         *        quarters. The <code>key</code> is the start of the quarter, <code>UTC</code> in epoch seconds.
         *        </p>
         *        <p>
         *        If the planned budget expires before 12 months for monthly or 4 quarters for quarterly, provide the
         *        <code>PlannedBudgetLimits</code> values only for the remaining periods.
         *        </p>
         *        <p>
         *        If the budget begins at a date in the future, provide <code>PlannedBudgetLimits</code> values from the
         *        start date of the budget.
         *        </p>
         *        <p>
         *        After all of the <code>BudgetLimit</code> values in <code>PlannedBudgetLimits</code> are used, the
         *        budget continues to use the last limit as the <code>BudgetLimit</code>. At that point, the planned
         *        budget provides the same experience as a fixed budget.
         *        </p>
         *        <p>
         *        <code>DescribeBudget</code> and <code>DescribeBudgets</code> response along with
         *        <code>PlannedBudgetLimits</code> will also contain <code>BudgetLimit</code> representing the current
         *        month or quarter limit present in <code>PlannedBudgetLimits</code>. This only applies to budgets
         *        created with <code>PlannedBudgetLimits</code>. Budgets created without
         *        <code>PlannedBudgetLimits</code> will only contain <code>BudgetLimit</code>, and no
         *        <code>PlannedBudgetLimits</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder plannedBudgetLimits(Map<String, Spend> plannedBudgetLimits);

        /**
         * <p>
         * The cost filters, such as service or tag, that are applied to a budget.
         * </p>
         * <p>
         * AWS Budgets supports the following services as a filter for RI budgets:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Amazon Elastic Compute Cloud - Compute
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Redshift
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Relational Database Service
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon ElastiCache
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Elasticsearch Service
         * </p>
         * </li>
         * </ul>
         * 
         * @param costFilters
         *        The cost filters, such as service or tag, that are applied to a budget.</p>
         *        <p>
         *        AWS Budgets supports the following services as a filter for RI budgets:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Amazon Elastic Compute Cloud - Compute
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Redshift
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Relational Database Service
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon ElastiCache
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Elasticsearch Service
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder costFilters(Map<String, ? extends Collection<String>> costFilters);

        /**
         * <p>
         * The types of costs that are included in this <code>COST</code> budget.
         * </p>
         * <p>
         * <code>USAGE</code>, <code>RI_UTILIZATION</code>, <code>RI_COVERAGE</code>,
         * <code>SAVINGS_PLANS_UTILIZATION</code>, and <code>SAVINGS_PLANS_COVERAGE</code> budgets do not have
         * <code>CostTypes</code>.
         * </p>
         * 
         * @param costTypes
         *        The types of costs that are included in this <code>COST</code> budget.</p>
         *        <p>
         *        <code>USAGE</code>, <code>RI_UTILIZATION</code>, <code>RI_COVERAGE</code>,
         *        <code>SAVINGS_PLANS_UTILIZATION</code>, and <code>SAVINGS_PLANS_COVERAGE</code> budgets do not have
         *        <code>CostTypes</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder costTypes(CostTypes costTypes);

        /**
         * <p>
         * The types of costs that are included in this <code>COST</code> budget.
         * </p>
         * <p>
         * <code>USAGE</code>, <code>RI_UTILIZATION</code>, <code>RI_COVERAGE</code>,
         * <code>SAVINGS_PLANS_UTILIZATION</code>, and <code>SAVINGS_PLANS_COVERAGE</code> budgets do not have
         * <code>CostTypes</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link CostTypes.Builder} avoiding the need to create
         * one manually via {@link CostTypes#builder()}.
         *
         * When the {@link Consumer} completes, {@link CostTypes.Builder#build()} is called immediately and its result
         * is passed to {@link #costTypes(CostTypes)}.
         * 
         * @param costTypes
         *        a consumer that will call methods on {@link CostTypes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #costTypes(CostTypes)
         */
        default Builder costTypes(Consumer<CostTypes.Builder> costTypes) {
            return costTypes(CostTypes.builder().applyMutation(costTypes).build());
        }

        /**
         * <p>
         * The length of time until a budget resets the actual and forecasted spend.
         * </p>
         * 
         * @param timeUnit
         *        The length of time until a budget resets the actual and forecasted spend.
         * @see TimeUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TimeUnit
         */
        Builder timeUnit(String timeUnit);

        /**
         * <p>
         * The length of time until a budget resets the actual and forecasted spend.
         * </p>
         * 
         * @param timeUnit
         *        The length of time until a budget resets the actual and forecasted spend.
         * @see TimeUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TimeUnit
         */
        Builder timeUnit(TimeUnit timeUnit);

        /**
         * <p>
         * The period of time that is covered by a budget. The period has a start date and an end date. The start date
         * must come before the end date. The end date must come before <code>06/15/87 00:00 UTC</code>.
         * </p>
         * <p>
         * If you create your budget and don't specify a start date, AWS defaults to the start of your chosen time
         * period (DAILY, MONTHLY, QUARTERLY, or ANNUALLY). For example, if you created your budget on January 24, 2018,
         * chose <code>DAILY</code>, and didn't set a start date, AWS set your start date to
         * <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, AWS set your start date to
         * <code>01/01/18 00:00 UTC</code>. If you didn't specify an end date, AWS set your end date to
         * <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost Management console
         * and the API.
         * </p>
         * <p>
         * You can change either date with the <code>UpdateBudget</code> operation.
         * </p>
         * <p>
         * After the end date, AWS deletes the budget and all associated notifications and subscribers.
         * </p>
         * 
         * @param timePeriod
         *        The period of time that is covered by a budget. The period has a start date and an end date. The start
         *        date must come before the end date. The end date must come before <code>06/15/87 00:00 UTC</code>.
         *        </p>
         *        <p>
         *        If you create your budget and don't specify a start date, AWS defaults to the start of your chosen
         *        time period (DAILY, MONTHLY, QUARTERLY, or ANNUALLY). For example, if you created your budget on
         *        January 24, 2018, chose <code>DAILY</code>, and didn't set a start date, AWS set your start date to
         *        <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, AWS set your start date to
         *        <code>01/01/18 00:00 UTC</code>. If you didn't specify an end date, AWS set your end date to
         *        <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost Management
         *        console and the API.
         *        </p>
         *        <p>
         *        You can change either date with the <code>UpdateBudget</code> operation.
         *        </p>
         *        <p>
         *        After the end date, AWS deletes the budget and all associated notifications and subscribers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timePeriod(TimePeriod timePeriod);

        /**
         * <p>
         * The period of time that is covered by a budget. The period has a start date and an end date. The start date
         * must come before the end date. The end date must come before <code>06/15/87 00:00 UTC</code>.
         * </p>
         * <p>
         * If you create your budget and don't specify a start date, AWS defaults to the start of your chosen time
         * period (DAILY, MONTHLY, QUARTERLY, or ANNUALLY). For example, if you created your budget on January 24, 2018,
         * chose <code>DAILY</code>, and didn't set a start date, AWS set your start date to
         * <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, AWS set your start date to
         * <code>01/01/18 00:00 UTC</code>. If you didn't specify an end date, AWS set your end date to
         * <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost Management console
         * and the API.
         * </p>
         * <p>
         * You can change either date with the <code>UpdateBudget</code> operation.
         * </p>
         * <p>
         * After the end date, AWS deletes the budget and all associated notifications and subscribers.
         * </p>
         * This is a convenience that creates an instance of the {@link TimePeriod.Builder} avoiding the need to create
         * one manually via {@link TimePeriod#builder()}.
         *
         * When the {@link Consumer} completes, {@link TimePeriod.Builder#build()} is called immediately and its result
         * is passed to {@link #timePeriod(TimePeriod)}.
         * 
         * @param timePeriod
         *        a consumer that will call methods on {@link TimePeriod.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #timePeriod(TimePeriod)
         */
        default Builder timePeriod(Consumer<TimePeriod.Builder> timePeriod) {
            return timePeriod(TimePeriod.builder().applyMutation(timePeriod).build());
        }

        /**
         * <p>
         * The actual and forecasted cost or usage that the budget tracks.
         * </p>
         * 
         * @param calculatedSpend
         *        The actual and forecasted cost or usage that the budget tracks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder calculatedSpend(CalculatedSpend calculatedSpend);

        /**
         * <p>
         * The actual and forecasted cost or usage that the budget tracks.
         * </p>
         * This is a convenience that creates an instance of the {@link CalculatedSpend.Builder} avoiding the need to
         * create one manually via {@link CalculatedSpend#builder()}.
         *
         * When the {@link Consumer} completes, {@link CalculatedSpend.Builder#build()} is called immediately and its
         * result is passed to {@link #calculatedSpend(CalculatedSpend)}.
         * 
         * @param calculatedSpend
         *        a consumer that will call methods on {@link CalculatedSpend.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #calculatedSpend(CalculatedSpend)
         */
        default Builder calculatedSpend(Consumer<CalculatedSpend.Builder> calculatedSpend) {
            return calculatedSpend(CalculatedSpend.builder().applyMutation(calculatedSpend).build());
        }

        /**
         * <p>
         * Whether this budget tracks costs, usage, RI utilization, RI coverage, Savings Plans utilization, or Savings
         * Plans coverage.
         * </p>
         * 
         * @param budgetType
         *        Whether this budget tracks costs, usage, RI utilization, RI coverage, Savings Plans utilization, or
         *        Savings Plans coverage.
         * @see BudgetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BudgetType
         */
        Builder budgetType(String budgetType);

        /**
         * <p>
         * Whether this budget tracks costs, usage, RI utilization, RI coverage, Savings Plans utilization, or Savings
         * Plans coverage.
         * </p>
         * 
         * @param budgetType
         *        Whether this budget tracks costs, usage, RI utilization, RI coverage, Savings Plans utilization, or
         *        Savings Plans coverage.
         * @see BudgetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BudgetType
         */
        Builder budgetType(BudgetType budgetType);

        /**
         * <p>
         * The last time that you updated this budget.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The last time that you updated this budget.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);
    }

    static final class BuilderImpl implements Builder {
        private String budgetName;

        private Spend budgetLimit;

        private Map<String, Spend> plannedBudgetLimits = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, List<String>> costFilters = DefaultSdkAutoConstructMap.getInstance();

        private CostTypes costTypes;

        private String timeUnit;

        private TimePeriod timePeriod;

        private CalculatedSpend calculatedSpend;

        private String budgetType;

        private Instant lastUpdatedTime;

        private BuilderImpl() {
        }

        private BuilderImpl(Budget model) {
            budgetName(model.budgetName);
            budgetLimit(model.budgetLimit);
            plannedBudgetLimits(model.plannedBudgetLimits);
            costFilters(model.costFilters);
            costTypes(model.costTypes);
            timeUnit(model.timeUnit);
            timePeriod(model.timePeriod);
            calculatedSpend(model.calculatedSpend);
            budgetType(model.budgetType);
            lastUpdatedTime(model.lastUpdatedTime);
        }

        public final String getBudgetName() {
            return budgetName;
        }

        @Override
        public final Builder budgetName(String budgetName) {
            this.budgetName = budgetName;
            return this;
        }

        public final void setBudgetName(String budgetName) {
            this.budgetName = budgetName;
        }

        public final Spend.Builder getBudgetLimit() {
            return budgetLimit != null ? budgetLimit.toBuilder() : null;
        }

        @Override
        public final Builder budgetLimit(Spend budgetLimit) {
            this.budgetLimit = budgetLimit;
            return this;
        }

        public final void setBudgetLimit(Spend.BuilderImpl budgetLimit) {
            this.budgetLimit = budgetLimit != null ? budgetLimit.build() : null;
        }

        public final Map<String, Spend.Builder> getPlannedBudgetLimits() {
            if (plannedBudgetLimits instanceof SdkAutoConstructMap) {
                return null;
            }
            return plannedBudgetLimits != null ? CollectionUtils.mapValues(plannedBudgetLimits, Spend::toBuilder) : null;
        }

        @Override
        public final Builder plannedBudgetLimits(Map<String, Spend> plannedBudgetLimits) {
            this.plannedBudgetLimits = PlannedBudgetLimitsCopier.copy(plannedBudgetLimits);
            return this;
        }

        public final void setPlannedBudgetLimits(Map<String, Spend.BuilderImpl> plannedBudgetLimits) {
            this.plannedBudgetLimits = PlannedBudgetLimitsCopier.copyFromBuilder(plannedBudgetLimits);
        }

        public final Map<String, ? extends Collection<String>> getCostFilters() {
            if (costFilters instanceof SdkAutoConstructMap) {
                return null;
            }
            return costFilters;
        }

        @Override
        public final Builder costFilters(Map<String, ? extends Collection<String>> costFilters) {
            this.costFilters = CostFiltersCopier.copy(costFilters);
            return this;
        }

        public final void setCostFilters(Map<String, ? extends Collection<String>> costFilters) {
            this.costFilters = CostFiltersCopier.copy(costFilters);
        }

        public final CostTypes.Builder getCostTypes() {
            return costTypes != null ? costTypes.toBuilder() : null;
        }

        @Override
        public final Builder costTypes(CostTypes costTypes) {
            this.costTypes = costTypes;
            return this;
        }

        public final void setCostTypes(CostTypes.BuilderImpl costTypes) {
            this.costTypes = costTypes != null ? costTypes.build() : null;
        }

        public final String getTimeUnit() {
            return timeUnit;
        }

        @Override
        public final Builder timeUnit(String timeUnit) {
            this.timeUnit = timeUnit;
            return this;
        }

        @Override
        public final Builder timeUnit(TimeUnit timeUnit) {
            this.timeUnit(timeUnit == null ? null : timeUnit.toString());
            return this;
        }

        public final void setTimeUnit(String timeUnit) {
            this.timeUnit = timeUnit;
        }

        public final TimePeriod.Builder getTimePeriod() {
            return timePeriod != null ? timePeriod.toBuilder() : null;
        }

        @Override
        public final Builder timePeriod(TimePeriod timePeriod) {
            this.timePeriod = timePeriod;
            return this;
        }

        public final void setTimePeriod(TimePeriod.BuilderImpl timePeriod) {
            this.timePeriod = timePeriod != null ? timePeriod.build() : null;
        }

        public final CalculatedSpend.Builder getCalculatedSpend() {
            return calculatedSpend != null ? calculatedSpend.toBuilder() : null;
        }

        @Override
        public final Builder calculatedSpend(CalculatedSpend calculatedSpend) {
            this.calculatedSpend = calculatedSpend;
            return this;
        }

        public final void setCalculatedSpend(CalculatedSpend.BuilderImpl calculatedSpend) {
            this.calculatedSpend = calculatedSpend != null ? calculatedSpend.build() : null;
        }

        public final String getBudgetType() {
            return budgetType;
        }

        @Override
        public final Builder budgetType(String budgetType) {
            this.budgetType = budgetType;
            return this;
        }

        @Override
        public final Builder budgetType(BudgetType budgetType) {
            this.budgetType(budgetType == null ? null : budgetType.toString());
            return this;
        }

        public final void setBudgetType(String budgetType) {
            this.budgetType = budgetType;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public Budget build() {
            return new Budget(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
