/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.budgets;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.SdkClient;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.budgets.model.AccessDeniedException;
import software.amazon.awssdk.services.budgets.model.BudgetsException;
import software.amazon.awssdk.services.budgets.model.CreateBudgetActionRequest;
import software.amazon.awssdk.services.budgets.model.CreateBudgetActionResponse;
import software.amazon.awssdk.services.budgets.model.CreateBudgetRequest;
import software.amazon.awssdk.services.budgets.model.CreateBudgetResponse;
import software.amazon.awssdk.services.budgets.model.CreateNotificationRequest;
import software.amazon.awssdk.services.budgets.model.CreateNotificationResponse;
import software.amazon.awssdk.services.budgets.model.CreateSubscriberRequest;
import software.amazon.awssdk.services.budgets.model.CreateSubscriberResponse;
import software.amazon.awssdk.services.budgets.model.CreationLimitExceededException;
import software.amazon.awssdk.services.budgets.model.DeleteBudgetActionRequest;
import software.amazon.awssdk.services.budgets.model.DeleteBudgetActionResponse;
import software.amazon.awssdk.services.budgets.model.DeleteBudgetRequest;
import software.amazon.awssdk.services.budgets.model.DeleteBudgetResponse;
import software.amazon.awssdk.services.budgets.model.DeleteNotificationRequest;
import software.amazon.awssdk.services.budgets.model.DeleteNotificationResponse;
import software.amazon.awssdk.services.budgets.model.DeleteSubscriberRequest;
import software.amazon.awssdk.services.budgets.model.DeleteSubscriberResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionHistoriesRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionHistoriesResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForAccountRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForAccountResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForBudgetRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForBudgetResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetPerformanceHistoryRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetPerformanceHistoryResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetsRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetsResponse;
import software.amazon.awssdk.services.budgets.model.DescribeNotificationsForBudgetRequest;
import software.amazon.awssdk.services.budgets.model.DescribeNotificationsForBudgetResponse;
import software.amazon.awssdk.services.budgets.model.DescribeSubscribersForNotificationRequest;
import software.amazon.awssdk.services.budgets.model.DescribeSubscribersForNotificationResponse;
import software.amazon.awssdk.services.budgets.model.DuplicateRecordException;
import software.amazon.awssdk.services.budgets.model.ExecuteBudgetActionRequest;
import software.amazon.awssdk.services.budgets.model.ExecuteBudgetActionResponse;
import software.amazon.awssdk.services.budgets.model.ExpiredNextTokenException;
import software.amazon.awssdk.services.budgets.model.InternalErrorException;
import software.amazon.awssdk.services.budgets.model.InvalidNextTokenException;
import software.amazon.awssdk.services.budgets.model.InvalidParameterException;
import software.amazon.awssdk.services.budgets.model.NotFoundException;
import software.amazon.awssdk.services.budgets.model.ResourceLockedException;
import software.amazon.awssdk.services.budgets.model.UpdateBudgetActionRequest;
import software.amazon.awssdk.services.budgets.model.UpdateBudgetActionResponse;
import software.amazon.awssdk.services.budgets.model.UpdateBudgetRequest;
import software.amazon.awssdk.services.budgets.model.UpdateBudgetResponse;
import software.amazon.awssdk.services.budgets.model.UpdateNotificationRequest;
import software.amazon.awssdk.services.budgets.model.UpdateNotificationResponse;
import software.amazon.awssdk.services.budgets.model.UpdateSubscriberRequest;
import software.amazon.awssdk.services.budgets.model.UpdateSubscriberResponse;
import software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionHistoriesIterable;
import software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionsForAccountIterable;
import software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionsForBudgetIterable;
import software.amazon.awssdk.services.budgets.paginators.DescribeBudgetPerformanceHistoryIterable;
import software.amazon.awssdk.services.budgets.paginators.DescribeBudgetsIterable;
import software.amazon.awssdk.services.budgets.paginators.DescribeNotificationsForBudgetIterable;
import software.amazon.awssdk.services.budgets.paginators.DescribeSubscribersForNotificationIterable;

/**
 * Service client for accessing AWSBudgets. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * The Amazon Web Services Budgets API enables you to use Amazon Web Services Budgets to plan your service usage,
 * service costs, and instance reservations. The API reference provides descriptions, syntax, and usage examples for
 * each of the actions and data types for Amazon Web Services Budgets.
 * </p>
 * <p>
 * Budgets provide you with a way to see the following information:
 * </p>
 * <ul>
 * <li>
 * <p>
 * How close your plan is to your budgeted amount or to the free tier limits
 * </p>
 * </li>
 * <li>
 * <p>
 * Your usage-to-date, including how much you've used of your Reserved Instances (RIs)
 * </p>
 * </li>
 * <li>
 * <p>
 * Your current estimated charges from Amazon Web Services, and how much your predicted usage will accrue in charges by
 * the end of the month
 * </p>
 * </li>
 * <li>
 * <p>
 * How much of your budget has been used
 * </p>
 * </li>
 * </ul>
 * <p>
 * Amazon Web Services updates your budget status several times a day. Budgets track your unblended costs,
 * subscriptions, refunds, and RIs. You can create the following types of budgets:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <b>Cost budgets</b> - Plan how much you want to spend on a service.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>Usage budgets</b> - Plan how much you want to use one or more services.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>RI utilization budgets</b> - Define a utilization threshold, and receive alerts when your RI usage falls below
 * that threshold. This lets you see if your RIs are unused or under-utilized.
 * </p>
 * </li>
 * <li>
 * <p>
 * <b>RI coverage budgets</b> - Define a coverage threshold, and receive alerts when the number of your instance hours
 * that are covered by RIs fall below that threshold. This lets you see how much of your instance usage is covered by a
 * reservation.
 * </p>
 * </li>
 * </ul>
 * <p>
 * Service Endpoint
 * </p>
 * <p>
 * The Amazon Web Services Budgets API provides the following endpoint:
 * </p>
 * <ul>
 * <li>
 * <p>
 * https://budgets.amazonaws.com
 * </p>
 * </li>
 * </ul>
 * <p>
 * For information about costs that are associated with the Amazon Web Services Budgets API, see <a
 * href="https://aws.amazon.com/aws-cost-management/pricing/">Amazon Web Services Cost Management Pricing</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface BudgetsClient extends SdkClient {
    String SERVICE_NAME = "budgets";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "budgets";

    /**
     * Create a {@link BudgetsClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static BudgetsClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link BudgetsClient}.
     */
    static BudgetsClientBuilder builder() {
        return new DefaultBudgetsClientBuilder();
    }

    /**
     * <p>
     * Creates a budget and, if included, notifications and subscribers.
     * </p>
     * <important>
     * <p>
     * Only one of <code>BudgetLimit</code> or <code>PlannedBudgetLimits</code> can be present in the syntax at one
     * time. Use the syntax that matches your case. The Request Syntax section shows the <code>BudgetLimit</code>
     * syntax. For <code>PlannedBudgetLimits</code>, see the <a href=
     * "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_CreateBudget.html#API_CreateBudget_Examples"
     * >Examples</a> section.
     * </p>
     * </important>
     *
     * @param createBudgetRequest
     *        Request of CreateBudget
     * @return Result of the CreateBudget operation returned by the service.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws CreationLimitExceededException
     *         You've exceeded the notification or subscriber limit.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.CreateBudget
     */
    default CreateBudgetResponse createBudget(CreateBudgetRequest createBudgetRequest) throws InvalidParameterException,
            InternalErrorException, CreationLimitExceededException, DuplicateRecordException, AccessDeniedException,
            AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a budget and, if included, notifications and subscribers.
     * </p>
     * <important>
     * <p>
     * Only one of <code>BudgetLimit</code> or <code>PlannedBudgetLimits</code> can be present in the syntax at one
     * time. Use the syntax that matches your case. The Request Syntax section shows the <code>BudgetLimit</code>
     * syntax. For <code>PlannedBudgetLimits</code>, see the <a href=
     * "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_CreateBudget.html#API_CreateBudget_Examples"
     * >Examples</a> section.
     * </p>
     * </important><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateBudgetRequest.Builder} avoiding the need to
     * create one manually via {@link CreateBudgetRequest#builder()}
     * </p>
     *
     * @param createBudgetRequest
     *        A {@link Consumer} that will call methods on {@link CreateBudgetRequest.Builder} to create a request.
     *        Request of CreateBudget
     * @return Result of the CreateBudget operation returned by the service.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws CreationLimitExceededException
     *         You've exceeded the notification or subscriber limit.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.CreateBudget
     */
    default CreateBudgetResponse createBudget(Consumer<CreateBudgetRequest.Builder> createBudgetRequest)
            throws InvalidParameterException, InternalErrorException, CreationLimitExceededException, DuplicateRecordException,
            AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        return createBudget(CreateBudgetRequest.builder().applyMutation(createBudgetRequest).build());
    }

    /**
     * <p>
     * Creates a budget action.
     * </p>
     *
     * @param createBudgetActionRequest
     * @return Result of the CreateBudgetAction operation returned by the service.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws CreationLimitExceededException
     *         You've exceeded the notification or subscriber limit.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.CreateBudgetAction
     */
    default CreateBudgetActionResponse createBudgetAction(CreateBudgetActionRequest createBudgetActionRequest)
            throws InvalidParameterException, InternalErrorException, CreationLimitExceededException, DuplicateRecordException,
            NotFoundException, AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a budget action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateBudgetActionRequest.Builder} avoiding the
     * need to create one manually via {@link CreateBudgetActionRequest#builder()}
     * </p>
     *
     * @param createBudgetActionRequest
     *        A {@link Consumer} that will call methods on {@link CreateBudgetActionRequest.Builder} to create a
     *        request.
     * @return Result of the CreateBudgetAction operation returned by the service.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws CreationLimitExceededException
     *         You've exceeded the notification or subscriber limit.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.CreateBudgetAction
     */
    default CreateBudgetActionResponse createBudgetAction(Consumer<CreateBudgetActionRequest.Builder> createBudgetActionRequest)
            throws InvalidParameterException, InternalErrorException, CreationLimitExceededException, DuplicateRecordException,
            NotFoundException, AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        return createBudgetAction(CreateBudgetActionRequest.builder().applyMutation(createBudgetActionRequest).build());
    }

    /**
     * <p>
     * Creates a notification. You must create the budget before you create the associated notification.
     * </p>
     *
     * @param createNotificationRequest
     *        Request of CreateNotification
     * @return Result of the CreateNotification operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws CreationLimitExceededException
     *         You've exceeded the notification or subscriber limit.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.CreateNotification
     */
    default CreateNotificationResponse createNotification(CreateNotificationRequest createNotificationRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, CreationLimitExceededException,
            DuplicateRecordException, AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a notification. You must create the budget before you create the associated notification.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateNotificationRequest.Builder} avoiding the
     * need to create one manually via {@link CreateNotificationRequest#builder()}
     * </p>
     *
     * @param createNotificationRequest
     *        A {@link Consumer} that will call methods on {@link CreateNotificationRequest.Builder} to create a
     *        request. Request of CreateNotification
     * @return Result of the CreateNotification operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws CreationLimitExceededException
     *         You've exceeded the notification or subscriber limit.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.CreateNotification
     */
    default CreateNotificationResponse createNotification(Consumer<CreateNotificationRequest.Builder> createNotificationRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, CreationLimitExceededException,
            DuplicateRecordException, AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        return createNotification(CreateNotificationRequest.builder().applyMutation(createNotificationRequest).build());
    }

    /**
     * <p>
     * Creates a subscriber. You must create the associated budget and notification before you create the subscriber.
     * </p>
     *
     * @param createSubscriberRequest
     *        Request of CreateSubscriber
     * @return Result of the CreateSubscriber operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws CreationLimitExceededException
     *         You've exceeded the notification or subscriber limit.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.CreateSubscriber
     */
    default CreateSubscriberResponse createSubscriber(CreateSubscriberRequest createSubscriberRequest)
            throws InternalErrorException, InvalidParameterException, CreationLimitExceededException, DuplicateRecordException,
            NotFoundException, AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a subscriber. You must create the associated budget and notification before you create the subscriber.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateSubscriberRequest.Builder} avoiding the need
     * to create one manually via {@link CreateSubscriberRequest#builder()}
     * </p>
     *
     * @param createSubscriberRequest
     *        A {@link Consumer} that will call methods on {@link CreateSubscriberRequest.Builder} to create a request.
     *        Request of CreateSubscriber
     * @return Result of the CreateSubscriber operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws CreationLimitExceededException
     *         You've exceeded the notification or subscriber limit.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.CreateSubscriber
     */
    default CreateSubscriberResponse createSubscriber(Consumer<CreateSubscriberRequest.Builder> createSubscriberRequest)
            throws InternalErrorException, InvalidParameterException, CreationLimitExceededException, DuplicateRecordException,
            NotFoundException, AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        return createSubscriber(CreateSubscriberRequest.builder().applyMutation(createSubscriberRequest).build());
    }

    /**
     * <p>
     * Deletes a budget. You can delete your budget at any time.
     * </p>
     * <important>
     * <p>
     * Deleting a budget also deletes the notifications and subscribers that are associated with that budget.
     * </p>
     * </important>
     *
     * @param deleteBudgetRequest
     *        Request of DeleteBudget
     * @return Result of the DeleteBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DeleteBudget
     */
    default DeleteBudgetResponse deleteBudget(DeleteBudgetRequest deleteBudgetRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a budget. You can delete your budget at any time.
     * </p>
     * <important>
     * <p>
     * Deleting a budget also deletes the notifications and subscribers that are associated with that budget.
     * </p>
     * </important><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteBudgetRequest.Builder} avoiding the need to
     * create one manually via {@link DeleteBudgetRequest#builder()}
     * </p>
     *
     * @param deleteBudgetRequest
     *        A {@link Consumer} that will call methods on {@link DeleteBudgetRequest.Builder} to create a request.
     *        Request of DeleteBudget
     * @return Result of the DeleteBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DeleteBudget
     */
    default DeleteBudgetResponse deleteBudget(Consumer<DeleteBudgetRequest.Builder> deleteBudgetRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, BudgetsException {
        return deleteBudget(DeleteBudgetRequest.builder().applyMutation(deleteBudgetRequest).build());
    }

    /**
     * <p>
     * Deletes a budget action.
     * </p>
     *
     * @param deleteBudgetActionRequest
     * @return Result of the DeleteBudgetAction operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ResourceLockedException
     *         The request was received and recognized by the server, but the server rejected that particular method for
     *         the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DeleteBudgetAction
     */
    default DeleteBudgetActionResponse deleteBudgetAction(DeleteBudgetActionRequest deleteBudgetActionRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            ResourceLockedException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a budget action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteBudgetActionRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteBudgetActionRequest#builder()}
     * </p>
     *
     * @param deleteBudgetActionRequest
     *        A {@link Consumer} that will call methods on {@link DeleteBudgetActionRequest.Builder} to create a
     *        request.
     * @return Result of the DeleteBudgetAction operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ResourceLockedException
     *         The request was received and recognized by the server, but the server rejected that particular method for
     *         the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DeleteBudgetAction
     */
    default DeleteBudgetActionResponse deleteBudgetAction(Consumer<DeleteBudgetActionRequest.Builder> deleteBudgetActionRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            ResourceLockedException, AwsServiceException, SdkClientException, BudgetsException {
        return deleteBudgetAction(DeleteBudgetActionRequest.builder().applyMutation(deleteBudgetActionRequest).build());
    }

    /**
     * <p>
     * Deletes a notification.
     * </p>
     * <important>
     * <p>
     * Deleting a notification also deletes the subscribers that are associated with the notification.
     * </p>
     * </important>
     *
     * @param deleteNotificationRequest
     *        Request of DeleteNotification
     * @return Result of the DeleteNotification operation returned by the service.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DeleteNotification
     */
    default DeleteNotificationResponse deleteNotification(DeleteNotificationRequest deleteNotificationRequest)
            throws InvalidParameterException, InternalErrorException, NotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a notification.
     * </p>
     * <important>
     * <p>
     * Deleting a notification also deletes the subscribers that are associated with the notification.
     * </p>
     * </important><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteNotificationRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteNotificationRequest#builder()}
     * </p>
     *
     * @param deleteNotificationRequest
     *        A {@link Consumer} that will call methods on {@link DeleteNotificationRequest.Builder} to create a
     *        request. Request of DeleteNotification
     * @return Result of the DeleteNotification operation returned by the service.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DeleteNotification
     */
    default DeleteNotificationResponse deleteNotification(Consumer<DeleteNotificationRequest.Builder> deleteNotificationRequest)
            throws InvalidParameterException, InternalErrorException, NotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, BudgetsException {
        return deleteNotification(DeleteNotificationRequest.builder().applyMutation(deleteNotificationRequest).build());
    }

    /**
     * <p>
     * Deletes a subscriber.
     * </p>
     * <important>
     * <p>
     * Deleting the last subscriber to a notification also deletes the notification.
     * </p>
     * </important>
     *
     * @param deleteSubscriberRequest
     *        Request of DeleteSubscriber
     * @return Result of the DeleteSubscriber operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DeleteSubscriber
     */
    default DeleteSubscriberResponse deleteSubscriber(DeleteSubscriberRequest deleteSubscriberRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a subscriber.
     * </p>
     * <important>
     * <p>
     * Deleting the last subscriber to a notification also deletes the notification.
     * </p>
     * </important><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteSubscriberRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteSubscriberRequest#builder()}
     * </p>
     *
     * @param deleteSubscriberRequest
     *        A {@link Consumer} that will call methods on {@link DeleteSubscriberRequest.Builder} to create a request.
     *        Request of DeleteSubscriber
     * @return Result of the DeleteSubscriber operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DeleteSubscriber
     */
    default DeleteSubscriberResponse deleteSubscriber(Consumer<DeleteSubscriberRequest.Builder> deleteSubscriberRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, BudgetsException {
        return deleteSubscriber(DeleteSubscriberRequest.builder().applyMutation(deleteSubscriberRequest).build());
    }

    /**
     * <p>
     * Describes a budget.
     * </p>
     * <important>
     * <p>
     * The Request Syntax section shows the <code>BudgetLimit</code> syntax. For <code>PlannedBudgetLimits</code>, see
     * the <a href=
     * "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_DescribeBudget.html#API_DescribeBudget_Examples"
     * >Examples</a> section.
     * </p>
     * </important>
     *
     * @param describeBudgetRequest
     *        Request of DescribeBudget
     * @return Result of the DescribeBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudget
     */
    default DescribeBudgetResponse describeBudget(DescribeBudgetRequest describeBudgetRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes a budget.
     * </p>
     * <important>
     * <p>
     * The Request Syntax section shows the <code>BudgetLimit</code> syntax. For <code>PlannedBudgetLimits</code>, see
     * the <a href=
     * "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_DescribeBudget.html#API_DescribeBudget_Examples"
     * >Examples</a> section.
     * </p>
     * </important><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeBudgetRequest.Builder} avoiding the need to
     * create one manually via {@link DescribeBudgetRequest#builder()}
     * </p>
     *
     * @param describeBudgetRequest
     *        A {@link Consumer} that will call methods on {@link DescribeBudgetRequest.Builder} to create a request.
     *        Request of DescribeBudget
     * @return Result of the DescribeBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudget
     */
    default DescribeBudgetResponse describeBudget(Consumer<DescribeBudgetRequest.Builder> describeBudgetRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, BudgetsException {
        return describeBudget(DescribeBudgetRequest.builder().applyMutation(describeBudgetRequest).build());
    }

    /**
     * <p>
     * Describes a budget action detail.
     * </p>
     *
     * @param describeBudgetActionRequest
     * @return Result of the DescribeBudgetAction operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetAction
     */
    default DescribeBudgetActionResponse describeBudgetAction(DescribeBudgetActionRequest describeBudgetActionRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes a budget action detail.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeBudgetActionRequest.Builder} avoiding the
     * need to create one manually via {@link DescribeBudgetActionRequest#builder()}
     * </p>
     *
     * @param describeBudgetActionRequest
     *        A {@link Consumer} that will call methods on {@link DescribeBudgetActionRequest.Builder} to create a
     *        request.
     * @return Result of the DescribeBudgetAction operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetAction
     */
    default DescribeBudgetActionResponse describeBudgetAction(
            Consumer<DescribeBudgetActionRequest.Builder> describeBudgetActionRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            BudgetsException {
        return describeBudgetAction(DescribeBudgetActionRequest.builder().applyMutation(describeBudgetActionRequest).build());
    }

    /**
     * <p>
     * Describes a budget action history detail.
     * </p>
     *
     * @param describeBudgetActionHistoriesRequest
     * @return Result of the DescribeBudgetActionHistories operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetActionHistories
     */
    default DescribeBudgetActionHistoriesResponse describeBudgetActionHistories(
            DescribeBudgetActionHistoriesRequest describeBudgetActionHistoriesRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, AccessDeniedException, InvalidNextTokenException, AwsServiceException,
            SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes a budget action history detail.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeBudgetActionHistoriesRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeBudgetActionHistoriesRequest#builder()}
     * </p>
     *
     * @param describeBudgetActionHistoriesRequest
     *        A {@link Consumer} that will call methods on {@link DescribeBudgetActionHistoriesRequest.Builder} to
     *        create a request.
     * @return Result of the DescribeBudgetActionHistories operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetActionHistories
     */
    default DescribeBudgetActionHistoriesResponse describeBudgetActionHistories(
            Consumer<DescribeBudgetActionHistoriesRequest.Builder> describeBudgetActionHistoriesRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            InvalidNextTokenException, AwsServiceException, SdkClientException, BudgetsException {
        return describeBudgetActionHistories(DescribeBudgetActionHistoriesRequest.builder()
                .applyMutation(describeBudgetActionHistoriesRequest).build());
    }

    /**
     * <p>
     * Describes a budget action history detail.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeBudgetActionHistories(software.amazon.awssdk.services.budgets.model.DescribeBudgetActionHistoriesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionHistoriesIterable responses = client.describeBudgetActionHistoriesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionHistoriesIterable responses = client
     *             .describeBudgetActionHistoriesPaginator(request);
     *     for (software.amazon.awssdk.services.budgets.model.DescribeBudgetActionHistoriesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionHistoriesIterable responses = client.describeBudgetActionHistoriesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeBudgetActionHistories(software.amazon.awssdk.services.budgets.model.DescribeBudgetActionHistoriesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeBudgetActionHistoriesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetActionHistories
     */
    default DescribeBudgetActionHistoriesIterable describeBudgetActionHistoriesPaginator(
            DescribeBudgetActionHistoriesRequest describeBudgetActionHistoriesRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, AccessDeniedException, InvalidNextTokenException, AwsServiceException,
            SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes a budget action history detail.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeBudgetActionHistories(software.amazon.awssdk.services.budgets.model.DescribeBudgetActionHistoriesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionHistoriesIterable responses = client.describeBudgetActionHistoriesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionHistoriesIterable responses = client
     *             .describeBudgetActionHistoriesPaginator(request);
     *     for (software.amazon.awssdk.services.budgets.model.DescribeBudgetActionHistoriesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionHistoriesIterable responses = client.describeBudgetActionHistoriesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeBudgetActionHistories(software.amazon.awssdk.services.budgets.model.DescribeBudgetActionHistoriesRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeBudgetActionHistoriesRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeBudgetActionHistoriesRequest#builder()}
     * </p>
     *
     * @param describeBudgetActionHistoriesRequest
     *        A {@link Consumer} that will call methods on {@link DescribeBudgetActionHistoriesRequest.Builder} to
     *        create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetActionHistories
     */
    default DescribeBudgetActionHistoriesIterable describeBudgetActionHistoriesPaginator(
            Consumer<DescribeBudgetActionHistoriesRequest.Builder> describeBudgetActionHistoriesRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            InvalidNextTokenException, AwsServiceException, SdkClientException, BudgetsException {
        return describeBudgetActionHistoriesPaginator(DescribeBudgetActionHistoriesRequest.builder()
                .applyMutation(describeBudgetActionHistoriesRequest).build());
    }

    /**
     * <p>
     * Describes all of the budget actions for an account.
     * </p>
     *
     * @param describeBudgetActionsForAccountRequest
     * @return Result of the DescribeBudgetActionsForAccount operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetActionsForAccount
     */
    default DescribeBudgetActionsForAccountResponse describeBudgetActionsForAccount(
            DescribeBudgetActionsForAccountRequest describeBudgetActionsForAccountRequest) throws InternalErrorException,
            InvalidParameterException, AccessDeniedException, InvalidNextTokenException, AwsServiceException, SdkClientException,
            BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes all of the budget actions for an account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeBudgetActionsForAccountRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeBudgetActionsForAccountRequest#builder()}
     * </p>
     *
     * @param describeBudgetActionsForAccountRequest
     *        A {@link Consumer} that will call methods on {@link DescribeBudgetActionsForAccountRequest.Builder} to
     *        create a request.
     * @return Result of the DescribeBudgetActionsForAccount operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetActionsForAccount
     */
    default DescribeBudgetActionsForAccountResponse describeBudgetActionsForAccount(
            Consumer<DescribeBudgetActionsForAccountRequest.Builder> describeBudgetActionsForAccountRequest)
            throws InternalErrorException, InvalidParameterException, AccessDeniedException, InvalidNextTokenException,
            AwsServiceException, SdkClientException, BudgetsException {
        return describeBudgetActionsForAccount(DescribeBudgetActionsForAccountRequest.builder()
                .applyMutation(describeBudgetActionsForAccountRequest).build());
    }

    /**
     * <p>
     * Describes all of the budget actions for an account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeBudgetActionsForAccount(software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForAccountRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionsForAccountIterable responses = client.describeBudgetActionsForAccountPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionsForAccountIterable responses = client
     *             .describeBudgetActionsForAccountPaginator(request);
     *     for (software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForAccountResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionsForAccountIterable responses = client.describeBudgetActionsForAccountPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeBudgetActionsForAccount(software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForAccountRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeBudgetActionsForAccountRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetActionsForAccount
     */
    default DescribeBudgetActionsForAccountIterable describeBudgetActionsForAccountPaginator(
            DescribeBudgetActionsForAccountRequest describeBudgetActionsForAccountRequest) throws InternalErrorException,
            InvalidParameterException, AccessDeniedException, InvalidNextTokenException, AwsServiceException, SdkClientException,
            BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes all of the budget actions for an account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeBudgetActionsForAccount(software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForAccountRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionsForAccountIterable responses = client.describeBudgetActionsForAccountPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionsForAccountIterable responses = client
     *             .describeBudgetActionsForAccountPaginator(request);
     *     for (software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForAccountResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionsForAccountIterable responses = client.describeBudgetActionsForAccountPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeBudgetActionsForAccount(software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForAccountRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeBudgetActionsForAccountRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeBudgetActionsForAccountRequest#builder()}
     * </p>
     *
     * @param describeBudgetActionsForAccountRequest
     *        A {@link Consumer} that will call methods on {@link DescribeBudgetActionsForAccountRequest.Builder} to
     *        create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetActionsForAccount
     */
    default DescribeBudgetActionsForAccountIterable describeBudgetActionsForAccountPaginator(
            Consumer<DescribeBudgetActionsForAccountRequest.Builder> describeBudgetActionsForAccountRequest)
            throws InternalErrorException, InvalidParameterException, AccessDeniedException, InvalidNextTokenException,
            AwsServiceException, SdkClientException, BudgetsException {
        return describeBudgetActionsForAccountPaginator(DescribeBudgetActionsForAccountRequest.builder()
                .applyMutation(describeBudgetActionsForAccountRequest).build());
    }

    /**
     * <p>
     * Describes all of the budget actions for a budget.
     * </p>
     *
     * @param describeBudgetActionsForBudgetRequest
     * @return Result of the DescribeBudgetActionsForBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetActionsForBudget
     */
    default DescribeBudgetActionsForBudgetResponse describeBudgetActionsForBudget(
            DescribeBudgetActionsForBudgetRequest describeBudgetActionsForBudgetRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, AccessDeniedException, InvalidNextTokenException, AwsServiceException,
            SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes all of the budget actions for a budget.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeBudgetActionsForBudgetRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeBudgetActionsForBudgetRequest#builder()}
     * </p>
     *
     * @param describeBudgetActionsForBudgetRequest
     *        A {@link Consumer} that will call methods on {@link DescribeBudgetActionsForBudgetRequest.Builder} to
     *        create a request.
     * @return Result of the DescribeBudgetActionsForBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetActionsForBudget
     */
    default DescribeBudgetActionsForBudgetResponse describeBudgetActionsForBudget(
            Consumer<DescribeBudgetActionsForBudgetRequest.Builder> describeBudgetActionsForBudgetRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            InvalidNextTokenException, AwsServiceException, SdkClientException, BudgetsException {
        return describeBudgetActionsForBudget(DescribeBudgetActionsForBudgetRequest.builder()
                .applyMutation(describeBudgetActionsForBudgetRequest).build());
    }

    /**
     * <p>
     * Describes all of the budget actions for a budget.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeBudgetActionsForBudget(software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForBudgetRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionsForBudgetIterable responses = client.describeBudgetActionsForBudgetPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionsForBudgetIterable responses = client
     *             .describeBudgetActionsForBudgetPaginator(request);
     *     for (software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForBudgetResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionsForBudgetIterable responses = client.describeBudgetActionsForBudgetPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeBudgetActionsForBudget(software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForBudgetRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeBudgetActionsForBudgetRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetActionsForBudget
     */
    default DescribeBudgetActionsForBudgetIterable describeBudgetActionsForBudgetPaginator(
            DescribeBudgetActionsForBudgetRequest describeBudgetActionsForBudgetRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, AccessDeniedException, InvalidNextTokenException, AwsServiceException,
            SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes all of the budget actions for a budget.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeBudgetActionsForBudget(software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForBudgetRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionsForBudgetIterable responses = client.describeBudgetActionsForBudgetPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionsForBudgetIterable responses = client
     *             .describeBudgetActionsForBudgetPaginator(request);
     *     for (software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForBudgetResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetActionsForBudgetIterable responses = client.describeBudgetActionsForBudgetPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeBudgetActionsForBudget(software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForBudgetRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeBudgetActionsForBudgetRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeBudgetActionsForBudgetRequest#builder()}
     * </p>
     *
     * @param describeBudgetActionsForBudgetRequest
     *        A {@link Consumer} that will call methods on {@link DescribeBudgetActionsForBudgetRequest.Builder} to
     *        create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetActionsForBudget
     */
    default DescribeBudgetActionsForBudgetIterable describeBudgetActionsForBudgetPaginator(
            Consumer<DescribeBudgetActionsForBudgetRequest.Builder> describeBudgetActionsForBudgetRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            InvalidNextTokenException, AwsServiceException, SdkClientException, BudgetsException {
        return describeBudgetActionsForBudgetPaginator(DescribeBudgetActionsForBudgetRequest.builder()
                .applyMutation(describeBudgetActionsForBudgetRequest).build());
    }

    /**
     * <p>
     * Describes the history for <code>DAILY</code>, <code>MONTHLY</code>, and <code>QUARTERLY</code> budgets. Budget
     * history isn't available for <code>ANNUAL</code> budgets.
     * </p>
     *
     * @param describeBudgetPerformanceHistoryRequest
     * @return Result of the DescribeBudgetPerformanceHistory operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetPerformanceHistory
     */
    default DescribeBudgetPerformanceHistoryResponse describeBudgetPerformanceHistory(
            DescribeBudgetPerformanceHistoryRequest describeBudgetPerformanceHistoryRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, InvalidNextTokenException, ExpiredNextTokenException,
            AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes the history for <code>DAILY</code>, <code>MONTHLY</code>, and <code>QUARTERLY</code> budgets. Budget
     * history isn't available for <code>ANNUAL</code> budgets.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeBudgetPerformanceHistoryRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeBudgetPerformanceHistoryRequest#builder()}
     * </p>
     *
     * @param describeBudgetPerformanceHistoryRequest
     *        A {@link Consumer} that will call methods on {@link DescribeBudgetPerformanceHistoryRequest.Builder} to
     *        create a request.
     * @return Result of the DescribeBudgetPerformanceHistory operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetPerformanceHistory
     */
    default DescribeBudgetPerformanceHistoryResponse describeBudgetPerformanceHistory(
            Consumer<DescribeBudgetPerformanceHistoryRequest.Builder> describeBudgetPerformanceHistoryRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        return describeBudgetPerformanceHistory(DescribeBudgetPerformanceHistoryRequest.builder()
                .applyMutation(describeBudgetPerformanceHistoryRequest).build());
    }

    /**
     * <p>
     * Describes the history for <code>DAILY</code>, <code>MONTHLY</code>, and <code>QUARTERLY</code> budgets. Budget
     * history isn't available for <code>ANNUAL</code> budgets.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeBudgetPerformanceHistory(software.amazon.awssdk.services.budgets.model.DescribeBudgetPerformanceHistoryRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetPerformanceHistoryIterable responses = client.describeBudgetPerformanceHistoryPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.budgets.paginators.DescribeBudgetPerformanceHistoryIterable responses = client
     *             .describeBudgetPerformanceHistoryPaginator(request);
     *     for (software.amazon.awssdk.services.budgets.model.DescribeBudgetPerformanceHistoryResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetPerformanceHistoryIterable responses = client.describeBudgetPerformanceHistoryPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeBudgetPerformanceHistory(software.amazon.awssdk.services.budgets.model.DescribeBudgetPerformanceHistoryRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeBudgetPerformanceHistoryRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetPerformanceHistory
     */
    default DescribeBudgetPerformanceHistoryIterable describeBudgetPerformanceHistoryPaginator(
            DescribeBudgetPerformanceHistoryRequest describeBudgetPerformanceHistoryRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, InvalidNextTokenException, ExpiredNextTokenException,
            AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Describes the history for <code>DAILY</code>, <code>MONTHLY</code>, and <code>QUARTERLY</code> budgets. Budget
     * history isn't available for <code>ANNUAL</code> budgets.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeBudgetPerformanceHistory(software.amazon.awssdk.services.budgets.model.DescribeBudgetPerformanceHistoryRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetPerformanceHistoryIterable responses = client.describeBudgetPerformanceHistoryPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.budgets.paginators.DescribeBudgetPerformanceHistoryIterable responses = client
     *             .describeBudgetPerformanceHistoryPaginator(request);
     *     for (software.amazon.awssdk.services.budgets.model.DescribeBudgetPerformanceHistoryResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetPerformanceHistoryIterable responses = client.describeBudgetPerformanceHistoryPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeBudgetPerformanceHistory(software.amazon.awssdk.services.budgets.model.DescribeBudgetPerformanceHistoryRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeBudgetPerformanceHistoryRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeBudgetPerformanceHistoryRequest#builder()}
     * </p>
     *
     * @param describeBudgetPerformanceHistoryRequest
     *        A {@link Consumer} that will call methods on {@link DescribeBudgetPerformanceHistoryRequest.Builder} to
     *        create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetPerformanceHistory
     */
    default DescribeBudgetPerformanceHistoryIterable describeBudgetPerformanceHistoryPaginator(
            Consumer<DescribeBudgetPerformanceHistoryRequest.Builder> describeBudgetPerformanceHistoryRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        return describeBudgetPerformanceHistoryPaginator(DescribeBudgetPerformanceHistoryRequest.builder()
                .applyMutation(describeBudgetPerformanceHistoryRequest).build());
    }

    /**
     * <p>
     * Lists the budgets that are associated with an account.
     * </p>
     * <important>
     * <p>
     * The Request Syntax section shows the <code>BudgetLimit</code> syntax. For <code>PlannedBudgetLimits</code>, see
     * the <a href=
     * "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_DescribeBudgets.html#API_DescribeBudgets_Examples"
     * >Examples</a> section.
     * </p>
     * </important>
     *
     * @param describeBudgetsRequest
     *        Request of DescribeBudgets
     * @return Result of the DescribeBudgets operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgets
     */
    default DescribeBudgetsResponse describeBudgets(DescribeBudgetsRequest describeBudgetsRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, InvalidNextTokenException, ExpiredNextTokenException,
            AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the budgets that are associated with an account.
     * </p>
     * <important>
     * <p>
     * The Request Syntax section shows the <code>BudgetLimit</code> syntax. For <code>PlannedBudgetLimits</code>, see
     * the <a href=
     * "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_DescribeBudgets.html#API_DescribeBudgets_Examples"
     * >Examples</a> section.
     * </p>
     * </important><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeBudgetsRequest.Builder} avoiding the need
     * to create one manually via {@link DescribeBudgetsRequest#builder()}
     * </p>
     *
     * @param describeBudgetsRequest
     *        A {@link Consumer} that will call methods on {@link DescribeBudgetsRequest.Builder} to create a request.
     *        Request of DescribeBudgets
     * @return Result of the DescribeBudgets operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgets
     */
    default DescribeBudgetsResponse describeBudgets(Consumer<DescribeBudgetsRequest.Builder> describeBudgetsRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        return describeBudgets(DescribeBudgetsRequest.builder().applyMutation(describeBudgetsRequest).build());
    }

    /**
     * <p>
     * Lists the budgets that are associated with an account.
     * </p>
     * <important>
     * <p>
     * The Request Syntax section shows the <code>BudgetLimit</code> syntax. For <code>PlannedBudgetLimits</code>, see
     * the <a href=
     * "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_DescribeBudgets.html#API_DescribeBudgets_Examples"
     * >Examples</a> section.
     * </p>
     * </important><br/>
     * <p>
     * This is a variant of
     * {@link #describeBudgets(software.amazon.awssdk.services.budgets.model.DescribeBudgetsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetsIterable responses = client.describeBudgetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.budgets.paginators.DescribeBudgetsIterable responses = client
     *             .describeBudgetsPaginator(request);
     *     for (software.amazon.awssdk.services.budgets.model.DescribeBudgetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetsIterable responses = client.describeBudgetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeBudgets(software.amazon.awssdk.services.budgets.model.DescribeBudgetsRequest)} operation.</b>
     * </p>
     *
     * @param describeBudgetsRequest
     *        Request of DescribeBudgets
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgets
     */
    default DescribeBudgetsIterable describeBudgetsPaginator(DescribeBudgetsRequest describeBudgetsRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the budgets that are associated with an account.
     * </p>
     * <important>
     * <p>
     * The Request Syntax section shows the <code>BudgetLimit</code> syntax. For <code>PlannedBudgetLimits</code>, see
     * the <a href=
     * "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_DescribeBudgets.html#API_DescribeBudgets_Examples"
     * >Examples</a> section.
     * </p>
     * </important><br/>
     * <p>
     * This is a variant of
     * {@link #describeBudgets(software.amazon.awssdk.services.budgets.model.DescribeBudgetsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetsIterable responses = client.describeBudgetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.budgets.paginators.DescribeBudgetsIterable responses = client
     *             .describeBudgetsPaginator(request);
     *     for (software.amazon.awssdk.services.budgets.model.DescribeBudgetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeBudgetsIterable responses = client.describeBudgetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeBudgets(software.amazon.awssdk.services.budgets.model.DescribeBudgetsRequest)} operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeBudgetsRequest.Builder} avoiding the need
     * to create one manually via {@link DescribeBudgetsRequest#builder()}
     * </p>
     *
     * @param describeBudgetsRequest
     *        A {@link Consumer} that will call methods on {@link DescribeBudgetsRequest.Builder} to create a request.
     *        Request of DescribeBudgets
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgets
     */
    default DescribeBudgetsIterable describeBudgetsPaginator(Consumer<DescribeBudgetsRequest.Builder> describeBudgetsRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        return describeBudgetsPaginator(DescribeBudgetsRequest.builder().applyMutation(describeBudgetsRequest).build());
    }

    /**
     * <p>
     * Lists the notifications that are associated with a budget.
     * </p>
     *
     * @param describeNotificationsForBudgetRequest
     *        Request of DescribeNotificationsForBudget
     * @return Result of the DescribeNotificationsForBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeNotificationsForBudget
     */
    default DescribeNotificationsForBudgetResponse describeNotificationsForBudget(
            DescribeNotificationsForBudgetRequest describeNotificationsForBudgetRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, InvalidNextTokenException, ExpiredNextTokenException,
            AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the notifications that are associated with a budget.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeNotificationsForBudgetRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeNotificationsForBudgetRequest#builder()}
     * </p>
     *
     * @param describeNotificationsForBudgetRequest
     *        A {@link Consumer} that will call methods on {@link DescribeNotificationsForBudgetRequest.Builder} to
     *        create a request. Request of DescribeNotificationsForBudget
     * @return Result of the DescribeNotificationsForBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeNotificationsForBudget
     */
    default DescribeNotificationsForBudgetResponse describeNotificationsForBudget(
            Consumer<DescribeNotificationsForBudgetRequest.Builder> describeNotificationsForBudgetRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        return describeNotificationsForBudget(DescribeNotificationsForBudgetRequest.builder()
                .applyMutation(describeNotificationsForBudgetRequest).build());
    }

    /**
     * <p>
     * Lists the notifications that are associated with a budget.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeNotificationsForBudget(software.amazon.awssdk.services.budgets.model.DescribeNotificationsForBudgetRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeNotificationsForBudgetIterable responses = client.describeNotificationsForBudgetPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.budgets.paginators.DescribeNotificationsForBudgetIterable responses = client
     *             .describeNotificationsForBudgetPaginator(request);
     *     for (software.amazon.awssdk.services.budgets.model.DescribeNotificationsForBudgetResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeNotificationsForBudgetIterable responses = client.describeNotificationsForBudgetPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeNotificationsForBudget(software.amazon.awssdk.services.budgets.model.DescribeNotificationsForBudgetRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeNotificationsForBudgetRequest
     *        Request of DescribeNotificationsForBudget
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeNotificationsForBudget
     */
    default DescribeNotificationsForBudgetIterable describeNotificationsForBudgetPaginator(
            DescribeNotificationsForBudgetRequest describeNotificationsForBudgetRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, InvalidNextTokenException, ExpiredNextTokenException,
            AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the notifications that are associated with a budget.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeNotificationsForBudget(software.amazon.awssdk.services.budgets.model.DescribeNotificationsForBudgetRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeNotificationsForBudgetIterable responses = client.describeNotificationsForBudgetPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.budgets.paginators.DescribeNotificationsForBudgetIterable responses = client
     *             .describeNotificationsForBudgetPaginator(request);
     *     for (software.amazon.awssdk.services.budgets.model.DescribeNotificationsForBudgetResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeNotificationsForBudgetIterable responses = client.describeNotificationsForBudgetPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeNotificationsForBudget(software.amazon.awssdk.services.budgets.model.DescribeNotificationsForBudgetRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeNotificationsForBudgetRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeNotificationsForBudgetRequest#builder()}
     * </p>
     *
     * @param describeNotificationsForBudgetRequest
     *        A {@link Consumer} that will call methods on {@link DescribeNotificationsForBudgetRequest.Builder} to
     *        create a request. Request of DescribeNotificationsForBudget
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeNotificationsForBudget
     */
    default DescribeNotificationsForBudgetIterable describeNotificationsForBudgetPaginator(
            Consumer<DescribeNotificationsForBudgetRequest.Builder> describeNotificationsForBudgetRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        return describeNotificationsForBudgetPaginator(DescribeNotificationsForBudgetRequest.builder()
                .applyMutation(describeNotificationsForBudgetRequest).build());
    }

    /**
     * <p>
     * Lists the subscribers that are associated with a notification.
     * </p>
     *
     * @param describeSubscribersForNotificationRequest
     *        Request of DescribeSubscribersForNotification
     * @return Result of the DescribeSubscribersForNotification operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeSubscribersForNotification
     */
    default DescribeSubscribersForNotificationResponse describeSubscribersForNotification(
            DescribeSubscribersForNotificationRequest describeSubscribersForNotificationRequest) throws InternalErrorException,
            NotFoundException, InvalidParameterException, InvalidNextTokenException, ExpiredNextTokenException,
            AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the subscribers that are associated with a notification.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeSubscribersForNotificationRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeSubscribersForNotificationRequest#builder()}
     * </p>
     *
     * @param describeSubscribersForNotificationRequest
     *        A {@link Consumer} that will call methods on {@link DescribeSubscribersForNotificationRequest.Builder} to
     *        create a request. Request of DescribeSubscribersForNotification
     * @return Result of the DescribeSubscribersForNotification operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeSubscribersForNotification
     */
    default DescribeSubscribersForNotificationResponse describeSubscribersForNotification(
            Consumer<DescribeSubscribersForNotificationRequest.Builder> describeSubscribersForNotificationRequest)
            throws InternalErrorException, NotFoundException, InvalidParameterException, InvalidNextTokenException,
            ExpiredNextTokenException, AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        return describeSubscribersForNotification(DescribeSubscribersForNotificationRequest.builder()
                .applyMutation(describeSubscribersForNotificationRequest).build());
    }

    /**
     * <p>
     * Lists the subscribers that are associated with a notification.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeSubscribersForNotification(software.amazon.awssdk.services.budgets.model.DescribeSubscribersForNotificationRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeSubscribersForNotificationIterable responses = client.describeSubscribersForNotificationPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.budgets.paginators.DescribeSubscribersForNotificationIterable responses = client
     *             .describeSubscribersForNotificationPaginator(request);
     *     for (software.amazon.awssdk.services.budgets.model.DescribeSubscribersForNotificationResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeSubscribersForNotificationIterable responses = client.describeSubscribersForNotificationPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeSubscribersForNotification(software.amazon.awssdk.services.budgets.model.DescribeSubscribersForNotificationRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeSubscribersForNotificationRequest
     *        Request of DescribeSubscribersForNotification
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeSubscribersForNotification
     */
    default DescribeSubscribersForNotificationIterable describeSubscribersForNotificationPaginator(
            DescribeSubscribersForNotificationRequest describeSubscribersForNotificationRequest) throws InternalErrorException,
            NotFoundException, InvalidParameterException, InvalidNextTokenException, ExpiredNextTokenException,
            AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists the subscribers that are associated with a notification.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeSubscribersForNotification(software.amazon.awssdk.services.budgets.model.DescribeSubscribersForNotificationRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeSubscribersForNotificationIterable responses = client.describeSubscribersForNotificationPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.budgets.paginators.DescribeSubscribersForNotificationIterable responses = client
     *             .describeSubscribersForNotificationPaginator(request);
     *     for (software.amazon.awssdk.services.budgets.model.DescribeSubscribersForNotificationResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.budgets.paginators.DescribeSubscribersForNotificationIterable responses = client.describeSubscribersForNotificationPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeSubscribersForNotification(software.amazon.awssdk.services.budgets.model.DescribeSubscribersForNotificationRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link DescribeSubscribersForNotificationRequest.Builder}
     * avoiding the need to create one manually via {@link DescribeSubscribersForNotificationRequest#builder()}
     * </p>
     *
     * @param describeSubscribersForNotificationRequest
     *        A {@link Consumer} that will call methods on {@link DescribeSubscribersForNotificationRequest.Builder} to
     *        create a request. Request of DescribeSubscribersForNotification
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeSubscribersForNotification
     */
    default DescribeSubscribersForNotificationIterable describeSubscribersForNotificationPaginator(
            Consumer<DescribeSubscribersForNotificationRequest.Builder> describeSubscribersForNotificationRequest)
            throws InternalErrorException, NotFoundException, InvalidParameterException, InvalidNextTokenException,
            ExpiredNextTokenException, AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        return describeSubscribersForNotificationPaginator(DescribeSubscribersForNotificationRequest.builder()
                .applyMutation(describeSubscribersForNotificationRequest).build());
    }

    /**
     * <p>
     * Executes a budget action.
     * </p>
     *
     * @param executeBudgetActionRequest
     * @return Result of the ExecuteBudgetAction operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ResourceLockedException
     *         The request was received and recognized by the server, but the server rejected that particular method for
     *         the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.ExecuteBudgetAction
     */
    default ExecuteBudgetActionResponse executeBudgetAction(ExecuteBudgetActionRequest executeBudgetActionRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            ResourceLockedException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Executes a budget action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ExecuteBudgetActionRequest.Builder} avoiding the
     * need to create one manually via {@link ExecuteBudgetActionRequest#builder()}
     * </p>
     *
     * @param executeBudgetActionRequest
     *        A {@link Consumer} that will call methods on {@link ExecuteBudgetActionRequest.Builder} to create a
     *        request.
     * @return Result of the ExecuteBudgetAction operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ResourceLockedException
     *         The request was received and recognized by the server, but the server rejected that particular method for
     *         the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.ExecuteBudgetAction
     */
    default ExecuteBudgetActionResponse executeBudgetAction(
            Consumer<ExecuteBudgetActionRequest.Builder> executeBudgetActionRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, AccessDeniedException, ResourceLockedException, AwsServiceException,
            SdkClientException, BudgetsException {
        return executeBudgetAction(ExecuteBudgetActionRequest.builder().applyMutation(executeBudgetActionRequest).build());
    }

    /**
     * <p>
     * Updates a budget. You can change every part of a budget except for the <code>budgetName</code> and the
     * <code>calculatedSpend</code>. When you modify a budget, the <code>calculatedSpend</code> drops to zero until
     * Amazon Web Services has new usage data to use for forecasting.
     * </p>
     * <important>
     * <p>
     * Only one of <code>BudgetLimit</code> or <code>PlannedBudgetLimits</code> can be present in the syntax at one
     * time. Use the syntax that matches your case. The Request Syntax section shows the <code>BudgetLimit</code>
     * syntax. For <code>PlannedBudgetLimits</code>, see the <a href=
     * "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_UpdateBudget.html#API_UpdateBudget_Examples"
     * >Examples</a> section.
     * </p>
     * </important>
     *
     * @param updateBudgetRequest
     *        Request of UpdateBudget
     * @return Result of the UpdateBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UpdateBudget
     */
    default UpdateBudgetResponse updateBudget(UpdateBudgetRequest updateBudgetRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, AccessDeniedException, AwsServiceException, SdkClientException,
            BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a budget. You can change every part of a budget except for the <code>budgetName</code> and the
     * <code>calculatedSpend</code>. When you modify a budget, the <code>calculatedSpend</code> drops to zero until
     * Amazon Web Services has new usage data to use for forecasting.
     * </p>
     * <important>
     * <p>
     * Only one of <code>BudgetLimit</code> or <code>PlannedBudgetLimits</code> can be present in the syntax at one
     * time. Use the syntax that matches your case. The Request Syntax section shows the <code>BudgetLimit</code>
     * syntax. For <code>PlannedBudgetLimits</code>, see the <a href=
     * "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_UpdateBudget.html#API_UpdateBudget_Examples"
     * >Examples</a> section.
     * </p>
     * </important><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateBudgetRequest.Builder} avoiding the need to
     * create one manually via {@link UpdateBudgetRequest#builder()}
     * </p>
     *
     * @param updateBudgetRequest
     *        A {@link Consumer} that will call methods on {@link UpdateBudgetRequest.Builder} to create a request.
     *        Request of UpdateBudget
     * @return Result of the UpdateBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UpdateBudget
     */
    default UpdateBudgetResponse updateBudget(Consumer<UpdateBudgetRequest.Builder> updateBudgetRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            AwsServiceException, SdkClientException, BudgetsException {
        return updateBudget(UpdateBudgetRequest.builder().applyMutation(updateBudgetRequest).build());
    }

    /**
     * <p>
     * Updates a budget action.
     * </p>
     *
     * @param updateBudgetActionRequest
     * @return Result of the UpdateBudgetAction operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ResourceLockedException
     *         The request was received and recognized by the server, but the server rejected that particular method for
     *         the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UpdateBudgetAction
     */
    default UpdateBudgetActionResponse updateBudgetAction(UpdateBudgetActionRequest updateBudgetActionRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            ResourceLockedException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a budget action.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateBudgetActionRequest.Builder} avoiding the
     * need to create one manually via {@link UpdateBudgetActionRequest#builder()}
     * </p>
     *
     * @param updateBudgetActionRequest
     *        A {@link Consumer} that will call methods on {@link UpdateBudgetActionRequest.Builder} to create a
     *        request.
     * @return Result of the UpdateBudgetAction operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ResourceLockedException
     *         The request was received and recognized by the server, but the server rejected that particular method for
     *         the requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UpdateBudgetAction
     */
    default UpdateBudgetActionResponse updateBudgetAction(Consumer<UpdateBudgetActionRequest.Builder> updateBudgetActionRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            ResourceLockedException, AwsServiceException, SdkClientException, BudgetsException {
        return updateBudgetAction(UpdateBudgetActionRequest.builder().applyMutation(updateBudgetActionRequest).build());
    }

    /**
     * <p>
     * Updates a notification.
     * </p>
     *
     * @param updateNotificationRequest
     *        Request of UpdateNotification
     * @return Result of the UpdateNotification operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UpdateNotification
     */
    default UpdateNotificationResponse updateNotification(UpdateNotificationRequest updateNotificationRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, DuplicateRecordException,
            AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a notification.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateNotificationRequest.Builder} avoiding the
     * need to create one manually via {@link UpdateNotificationRequest#builder()}
     * </p>
     *
     * @param updateNotificationRequest
     *        A {@link Consumer} that will call methods on {@link UpdateNotificationRequest.Builder} to create a
     *        request. Request of UpdateNotification
     * @return Result of the UpdateNotification operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UpdateNotification
     */
    default UpdateNotificationResponse updateNotification(Consumer<UpdateNotificationRequest.Builder> updateNotificationRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, DuplicateRecordException,
            AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        return updateNotification(UpdateNotificationRequest.builder().applyMutation(updateNotificationRequest).build());
    }

    /**
     * <p>
     * Updates a subscriber.
     * </p>
     *
     * @param updateSubscriberRequest
     *        Request of UpdateSubscriber
     * @return Result of the UpdateSubscriber operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UpdateSubscriber
     */
    default UpdateSubscriberResponse updateSubscriber(UpdateSubscriberRequest updateSubscriberRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, DuplicateRecordException,
            AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates a subscriber.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateSubscriberRequest.Builder} avoiding the need
     * to create one manually via {@link UpdateSubscriberRequest#builder()}
     * </p>
     *
     * @param updateSubscriberRequest
     *        A {@link Consumer} that will call methods on {@link UpdateSubscriberRequest.Builder} to create a request.
     *        Request of UpdateSubscriber
     * @return Result of the UpdateSubscriber operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UpdateSubscriber
     */
    default UpdateSubscriberResponse updateSubscriber(Consumer<UpdateSubscriberRequest.Builder> updateSubscriberRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, DuplicateRecordException,
            AccessDeniedException, AwsServiceException, SdkClientException, BudgetsException {
        return updateSubscriber(UpdateSubscriberRequest.builder().applyMutation(updateSubscriberRequest).build());
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }
}
