/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.budgets.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Request of CreateNotification
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateNotificationRequest extends BudgetsRequest implements
        ToCopyableBuilder<CreateNotificationRequest.Builder, CreateNotificationRequest> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountId").getter(getter(CreateNotificationRequest::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountId").build()).build();

    private static final SdkField<String> BUDGET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BudgetName").getter(getter(CreateNotificationRequest::budgetName)).setter(setter(Builder::budgetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BudgetName").build()).build();

    private static final SdkField<Notification> NOTIFICATION_FIELD = SdkField.<Notification> builder(MarshallingType.SDK_POJO)
            .memberName("Notification").getter(getter(CreateNotificationRequest::notification))
            .setter(setter(Builder::notification)).constructor(Notification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Notification").build()).build();

    private static final SdkField<List<Subscriber>> SUBSCRIBERS_FIELD = SdkField
            .<List<Subscriber>> builder(MarshallingType.LIST)
            .memberName("Subscribers")
            .getter(getter(CreateNotificationRequest::subscribers))
            .setter(setter(Builder::subscribers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subscribers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Subscriber> builder(MarshallingType.SDK_POJO)
                                            .constructor(Subscriber::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            BUDGET_NAME_FIELD, NOTIFICATION_FIELD, SUBSCRIBERS_FIELD));

    private final String accountId;

    private final String budgetName;

    private final Notification notification;

    private final List<Subscriber> subscribers;

    private CreateNotificationRequest(BuilderImpl builder) {
        super(builder);
        this.accountId = builder.accountId;
        this.budgetName = builder.budgetName;
        this.notification = builder.notification;
        this.subscribers = builder.subscribers;
    }

    /**
     * <p>
     * The <code>accountId</code> that is associated with the budget that you want to create a notification for.
     * </p>
     * 
     * @return The <code>accountId</code> that is associated with the budget that you want to create a notification for.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The name of the budget that you want Amazon Web Services to notify you about. Budget names must be unique within
     * an account.
     * </p>
     * 
     * @return The name of the budget that you want Amazon Web Services to notify you about. Budget names must be unique
     *         within an account.
     */
    public final String budgetName() {
        return budgetName;
    }

    /**
     * <p>
     * The notification that you want to create.
     * </p>
     * 
     * @return The notification that you want to create.
     */
    public final Notification notification() {
        return notification;
    }

    /**
     * For responses, this returns true if the service returned a value for the Subscribers property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSubscribers() {
        return subscribers != null && !(subscribers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of subscribers that you want to associate with the notification. Each notification can have one SNS
     * subscriber and up to 10 email subscribers.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSubscribers} method.
     * </p>
     * 
     * @return A list of subscribers that you want to associate with the notification. Each notification can have one
     *         SNS subscriber and up to 10 email subscribers.
     */
    public final List<Subscriber> subscribers() {
        return subscribers;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(budgetName());
        hashCode = 31 * hashCode + Objects.hashCode(notification());
        hashCode = 31 * hashCode + Objects.hashCode(hasSubscribers() ? subscribers() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateNotificationRequest)) {
            return false;
        }
        CreateNotificationRequest other = (CreateNotificationRequest) obj;
        return Objects.equals(accountId(), other.accountId()) && Objects.equals(budgetName(), other.budgetName())
                && Objects.equals(notification(), other.notification()) && hasSubscribers() == other.hasSubscribers()
                && Objects.equals(subscribers(), other.subscribers());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateNotificationRequest").add("AccountId", accountId()).add("BudgetName", budgetName())
                .add("Notification", notification()).add("Subscribers", hasSubscribers() ? subscribers() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "BudgetName":
            return Optional.ofNullable(clazz.cast(budgetName()));
        case "Notification":
            return Optional.ofNullable(clazz.cast(notification()));
        case "Subscribers":
            return Optional.ofNullable(clazz.cast(subscribers()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateNotificationRequest, T> g) {
        return obj -> g.apply((CreateNotificationRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends BudgetsRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateNotificationRequest> {
        /**
         * <p>
         * The <code>accountId</code> that is associated with the budget that you want to create a notification for.
         * </p>
         * 
         * @param accountId
         *        The <code>accountId</code> that is associated with the budget that you want to create a notification
         *        for.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The name of the budget that you want Amazon Web Services to notify you about. Budget names must be unique
         * within an account.
         * </p>
         * 
         * @param budgetName
         *        The name of the budget that you want Amazon Web Services to notify you about. Budget names must be
         *        unique within an account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder budgetName(String budgetName);

        /**
         * <p>
         * The notification that you want to create.
         * </p>
         * 
         * @param notification
         *        The notification that you want to create.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notification(Notification notification);

        /**
         * <p>
         * The notification that you want to create.
         * </p>
         * This is a convenience method that creates an instance of the {@link Notification.Builder} avoiding the need
         * to create one manually via {@link Notification#builder()}.
         *
         * When the {@link Consumer} completes, {@link Notification.Builder#build()} is called immediately and its
         * result is passed to {@link #notification(Notification)}.
         * 
         * @param notification
         *        a consumer that will call methods on {@link Notification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notification(Notification)
         */
        default Builder notification(Consumer<Notification.Builder> notification) {
            return notification(Notification.builder().applyMutation(notification).build());
        }

        /**
         * <p>
         * A list of subscribers that you want to associate with the notification. Each notification can have one SNS
         * subscriber and up to 10 email subscribers.
         * </p>
         * 
         * @param subscribers
         *        A list of subscribers that you want to associate with the notification. Each notification can have one
         *        SNS subscriber and up to 10 email subscribers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subscribers(Collection<Subscriber> subscribers);

        /**
         * <p>
         * A list of subscribers that you want to associate with the notification. Each notification can have one SNS
         * subscriber and up to 10 email subscribers.
         * </p>
         * 
         * @param subscribers
         *        A list of subscribers that you want to associate with the notification. Each notification can have one
         *        SNS subscriber and up to 10 email subscribers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subscribers(Subscriber... subscribers);

        /**
         * <p>
         * A list of subscribers that you want to associate with the notification. Each notification can have one SNS
         * subscriber and up to 10 email subscribers.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Subscriber>.Builder} avoiding the
         * need to create one manually via {@link List<Subscriber>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Subscriber>.Builder#build()} is called immediately and its
         * result is passed to {@link #subscribers(List<Subscriber>)}.
         * 
         * @param subscribers
         *        a consumer that will call methods on {@link List<Subscriber>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #subscribers(List<Subscriber>)
         */
        Builder subscribers(Consumer<Subscriber.Builder>... subscribers);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends BudgetsRequest.BuilderImpl implements Builder {
        private String accountId;

        private String budgetName;

        private Notification notification;

        private List<Subscriber> subscribers = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateNotificationRequest model) {
            super(model);
            accountId(model.accountId);
            budgetName(model.budgetName);
            notification(model.notification);
            subscribers(model.subscribers);
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getBudgetName() {
            return budgetName;
        }

        public final void setBudgetName(String budgetName) {
            this.budgetName = budgetName;
        }

        @Override
        public final Builder budgetName(String budgetName) {
            this.budgetName = budgetName;
            return this;
        }

        public final Notification.Builder getNotification() {
            return notification != null ? notification.toBuilder() : null;
        }

        public final void setNotification(Notification.BuilderImpl notification) {
            this.notification = notification != null ? notification.build() : null;
        }

        @Override
        public final Builder notification(Notification notification) {
            this.notification = notification;
            return this;
        }

        public final List<Subscriber.Builder> getSubscribers() {
            List<Subscriber.Builder> result = SubscribersCopier.copyToBuilder(this.subscribers);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setSubscribers(Collection<Subscriber.BuilderImpl> subscribers) {
            this.subscribers = SubscribersCopier.copyFromBuilder(subscribers);
        }

        @Override
        public final Builder subscribers(Collection<Subscriber> subscribers) {
            this.subscribers = SubscribersCopier.copy(subscribers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subscribers(Subscriber... subscribers) {
            subscribers(Arrays.asList(subscribers));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subscribers(Consumer<Subscriber.Builder>... subscribers) {
            subscribers(Stream.of(subscribers).map(c -> Subscriber.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateNotificationRequest build() {
            return new CreateNotificationRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
