/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.budgets.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the output of the <code>CreateBudget</code> operation. The content consists of the detailed metadata and
 * data file information, and the current status of the <code>budget</code> object.
 * </p>
 * <p>
 * This is the ARN pattern for a budget:
 * </p>
 * <p>
 * <code>arn:aws:budgetservice::AccountId:budget/budgetName</code>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Budget implements SdkPojo, Serializable, ToCopyableBuilder<Budget.Builder, Budget> {
    private static final SdkField<String> BUDGET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Budget::budgetName)).setter(setter(Builder::budgetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BudgetName").build()).build();

    private static final SdkField<Spend> BUDGET_LIMIT_FIELD = SdkField.<Spend> builder(MarshallingType.SDK_POJO)
            .getter(getter(Budget::budgetLimit)).setter(setter(Builder::budgetLimit)).constructor(Spend::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BudgetLimit").build()).build();

    private static final SdkField<Map<String, List<String>>> COST_FILTERS_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .getter(getter(Budget::costFilters))
            .setter(setter(Builder::costFilters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CostFilters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<CostTypes> COST_TYPES_FIELD = SdkField.<CostTypes> builder(MarshallingType.SDK_POJO)
            .getter(getter(Budget::costTypes)).setter(setter(Builder::costTypes)).constructor(CostTypes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CostTypes").build()).build();

    private static final SdkField<String> TIME_UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Budget::timeUnitAsString)).setter(setter(Builder::timeUnit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeUnit").build()).build();

    private static final SdkField<TimePeriod> TIME_PERIOD_FIELD = SdkField.<TimePeriod> builder(MarshallingType.SDK_POJO)
            .getter(getter(Budget::timePeriod)).setter(setter(Builder::timePeriod)).constructor(TimePeriod::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimePeriod").build()).build();

    private static final SdkField<CalculatedSpend> CALCULATED_SPEND_FIELD = SdkField
            .<CalculatedSpend> builder(MarshallingType.SDK_POJO).getter(getter(Budget::calculatedSpend))
            .setter(setter(Builder::calculatedSpend)).constructor(CalculatedSpend::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CalculatedSpend").build()).build();

    private static final SdkField<String> BUDGET_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Budget::budgetTypeAsString)).setter(setter(Builder::budgetType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BudgetType").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Budget::lastUpdatedTime)).setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUDGET_NAME_FIELD,
            BUDGET_LIMIT_FIELD, COST_FILTERS_FIELD, COST_TYPES_FIELD, TIME_UNIT_FIELD, TIME_PERIOD_FIELD, CALCULATED_SPEND_FIELD,
            BUDGET_TYPE_FIELD, LAST_UPDATED_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String budgetName;

    private final Spend budgetLimit;

    private final Map<String, List<String>> costFilters;

    private final CostTypes costTypes;

    private final String timeUnit;

    private final TimePeriod timePeriod;

    private final CalculatedSpend calculatedSpend;

    private final String budgetType;

    private final Instant lastUpdatedTime;

    private Budget(BuilderImpl builder) {
        this.budgetName = builder.budgetName;
        this.budgetLimit = builder.budgetLimit;
        this.costFilters = builder.costFilters;
        this.costTypes = builder.costTypes;
        this.timeUnit = builder.timeUnit;
        this.timePeriod = builder.timePeriod;
        this.calculatedSpend = builder.calculatedSpend;
        this.budgetType = builder.budgetType;
        this.lastUpdatedTime = builder.lastUpdatedTime;
    }

    /**
     * <p>
     * The name of a budget. The name must be unique within accounts. The <code>:</code> and <code>\</code> characters
     * aren't allowed in <code>BudgetName</code>.
     * </p>
     * 
     * @return The name of a budget. The name must be unique within accounts. The <code>:</code> and <code>\</code>
     *         characters aren't allowed in <code>BudgetName</code>.
     */
    public String budgetName() {
        return budgetName;
    }

    /**
     * <p>
     * The total amount of cost, usage, RI utilization, or RI coverage that you want to track with your budget.
     * </p>
     * <p>
     * <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI utilization or coverage
     * budgets. RI utilization or coverage budgets default to <code>100</code>, which is the only valid value for RI
     * utilization or coverage budgets.
     * </p>
     * 
     * @return The total amount of cost, usage, RI utilization, or RI coverage that you want to track with your
     *         budget.</p>
     *         <p>
     *         <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI utilization or
     *         coverage budgets. RI utilization or coverage budgets default to <code>100</code>, which is the only valid
     *         value for RI utilization or coverage budgets.
     */
    public Spend budgetLimit() {
        return budgetLimit;
    }

    /**
     * <p>
     * The cost filters, such as service or region, that are applied to a budget.
     * </p>
     * <p>
     * AWS Budgets supports the following services as a filter for RI budgets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon Elastic Compute Cloud - Compute
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Redshift
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Relational Database Service
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon ElastiCache
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Elasticsearch Service
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The cost filters, such as service or region, that are applied to a budget.</p>
     *         <p>
     *         AWS Budgets supports the following services as a filter for RI budgets:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Amazon Elastic Compute Cloud - Compute
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Redshift
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Relational Database Service
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon ElastiCache
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Elasticsearch Service
     *         </p>
     *         </li>
     */
    public Map<String, List<String>> costFilters() {
        return costFilters;
    }

    /**
     * <p>
     * The types of costs that are included in this <code>COST</code> budget.
     * </p>
     * <p>
     * <code>USAGE</code>, <code>RI_UTILIZATION</code>, and <code>RI_COVERAGE</code> budgets do not have
     * <code>CostTypes</code>.
     * </p>
     * 
     * @return The types of costs that are included in this <code>COST</code> budget.</p>
     *         <p>
     *         <code>USAGE</code>, <code>RI_UTILIZATION</code>, and <code>RI_COVERAGE</code> budgets do not have
     *         <code>CostTypes</code>.
     */
    public CostTypes costTypes() {
        return costTypes;
    }

    /**
     * <p>
     * The length of time until a budget resets the actual and forecasted spend. <code>DAILY</code> is available only
     * for <code>RI_UTILIZATION</code> and <code>RI_COVERAGE</code> budgets.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #timeUnit} will
     * return {@link TimeUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #timeUnitAsString}.
     * </p>
     * 
     * @return The length of time until a budget resets the actual and forecasted spend. <code>DAILY</code> is available
     *         only for <code>RI_UTILIZATION</code> and <code>RI_COVERAGE</code> budgets.
     * @see TimeUnit
     */
    public TimeUnit timeUnit() {
        return TimeUnit.fromValue(timeUnit);
    }

    /**
     * <p>
     * The length of time until a budget resets the actual and forecasted spend. <code>DAILY</code> is available only
     * for <code>RI_UTILIZATION</code> and <code>RI_COVERAGE</code> budgets.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #timeUnit} will
     * return {@link TimeUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #timeUnitAsString}.
     * </p>
     * 
     * @return The length of time until a budget resets the actual and forecasted spend. <code>DAILY</code> is available
     *         only for <code>RI_UTILIZATION</code> and <code>RI_COVERAGE</code> budgets.
     * @see TimeUnit
     */
    public String timeUnitAsString() {
        return timeUnit;
    }

    /**
     * <p>
     * The period of time that is covered by a budget. The period has a start date and an end date. The start date must
     * come before the end date. The end date must come before <code>06/15/87 00:00 UTC</code>.
     * </p>
     * <p>
     * If you create your budget and don't specify a start date, AWS defaults to the start of your chosen time period
     * (DAILY, MONTHLY, QUARTERLY, or ANNUALLY). For example, if you created your budget on January 24, 2018, chose
     * <code>DAILY</code>, and didn't set a start date, AWS set your start date to <code>01/24/18 00:00 UTC</code>. If
     * you chose <code>MONTHLY</code>, AWS set your start date to <code>01/01/18 00:00 UTC</code>. If you didn't specify
     * an end date, AWS set your end date to <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS
     * Billing and Cost Management console and the API.
     * </p>
     * <p>
     * You can change either date with the <code>UpdateBudget</code> operation.
     * </p>
     * <p>
     * After the end date, AWS deletes the budget and all associated notifications and subscribers.
     * </p>
     * 
     * @return The period of time that is covered by a budget. The period has a start date and an end date. The start
     *         date must come before the end date. The end date must come before <code>06/15/87 00:00 UTC</code>. </p>
     *         <p>
     *         If you create your budget and don't specify a start date, AWS defaults to the start of your chosen time
     *         period (DAILY, MONTHLY, QUARTERLY, or ANNUALLY). For example, if you created your budget on January 24,
     *         2018, chose <code>DAILY</code>, and didn't set a start date, AWS set your start date to
     *         <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, AWS set your start date to
     *         <code>01/01/18 00:00 UTC</code>. If you didn't specify an end date, AWS set your end date to
     *         <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost Management
     *         console and the API.
     *         </p>
     *         <p>
     *         You can change either date with the <code>UpdateBudget</code> operation.
     *         </p>
     *         <p>
     *         After the end date, AWS deletes the budget and all associated notifications and subscribers.
     */
    public TimePeriod timePeriod() {
        return timePeriod;
    }

    /**
     * <p>
     * The actual and forecasted cost or usage that the budget tracks.
     * </p>
     * 
     * @return The actual and forecasted cost or usage that the budget tracks.
     */
    public CalculatedSpend calculatedSpend() {
        return calculatedSpend;
    }

    /**
     * <p>
     * Whether this budget tracks monetary costs, usage, RI utilization, or RI coverage.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #budgetType} will
     * return {@link BudgetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #budgetTypeAsString}.
     * </p>
     * 
     * @return Whether this budget tracks monetary costs, usage, RI utilization, or RI coverage.
     * @see BudgetType
     */
    public BudgetType budgetType() {
        return BudgetType.fromValue(budgetType);
    }

    /**
     * <p>
     * Whether this budget tracks monetary costs, usage, RI utilization, or RI coverage.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #budgetType} will
     * return {@link BudgetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #budgetTypeAsString}.
     * </p>
     * 
     * @return Whether this budget tracks monetary costs, usage, RI utilization, or RI coverage.
     * @see BudgetType
     */
    public String budgetTypeAsString() {
        return budgetType;
    }

    /**
     * <p>
     * The last time that you updated this budget.
     * </p>
     * 
     * @return The last time that you updated this budget.
     */
    public Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(budgetName());
        hashCode = 31 * hashCode + Objects.hashCode(budgetLimit());
        hashCode = 31 * hashCode + Objects.hashCode(costFilters());
        hashCode = 31 * hashCode + Objects.hashCode(costTypes());
        hashCode = 31 * hashCode + Objects.hashCode(timeUnitAsString());
        hashCode = 31 * hashCode + Objects.hashCode(timePeriod());
        hashCode = 31 * hashCode + Objects.hashCode(calculatedSpend());
        hashCode = 31 * hashCode + Objects.hashCode(budgetTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Budget)) {
            return false;
        }
        Budget other = (Budget) obj;
        return Objects.equals(budgetName(), other.budgetName()) && Objects.equals(budgetLimit(), other.budgetLimit())
                && Objects.equals(costFilters(), other.costFilters()) && Objects.equals(costTypes(), other.costTypes())
                && Objects.equals(timeUnitAsString(), other.timeUnitAsString())
                && Objects.equals(timePeriod(), other.timePeriod()) && Objects.equals(calculatedSpend(), other.calculatedSpend())
                && Objects.equals(budgetTypeAsString(), other.budgetTypeAsString())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime());
    }

    @Override
    public String toString() {
        return ToString.builder("Budget").add("BudgetName", budgetName()).add("BudgetLimit", budgetLimit())
                .add("CostFilters", costFilters()).add("CostTypes", costTypes()).add("TimeUnit", timeUnitAsString())
                .add("TimePeriod", timePeriod()).add("CalculatedSpend", calculatedSpend())
                .add("BudgetType", budgetTypeAsString()).add("LastUpdatedTime", lastUpdatedTime()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BudgetName":
            return Optional.ofNullable(clazz.cast(budgetName()));
        case "BudgetLimit":
            return Optional.ofNullable(clazz.cast(budgetLimit()));
        case "CostFilters":
            return Optional.ofNullable(clazz.cast(costFilters()));
        case "CostTypes":
            return Optional.ofNullable(clazz.cast(costTypes()));
        case "TimeUnit":
            return Optional.ofNullable(clazz.cast(timeUnitAsString()));
        case "TimePeriod":
            return Optional.ofNullable(clazz.cast(timePeriod()));
        case "CalculatedSpend":
            return Optional.ofNullable(clazz.cast(calculatedSpend()));
        case "BudgetType":
            return Optional.ofNullable(clazz.cast(budgetTypeAsString()));
        case "LastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Budget, T> g) {
        return obj -> g.apply((Budget) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Budget> {
        /**
         * <p>
         * The name of a budget. The name must be unique within accounts. The <code>:</code> and <code>\</code>
         * characters aren't allowed in <code>BudgetName</code>.
         * </p>
         * 
         * @param budgetName
         *        The name of a budget. The name must be unique within accounts. The <code>:</code> and <code>\</code>
         *        characters aren't allowed in <code>BudgetName</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder budgetName(String budgetName);

        /**
         * <p>
         * The total amount of cost, usage, RI utilization, or RI coverage that you want to track with your budget.
         * </p>
         * <p>
         * <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI utilization or coverage
         * budgets. RI utilization or coverage budgets default to <code>100</code>, which is the only valid value for RI
         * utilization or coverage budgets.
         * </p>
         * 
         * @param budgetLimit
         *        The total amount of cost, usage, RI utilization, or RI coverage that you want to track with your
         *        budget.</p>
         *        <p>
         *        <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI utilization or
         *        coverage budgets. RI utilization or coverage budgets default to <code>100</code>, which is the only
         *        valid value for RI utilization or coverage budgets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder budgetLimit(Spend budgetLimit);

        /**
         * <p>
         * The total amount of cost, usage, RI utilization, or RI coverage that you want to track with your budget.
         * </p>
         * <p>
         * <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI utilization or coverage
         * budgets. RI utilization or coverage budgets default to <code>100</code>, which is the only valid value for RI
         * utilization or coverage budgets.
         * </p>
         * This is a convenience that creates an instance of the {@link Spend.Builder} avoiding the need to create one
         * manually via {@link Spend#builder()}.
         *
         * When the {@link Consumer} completes, {@link Spend.Builder#build()} is called immediately and its result is
         * passed to {@link #budgetLimit(Spend)}.
         * 
         * @param budgetLimit
         *        a consumer that will call methods on {@link Spend.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #budgetLimit(Spend)
         */
        default Builder budgetLimit(Consumer<Spend.Builder> budgetLimit) {
            return budgetLimit(Spend.builder().applyMutation(budgetLimit).build());
        }

        /**
         * <p>
         * The cost filters, such as service or region, that are applied to a budget.
         * </p>
         * <p>
         * AWS Budgets supports the following services as a filter for RI budgets:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Amazon Elastic Compute Cloud - Compute
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Redshift
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Relational Database Service
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon ElastiCache
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Elasticsearch Service
         * </p>
         * </li>
         * </ul>
         * 
         * @param costFilters
         *        The cost filters, such as service or region, that are applied to a budget.</p>
         *        <p>
         *        AWS Budgets supports the following services as a filter for RI budgets:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Amazon Elastic Compute Cloud - Compute
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Redshift
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Relational Database Service
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon ElastiCache
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Elasticsearch Service
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder costFilters(Map<String, ? extends Collection<String>> costFilters);

        /**
         * <p>
         * The types of costs that are included in this <code>COST</code> budget.
         * </p>
         * <p>
         * <code>USAGE</code>, <code>RI_UTILIZATION</code>, and <code>RI_COVERAGE</code> budgets do not have
         * <code>CostTypes</code>.
         * </p>
         * 
         * @param costTypes
         *        The types of costs that are included in this <code>COST</code> budget.</p>
         *        <p>
         *        <code>USAGE</code>, <code>RI_UTILIZATION</code>, and <code>RI_COVERAGE</code> budgets do not have
         *        <code>CostTypes</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder costTypes(CostTypes costTypes);

        /**
         * <p>
         * The types of costs that are included in this <code>COST</code> budget.
         * </p>
         * <p>
         * <code>USAGE</code>, <code>RI_UTILIZATION</code>, and <code>RI_COVERAGE</code> budgets do not have
         * <code>CostTypes</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link CostTypes.Builder} avoiding the need to create
         * one manually via {@link CostTypes#builder()}.
         *
         * When the {@link Consumer} completes, {@link CostTypes.Builder#build()} is called immediately and its result
         * is passed to {@link #costTypes(CostTypes)}.
         * 
         * @param costTypes
         *        a consumer that will call methods on {@link CostTypes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #costTypes(CostTypes)
         */
        default Builder costTypes(Consumer<CostTypes.Builder> costTypes) {
            return costTypes(CostTypes.builder().applyMutation(costTypes).build());
        }

        /**
         * <p>
         * The length of time until a budget resets the actual and forecasted spend. <code>DAILY</code> is available
         * only for <code>RI_UTILIZATION</code> and <code>RI_COVERAGE</code> budgets.
         * </p>
         * 
         * @param timeUnit
         *        The length of time until a budget resets the actual and forecasted spend. <code>DAILY</code> is
         *        available only for <code>RI_UTILIZATION</code> and <code>RI_COVERAGE</code> budgets.
         * @see TimeUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TimeUnit
         */
        Builder timeUnit(String timeUnit);

        /**
         * <p>
         * The length of time until a budget resets the actual and forecasted spend. <code>DAILY</code> is available
         * only for <code>RI_UTILIZATION</code> and <code>RI_COVERAGE</code> budgets.
         * </p>
         * 
         * @param timeUnit
         *        The length of time until a budget resets the actual and forecasted spend. <code>DAILY</code> is
         *        available only for <code>RI_UTILIZATION</code> and <code>RI_COVERAGE</code> budgets.
         * @see TimeUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TimeUnit
         */
        Builder timeUnit(TimeUnit timeUnit);

        /**
         * <p>
         * The period of time that is covered by a budget. The period has a start date and an end date. The start date
         * must come before the end date. The end date must come before <code>06/15/87 00:00 UTC</code>.
         * </p>
         * <p>
         * If you create your budget and don't specify a start date, AWS defaults to the start of your chosen time
         * period (DAILY, MONTHLY, QUARTERLY, or ANNUALLY). For example, if you created your budget on January 24, 2018,
         * chose <code>DAILY</code>, and didn't set a start date, AWS set your start date to
         * <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, AWS set your start date to
         * <code>01/01/18 00:00 UTC</code>. If you didn't specify an end date, AWS set your end date to
         * <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost Management console
         * and the API.
         * </p>
         * <p>
         * You can change either date with the <code>UpdateBudget</code> operation.
         * </p>
         * <p>
         * After the end date, AWS deletes the budget and all associated notifications and subscribers.
         * </p>
         * 
         * @param timePeriod
         *        The period of time that is covered by a budget. The period has a start date and an end date. The start
         *        date must come before the end date. The end date must come before <code>06/15/87 00:00 UTC</code>.
         *        </p>
         *        <p>
         *        If you create your budget and don't specify a start date, AWS defaults to the start of your chosen
         *        time period (DAILY, MONTHLY, QUARTERLY, or ANNUALLY). For example, if you created your budget on
         *        January 24, 2018, chose <code>DAILY</code>, and didn't set a start date, AWS set your start date to
         *        <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, AWS set your start date to
         *        <code>01/01/18 00:00 UTC</code>. If you didn't specify an end date, AWS set your end date to
         *        <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost Management
         *        console and the API.
         *        </p>
         *        <p>
         *        You can change either date with the <code>UpdateBudget</code> operation.
         *        </p>
         *        <p>
         *        After the end date, AWS deletes the budget and all associated notifications and subscribers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timePeriod(TimePeriod timePeriod);

        /**
         * <p>
         * The period of time that is covered by a budget. The period has a start date and an end date. The start date
         * must come before the end date. The end date must come before <code>06/15/87 00:00 UTC</code>.
         * </p>
         * <p>
         * If you create your budget and don't specify a start date, AWS defaults to the start of your chosen time
         * period (DAILY, MONTHLY, QUARTERLY, or ANNUALLY). For example, if you created your budget on January 24, 2018,
         * chose <code>DAILY</code>, and didn't set a start date, AWS set your start date to
         * <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, AWS set your start date to
         * <code>01/01/18 00:00 UTC</code>. If you didn't specify an end date, AWS set your end date to
         * <code>06/15/87 00:00 UTC</code>. The defaults are the same for the AWS Billing and Cost Management console
         * and the API.
         * </p>
         * <p>
         * You can change either date with the <code>UpdateBudget</code> operation.
         * </p>
         * <p>
         * After the end date, AWS deletes the budget and all associated notifications and subscribers.
         * </p>
         * This is a convenience that creates an instance of the {@link TimePeriod.Builder} avoiding the need to create
         * one manually via {@link TimePeriod#builder()}.
         *
         * When the {@link Consumer} completes, {@link TimePeriod.Builder#build()} is called immediately and its result
         * is passed to {@link #timePeriod(TimePeriod)}.
         * 
         * @param timePeriod
         *        a consumer that will call methods on {@link TimePeriod.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #timePeriod(TimePeriod)
         */
        default Builder timePeriod(Consumer<TimePeriod.Builder> timePeriod) {
            return timePeriod(TimePeriod.builder().applyMutation(timePeriod).build());
        }

        /**
         * <p>
         * The actual and forecasted cost or usage that the budget tracks.
         * </p>
         * 
         * @param calculatedSpend
         *        The actual and forecasted cost or usage that the budget tracks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder calculatedSpend(CalculatedSpend calculatedSpend);

        /**
         * <p>
         * The actual and forecasted cost or usage that the budget tracks.
         * </p>
         * This is a convenience that creates an instance of the {@link CalculatedSpend.Builder} avoiding the need to
         * create one manually via {@link CalculatedSpend#builder()}.
         *
         * When the {@link Consumer} completes, {@link CalculatedSpend.Builder#build()} is called immediately and its
         * result is passed to {@link #calculatedSpend(CalculatedSpend)}.
         * 
         * @param calculatedSpend
         *        a consumer that will call methods on {@link CalculatedSpend.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #calculatedSpend(CalculatedSpend)
         */
        default Builder calculatedSpend(Consumer<CalculatedSpend.Builder> calculatedSpend) {
            return calculatedSpend(CalculatedSpend.builder().applyMutation(calculatedSpend).build());
        }

        /**
         * <p>
         * Whether this budget tracks monetary costs, usage, RI utilization, or RI coverage.
         * </p>
         * 
         * @param budgetType
         *        Whether this budget tracks monetary costs, usage, RI utilization, or RI coverage.
         * @see BudgetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BudgetType
         */
        Builder budgetType(String budgetType);

        /**
         * <p>
         * Whether this budget tracks monetary costs, usage, RI utilization, or RI coverage.
         * </p>
         * 
         * @param budgetType
         *        Whether this budget tracks monetary costs, usage, RI utilization, or RI coverage.
         * @see BudgetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BudgetType
         */
        Builder budgetType(BudgetType budgetType);

        /**
         * <p>
         * The last time that you updated this budget.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The last time that you updated this budget.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);
    }

    static final class BuilderImpl implements Builder {
        private String budgetName;

        private Spend budgetLimit;

        private Map<String, List<String>> costFilters = DefaultSdkAutoConstructMap.getInstance();

        private CostTypes costTypes;

        private String timeUnit;

        private TimePeriod timePeriod;

        private CalculatedSpend calculatedSpend;

        private String budgetType;

        private Instant lastUpdatedTime;

        private BuilderImpl() {
        }

        private BuilderImpl(Budget model) {
            budgetName(model.budgetName);
            budgetLimit(model.budgetLimit);
            costFilters(model.costFilters);
            costTypes(model.costTypes);
            timeUnit(model.timeUnit);
            timePeriod(model.timePeriod);
            calculatedSpend(model.calculatedSpend);
            budgetType(model.budgetType);
            lastUpdatedTime(model.lastUpdatedTime);
        }

        public final String getBudgetName() {
            return budgetName;
        }

        @Override
        public final Builder budgetName(String budgetName) {
            this.budgetName = budgetName;
            return this;
        }

        public final void setBudgetName(String budgetName) {
            this.budgetName = budgetName;
        }

        public final Spend.Builder getBudgetLimit() {
            return budgetLimit != null ? budgetLimit.toBuilder() : null;
        }

        @Override
        public final Builder budgetLimit(Spend budgetLimit) {
            this.budgetLimit = budgetLimit;
            return this;
        }

        public final void setBudgetLimit(Spend.BuilderImpl budgetLimit) {
            this.budgetLimit = budgetLimit != null ? budgetLimit.build() : null;
        }

        public final Map<String, ? extends Collection<String>> getCostFilters() {
            return costFilters;
        }

        @Override
        public final Builder costFilters(Map<String, ? extends Collection<String>> costFilters) {
            this.costFilters = CostFiltersCopier.copy(costFilters);
            return this;
        }

        public final void setCostFilters(Map<String, ? extends Collection<String>> costFilters) {
            this.costFilters = CostFiltersCopier.copy(costFilters);
        }

        public final CostTypes.Builder getCostTypes() {
            return costTypes != null ? costTypes.toBuilder() : null;
        }

        @Override
        public final Builder costTypes(CostTypes costTypes) {
            this.costTypes = costTypes;
            return this;
        }

        public final void setCostTypes(CostTypes.BuilderImpl costTypes) {
            this.costTypes = costTypes != null ? costTypes.build() : null;
        }

        public final String getTimeUnitAsString() {
            return timeUnit;
        }

        @Override
        public final Builder timeUnit(String timeUnit) {
            this.timeUnit = timeUnit;
            return this;
        }

        @Override
        public final Builder timeUnit(TimeUnit timeUnit) {
            this.timeUnit(timeUnit.toString());
            return this;
        }

        public final void setTimeUnit(String timeUnit) {
            this.timeUnit = timeUnit;
        }

        public final TimePeriod.Builder getTimePeriod() {
            return timePeriod != null ? timePeriod.toBuilder() : null;
        }

        @Override
        public final Builder timePeriod(TimePeriod timePeriod) {
            this.timePeriod = timePeriod;
            return this;
        }

        public final void setTimePeriod(TimePeriod.BuilderImpl timePeriod) {
            this.timePeriod = timePeriod != null ? timePeriod.build() : null;
        }

        public final CalculatedSpend.Builder getCalculatedSpend() {
            return calculatedSpend != null ? calculatedSpend.toBuilder() : null;
        }

        @Override
        public final Builder calculatedSpend(CalculatedSpend calculatedSpend) {
            this.calculatedSpend = calculatedSpend;
            return this;
        }

        public final void setCalculatedSpend(CalculatedSpend.BuilderImpl calculatedSpend) {
            this.calculatedSpend = calculatedSpend != null ? calculatedSpend.build() : null;
        }

        public final String getBudgetTypeAsString() {
            return budgetType;
        }

        @Override
        public final Builder budgetType(String budgetType) {
            this.budgetType = budgetType;
            return this;
        }

        @Override
        public final Builder budgetType(BudgetType budgetType) {
            this.budgetType(budgetType.toString());
            return this;
        }

        public final void setBudgetType(String budgetType) {
            this.budgetType = budgetType;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public Budget build() {
            return new Budget(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
