/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.budgets.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the output of the <code>CreateBudget</code> operation. The content consists of the detailed metadata and
 * data file information, and the current status of the <code>budget</code> object.
 * </p>
 * <p>
 * This is the Amazon Resource Name (ARN) pattern for a budget:
 * </p>
 * <p>
 * <code>arn:aws:budgets::AccountId:budget/budgetName</code>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Budget implements SdkPojo, Serializable, ToCopyableBuilder<Budget.Builder, Budget> {
    private static final SdkField<String> BUDGET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BudgetName").getter(getter(Budget::budgetName)).setter(setter(Builder::budgetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BudgetName").build()).build();

    private static final SdkField<Spend> BUDGET_LIMIT_FIELD = SdkField.<Spend> builder(MarshallingType.SDK_POJO)
            .memberName("BudgetLimit").getter(getter(Budget::budgetLimit)).setter(setter(Builder::budgetLimit))
            .constructor(Spend::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BudgetLimit").build()).build();

    private static final SdkField<Map<String, Spend>> PLANNED_BUDGET_LIMITS_FIELD = SdkField
            .<Map<String, Spend>> builder(MarshallingType.MAP)
            .memberName("PlannedBudgetLimits")
            .getter(getter(Budget::plannedBudgetLimits))
            .setter(setter(Builder::plannedBudgetLimits))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlannedBudgetLimits").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Spend> builder(MarshallingType.SDK_POJO)
                                            .constructor(Spend::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Map<String, List<String>>> COST_FILTERS_FIELD = SdkField
            .<Map<String, List<String>>> builder(MarshallingType.MAP)
            .memberName("CostFilters")
            .getter(getter(Budget::costFilters))
            .setter(setter(Builder::costFilters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CostFilters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<List<String>> builder(MarshallingType.LIST)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build(),
                                                    ListTrait
                                                            .builder()
                                                            .memberLocationName(null)
                                                            .memberFieldInfo(
                                                                    SdkField.<String> builder(MarshallingType.STRING)
                                                                            .traits(LocationTrait.builder()
                                                                                    .location(MarshallLocation.PAYLOAD)
                                                                                    .locationName("member").build()).build())
                                                            .build()).build()).build()).build();

    private static final SdkField<CostTypes> COST_TYPES_FIELD = SdkField.<CostTypes> builder(MarshallingType.SDK_POJO)
            .memberName("CostTypes").getter(getter(Budget::costTypes)).setter(setter(Builder::costTypes))
            .constructor(CostTypes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CostTypes").build()).build();

    private static final SdkField<String> TIME_UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TimeUnit").getter(getter(Budget::timeUnitAsString)).setter(setter(Builder::timeUnit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeUnit").build()).build();

    private static final SdkField<TimePeriod> TIME_PERIOD_FIELD = SdkField.<TimePeriod> builder(MarshallingType.SDK_POJO)
            .memberName("TimePeriod").getter(getter(Budget::timePeriod)).setter(setter(Builder::timePeriod))
            .constructor(TimePeriod::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimePeriod").build()).build();

    private static final SdkField<CalculatedSpend> CALCULATED_SPEND_FIELD = SdkField
            .<CalculatedSpend> builder(MarshallingType.SDK_POJO).memberName("CalculatedSpend")
            .getter(getter(Budget::calculatedSpend)).setter(setter(Builder::calculatedSpend))
            .constructor(CalculatedSpend::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CalculatedSpend").build()).build();

    private static final SdkField<String> BUDGET_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BudgetType").getter(getter(Budget::budgetTypeAsString)).setter(setter(Builder::budgetType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BudgetType").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedTime").getter(getter(Budget::lastUpdatedTime)).setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedTime").build()).build();

    private static final SdkField<AutoAdjustData> AUTO_ADJUST_DATA_FIELD = SdkField
            .<AutoAdjustData> builder(MarshallingType.SDK_POJO).memberName("AutoAdjustData")
            .getter(getter(Budget::autoAdjustData)).setter(setter(Builder::autoAdjustData)).constructor(AutoAdjustData::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoAdjustData").build()).build();

    private static final SdkField<Expression> FILTER_EXPRESSION_FIELD = SdkField.<Expression> builder(MarshallingType.SDK_POJO)
            .memberName("FilterExpression").getter(getter(Budget::filterExpression)).setter(setter(Builder::filterExpression))
            .constructor(Expression::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FilterExpression").build()).build();

    private static final SdkField<List<String>> METRICS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Metrics")
            .getter(getter(Budget::metricsAsStrings))
            .setter(setter(Builder::metricsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Metrics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUDGET_NAME_FIELD,
            BUDGET_LIMIT_FIELD, PLANNED_BUDGET_LIMITS_FIELD, COST_FILTERS_FIELD, COST_TYPES_FIELD, TIME_UNIT_FIELD,
            TIME_PERIOD_FIELD, CALCULATED_SPEND_FIELD, BUDGET_TYPE_FIELD, LAST_UPDATED_TIME_FIELD, AUTO_ADJUST_DATA_FIELD,
            FILTER_EXPRESSION_FIELD, METRICS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String budgetName;

    private final Spend budgetLimit;

    private final Map<String, Spend> plannedBudgetLimits;

    private final Map<String, List<String>> costFilters;

    private final CostTypes costTypes;

    private final String timeUnit;

    private final TimePeriod timePeriod;

    private final CalculatedSpend calculatedSpend;

    private final String budgetType;

    private final Instant lastUpdatedTime;

    private final AutoAdjustData autoAdjustData;

    private final Expression filterExpression;

    private final List<String> metrics;

    private Budget(BuilderImpl builder) {
        this.budgetName = builder.budgetName;
        this.budgetLimit = builder.budgetLimit;
        this.plannedBudgetLimits = builder.plannedBudgetLimits;
        this.costFilters = builder.costFilters;
        this.costTypes = builder.costTypes;
        this.timeUnit = builder.timeUnit;
        this.timePeriod = builder.timePeriod;
        this.calculatedSpend = builder.calculatedSpend;
        this.budgetType = builder.budgetType;
        this.lastUpdatedTime = builder.lastUpdatedTime;
        this.autoAdjustData = builder.autoAdjustData;
        this.filterExpression = builder.filterExpression;
        this.metrics = builder.metrics;
    }

    /**
     * <p>
     * The name of a budget. The name must be unique within an account. The <code>:</code> and <code>\</code>
     * characters, and the "/action/" substring, aren't allowed in <code>BudgetName</code>.
     * </p>
     * 
     * @return The name of a budget. The name must be unique within an account. The <code>:</code> and <code>\</code>
     *         characters, and the "/action/" substring, aren't allowed in <code>BudgetName</code>.
     */
    public final String budgetName() {
        return budgetName;
    }

    /**
     * <p>
     * The total amount of cost, usage, RI utilization, RI coverage, Savings Plans utilization, or Savings Plans
     * coverage that you want to track with your budget.
     * </p>
     * <p>
     * <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI or Savings Plans utilization
     * or coverage budgets. RI and Savings Plans utilization or coverage budgets default to <code>100</code>. This is
     * the only valid value for RI or Savings Plans utilization or coverage budgets. You can't use
     * <code>BudgetLimit</code> with <code>PlannedBudgetLimits</code> for <code>CreateBudget</code> and
     * <code>UpdateBudget</code> actions.
     * </p>
     * 
     * @return The total amount of cost, usage, RI utilization, RI coverage, Savings Plans utilization, or Savings Plans
     *         coverage that you want to track with your budget.</p>
     *         <p>
     *         <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI or Savings Plans
     *         utilization or coverage budgets. RI and Savings Plans utilization or coverage budgets default to
     *         <code>100</code>. This is the only valid value for RI or Savings Plans utilization or coverage budgets.
     *         You can't use <code>BudgetLimit</code> with <code>PlannedBudgetLimits</code> for
     *         <code>CreateBudget</code> and <code>UpdateBudget</code> actions.
     */
    public final Spend budgetLimit() {
        return budgetLimit;
    }

    /**
     * For responses, this returns true if the service returned a value for the PlannedBudgetLimits property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasPlannedBudgetLimits() {
        return plannedBudgetLimits != null && !(plannedBudgetLimits instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map containing multiple <code>BudgetLimit</code>, including current or future limits.
     * </p>
     * <p>
     * <code>PlannedBudgetLimits</code> is available for cost or usage budget and supports both monthly and quarterly
     * <code>TimeUnit</code>.
     * </p>
     * <p>
     * For monthly budgets, provide 12 months of <code>PlannedBudgetLimits</code> values. This must start from the
     * current month and include the next 11 months. The <code>key</code> is the start of the month, <code>UTC</code> in
     * epoch seconds.
     * </p>
     * <p>
     * For quarterly budgets, provide four quarters of <code>PlannedBudgetLimits</code> value entries in standard
     * calendar quarter increments. This must start from the current quarter and include the next three quarters. The
     * <code>key</code> is the start of the quarter, <code>UTC</code> in epoch seconds.
     * </p>
     * <p>
     * If the planned budget expires before 12 months for monthly or four quarters for quarterly, provide the
     * <code>PlannedBudgetLimits</code> values only for the remaining periods.
     * </p>
     * <p>
     * If the budget begins at a date in the future, provide <code>PlannedBudgetLimits</code> values from the start date
     * of the budget.
     * </p>
     * <p>
     * After all of the <code>BudgetLimit</code> values in <code>PlannedBudgetLimits</code> are used, the budget
     * continues to use the last limit as the <code>BudgetLimit</code>. At that point, the planned budget provides the
     * same experience as a fixed budget.
     * </p>
     * <p>
     * <code>DescribeBudget</code> and <code>DescribeBudgets</code> response along with <code>PlannedBudgetLimits</code>
     * also contain <code>BudgetLimit</code> representing the current month or quarter limit present in
     * <code>PlannedBudgetLimits</code>. This only applies to budgets that are created with
     * <code>PlannedBudgetLimits</code>. Budgets that are created without <code>PlannedBudgetLimits</code> only contain
     * <code>BudgetLimit</code>. They don't contain <code>PlannedBudgetLimits</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPlannedBudgetLimits} method.
     * </p>
     * 
     * @return A map containing multiple <code>BudgetLimit</code>, including current or future limits.</p>
     *         <p>
     *         <code>PlannedBudgetLimits</code> is available for cost or usage budget and supports both monthly and
     *         quarterly <code>TimeUnit</code>.
     *         </p>
     *         <p>
     *         For monthly budgets, provide 12 months of <code>PlannedBudgetLimits</code> values. This must start from
     *         the current month and include the next 11 months. The <code>key</code> is the start of the month,
     *         <code>UTC</code> in epoch seconds.
     *         </p>
     *         <p>
     *         For quarterly budgets, provide four quarters of <code>PlannedBudgetLimits</code> value entries in
     *         standard calendar quarter increments. This must start from the current quarter and include the next three
     *         quarters. The <code>key</code> is the start of the quarter, <code>UTC</code> in epoch seconds.
     *         </p>
     *         <p>
     *         If the planned budget expires before 12 months for monthly or four quarters for quarterly, provide the
     *         <code>PlannedBudgetLimits</code> values only for the remaining periods.
     *         </p>
     *         <p>
     *         If the budget begins at a date in the future, provide <code>PlannedBudgetLimits</code> values from the
     *         start date of the budget.
     *         </p>
     *         <p>
     *         After all of the <code>BudgetLimit</code> values in <code>PlannedBudgetLimits</code> are used, the budget
     *         continues to use the last limit as the <code>BudgetLimit</code>. At that point, the planned budget
     *         provides the same experience as a fixed budget.
     *         </p>
     *         <p>
     *         <code>DescribeBudget</code> and <code>DescribeBudgets</code> response along with
     *         <code>PlannedBudgetLimits</code> also contain <code>BudgetLimit</code> representing the current month or
     *         quarter limit present in <code>PlannedBudgetLimits</code>. This only applies to budgets that are created
     *         with <code>PlannedBudgetLimits</code>. Budgets that are created without <code>PlannedBudgetLimits</code>
     *         only contain <code>BudgetLimit</code>. They don't contain <code>PlannedBudgetLimits</code>.
     */
    public final Map<String, Spend> plannedBudgetLimits() {
        return plannedBudgetLimits;
    }

    /**
     * For responses, this returns true if the service returned a value for the CostFilters property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     *
     * @deprecated CostFilters lack support for newer dimensions and filtering options. Please consider using the new
     *             'FilterExpression' field.
     */
    @Deprecated
    public final boolean hasCostFilters() {
        return costFilters != null && !(costFilters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The cost filters, such as <code>Region</code>, <code>Service</code>, <code>LinkedAccount</code>, <code>Tag</code>
     * , or <code>CostCategory</code>, that are applied to a budget.
     * </p>
     * <p>
     * Amazon Web Services Budgets supports the following services as a <code>Service</code> filter for RI budgets:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon EC2
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Redshift
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Relational Database Service
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon ElastiCache
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon OpenSearch Service
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCostFilters} method.
     * </p>
     * 
     * @return The cost filters, such as <code>Region</code>, <code>Service</code>, <code>LinkedAccount</code>,
     *         <code>Tag</code>, or <code>CostCategory</code>, that are applied to a budget.</p>
     *         <p>
     *         Amazon Web Services Budgets supports the following services as a <code>Service</code> filter for RI
     *         budgets:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Amazon EC2
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Redshift
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Relational Database Service
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon ElastiCache
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon OpenSearch Service
     *         </p>
     *         </li>
     * @deprecated CostFilters lack support for newer dimensions and filtering options. Please consider using the new
     *             'FilterExpression' field.
     */
    @Deprecated
    public final Map<String, List<String>> costFilters() {
        return costFilters;
    }

    /**
     * <p>
     * The types of costs that are included in this <code>COST</code> budget.
     * </p>
     * <p>
     * <code>USAGE</code>, <code>RI_UTILIZATION</code>, <code>RI_COVERAGE</code>, <code>SAVINGS_PLANS_UTILIZATION</code>
     * , and <code>SAVINGS_PLANS_COVERAGE</code> budgets do not have <code>CostTypes</code>.
     * </p>
     * 
     * @return The types of costs that are included in this <code>COST</code> budget.</p>
     *         <p>
     *         <code>USAGE</code>, <code>RI_UTILIZATION</code>, <code>RI_COVERAGE</code>,
     *         <code>SAVINGS_PLANS_UTILIZATION</code>, and <code>SAVINGS_PLANS_COVERAGE</code> budgets do not have
     *         <code>CostTypes</code>.
     * @deprecated CostTypes lack support for newer record type dimensions and filtering options. Please consider using
     *             the new 'Metrics' field.
     */
    @Deprecated
    public final CostTypes costTypes() {
        return costTypes;
    }

    /**
     * <p>
     * The length of time until a budget resets the actual and forecasted spend.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #timeUnit} will
     * return {@link TimeUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #timeUnitAsString}.
     * </p>
     * 
     * @return The length of time until a budget resets the actual and forecasted spend.
     * @see TimeUnit
     */
    public final TimeUnit timeUnit() {
        return TimeUnit.fromValue(timeUnit);
    }

    /**
     * <p>
     * The length of time until a budget resets the actual and forecasted spend.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #timeUnit} will
     * return {@link TimeUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #timeUnitAsString}.
     * </p>
     * 
     * @return The length of time until a budget resets the actual and forecasted spend.
     * @see TimeUnit
     */
    public final String timeUnitAsString() {
        return timeUnit;
    }

    /**
     * <p>
     * The period of time that's covered by a budget. You set the start date and end date. The start date must come
     * before the end date. The end date must come before <code>06/15/87 00:00 UTC</code>.
     * </p>
     * <p>
     * If you create your budget and don't specify a start date, Amazon Web Services defaults to the start of your
     * chosen time period (DAILY, MONTHLY, QUARTERLY, or ANNUALLY). For example, if you created your budget on January
     * 24, 2018, chose <code>DAILY</code>, and didn't set a start date, Amazon Web Services set your start date to
     * <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, Amazon Web Services set your start date to
     * <code>01/01/18 00:00 UTC</code>. If you didn't specify an end date, Amazon Web Services set your end date to
     * <code>06/15/87 00:00 UTC</code>. The defaults are the same for the Billing and Cost Management console and the
     * API.
     * </p>
     * <p>
     * You can change either date with the <code>UpdateBudget</code> operation.
     * </p>
     * <p>
     * After the end date, Amazon Web Services deletes the budget and all the associated notifications and subscribers.
     * </p>
     * 
     * @return The period of time that's covered by a budget. You set the start date and end date. The start date must
     *         come before the end date. The end date must come before <code>06/15/87 00:00 UTC</code>. </p>
     *         <p>
     *         If you create your budget and don't specify a start date, Amazon Web Services defaults to the start of
     *         your chosen time period (DAILY, MONTHLY, QUARTERLY, or ANNUALLY). For example, if you created your budget
     *         on January 24, 2018, chose <code>DAILY</code>, and didn't set a start date, Amazon Web Services set your
     *         start date to <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, Amazon Web Services set
     *         your start date to <code>01/01/18 00:00 UTC</code>. If you didn't specify an end date, Amazon Web
     *         Services set your end date to <code>06/15/87 00:00 UTC</code>. The defaults are the same for the Billing
     *         and Cost Management console and the API.
     *         </p>
     *         <p>
     *         You can change either date with the <code>UpdateBudget</code> operation.
     *         </p>
     *         <p>
     *         After the end date, Amazon Web Services deletes the budget and all the associated notifications and
     *         subscribers.
     */
    public final TimePeriod timePeriod() {
        return timePeriod;
    }

    /**
     * <p>
     * The actual and forecasted cost or usage that the budget tracks.
     * </p>
     * 
     * @return The actual and forecasted cost or usage that the budget tracks.
     */
    public final CalculatedSpend calculatedSpend() {
        return calculatedSpend;
    }

    /**
     * <p>
     * Specifies whether this budget tracks costs, usage, RI utilization, RI coverage, Savings Plans utilization, or
     * Savings Plans coverage.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #budgetType} will
     * return {@link BudgetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #budgetTypeAsString}.
     * </p>
     * 
     * @return Specifies whether this budget tracks costs, usage, RI utilization, RI coverage, Savings Plans
     *         utilization, or Savings Plans coverage.
     * @see BudgetType
     */
    public final BudgetType budgetType() {
        return BudgetType.fromValue(budgetType);
    }

    /**
     * <p>
     * Specifies whether this budget tracks costs, usage, RI utilization, RI coverage, Savings Plans utilization, or
     * Savings Plans coverage.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #budgetType} will
     * return {@link BudgetType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #budgetTypeAsString}.
     * </p>
     * 
     * @return Specifies whether this budget tracks costs, usage, RI utilization, RI coverage, Savings Plans
     *         utilization, or Savings Plans coverage.
     * @see BudgetType
     */
    public final String budgetTypeAsString() {
        return budgetType;
    }

    /**
     * <p>
     * The last time that you updated this budget.
     * </p>
     * 
     * @return The last time that you updated this budget.
     */
    public final Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    /**
     * <p>
     * The parameters that determine the budget amount for an auto-adjusting budget.
     * </p>
     * 
     * @return The parameters that determine the budget amount for an auto-adjusting budget.
     */
    public final AutoAdjustData autoAdjustData() {
        return autoAdjustData;
    }

    /**
     * <p>
     * The filtering dimensions for the budget and their corresponding values.
     * </p>
     * 
     * @return The filtering dimensions for the budget and their corresponding values.
     */
    public final Expression filterExpression() {
        return filterExpression;
    }

    /**
     * <p>
     * The definition for how the budget data is aggregated.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetrics} method.
     * </p>
     * 
     * @return The definition for how the budget data is aggregated.
     */
    public final List<Metric> metrics() {
        return MetricsCopier.copyStringToEnum(metrics);
    }

    /**
     * For responses, this returns true if the service returned a value for the Metrics property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasMetrics() {
        return metrics != null && !(metrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The definition for how the budget data is aggregated.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetrics} method.
     * </p>
     * 
     * @return The definition for how the budget data is aggregated.
     */
    public final List<String> metricsAsStrings() {
        return metrics;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(budgetName());
        hashCode = 31 * hashCode + Objects.hashCode(budgetLimit());
        hashCode = 31 * hashCode + Objects.hashCode(hasPlannedBudgetLimits() ? plannedBudgetLimits() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasCostFilters() ? costFilters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(costTypes());
        hashCode = 31 * hashCode + Objects.hashCode(timeUnitAsString());
        hashCode = 31 * hashCode + Objects.hashCode(timePeriod());
        hashCode = 31 * hashCode + Objects.hashCode(calculatedSpend());
        hashCode = 31 * hashCode + Objects.hashCode(budgetTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(autoAdjustData());
        hashCode = 31 * hashCode + Objects.hashCode(filterExpression());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetrics() ? metricsAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Budget)) {
            return false;
        }
        Budget other = (Budget) obj;
        return Objects.equals(budgetName(), other.budgetName()) && Objects.equals(budgetLimit(), other.budgetLimit())
                && hasPlannedBudgetLimits() == other.hasPlannedBudgetLimits()
                && Objects.equals(plannedBudgetLimits(), other.plannedBudgetLimits())
                && hasCostFilters() == other.hasCostFilters() && Objects.equals(costFilters(), other.costFilters())
                && Objects.equals(costTypes(), other.costTypes()) && Objects.equals(timeUnitAsString(), other.timeUnitAsString())
                && Objects.equals(timePeriod(), other.timePeriod()) && Objects.equals(calculatedSpend(), other.calculatedSpend())
                && Objects.equals(budgetTypeAsString(), other.budgetTypeAsString())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime())
                && Objects.equals(autoAdjustData(), other.autoAdjustData())
                && Objects.equals(filterExpression(), other.filterExpression()) && hasMetrics() == other.hasMetrics()
                && Objects.equals(metricsAsStrings(), other.metricsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Budget").add("BudgetName", budgetName()).add("BudgetLimit", budgetLimit())
                .add("PlannedBudgetLimits", hasPlannedBudgetLimits() ? plannedBudgetLimits() : null)
                .add("CostFilters", hasCostFilters() ? costFilters() : null).add("CostTypes", costTypes())
                .add("TimeUnit", timeUnitAsString()).add("TimePeriod", timePeriod()).add("CalculatedSpend", calculatedSpend())
                .add("BudgetType", budgetTypeAsString()).add("LastUpdatedTime", lastUpdatedTime())
                .add("AutoAdjustData", autoAdjustData()).add("FilterExpression", filterExpression())
                .add("Metrics", hasMetrics() ? metricsAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BudgetName":
            return Optional.ofNullable(clazz.cast(budgetName()));
        case "BudgetLimit":
            return Optional.ofNullable(clazz.cast(budgetLimit()));
        case "PlannedBudgetLimits":
            return Optional.ofNullable(clazz.cast(plannedBudgetLimits()));
        case "CostFilters":
            return Optional.ofNullable(clazz.cast(costFilters()));
        case "CostTypes":
            return Optional.ofNullable(clazz.cast(costTypes()));
        case "TimeUnit":
            return Optional.ofNullable(clazz.cast(timeUnitAsString()));
        case "TimePeriod":
            return Optional.ofNullable(clazz.cast(timePeriod()));
        case "CalculatedSpend":
            return Optional.ofNullable(clazz.cast(calculatedSpend()));
        case "BudgetType":
            return Optional.ofNullable(clazz.cast(budgetTypeAsString()));
        case "LastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        case "AutoAdjustData":
            return Optional.ofNullable(clazz.cast(autoAdjustData()));
        case "FilterExpression":
            return Optional.ofNullable(clazz.cast(filterExpression()));
        case "Metrics":
            return Optional.ofNullable(clazz.cast(metricsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("BudgetName", BUDGET_NAME_FIELD);
        map.put("BudgetLimit", BUDGET_LIMIT_FIELD);
        map.put("PlannedBudgetLimits", PLANNED_BUDGET_LIMITS_FIELD);
        map.put("CostFilters", COST_FILTERS_FIELD);
        map.put("CostTypes", COST_TYPES_FIELD);
        map.put("TimeUnit", TIME_UNIT_FIELD);
        map.put("TimePeriod", TIME_PERIOD_FIELD);
        map.put("CalculatedSpend", CALCULATED_SPEND_FIELD);
        map.put("BudgetType", BUDGET_TYPE_FIELD);
        map.put("LastUpdatedTime", LAST_UPDATED_TIME_FIELD);
        map.put("AutoAdjustData", AUTO_ADJUST_DATA_FIELD);
        map.put("FilterExpression", FILTER_EXPRESSION_FIELD);
        map.put("Metrics", METRICS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Budget, T> g) {
        return obj -> g.apply((Budget) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Budget> {
        /**
         * <p>
         * The name of a budget. The name must be unique within an account. The <code>:</code> and <code>\</code>
         * characters, and the "/action/" substring, aren't allowed in <code>BudgetName</code>.
         * </p>
         * 
         * @param budgetName
         *        The name of a budget. The name must be unique within an account. The <code>:</code> and <code>\</code>
         *        characters, and the "/action/" substring, aren't allowed in <code>BudgetName</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder budgetName(String budgetName);

        /**
         * <p>
         * The total amount of cost, usage, RI utilization, RI coverage, Savings Plans utilization, or Savings Plans
         * coverage that you want to track with your budget.
         * </p>
         * <p>
         * <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI or Savings Plans
         * utilization or coverage budgets. RI and Savings Plans utilization or coverage budgets default to
         * <code>100</code>. This is the only valid value for RI or Savings Plans utilization or coverage budgets. You
         * can't use <code>BudgetLimit</code> with <code>PlannedBudgetLimits</code> for <code>CreateBudget</code> and
         * <code>UpdateBudget</code> actions.
         * </p>
         * 
         * @param budgetLimit
         *        The total amount of cost, usage, RI utilization, RI coverage, Savings Plans utilization, or Savings
         *        Plans coverage that you want to track with your budget.</p>
         *        <p>
         *        <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI or Savings Plans
         *        utilization or coverage budgets. RI and Savings Plans utilization or coverage budgets default to
         *        <code>100</code>. This is the only valid value for RI or Savings Plans utilization or coverage
         *        budgets. You can't use <code>BudgetLimit</code> with <code>PlannedBudgetLimits</code> for
         *        <code>CreateBudget</code> and <code>UpdateBudget</code> actions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder budgetLimit(Spend budgetLimit);

        /**
         * <p>
         * The total amount of cost, usage, RI utilization, RI coverage, Savings Plans utilization, or Savings Plans
         * coverage that you want to track with your budget.
         * </p>
         * <p>
         * <code>BudgetLimit</code> is required for cost or usage budgets, but optional for RI or Savings Plans
         * utilization or coverage budgets. RI and Savings Plans utilization or coverage budgets default to
         * <code>100</code>. This is the only valid value for RI or Savings Plans utilization or coverage budgets. You
         * can't use <code>BudgetLimit</code> with <code>PlannedBudgetLimits</code> for <code>CreateBudget</code> and
         * <code>UpdateBudget</code> actions.
         * </p>
         * This is a convenience method that creates an instance of the {@link Spend.Builder} avoiding the need to
         * create one manually via {@link Spend#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Spend.Builder#build()} is called immediately and its result is
         * passed to {@link #budgetLimit(Spend)}.
         * 
         * @param budgetLimit
         *        a consumer that will call methods on {@link Spend.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #budgetLimit(Spend)
         */
        default Builder budgetLimit(Consumer<Spend.Builder> budgetLimit) {
            return budgetLimit(Spend.builder().applyMutation(budgetLimit).build());
        }

        /**
         * <p>
         * A map containing multiple <code>BudgetLimit</code>, including current or future limits.
         * </p>
         * <p>
         * <code>PlannedBudgetLimits</code> is available for cost or usage budget and supports both monthly and
         * quarterly <code>TimeUnit</code>.
         * </p>
         * <p>
         * For monthly budgets, provide 12 months of <code>PlannedBudgetLimits</code> values. This must start from the
         * current month and include the next 11 months. The <code>key</code> is the start of the month,
         * <code>UTC</code> in epoch seconds.
         * </p>
         * <p>
         * For quarterly budgets, provide four quarters of <code>PlannedBudgetLimits</code> value entries in standard
         * calendar quarter increments. This must start from the current quarter and include the next three quarters.
         * The <code>key</code> is the start of the quarter, <code>UTC</code> in epoch seconds.
         * </p>
         * <p>
         * If the planned budget expires before 12 months for monthly or four quarters for quarterly, provide the
         * <code>PlannedBudgetLimits</code> values only for the remaining periods.
         * </p>
         * <p>
         * If the budget begins at a date in the future, provide <code>PlannedBudgetLimits</code> values from the start
         * date of the budget.
         * </p>
         * <p>
         * After all of the <code>BudgetLimit</code> values in <code>PlannedBudgetLimits</code> are used, the budget
         * continues to use the last limit as the <code>BudgetLimit</code>. At that point, the planned budget provides
         * the same experience as a fixed budget.
         * </p>
         * <p>
         * <code>DescribeBudget</code> and <code>DescribeBudgets</code> response along with
         * <code>PlannedBudgetLimits</code> also contain <code>BudgetLimit</code> representing the current month or
         * quarter limit present in <code>PlannedBudgetLimits</code>. This only applies to budgets that are created with
         * <code>PlannedBudgetLimits</code>. Budgets that are created without <code>PlannedBudgetLimits</code> only
         * contain <code>BudgetLimit</code>. They don't contain <code>PlannedBudgetLimits</code>.
         * </p>
         * 
         * @param plannedBudgetLimits
         *        A map containing multiple <code>BudgetLimit</code>, including current or future limits.</p>
         *        <p>
         *        <code>PlannedBudgetLimits</code> is available for cost or usage budget and supports both monthly and
         *        quarterly <code>TimeUnit</code>.
         *        </p>
         *        <p>
         *        For monthly budgets, provide 12 months of <code>PlannedBudgetLimits</code> values. This must start
         *        from the current month and include the next 11 months. The <code>key</code> is the start of the month,
         *        <code>UTC</code> in epoch seconds.
         *        </p>
         *        <p>
         *        For quarterly budgets, provide four quarters of <code>PlannedBudgetLimits</code> value entries in
         *        standard calendar quarter increments. This must start from the current quarter and include the next
         *        three quarters. The <code>key</code> is the start of the quarter, <code>UTC</code> in epoch seconds.
         *        </p>
         *        <p>
         *        If the planned budget expires before 12 months for monthly or four quarters for quarterly, provide the
         *        <code>PlannedBudgetLimits</code> values only for the remaining periods.
         *        </p>
         *        <p>
         *        If the budget begins at a date in the future, provide <code>PlannedBudgetLimits</code> values from the
         *        start date of the budget.
         *        </p>
         *        <p>
         *        After all of the <code>BudgetLimit</code> values in <code>PlannedBudgetLimits</code> are used, the
         *        budget continues to use the last limit as the <code>BudgetLimit</code>. At that point, the planned
         *        budget provides the same experience as a fixed budget.
         *        </p>
         *        <p>
         *        <code>DescribeBudget</code> and <code>DescribeBudgets</code> response along with
         *        <code>PlannedBudgetLimits</code> also contain <code>BudgetLimit</code> representing the current month
         *        or quarter limit present in <code>PlannedBudgetLimits</code>. This only applies to budgets that are
         *        created with <code>PlannedBudgetLimits</code>. Budgets that are created without
         *        <code>PlannedBudgetLimits</code> only contain <code>BudgetLimit</code>. They don't contain
         *        <code>PlannedBudgetLimits</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder plannedBudgetLimits(Map<String, Spend> plannedBudgetLimits);

        /**
         * <p>
         * The cost filters, such as <code>Region</code>, <code>Service</code>, <code>LinkedAccount</code>,
         * <code>Tag</code>, or <code>CostCategory</code>, that are applied to a budget.
         * </p>
         * <p>
         * Amazon Web Services Budgets supports the following services as a <code>Service</code> filter for RI budgets:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Amazon EC2
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Redshift
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Relational Database Service
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon ElastiCache
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon OpenSearch Service
         * </p>
         * </li>
         * </ul>
         * 
         * @param costFilters
         *        The cost filters, such as <code>Region</code>, <code>Service</code>, <code>LinkedAccount</code>,
         *        <code>Tag</code>, or <code>CostCategory</code>, that are applied to a budget.</p>
         *        <p>
         *        Amazon Web Services Budgets supports the following services as a <code>Service</code> filter for RI
         *        budgets:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Amazon EC2
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Redshift
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Relational Database Service
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon ElastiCache
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon OpenSearch Service
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated CostFilters lack support for newer dimensions and filtering options. Please consider using the
         *             new 'FilterExpression' field.
         */
        @Deprecated
        Builder costFilters(Map<String, ? extends Collection<String>> costFilters);

        /**
         * <p>
         * The types of costs that are included in this <code>COST</code> budget.
         * </p>
         * <p>
         * <code>USAGE</code>, <code>RI_UTILIZATION</code>, <code>RI_COVERAGE</code>,
         * <code>SAVINGS_PLANS_UTILIZATION</code>, and <code>SAVINGS_PLANS_COVERAGE</code> budgets do not have
         * <code>CostTypes</code>.
         * </p>
         * 
         * @param costTypes
         *        The types of costs that are included in this <code>COST</code> budget.</p>
         *        <p>
         *        <code>USAGE</code>, <code>RI_UTILIZATION</code>, <code>RI_COVERAGE</code>,
         *        <code>SAVINGS_PLANS_UTILIZATION</code>, and <code>SAVINGS_PLANS_COVERAGE</code> budgets do not have
         *        <code>CostTypes</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated CostTypes lack support for newer record type dimensions and filtering options. Please consider
         *             using the new 'Metrics' field.
         */
        @Deprecated
        Builder costTypes(CostTypes costTypes);

        /**
         * <p>
         * The types of costs that are included in this <code>COST</code> budget.
         * </p>
         * <p>
         * <code>USAGE</code>, <code>RI_UTILIZATION</code>, <code>RI_COVERAGE</code>,
         * <code>SAVINGS_PLANS_UTILIZATION</code>, and <code>SAVINGS_PLANS_COVERAGE</code> budgets do not have
         * <code>CostTypes</code>.
         * </p>
         * This is a convenience method that creates an instance of the {@link CostTypes.Builder} avoiding the need to
         * create one manually via {@link CostTypes#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CostTypes.Builder#build()} is called immediately and its result
         * is passed to {@link #costTypes(CostTypes)}.
         * 
         * @param costTypes
         *        a consumer that will call methods on {@link CostTypes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #costTypes(CostTypes)
         * @deprecated CostTypes lack support for newer record type dimensions and filtering options. Please consider
         *             using the new 'Metrics' field.
         */
        @Deprecated
        default Builder costTypes(Consumer<CostTypes.Builder> costTypes) {
            return costTypes(CostTypes.builder().applyMutation(costTypes).build());
        }

        /**
         * <p>
         * The length of time until a budget resets the actual and forecasted spend.
         * </p>
         * 
         * @param timeUnit
         *        The length of time until a budget resets the actual and forecasted spend.
         * @see TimeUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TimeUnit
         */
        Builder timeUnit(String timeUnit);

        /**
         * <p>
         * The length of time until a budget resets the actual and forecasted spend.
         * </p>
         * 
         * @param timeUnit
         *        The length of time until a budget resets the actual and forecasted spend.
         * @see TimeUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TimeUnit
         */
        Builder timeUnit(TimeUnit timeUnit);

        /**
         * <p>
         * The period of time that's covered by a budget. You set the start date and end date. The start date must come
         * before the end date. The end date must come before <code>06/15/87 00:00 UTC</code>.
         * </p>
         * <p>
         * If you create your budget and don't specify a start date, Amazon Web Services defaults to the start of your
         * chosen time period (DAILY, MONTHLY, QUARTERLY, or ANNUALLY). For example, if you created your budget on
         * January 24, 2018, chose <code>DAILY</code>, and didn't set a start date, Amazon Web Services set your start
         * date to <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, Amazon Web Services set your
         * start date to <code>01/01/18 00:00 UTC</code>. If you didn't specify an end date, Amazon Web Services set
         * your end date to <code>06/15/87 00:00 UTC</code>. The defaults are the same for the Billing and Cost
         * Management console and the API.
         * </p>
         * <p>
         * You can change either date with the <code>UpdateBudget</code> operation.
         * </p>
         * <p>
         * After the end date, Amazon Web Services deletes the budget and all the associated notifications and
         * subscribers.
         * </p>
         * 
         * @param timePeriod
         *        The period of time that's covered by a budget. You set the start date and end date. The start date
         *        must come before the end date. The end date must come before <code>06/15/87 00:00 UTC</code>. </p>
         *        <p>
         *        If you create your budget and don't specify a start date, Amazon Web Services defaults to the start of
         *        your chosen time period (DAILY, MONTHLY, QUARTERLY, or ANNUALLY). For example, if you created your
         *        budget on January 24, 2018, chose <code>DAILY</code>, and didn't set a start date, Amazon Web Services
         *        set your start date to <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, Amazon Web
         *        Services set your start date to <code>01/01/18 00:00 UTC</code>. If you didn't specify an end date,
         *        Amazon Web Services set your end date to <code>06/15/87 00:00 UTC</code>. The defaults are the same
         *        for the Billing and Cost Management console and the API.
         *        </p>
         *        <p>
         *        You can change either date with the <code>UpdateBudget</code> operation.
         *        </p>
         *        <p>
         *        After the end date, Amazon Web Services deletes the budget and all the associated notifications and
         *        subscribers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timePeriod(TimePeriod timePeriod);

        /**
         * <p>
         * The period of time that's covered by a budget. You set the start date and end date. The start date must come
         * before the end date. The end date must come before <code>06/15/87 00:00 UTC</code>.
         * </p>
         * <p>
         * If you create your budget and don't specify a start date, Amazon Web Services defaults to the start of your
         * chosen time period (DAILY, MONTHLY, QUARTERLY, or ANNUALLY). For example, if you created your budget on
         * January 24, 2018, chose <code>DAILY</code>, and didn't set a start date, Amazon Web Services set your start
         * date to <code>01/24/18 00:00 UTC</code>. If you chose <code>MONTHLY</code>, Amazon Web Services set your
         * start date to <code>01/01/18 00:00 UTC</code>. If you didn't specify an end date, Amazon Web Services set
         * your end date to <code>06/15/87 00:00 UTC</code>. The defaults are the same for the Billing and Cost
         * Management console and the API.
         * </p>
         * <p>
         * You can change either date with the <code>UpdateBudget</code> operation.
         * </p>
         * <p>
         * After the end date, Amazon Web Services deletes the budget and all the associated notifications and
         * subscribers.
         * </p>
         * This is a convenience method that creates an instance of the {@link TimePeriod.Builder} avoiding the need to
         * create one manually via {@link TimePeriod#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TimePeriod.Builder#build()} is called immediately and its result
         * is passed to {@link #timePeriod(TimePeriod)}.
         * 
         * @param timePeriod
         *        a consumer that will call methods on {@link TimePeriod.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #timePeriod(TimePeriod)
         */
        default Builder timePeriod(Consumer<TimePeriod.Builder> timePeriod) {
            return timePeriod(TimePeriod.builder().applyMutation(timePeriod).build());
        }

        /**
         * <p>
         * The actual and forecasted cost or usage that the budget tracks.
         * </p>
         * 
         * @param calculatedSpend
         *        The actual and forecasted cost or usage that the budget tracks.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder calculatedSpend(CalculatedSpend calculatedSpend);

        /**
         * <p>
         * The actual and forecasted cost or usage that the budget tracks.
         * </p>
         * This is a convenience method that creates an instance of the {@link CalculatedSpend.Builder} avoiding the
         * need to create one manually via {@link CalculatedSpend#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CalculatedSpend.Builder#build()} is called immediately and its
         * result is passed to {@link #calculatedSpend(CalculatedSpend)}.
         * 
         * @param calculatedSpend
         *        a consumer that will call methods on {@link CalculatedSpend.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #calculatedSpend(CalculatedSpend)
         */
        default Builder calculatedSpend(Consumer<CalculatedSpend.Builder> calculatedSpend) {
            return calculatedSpend(CalculatedSpend.builder().applyMutation(calculatedSpend).build());
        }

        /**
         * <p>
         * Specifies whether this budget tracks costs, usage, RI utilization, RI coverage, Savings Plans utilization, or
         * Savings Plans coverage.
         * </p>
         * 
         * @param budgetType
         *        Specifies whether this budget tracks costs, usage, RI utilization, RI coverage, Savings Plans
         *        utilization, or Savings Plans coverage.
         * @see BudgetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BudgetType
         */
        Builder budgetType(String budgetType);

        /**
         * <p>
         * Specifies whether this budget tracks costs, usage, RI utilization, RI coverage, Savings Plans utilization, or
         * Savings Plans coverage.
         * </p>
         * 
         * @param budgetType
         *        Specifies whether this budget tracks costs, usage, RI utilization, RI coverage, Savings Plans
         *        utilization, or Savings Plans coverage.
         * @see BudgetType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BudgetType
         */
        Builder budgetType(BudgetType budgetType);

        /**
         * <p>
         * The last time that you updated this budget.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The last time that you updated this budget.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);

        /**
         * <p>
         * The parameters that determine the budget amount for an auto-adjusting budget.
         * </p>
         * 
         * @param autoAdjustData
         *        The parameters that determine the budget amount for an auto-adjusting budget.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoAdjustData(AutoAdjustData autoAdjustData);

        /**
         * <p>
         * The parameters that determine the budget amount for an auto-adjusting budget.
         * </p>
         * This is a convenience method that creates an instance of the {@link AutoAdjustData.Builder} avoiding the need
         * to create one manually via {@link AutoAdjustData#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AutoAdjustData.Builder#build()} is called immediately and its
         * result is passed to {@link #autoAdjustData(AutoAdjustData)}.
         * 
         * @param autoAdjustData
         *        a consumer that will call methods on {@link AutoAdjustData.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #autoAdjustData(AutoAdjustData)
         */
        default Builder autoAdjustData(Consumer<AutoAdjustData.Builder> autoAdjustData) {
            return autoAdjustData(AutoAdjustData.builder().applyMutation(autoAdjustData).build());
        }

        /**
         * <p>
         * The filtering dimensions for the budget and their corresponding values.
         * </p>
         * 
         * @param filterExpression
         *        The filtering dimensions for the budget and their corresponding values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filterExpression(Expression filterExpression);

        /**
         * <p>
         * The filtering dimensions for the budget and their corresponding values.
         * </p>
         * This is a convenience method that creates an instance of the {@link Expression.Builder} avoiding the need to
         * create one manually via {@link Expression#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Expression.Builder#build()} is called immediately and its result
         * is passed to {@link #filterExpression(Expression)}.
         * 
         * @param filterExpression
         *        a consumer that will call methods on {@link Expression.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filterExpression(Expression)
         */
        default Builder filterExpression(Consumer<Expression.Builder> filterExpression) {
            return filterExpression(Expression.builder().applyMutation(filterExpression).build());
        }

        /**
         * <p>
         * The definition for how the budget data is aggregated.
         * </p>
         * 
         * @param metrics
         *        The definition for how the budget data is aggregated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricsWithStrings(Collection<String> metrics);

        /**
         * <p>
         * The definition for how the budget data is aggregated.
         * </p>
         * 
         * @param metrics
         *        The definition for how the budget data is aggregated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricsWithStrings(String... metrics);

        /**
         * <p>
         * The definition for how the budget data is aggregated.
         * </p>
         * 
         * @param metrics
         *        The definition for how the budget data is aggregated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metrics(Collection<Metric> metrics);

        /**
         * <p>
         * The definition for how the budget data is aggregated.
         * </p>
         * 
         * @param metrics
         *        The definition for how the budget data is aggregated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metrics(Metric... metrics);
    }

    static final class BuilderImpl implements Builder {
        private String budgetName;

        private Spend budgetLimit;

        private Map<String, Spend> plannedBudgetLimits = DefaultSdkAutoConstructMap.getInstance();

        private Map<String, List<String>> costFilters = DefaultSdkAutoConstructMap.getInstance();

        private CostTypes costTypes;

        private String timeUnit;

        private TimePeriod timePeriod;

        private CalculatedSpend calculatedSpend;

        private String budgetType;

        private Instant lastUpdatedTime;

        private AutoAdjustData autoAdjustData;

        private Expression filterExpression;

        private List<String> metrics = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Budget model) {
            budgetName(model.budgetName);
            budgetLimit(model.budgetLimit);
            plannedBudgetLimits(model.plannedBudgetLimits);
            costFilters(model.costFilters);
            costTypes(model.costTypes);
            timeUnit(model.timeUnit);
            timePeriod(model.timePeriod);
            calculatedSpend(model.calculatedSpend);
            budgetType(model.budgetType);
            lastUpdatedTime(model.lastUpdatedTime);
            autoAdjustData(model.autoAdjustData);
            filterExpression(model.filterExpression);
            metricsWithStrings(model.metrics);
        }

        public final String getBudgetName() {
            return budgetName;
        }

        public final void setBudgetName(String budgetName) {
            this.budgetName = budgetName;
        }

        @Override
        public final Builder budgetName(String budgetName) {
            this.budgetName = budgetName;
            return this;
        }

        public final Spend.Builder getBudgetLimit() {
            return budgetLimit != null ? budgetLimit.toBuilder() : null;
        }

        public final void setBudgetLimit(Spend.BuilderImpl budgetLimit) {
            this.budgetLimit = budgetLimit != null ? budgetLimit.build() : null;
        }

        @Override
        public final Builder budgetLimit(Spend budgetLimit) {
            this.budgetLimit = budgetLimit;
            return this;
        }

        public final Map<String, Spend.Builder> getPlannedBudgetLimits() {
            Map<String, Spend.Builder> result = PlannedBudgetLimitsCopier.copyToBuilder(this.plannedBudgetLimits);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setPlannedBudgetLimits(Map<String, Spend.BuilderImpl> plannedBudgetLimits) {
            this.plannedBudgetLimits = PlannedBudgetLimitsCopier.copyFromBuilder(plannedBudgetLimits);
        }

        @Override
        public final Builder plannedBudgetLimits(Map<String, Spend> plannedBudgetLimits) {
            this.plannedBudgetLimits = PlannedBudgetLimitsCopier.copy(plannedBudgetLimits);
            return this;
        }

        @Deprecated
        public final Map<String, ? extends Collection<String>> getCostFilters() {
            if (costFilters instanceof SdkAutoConstructMap) {
                return null;
            }
            return costFilters;
        }

        @Deprecated
        public final void setCostFilters(Map<String, ? extends Collection<String>> costFilters) {
            this.costFilters = CostFiltersCopier.copy(costFilters);
        }

        @Override
        @Deprecated
        public final Builder costFilters(Map<String, ? extends Collection<String>> costFilters) {
            this.costFilters = CostFiltersCopier.copy(costFilters);
            return this;
        }

        @Deprecated
        public final CostTypes.Builder getCostTypes() {
            return costTypes != null ? costTypes.toBuilder() : null;
        }

        @Deprecated
        public final void setCostTypes(CostTypes.BuilderImpl costTypes) {
            this.costTypes = costTypes != null ? costTypes.build() : null;
        }

        @Override
        @Deprecated
        public final Builder costTypes(CostTypes costTypes) {
            this.costTypes = costTypes;
            return this;
        }

        public final String getTimeUnit() {
            return timeUnit;
        }

        public final void setTimeUnit(String timeUnit) {
            this.timeUnit = timeUnit;
        }

        @Override
        public final Builder timeUnit(String timeUnit) {
            this.timeUnit = timeUnit;
            return this;
        }

        @Override
        public final Builder timeUnit(TimeUnit timeUnit) {
            this.timeUnit(timeUnit == null ? null : timeUnit.toString());
            return this;
        }

        public final TimePeriod.Builder getTimePeriod() {
            return timePeriod != null ? timePeriod.toBuilder() : null;
        }

        public final void setTimePeriod(TimePeriod.BuilderImpl timePeriod) {
            this.timePeriod = timePeriod != null ? timePeriod.build() : null;
        }

        @Override
        public final Builder timePeriod(TimePeriod timePeriod) {
            this.timePeriod = timePeriod;
            return this;
        }

        public final CalculatedSpend.Builder getCalculatedSpend() {
            return calculatedSpend != null ? calculatedSpend.toBuilder() : null;
        }

        public final void setCalculatedSpend(CalculatedSpend.BuilderImpl calculatedSpend) {
            this.calculatedSpend = calculatedSpend != null ? calculatedSpend.build() : null;
        }

        @Override
        public final Builder calculatedSpend(CalculatedSpend calculatedSpend) {
            this.calculatedSpend = calculatedSpend;
            return this;
        }

        public final String getBudgetType() {
            return budgetType;
        }

        public final void setBudgetType(String budgetType) {
            this.budgetType = budgetType;
        }

        @Override
        public final Builder budgetType(String budgetType) {
            this.budgetType = budgetType;
            return this;
        }

        @Override
        public final Builder budgetType(BudgetType budgetType) {
            this.budgetType(budgetType == null ? null : budgetType.toString());
            return this;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final AutoAdjustData.Builder getAutoAdjustData() {
            return autoAdjustData != null ? autoAdjustData.toBuilder() : null;
        }

        public final void setAutoAdjustData(AutoAdjustData.BuilderImpl autoAdjustData) {
            this.autoAdjustData = autoAdjustData != null ? autoAdjustData.build() : null;
        }

        @Override
        public final Builder autoAdjustData(AutoAdjustData autoAdjustData) {
            this.autoAdjustData = autoAdjustData;
            return this;
        }

        public final Expression.Builder getFilterExpression() {
            return filterExpression != null ? filterExpression.toBuilder() : null;
        }

        public final void setFilterExpression(Expression.BuilderImpl filterExpression) {
            this.filterExpression = filterExpression != null ? filterExpression.build() : null;
        }

        @Override
        public final Builder filterExpression(Expression filterExpression) {
            this.filterExpression = filterExpression;
            return this;
        }

        public final Collection<String> getMetrics() {
            if (metrics instanceof SdkAutoConstructList) {
                return null;
            }
            return metrics;
        }

        public final void setMetrics(Collection<String> metrics) {
            this.metrics = MetricsCopier.copy(metrics);
        }

        @Override
        public final Builder metricsWithStrings(Collection<String> metrics) {
            this.metrics = MetricsCopier.copy(metrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricsWithStrings(String... metrics) {
            metricsWithStrings(Arrays.asList(metrics));
            return this;
        }

        @Override
        public final Builder metrics(Collection<Metric> metrics) {
            this.metrics = MetricsCopier.copyEnumToString(metrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metrics(Metric... metrics) {
            metrics(Arrays.asList(metrics));
            return this;
        }

        @Override
        public Budget build() {
            return new Budget(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
