/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.budgets;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.budgets.internal.BudgetsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.budgets.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.budgets.model.AccessDeniedException;
import software.amazon.awssdk.services.budgets.model.BillingViewHealthStatusException;
import software.amazon.awssdk.services.budgets.model.BudgetsException;
import software.amazon.awssdk.services.budgets.model.CreateBudgetActionRequest;
import software.amazon.awssdk.services.budgets.model.CreateBudgetActionResponse;
import software.amazon.awssdk.services.budgets.model.CreateBudgetRequest;
import software.amazon.awssdk.services.budgets.model.CreateBudgetResponse;
import software.amazon.awssdk.services.budgets.model.CreateNotificationRequest;
import software.amazon.awssdk.services.budgets.model.CreateNotificationResponse;
import software.amazon.awssdk.services.budgets.model.CreateSubscriberRequest;
import software.amazon.awssdk.services.budgets.model.CreateSubscriberResponse;
import software.amazon.awssdk.services.budgets.model.CreationLimitExceededException;
import software.amazon.awssdk.services.budgets.model.DeleteBudgetActionRequest;
import software.amazon.awssdk.services.budgets.model.DeleteBudgetActionResponse;
import software.amazon.awssdk.services.budgets.model.DeleteBudgetRequest;
import software.amazon.awssdk.services.budgets.model.DeleteBudgetResponse;
import software.amazon.awssdk.services.budgets.model.DeleteNotificationRequest;
import software.amazon.awssdk.services.budgets.model.DeleteNotificationResponse;
import software.amazon.awssdk.services.budgets.model.DeleteSubscriberRequest;
import software.amazon.awssdk.services.budgets.model.DeleteSubscriberResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionHistoriesRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionHistoriesResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForAccountRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForAccountResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForBudgetRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetActionsForBudgetResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetNotificationsForAccountRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetNotificationsForAccountResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetPerformanceHistoryRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetPerformanceHistoryResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetResponse;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetsRequest;
import software.amazon.awssdk.services.budgets.model.DescribeBudgetsResponse;
import software.amazon.awssdk.services.budgets.model.DescribeNotificationsForBudgetRequest;
import software.amazon.awssdk.services.budgets.model.DescribeNotificationsForBudgetResponse;
import software.amazon.awssdk.services.budgets.model.DescribeSubscribersForNotificationRequest;
import software.amazon.awssdk.services.budgets.model.DescribeSubscribersForNotificationResponse;
import software.amazon.awssdk.services.budgets.model.DuplicateRecordException;
import software.amazon.awssdk.services.budgets.model.ExecuteBudgetActionRequest;
import software.amazon.awssdk.services.budgets.model.ExecuteBudgetActionResponse;
import software.amazon.awssdk.services.budgets.model.ExpiredNextTokenException;
import software.amazon.awssdk.services.budgets.model.InternalErrorException;
import software.amazon.awssdk.services.budgets.model.InvalidNextTokenException;
import software.amazon.awssdk.services.budgets.model.InvalidParameterException;
import software.amazon.awssdk.services.budgets.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.budgets.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.budgets.model.NotFoundException;
import software.amazon.awssdk.services.budgets.model.ResourceLockedException;
import software.amazon.awssdk.services.budgets.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.budgets.model.TagResourceRequest;
import software.amazon.awssdk.services.budgets.model.TagResourceResponse;
import software.amazon.awssdk.services.budgets.model.ThrottlingException;
import software.amazon.awssdk.services.budgets.model.UntagResourceRequest;
import software.amazon.awssdk.services.budgets.model.UntagResourceResponse;
import software.amazon.awssdk.services.budgets.model.UpdateBudgetActionRequest;
import software.amazon.awssdk.services.budgets.model.UpdateBudgetActionResponse;
import software.amazon.awssdk.services.budgets.model.UpdateBudgetRequest;
import software.amazon.awssdk.services.budgets.model.UpdateBudgetResponse;
import software.amazon.awssdk.services.budgets.model.UpdateNotificationRequest;
import software.amazon.awssdk.services.budgets.model.UpdateNotificationResponse;
import software.amazon.awssdk.services.budgets.model.UpdateSubscriberRequest;
import software.amazon.awssdk.services.budgets.model.UpdateSubscriberResponse;
import software.amazon.awssdk.services.budgets.transform.CreateBudgetActionRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.CreateBudgetRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.CreateNotificationRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.CreateSubscriberRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DeleteBudgetActionRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DeleteBudgetRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DeleteNotificationRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DeleteSubscriberRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetActionHistoriesRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetActionRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetActionsForAccountRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetActionsForBudgetRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetNotificationsForAccountRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetPerformanceHistoryRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeBudgetsRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeNotificationsForBudgetRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.DescribeSubscribersForNotificationRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.ExecuteBudgetActionRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.UpdateBudgetActionRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.UpdateBudgetRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.UpdateNotificationRequestMarshaller;
import software.amazon.awssdk.services.budgets.transform.UpdateSubscriberRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link BudgetsClient}.
 *
 * @see BudgetsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBudgetsClient implements BudgetsClient {
    private static final Logger log = Logger.loggerFor(DefaultBudgetsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultBudgetsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Budgets" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a budget and, if included, notifications and subscribers.
     * </p>
     * <important>
     * <p>
     * Only one of <code>BudgetLimit</code> or <code>PlannedBudgetLimits</code> can be present in the syntax at one
     * time. Use the syntax that matches your use case. The Request Syntax section shows the <code>BudgetLimit</code>
     * syntax. For <code>PlannedBudgetLimits</code>, see the <a href=
     * "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_CreateBudget.html#API_CreateBudget_Examples"
     * >Examples</a> section.
     * </p>
     * <p>
     * Similarly, only one set of filter and metric selections can be present in the syntax at one time. Either
     * <code>FilterExpression</code> and <code>Metrics</code> or <code>CostFilters</code> and <code>CostTypes</code>,
     * not both or a different combination. We recommend using <code>FilterExpression</code> and <code>Metrics</code> as
     * they provide more flexible and powerful filtering capabilities. The Request Syntax section shows the
     * <code>FilterExpression</code>/<code>Metrics</code> syntax.
     * </p>
     * </important>
     *
     * @param createBudgetRequest
     *        Request of CreateBudget
     * @return Result of the CreateBudget operation returned by the service.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws BillingViewHealthStatusException
     *         The billing view status must be HEALTHY to perform this action. Try again when the status is HEALTHY.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws CreationLimitExceededException
     *         You've exceeded the notification or subscriber limit.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws ServiceQuotaExceededException
     *         You've reached a Service Quota limit on this resource.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.CreateBudget
     */
    @Override
    public CreateBudgetResponse createBudget(CreateBudgetRequest createBudgetRequest) throws InvalidParameterException,
            BillingViewHealthStatusException, InternalErrorException, CreationLimitExceededException, DuplicateRecordException,
            AccessDeniedException, ThrottlingException, ServiceQuotaExceededException, NotFoundException, AwsServiceException,
            SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBudgetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateBudgetResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBudgetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBudgetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBudget");

            return clientHandler.execute(new ClientExecutionParams<CreateBudgetRequest, CreateBudgetResponse>()
                    .withOperationName("CreateBudget").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createBudgetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateBudgetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a budget action.
     * </p>
     *
     * @param createBudgetActionRequest
     * @return Result of the CreateBudgetAction operation returned by the service.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws CreationLimitExceededException
     *         You've exceeded the notification or subscriber limit.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws ServiceQuotaExceededException
     *         You've reached a Service Quota limit on this resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.CreateBudgetAction
     */
    @Override
    public CreateBudgetActionResponse createBudgetAction(CreateBudgetActionRequest createBudgetActionRequest)
            throws InvalidParameterException, InternalErrorException, CreationLimitExceededException, DuplicateRecordException,
            NotFoundException, AccessDeniedException, ThrottlingException, ServiceQuotaExceededException, AwsServiceException,
            SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBudgetActionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateBudgetActionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBudgetActionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBudgetActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBudgetAction");

            return clientHandler.execute(new ClientExecutionParams<CreateBudgetActionRequest, CreateBudgetActionResponse>()
                    .withOperationName("CreateBudgetAction").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createBudgetActionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateBudgetActionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a notification. You must create the budget before you create the associated notification.
     * </p>
     *
     * @param createNotificationRequest
     *        Request of CreateNotification
     * @return Result of the CreateNotification operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws CreationLimitExceededException
     *         You've exceeded the notification or subscriber limit.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.CreateNotification
     */
    @Override
    public CreateNotificationResponse createNotification(CreateNotificationRequest createNotificationRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, CreationLimitExceededException,
            DuplicateRecordException, AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException,
            BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateNotificationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createNotificationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createNotificationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateNotification");

            return clientHandler.execute(new ClientExecutionParams<CreateNotificationRequest, CreateNotificationResponse>()
                    .withOperationName("CreateNotification").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createNotificationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateNotificationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a subscriber. You must create the associated budget and notification before you create the subscriber.
     * </p>
     *
     * @param createSubscriberRequest
     *        Request of CreateSubscriber
     * @return Result of the CreateSubscriber operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws CreationLimitExceededException
     *         You've exceeded the notification or subscriber limit.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.CreateSubscriber
     */
    @Override
    public CreateSubscriberResponse createSubscriber(CreateSubscriberRequest createSubscriberRequest)
            throws InternalErrorException, InvalidParameterException, CreationLimitExceededException, DuplicateRecordException,
            NotFoundException, AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException,
            BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateSubscriberResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSubscriberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSubscriberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSubscriber");

            return clientHandler.execute(new ClientExecutionParams<CreateSubscriberRequest, CreateSubscriberResponse>()
                    .withOperationName("CreateSubscriber").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createSubscriberRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateSubscriberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a budget. You can delete your budget at any time.
     * </p>
     * <important>
     * <p>
     * Deleting a budget also deletes the notifications and subscribers that are associated with that budget.
     * </p>
     * </important>
     *
     * @param deleteBudgetRequest
     *        Request of DeleteBudget
     * @return Result of the DeleteBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DeleteBudget
     */
    @Override
    public DeleteBudgetResponse deleteBudget(DeleteBudgetRequest deleteBudgetRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBudgetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteBudgetResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBudgetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBudgetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBudget");

            return clientHandler.execute(new ClientExecutionParams<DeleteBudgetRequest, DeleteBudgetResponse>()
                    .withOperationName("DeleteBudget").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteBudgetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteBudgetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a budget action.
     * </p>
     *
     * @param deleteBudgetActionRequest
     * @return Result of the DeleteBudgetAction operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ResourceLockedException
     *         The request was received and recognized by the server, but the server rejected that particular method for
     *         the requested resource.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DeleteBudgetAction
     */
    @Override
    public DeleteBudgetActionResponse deleteBudgetAction(DeleteBudgetActionRequest deleteBudgetActionRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            ResourceLockedException, ThrottlingException, AwsServiceException, SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBudgetActionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteBudgetActionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBudgetActionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBudgetActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBudgetAction");

            return clientHandler.execute(new ClientExecutionParams<DeleteBudgetActionRequest, DeleteBudgetActionResponse>()
                    .withOperationName("DeleteBudgetAction").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteBudgetActionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteBudgetActionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a notification.
     * </p>
     * <important>
     * <p>
     * Deleting a notification also deletes the subscribers that are associated with the notification.
     * </p>
     * </important>
     *
     * @param deleteNotificationRequest
     *        Request of DeleteNotification
     * @return Result of the DeleteNotification operation returned by the service.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DeleteNotification
     */
    @Override
    public DeleteNotificationResponse deleteNotification(DeleteNotificationRequest deleteNotificationRequest)
            throws InvalidParameterException, InternalErrorException, NotFoundException, AccessDeniedException,
            ThrottlingException, AwsServiceException, SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteNotificationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteNotificationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteNotificationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteNotification");

            return clientHandler.execute(new ClientExecutionParams<DeleteNotificationRequest, DeleteNotificationResponse>()
                    .withOperationName("DeleteNotification").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteNotificationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteNotificationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a subscriber.
     * </p>
     * <important>
     * <p>
     * Deleting the last subscriber to a notification also deletes the notification.
     * </p>
     * </important>
     *
     * @param deleteSubscriberRequest
     *        Request of DeleteSubscriber
     * @return Result of the DeleteSubscriber operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DeleteSubscriber
     */
    @Override
    public DeleteSubscriberResponse deleteSubscriber(DeleteSubscriberRequest deleteSubscriberRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            ThrottlingException, AwsServiceException, SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteSubscriberResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSubscriberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSubscriberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSubscriber");

            return clientHandler.execute(new ClientExecutionParams<DeleteSubscriberRequest, DeleteSubscriberResponse>()
                    .withOperationName("DeleteSubscriber").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteSubscriberRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSubscriberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes a budget.
     * </p>
     * <important>
     * <p>
     * The Request Syntax section shows the <code>BudgetLimit</code> syntax. For <code>PlannedBudgetLimits</code>, see
     * the <a href=
     * "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_DescribeBudget.html#API_DescribeBudget_Examples"
     * >Examples</a> section.
     * </p>
     * </important>
     *
     * @param describeBudgetRequest
     *        Request of DescribeBudget
     * @return Result of the DescribeBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudget
     */
    @Override
    public DescribeBudgetResponse describeBudget(DescribeBudgetRequest describeBudgetRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, AccessDeniedException, ThrottlingException, AwsServiceException,
            SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBudgetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeBudgetResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBudgetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBudgetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBudget");

            return clientHandler.execute(new ClientExecutionParams<DescribeBudgetRequest, DescribeBudgetResponse>()
                    .withOperationName("DescribeBudget").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeBudgetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeBudgetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes a budget action detail.
     * </p>
     *
     * @param describeBudgetActionRequest
     * @return Result of the DescribeBudgetAction operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetAction
     */
    @Override
    public DescribeBudgetActionResponse describeBudgetAction(DescribeBudgetActionRequest describeBudgetActionRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            ThrottlingException, AwsServiceException, SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBudgetActionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeBudgetActionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBudgetActionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBudgetActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBudgetAction");

            return clientHandler.execute(new ClientExecutionParams<DescribeBudgetActionRequest, DescribeBudgetActionResponse>()
                    .withOperationName("DescribeBudgetAction").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeBudgetActionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeBudgetActionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes a budget action history detail.
     * </p>
     *
     * @param describeBudgetActionHistoriesRequest
     * @return Result of the DescribeBudgetActionHistories operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetActionHistories
     */
    @Override
    public DescribeBudgetActionHistoriesResponse describeBudgetActionHistories(
            DescribeBudgetActionHistoriesRequest describeBudgetActionHistoriesRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, AccessDeniedException, InvalidNextTokenException, ThrottlingException,
            AwsServiceException, SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBudgetActionHistoriesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeBudgetActionHistoriesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBudgetActionHistoriesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeBudgetActionHistoriesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBudgetActionHistories");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeBudgetActionHistoriesRequest, DescribeBudgetActionHistoriesResponse>()
                            .withOperationName("DescribeBudgetActionHistories").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeBudgetActionHistoriesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeBudgetActionHistoriesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes all of the budget actions for an account.
     * </p>
     *
     * @param describeBudgetActionsForAccountRequest
     * @return Result of the DescribeBudgetActionsForAccount operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetActionsForAccount
     */
    @Override
    public DescribeBudgetActionsForAccountResponse describeBudgetActionsForAccount(
            DescribeBudgetActionsForAccountRequest describeBudgetActionsForAccountRequest) throws InternalErrorException,
            InvalidParameterException, AccessDeniedException, InvalidNextTokenException, ThrottlingException,
            AwsServiceException, SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBudgetActionsForAccountResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeBudgetActionsForAccountResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBudgetActionsForAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeBudgetActionsForAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBudgetActionsForAccount");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeBudgetActionsForAccountRequest, DescribeBudgetActionsForAccountResponse>()
                            .withOperationName("DescribeBudgetActionsForAccount").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeBudgetActionsForAccountRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeBudgetActionsForAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes all of the budget actions for a budget.
     * </p>
     *
     * @param describeBudgetActionsForBudgetRequest
     * @return Result of the DescribeBudgetActionsForBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetActionsForBudget
     */
    @Override
    public DescribeBudgetActionsForBudgetResponse describeBudgetActionsForBudget(
            DescribeBudgetActionsForBudgetRequest describeBudgetActionsForBudgetRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, AccessDeniedException, InvalidNextTokenException, ThrottlingException,
            AwsServiceException, SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBudgetActionsForBudgetResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeBudgetActionsForBudgetResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBudgetActionsForBudgetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeBudgetActionsForBudgetRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBudgetActionsForBudget");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeBudgetActionsForBudgetRequest, DescribeBudgetActionsForBudgetResponse>()
                            .withOperationName("DescribeBudgetActionsForBudget").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeBudgetActionsForBudgetRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeBudgetActionsForBudgetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the budget names and notifications that are associated with an account.
     * </p>
     *
     * @param describeBudgetNotificationsForAccountRequest
     * @return Result of the DescribeBudgetNotificationsForAccount operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetNotificationsForAccount
     */
    @Override
    public DescribeBudgetNotificationsForAccountResponse describeBudgetNotificationsForAccount(
            DescribeBudgetNotificationsForAccountRequest describeBudgetNotificationsForAccountRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, InvalidNextTokenException,
            ExpiredNextTokenException, AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException,
            BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBudgetNotificationsForAccountResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DescribeBudgetNotificationsForAccountResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBudgetNotificationsForAccountRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeBudgetNotificationsForAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBudgetNotificationsForAccount");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeBudgetNotificationsForAccountRequest, DescribeBudgetNotificationsForAccountResponse>()
                            .withOperationName("DescribeBudgetNotificationsForAccount").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(describeBudgetNotificationsForAccountRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeBudgetNotificationsForAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the history for <code>DAILY</code>, <code>MONTHLY</code>, and <code>QUARTERLY</code> budgets. Budget
     * history isn't available for <code>ANNUAL</code> budgets.
     * </p>
     *
     * @param describeBudgetPerformanceHistoryRequest
     * @return Result of the DescribeBudgetPerformanceHistory operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws BillingViewHealthStatusException
     *         The billing view status must be HEALTHY to perform this action. Try again when the status is HEALTHY.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgetPerformanceHistory
     */
    @Override
    public DescribeBudgetPerformanceHistoryResponse describeBudgetPerformanceHistory(
            DescribeBudgetPerformanceHistoryRequest describeBudgetPerformanceHistoryRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, InvalidNextTokenException, ExpiredNextTokenException,
            AccessDeniedException, ThrottlingException, BillingViewHealthStatusException, AwsServiceException,
            SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBudgetPerformanceHistoryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeBudgetPerformanceHistoryResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBudgetPerformanceHistoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeBudgetPerformanceHistoryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBudgetPerformanceHistory");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeBudgetPerformanceHistoryRequest, DescribeBudgetPerformanceHistoryResponse>()
                            .withOperationName("DescribeBudgetPerformanceHistory").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeBudgetPerformanceHistoryRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeBudgetPerformanceHistoryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the budgets that are associated with an account.
     * </p>
     * <important>
     * <p>
     * The Request Syntax section shows the <code>BudgetLimit</code> syntax. For <code>PlannedBudgetLimits</code>, see
     * the <a href=
     * "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_DescribeBudgets.html#API_DescribeBudgets_Examples"
     * >Examples</a> section.
     * </p>
     * </important>
     *
     * @param describeBudgetsRequest
     *        Request of DescribeBudgets
     * @return Result of the DescribeBudgets operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeBudgets
     */
    @Override
    public DescribeBudgetsResponse describeBudgets(DescribeBudgetsRequest describeBudgetsRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, InvalidNextTokenException, ExpiredNextTokenException,
            AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBudgetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeBudgetsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBudgetsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBudgetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBudgets");

            return clientHandler.execute(new ClientExecutionParams<DescribeBudgetsRequest, DescribeBudgetsResponse>()
                    .withOperationName("DescribeBudgets").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeBudgetsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeBudgetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the notifications that are associated with a budget.
     * </p>
     *
     * @param describeNotificationsForBudgetRequest
     *        Request of DescribeNotificationsForBudget
     * @return Result of the DescribeNotificationsForBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeNotificationsForBudget
     */
    @Override
    public DescribeNotificationsForBudgetResponse describeNotificationsForBudget(
            DescribeNotificationsForBudgetRequest describeNotificationsForBudgetRequest) throws InternalErrorException,
            InvalidParameterException, NotFoundException, InvalidNextTokenException, ExpiredNextTokenException,
            AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeNotificationsForBudgetResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeNotificationsForBudgetResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeNotificationsForBudgetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeNotificationsForBudgetRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeNotificationsForBudget");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeNotificationsForBudgetRequest, DescribeNotificationsForBudgetResponse>()
                            .withOperationName("DescribeNotificationsForBudget").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeNotificationsForBudgetRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeNotificationsForBudgetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the subscribers that are associated with a notification.
     * </p>
     *
     * @param describeSubscribersForNotificationRequest
     *        Request of DescribeSubscribersForNotification
     * @return Result of the DescribeSubscribersForNotification operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid.
     * @throws ExpiredNextTokenException
     *         The pagination token expired.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.DescribeSubscribersForNotification
     */
    @Override
    public DescribeSubscribersForNotificationResponse describeSubscribersForNotification(
            DescribeSubscribersForNotificationRequest describeSubscribersForNotificationRequest) throws InternalErrorException,
            NotFoundException, InvalidParameterException, InvalidNextTokenException, ExpiredNextTokenException,
            AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSubscribersForNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeSubscribersForNotificationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeSubscribersForNotificationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeSubscribersForNotificationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSubscribersForNotification");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeSubscribersForNotificationRequest, DescribeSubscribersForNotificationResponse>()
                            .withOperationName("DescribeSubscribersForNotification").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeSubscribersForNotificationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeSubscribersForNotificationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Executes a budget action.
     * </p>
     *
     * @param executeBudgetActionRequest
     * @return Result of the ExecuteBudgetAction operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ResourceLockedException
     *         The request was received and recognized by the server, but the server rejected that particular method for
     *         the requested resource.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.ExecuteBudgetAction
     */
    @Override
    public ExecuteBudgetActionResponse executeBudgetAction(ExecuteBudgetActionRequest executeBudgetActionRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            ResourceLockedException, ThrottlingException, AwsServiceException, SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExecuteBudgetActionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ExecuteBudgetActionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(executeBudgetActionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, executeBudgetActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExecuteBudgetAction");

            return clientHandler.execute(new ClientExecutionParams<ExecuteBudgetActionRequest, ExecuteBudgetActionResponse>()
                    .withOperationName("ExecuteBudgetAction").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(executeBudgetActionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ExecuteBudgetActionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists tags associated with a budget or budget action resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.ListTagsForResource
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ThrottlingException, NotFoundException, AccessDeniedException, InternalErrorException,
            InvalidParameterException, AwsServiceException, SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates tags for a budget or budget action resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws ServiceQuotaExceededException
     *         You've reached a Service Quota limit on this resource.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.TagResource
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ThrottlingException,
            ServiceQuotaExceededException, NotFoundException, AccessDeniedException, InternalErrorException,
            InvalidParameterException, AwsServiceException, SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes tags associated with a budget or budget action resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UntagResource
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ThrottlingException,
            NotFoundException, AccessDeniedException, InternalErrorException, InvalidParameterException, AwsServiceException,
            SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a budget. You can change every part of a budget except for the <code>budgetName</code> and the
     * <code>calculatedSpend</code>. When you modify a budget, the <code>calculatedSpend</code> drops to zero until
     * Amazon Web Services has new usage data to use for forecasting.
     * </p>
     * <important>
     * <p>
     * Only one of <code>BudgetLimit</code> or <code>PlannedBudgetLimits</code> can be present in the syntax at one
     * time. Use the syntax that matches your case. The Request Syntax section shows the <code>BudgetLimit</code>
     * syntax. For <code>PlannedBudgetLimits</code>, see the <a href=
     * "https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_budgets_UpdateBudget.html#API_UpdateBudget_Examples"
     * >Examples</a> section.
     * </p>
     * <p>
     * Similarly, only one set of filter and metric selections can be present in the syntax at one time. Either
     * <code>FilterExpression</code> and <code>Metrics</code> or <code>CostFilters</code> and <code>CostTypes</code>,
     * not both or a different combination. We recommend using <code>FilterExpression</code> and <code>Metrics</code> as
     * they provide more flexible and powerful filtering capabilities. The Request Syntax section shows the
     * <code>FilterExpression</code>/<code>Metrics</code> syntax.
     * </p>
     * </important>
     *
     * @param updateBudgetRequest
     *        Request of UpdateBudget
     * @return Result of the UpdateBudget operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws BillingViewHealthStatusException
     *         The billing view status must be HEALTHY to perform this action. Try again when the status is HEALTHY.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws ServiceQuotaExceededException
     *         You've reached a Service Quota limit on this resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UpdateBudget
     */
    @Override
    public UpdateBudgetResponse updateBudget(UpdateBudgetRequest updateBudgetRequest) throws InternalErrorException,
            InvalidParameterException, BillingViewHealthStatusException, NotFoundException, AccessDeniedException,
            ThrottlingException, ServiceQuotaExceededException, AwsServiceException, SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateBudgetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateBudgetResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateBudgetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBudgetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBudget");

            return clientHandler.execute(new ClientExecutionParams<UpdateBudgetRequest, UpdateBudgetResponse>()
                    .withOperationName("UpdateBudget").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateBudgetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateBudgetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a budget action.
     * </p>
     *
     * @param updateBudgetActionRequest
     * @return Result of the UpdateBudgetAction operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ResourceLockedException
     *         The request was received and recognized by the server, but the server rejected that particular method for
     *         the requested resource.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UpdateBudgetAction
     */
    @Override
    public UpdateBudgetActionResponse updateBudgetAction(UpdateBudgetActionRequest updateBudgetActionRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, AccessDeniedException,
            ResourceLockedException, ThrottlingException, AwsServiceException, SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateBudgetActionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateBudgetActionResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateBudgetActionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBudgetActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBudgetAction");

            return clientHandler.execute(new ClientExecutionParams<UpdateBudgetActionRequest, UpdateBudgetActionResponse>()
                    .withOperationName("UpdateBudgetAction").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateBudgetActionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateBudgetActionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a notification.
     * </p>
     *
     * @param updateNotificationRequest
     *        Request of UpdateNotification
     * @return Result of the UpdateNotification operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UpdateNotification
     */
    @Override
    public UpdateNotificationResponse updateNotification(UpdateNotificationRequest updateNotificationRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, DuplicateRecordException,
            AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateNotificationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateNotificationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateNotificationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateNotificationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateNotification");

            return clientHandler.execute(new ClientExecutionParams<UpdateNotificationRequest, UpdateNotificationResponse>()
                    .withOperationName("UpdateNotification").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateNotificationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateNotificationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a subscriber.
     * </p>
     *
     * @param updateSubscriberRequest
     *        Request of UpdateSubscriber
     * @return Result of the UpdateSubscriber operation returned by the service.
     * @throws InternalErrorException
     *         An error on the server occurred during the processing of your request. Try again later.
     * @throws InvalidParameterException
     *         An error on the client occurred. Typically, the cause is an invalid input value.
     * @throws NotFoundException
     *         We can’t locate the resource that you specified.
     * @throws DuplicateRecordException
     *         The budget name already exists. Budget names must be unique within an account.
     * @throws AccessDeniedException
     *         You are not authorized to use this operation with the given parameters.
     * @throws ThrottlingException
     *         The number of API requests has exceeded the maximum allowed API request throttling limit for the account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BudgetsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BudgetsClient.UpdateSubscriber
     */
    @Override
    public UpdateSubscriberResponse updateSubscriber(UpdateSubscriberRequest updateSubscriberRequest)
            throws InternalErrorException, InvalidParameterException, NotFoundException, DuplicateRecordException,
            AccessDeniedException, ThrottlingException, AwsServiceException, SdkClientException, BudgetsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSubscriberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateSubscriberResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidParameterException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidParameterException::builder).build());
            case "ThrottlingException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottlingException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ThrottlingException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceQuotaExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build());
            case "DuplicateRecordException":
                return Optional.of(ExceptionMetadata.builder().errorCode("DuplicateRecordException").httpStatusCode(400)
                        .exceptionBuilderSupplier(DuplicateRecordException::builder).build());
            case "InternalErrorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalErrorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InternalErrorException::builder).build());
            case "CreationLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("CreationLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(CreationLimitExceededException::builder).build());
            case "InvalidNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNextTokenException::builder).build());
            case "ExpiredNextTokenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredNextTokenException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredNextTokenException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "BillingViewHealthStatusException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BillingViewHealthStatusException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BillingViewHealthStatusException::builder).build());
            case "ResourceLockedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLockedException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLockedException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSubscriberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSubscriberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Budgets");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSubscriber");

            return clientHandler.execute(new ClientExecutionParams<UpdateSubscriberRequest, UpdateSubscriberResponse>()
                    .withOperationName("UpdateSubscriber").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateSubscriberRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateSubscriberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        BudgetsServiceClientConfigurationBuilder serviceConfigBuilder = new BudgetsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(BudgetsException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON).protocolVersion("1.1");
    }

    @Override
    public final BudgetsServiceClientConfiguration serviceClientConfiguration() {
        return new BudgetsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
