/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chatbot.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An AWS Chatbot configuration for Slack.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SlackChannelConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<SlackChannelConfiguration.Builder, SlackChannelConfiguration> {
    private static final SdkField<String> SLACK_TEAM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SlackTeamName").getter(getter(SlackChannelConfiguration::slackTeamName))
            .setter(setter(Builder::slackTeamName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SlackTeamName").build()).build();

    private static final SdkField<String> SLACK_TEAM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SlackTeamId").getter(getter(SlackChannelConfiguration::slackTeamId))
            .setter(setter(Builder::slackTeamId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SlackTeamId").build()).build();

    private static final SdkField<String> SLACK_CHANNEL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SlackChannelId").getter(getter(SlackChannelConfiguration::slackChannelId))
            .setter(setter(Builder::slackChannelId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SlackChannelId").build()).build();

    private static final SdkField<String> SLACK_CHANNEL_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SlackChannelName").getter(getter(SlackChannelConfiguration::slackChannelName))
            .setter(setter(Builder::slackChannelName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SlackChannelName").build()).build();

    private static final SdkField<String> CHAT_CONFIGURATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ChatConfigurationArn").getter(getter(SlackChannelConfiguration::chatConfigurationArn))
            .setter(setter(Builder::chatConfigurationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChatConfigurationArn").build())
            .build();

    private static final SdkField<String> IAM_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IamRoleArn").getter(getter(SlackChannelConfiguration::iamRoleArn)).setter(setter(Builder::iamRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IamRoleArn").build()).build();

    private static final SdkField<List<String>> SNS_TOPIC_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SnsTopicArns")
            .getter(getter(SlackChannelConfiguration::snsTopicArns))
            .setter(setter(Builder::snsTopicArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnsTopicArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> CONFIGURATION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConfigurationName").getter(getter(SlackChannelConfiguration::configurationName))
            .setter(setter(Builder::configurationName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigurationName").build()).build();

    private static final SdkField<String> LOGGING_LEVEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LoggingLevel").getter(getter(SlackChannelConfiguration::loggingLevel))
            .setter(setter(Builder::loggingLevel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LoggingLevel").build()).build();

    private static final SdkField<List<String>> GUARDRAIL_POLICY_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("GuardrailPolicyArns")
            .getter(getter(SlackChannelConfiguration::guardrailPolicyArns))
            .setter(setter(Builder::guardrailPolicyArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GuardrailPolicyArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> USER_AUTHORIZATION_REQUIRED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).memberName("UserAuthorizationRequired")
            .getter(getter(SlackChannelConfiguration::userAuthorizationRequired))
            .setter(setter(Builder::userAuthorizationRequired))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserAuthorizationRequired").build())
            .build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(SlackChannelConfiguration::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(SlackChannelConfiguration::state)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<String> STATE_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StateReason").getter(getter(SlackChannelConfiguration::stateReason))
            .setter(setter(Builder::stateReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateReason").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SLACK_TEAM_NAME_FIELD,
            SLACK_TEAM_ID_FIELD, SLACK_CHANNEL_ID_FIELD, SLACK_CHANNEL_NAME_FIELD, CHAT_CONFIGURATION_ARN_FIELD,
            IAM_ROLE_ARN_FIELD, SNS_TOPIC_ARNS_FIELD, CONFIGURATION_NAME_FIELD, LOGGING_LEVEL_FIELD, GUARDRAIL_POLICY_ARNS_FIELD,
            USER_AUTHORIZATION_REQUIRED_FIELD, TAGS_FIELD, STATE_FIELD, STATE_REASON_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("SlackTeamName", SLACK_TEAM_NAME_FIELD);
                    put("SlackTeamId", SLACK_TEAM_ID_FIELD);
                    put("SlackChannelId", SLACK_CHANNEL_ID_FIELD);
                    put("SlackChannelName", SLACK_CHANNEL_NAME_FIELD);
                    put("ChatConfigurationArn", CHAT_CONFIGURATION_ARN_FIELD);
                    put("IamRoleArn", IAM_ROLE_ARN_FIELD);
                    put("SnsTopicArns", SNS_TOPIC_ARNS_FIELD);
                    put("ConfigurationName", CONFIGURATION_NAME_FIELD);
                    put("LoggingLevel", LOGGING_LEVEL_FIELD);
                    put("GuardrailPolicyArns", GUARDRAIL_POLICY_ARNS_FIELD);
                    put("UserAuthorizationRequired", USER_AUTHORIZATION_REQUIRED_FIELD);
                    put("Tags", TAGS_FIELD);
                    put("State", STATE_FIELD);
                    put("StateReason", STATE_REASON_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String slackTeamName;

    private final String slackTeamId;

    private final String slackChannelId;

    private final String slackChannelName;

    private final String chatConfigurationArn;

    private final String iamRoleArn;

    private final List<String> snsTopicArns;

    private final String configurationName;

    private final String loggingLevel;

    private final List<String> guardrailPolicyArns;

    private final Boolean userAuthorizationRequired;

    private final List<Tag> tags;

    private final String state;

    private final String stateReason;

    private SlackChannelConfiguration(BuilderImpl builder) {
        this.slackTeamName = builder.slackTeamName;
        this.slackTeamId = builder.slackTeamId;
        this.slackChannelId = builder.slackChannelId;
        this.slackChannelName = builder.slackChannelName;
        this.chatConfigurationArn = builder.chatConfigurationArn;
        this.iamRoleArn = builder.iamRoleArn;
        this.snsTopicArns = builder.snsTopicArns;
        this.configurationName = builder.configurationName;
        this.loggingLevel = builder.loggingLevel;
        this.guardrailPolicyArns = builder.guardrailPolicyArns;
        this.userAuthorizationRequired = builder.userAuthorizationRequired;
        this.tags = builder.tags;
        this.state = builder.state;
        this.stateReason = builder.stateReason;
    }

    /**
     * <p>
     * Name of the Slack workspace.
     * </p>
     * 
     * @return Name of the Slack workspace.
     */
    public final String slackTeamName() {
        return slackTeamName;
    }

    /**
     * <p>
     * The ID of the Slack workspace authorized with Amazon Chime.
     * </p>
     * 
     * @return The ID of the Slack workspace authorized with Amazon Chime.
     */
    public final String slackTeamId() {
        return slackTeamId;
    }

    /**
     * <p>
     * The ID of the Slack channel.
     * </p>
     * <p>
     * To get this ID, open Slack, right click on the channel name in the left pane, then choose Copy Link. The channel
     * ID is the 9-character string at the end of the URL. For example, ABCBBLZZZ.
     * </p>
     * 
     * @return The ID of the Slack channel.</p>
     *         <p>
     *         To get this ID, open Slack, right click on the channel name in the left pane, then choose Copy Link. The
     *         channel ID is the 9-character string at the end of the URL. For example, ABCBBLZZZ.
     */
    public final String slackChannelId() {
        return slackChannelId;
    }

    /**
     * <p>
     * The name of the Slack channel.
     * </p>
     * 
     * @return The name of the Slack channel.
     */
    public final String slackChannelName() {
        return slackChannelName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the SlackChannelConfiguration.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the SlackChannelConfiguration.
     */
    public final String chatConfigurationArn() {
        return chatConfigurationArn;
    }

    /**
     * <p>
     * A user-defined role that AWS Chatbot assumes. This is not the service-linked role.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/chatbot/latest/adminguide/chatbot-iam-policies.html">IAM policies for AWS
     * Chatbot</a> in the <i> AWS Chatbot Administrator Guide</i>.
     * </p>
     * 
     * @return A user-defined role that AWS Chatbot assumes. This is not the service-linked role.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/chatbot/latest/adminguide/chatbot-iam-policies.html">IAM policies for
     *         AWS Chatbot</a> in the <i> AWS Chatbot Administrator Guide</i>.
     */
    public final String iamRoleArn() {
        return iamRoleArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the SnsTopicArns property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSnsTopicArns() {
        return snsTopicArns != null && !(snsTopicArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The ARNs of the SNS topics that deliver notifications to AWS Chatbot.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSnsTopicArns} method.
     * </p>
     * 
     * @return The ARNs of the SNS topics that deliver notifications to AWS Chatbot.
     */
    public final List<String> snsTopicArns() {
        return snsTopicArns;
    }

    /**
     * <p>
     * The name of the configuration.
     * </p>
     * 
     * @return The name of the configuration.
     */
    public final String configurationName() {
        return configurationName;
    }

    /**
     * <p>
     * Logging levels include <code>ERROR</code>, <code>INFO</code>, or <code>NONE</code>.
     * </p>
     * 
     * @return Logging levels include <code>ERROR</code>, <code>INFO</code>, or <code>NONE</code>.
     */
    public final String loggingLevel() {
        return loggingLevel;
    }

    /**
     * For responses, this returns true if the service returned a value for the GuardrailPolicyArns property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasGuardrailPolicyArns() {
        return guardrailPolicyArns != null && !(guardrailPolicyArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of IAM policy ARNs that are applied as channel guardrails. The AWS managed
     * <code>AdministratorAccess</code> policy is applied by default if this is not set.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGuardrailPolicyArns} method.
     * </p>
     * 
     * @return The list of IAM policy ARNs that are applied as channel guardrails. The AWS managed
     *         <code>AdministratorAccess</code> policy is applied by default if this is not set.
     */
    public final List<String> guardrailPolicyArns() {
        return guardrailPolicyArns;
    }

    /**
     * <p>
     * Enables use of a user role requirement in your chat configuration.
     * </p>
     * 
     * @return Enables use of a user role requirement in your chat configuration.
     */
    public final Boolean userAuthorizationRequired() {
        return userAuthorizationRequired;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A map of tags assigned to a resource. A tag is a string-to-string map of key-value pairs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A map of tags assigned to a resource. A tag is a string-to-string map of key-value pairs.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Either <code>ENABLED</code> or <code>DISABLED</code>. The resource returns <code>DISABLED</code> if the
     * organization's AWS Chatbot policy has explicitly denied that configuration. For example, if Amazon Chime is
     * disabled.
     * </p>
     * 
     * @return Either <code>ENABLED</code> or <code>DISABLED</code>. The resource returns <code>DISABLED</code> if the
     *         organization's AWS Chatbot policy has explicitly denied that configuration. For example, if Amazon Chime
     *         is disabled.
     */
    public final String state() {
        return state;
    }

    /**
     * <p>
     * Provided if State is <code>DISABLED</code>. Provides context as to why the resource is disabled.
     * </p>
     * 
     * @return Provided if State is <code>DISABLED</code>. Provides context as to why the resource is disabled.
     */
    public final String stateReason() {
        return stateReason;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(slackTeamName());
        hashCode = 31 * hashCode + Objects.hashCode(slackTeamId());
        hashCode = 31 * hashCode + Objects.hashCode(slackChannelId());
        hashCode = 31 * hashCode + Objects.hashCode(slackChannelName());
        hashCode = 31 * hashCode + Objects.hashCode(chatConfigurationArn());
        hashCode = 31 * hashCode + Objects.hashCode(iamRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasSnsTopicArns() ? snsTopicArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(configurationName());
        hashCode = 31 * hashCode + Objects.hashCode(loggingLevel());
        hashCode = 31 * hashCode + Objects.hashCode(hasGuardrailPolicyArns() ? guardrailPolicyArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(userAuthorizationRequired());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(state());
        hashCode = 31 * hashCode + Objects.hashCode(stateReason());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SlackChannelConfiguration)) {
            return false;
        }
        SlackChannelConfiguration other = (SlackChannelConfiguration) obj;
        return Objects.equals(slackTeamName(), other.slackTeamName()) && Objects.equals(slackTeamId(), other.slackTeamId())
                && Objects.equals(slackChannelId(), other.slackChannelId())
                && Objects.equals(slackChannelName(), other.slackChannelName())
                && Objects.equals(chatConfigurationArn(), other.chatConfigurationArn())
                && Objects.equals(iamRoleArn(), other.iamRoleArn()) && hasSnsTopicArns() == other.hasSnsTopicArns()
                && Objects.equals(snsTopicArns(), other.snsTopicArns())
                && Objects.equals(configurationName(), other.configurationName())
                && Objects.equals(loggingLevel(), other.loggingLevel())
                && hasGuardrailPolicyArns() == other.hasGuardrailPolicyArns()
                && Objects.equals(guardrailPolicyArns(), other.guardrailPolicyArns())
                && Objects.equals(userAuthorizationRequired(), other.userAuthorizationRequired()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(state(), other.state())
                && Objects.equals(stateReason(), other.stateReason());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SlackChannelConfiguration").add("SlackTeamName", slackTeamName())
                .add("SlackTeamId", slackTeamId()).add("SlackChannelId", slackChannelId())
                .add("SlackChannelName", slackChannelName() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ChatConfigurationArn", chatConfigurationArn()).add("IamRoleArn", iamRoleArn())
                .add("SnsTopicArns", hasSnsTopicArns() ? snsTopicArns() : null).add("ConfigurationName", configurationName())
                .add("LoggingLevel", loggingLevel())
                .add("GuardrailPolicyArns", hasGuardrailPolicyArns() ? guardrailPolicyArns() : null)
                .add("UserAuthorizationRequired", userAuthorizationRequired()).add("Tags", hasTags() ? tags() : null)
                .add("State", state()).add("StateReason", stateReason()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SlackTeamName":
            return Optional.ofNullable(clazz.cast(slackTeamName()));
        case "SlackTeamId":
            return Optional.ofNullable(clazz.cast(slackTeamId()));
        case "SlackChannelId":
            return Optional.ofNullable(clazz.cast(slackChannelId()));
        case "SlackChannelName":
            return Optional.ofNullable(clazz.cast(slackChannelName()));
        case "ChatConfigurationArn":
            return Optional.ofNullable(clazz.cast(chatConfigurationArn()));
        case "IamRoleArn":
            return Optional.ofNullable(clazz.cast(iamRoleArn()));
        case "SnsTopicArns":
            return Optional.ofNullable(clazz.cast(snsTopicArns()));
        case "ConfigurationName":
            return Optional.ofNullable(clazz.cast(configurationName()));
        case "LoggingLevel":
            return Optional.ofNullable(clazz.cast(loggingLevel()));
        case "GuardrailPolicyArns":
            return Optional.ofNullable(clazz.cast(guardrailPolicyArns()));
        case "UserAuthorizationRequired":
            return Optional.ofNullable(clazz.cast(userAuthorizationRequired()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "State":
            return Optional.ofNullable(clazz.cast(state()));
        case "StateReason":
            return Optional.ofNullable(clazz.cast(stateReason()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<SlackChannelConfiguration, T> g) {
        return obj -> g.apply((SlackChannelConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SlackChannelConfiguration> {
        /**
         * <p>
         * Name of the Slack workspace.
         * </p>
         * 
         * @param slackTeamName
         *        Name of the Slack workspace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slackTeamName(String slackTeamName);

        /**
         * <p>
         * The ID of the Slack workspace authorized with Amazon Chime.
         * </p>
         * 
         * @param slackTeamId
         *        The ID of the Slack workspace authorized with Amazon Chime.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slackTeamId(String slackTeamId);

        /**
         * <p>
         * The ID of the Slack channel.
         * </p>
         * <p>
         * To get this ID, open Slack, right click on the channel name in the left pane, then choose Copy Link. The
         * channel ID is the 9-character string at the end of the URL. For example, ABCBBLZZZ.
         * </p>
         * 
         * @param slackChannelId
         *        The ID of the Slack channel.</p>
         *        <p>
         *        To get this ID, open Slack, right click on the channel name in the left pane, then choose Copy Link.
         *        The channel ID is the 9-character string at the end of the URL. For example, ABCBBLZZZ.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slackChannelId(String slackChannelId);

        /**
         * <p>
         * The name of the Slack channel.
         * </p>
         * 
         * @param slackChannelName
         *        The name of the Slack channel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder slackChannelName(String slackChannelName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the SlackChannelConfiguration.
         * </p>
         * 
         * @param chatConfigurationArn
         *        The Amazon Resource Name (ARN) of the SlackChannelConfiguration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder chatConfigurationArn(String chatConfigurationArn);

        /**
         * <p>
         * A user-defined role that AWS Chatbot assumes. This is not the service-linked role.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/chatbot/latest/adminguide/chatbot-iam-policies.html">IAM policies for AWS
         * Chatbot</a> in the <i> AWS Chatbot Administrator Guide</i>.
         * </p>
         * 
         * @param iamRoleArn
         *        A user-defined role that AWS Chatbot assumes. This is not the service-linked role.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/chatbot/latest/adminguide/chatbot-iam-policies.html">IAM policies
         *        for AWS Chatbot</a> in the <i> AWS Chatbot Administrator Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamRoleArn(String iamRoleArn);

        /**
         * <p>
         * The ARNs of the SNS topics that deliver notifications to AWS Chatbot.
         * </p>
         * 
         * @param snsTopicArns
         *        The ARNs of the SNS topics that deliver notifications to AWS Chatbot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snsTopicArns(Collection<String> snsTopicArns);

        /**
         * <p>
         * The ARNs of the SNS topics that deliver notifications to AWS Chatbot.
         * </p>
         * 
         * @param snsTopicArns
         *        The ARNs of the SNS topics that deliver notifications to AWS Chatbot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snsTopicArns(String... snsTopicArns);

        /**
         * <p>
         * The name of the configuration.
         * </p>
         * 
         * @param configurationName
         *        The name of the configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationName(String configurationName);

        /**
         * <p>
         * Logging levels include <code>ERROR</code>, <code>INFO</code>, or <code>NONE</code>.
         * </p>
         * 
         * @param loggingLevel
         *        Logging levels include <code>ERROR</code>, <code>INFO</code>, or <code>NONE</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loggingLevel(String loggingLevel);

        /**
         * <p>
         * The list of IAM policy ARNs that are applied as channel guardrails. The AWS managed
         * <code>AdministratorAccess</code> policy is applied by default if this is not set.
         * </p>
         * 
         * @param guardrailPolicyArns
         *        The list of IAM policy ARNs that are applied as channel guardrails. The AWS managed
         *        <code>AdministratorAccess</code> policy is applied by default if this is not set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder guardrailPolicyArns(Collection<String> guardrailPolicyArns);

        /**
         * <p>
         * The list of IAM policy ARNs that are applied as channel guardrails. The AWS managed
         * <code>AdministratorAccess</code> policy is applied by default if this is not set.
         * </p>
         * 
         * @param guardrailPolicyArns
         *        The list of IAM policy ARNs that are applied as channel guardrails. The AWS managed
         *        <code>AdministratorAccess</code> policy is applied by default if this is not set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder guardrailPolicyArns(String... guardrailPolicyArns);

        /**
         * <p>
         * Enables use of a user role requirement in your chat configuration.
         * </p>
         * 
         * @param userAuthorizationRequired
         *        Enables use of a user role requirement in your chat configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAuthorizationRequired(Boolean userAuthorizationRequired);

        /**
         * <p>
         * A map of tags assigned to a resource. A tag is a string-to-string map of key-value pairs.
         * </p>
         * 
         * @param tags
         *        A map of tags assigned to a resource. A tag is a string-to-string map of key-value pairs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A map of tags assigned to a resource. A tag is a string-to-string map of key-value pairs.
         * </p>
         * 
         * @param tags
         *        A map of tags assigned to a resource. A tag is a string-to-string map of key-value pairs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A map of tags assigned to a resource. A tag is a string-to-string map of key-value pairs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.chatbot.model.Tag.Builder} avoiding the need to create one manually
         * via {@link software.amazon.awssdk.services.chatbot.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.chatbot.model.Tag.Builder#build()} is called immediately and its
         * result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.chatbot.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Either <code>ENABLED</code> or <code>DISABLED</code>. The resource returns <code>DISABLED</code> if the
         * organization's AWS Chatbot policy has explicitly denied that configuration. For example, if Amazon Chime is
         * disabled.
         * </p>
         * 
         * @param state
         *        Either <code>ENABLED</code> or <code>DISABLED</code>. The resource returns <code>DISABLED</code> if
         *        the organization's AWS Chatbot policy has explicitly denied that configuration. For example, if Amazon
         *        Chime is disabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder state(String state);

        /**
         * <p>
         * Provided if State is <code>DISABLED</code>. Provides context as to why the resource is disabled.
         * </p>
         * 
         * @param stateReason
         *        Provided if State is <code>DISABLED</code>. Provides context as to why the resource is disabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateReason(String stateReason);
    }

    static final class BuilderImpl implements Builder {
        private String slackTeamName;

        private String slackTeamId;

        private String slackChannelId;

        private String slackChannelName;

        private String chatConfigurationArn;

        private String iamRoleArn;

        private List<String> snsTopicArns = DefaultSdkAutoConstructList.getInstance();

        private String configurationName;

        private String loggingLevel;

        private List<String> guardrailPolicyArns = DefaultSdkAutoConstructList.getInstance();

        private Boolean userAuthorizationRequired;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String state;

        private String stateReason;

        private BuilderImpl() {
        }

        private BuilderImpl(SlackChannelConfiguration model) {
            slackTeamName(model.slackTeamName);
            slackTeamId(model.slackTeamId);
            slackChannelId(model.slackChannelId);
            slackChannelName(model.slackChannelName);
            chatConfigurationArn(model.chatConfigurationArn);
            iamRoleArn(model.iamRoleArn);
            snsTopicArns(model.snsTopicArns);
            configurationName(model.configurationName);
            loggingLevel(model.loggingLevel);
            guardrailPolicyArns(model.guardrailPolicyArns);
            userAuthorizationRequired(model.userAuthorizationRequired);
            tags(model.tags);
            state(model.state);
            stateReason(model.stateReason);
        }

        public final String getSlackTeamName() {
            return slackTeamName;
        }

        public final void setSlackTeamName(String slackTeamName) {
            this.slackTeamName = slackTeamName;
        }

        @Override
        public final Builder slackTeamName(String slackTeamName) {
            this.slackTeamName = slackTeamName;
            return this;
        }

        public final String getSlackTeamId() {
            return slackTeamId;
        }

        public final void setSlackTeamId(String slackTeamId) {
            this.slackTeamId = slackTeamId;
        }

        @Override
        public final Builder slackTeamId(String slackTeamId) {
            this.slackTeamId = slackTeamId;
            return this;
        }

        public final String getSlackChannelId() {
            return slackChannelId;
        }

        public final void setSlackChannelId(String slackChannelId) {
            this.slackChannelId = slackChannelId;
        }

        @Override
        public final Builder slackChannelId(String slackChannelId) {
            this.slackChannelId = slackChannelId;
            return this;
        }

        public final String getSlackChannelName() {
            return slackChannelName;
        }

        public final void setSlackChannelName(String slackChannelName) {
            this.slackChannelName = slackChannelName;
        }

        @Override
        public final Builder slackChannelName(String slackChannelName) {
            this.slackChannelName = slackChannelName;
            return this;
        }

        public final String getChatConfigurationArn() {
            return chatConfigurationArn;
        }

        public final void setChatConfigurationArn(String chatConfigurationArn) {
            this.chatConfigurationArn = chatConfigurationArn;
        }

        @Override
        public final Builder chatConfigurationArn(String chatConfigurationArn) {
            this.chatConfigurationArn = chatConfigurationArn;
            return this;
        }

        public final String getIamRoleArn() {
            return iamRoleArn;
        }

        public final void setIamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
        }

        @Override
        public final Builder iamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
            return this;
        }

        public final Collection<String> getSnsTopicArns() {
            if (snsTopicArns instanceof SdkAutoConstructList) {
                return null;
            }
            return snsTopicArns;
        }

        public final void setSnsTopicArns(Collection<String> snsTopicArns) {
            this.snsTopicArns = SnsTopicArnListCopier.copy(snsTopicArns);
        }

        @Override
        public final Builder snsTopicArns(Collection<String> snsTopicArns) {
            this.snsTopicArns = SnsTopicArnListCopier.copy(snsTopicArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder snsTopicArns(String... snsTopicArns) {
            snsTopicArns(Arrays.asList(snsTopicArns));
            return this;
        }

        public final String getConfigurationName() {
            return configurationName;
        }

        public final void setConfigurationName(String configurationName) {
            this.configurationName = configurationName;
        }

        @Override
        public final Builder configurationName(String configurationName) {
            this.configurationName = configurationName;
            return this;
        }

        public final String getLoggingLevel() {
            return loggingLevel;
        }

        public final void setLoggingLevel(String loggingLevel) {
            this.loggingLevel = loggingLevel;
        }

        @Override
        public final Builder loggingLevel(String loggingLevel) {
            this.loggingLevel = loggingLevel;
            return this;
        }

        public final Collection<String> getGuardrailPolicyArns() {
            if (guardrailPolicyArns instanceof SdkAutoConstructList) {
                return null;
            }
            return guardrailPolicyArns;
        }

        public final void setGuardrailPolicyArns(Collection<String> guardrailPolicyArns) {
            this.guardrailPolicyArns = GuardrailPolicyArnListCopier.copy(guardrailPolicyArns);
        }

        @Override
        public final Builder guardrailPolicyArns(Collection<String> guardrailPolicyArns) {
            this.guardrailPolicyArns = GuardrailPolicyArnListCopier.copy(guardrailPolicyArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder guardrailPolicyArns(String... guardrailPolicyArns) {
            guardrailPolicyArns(Arrays.asList(guardrailPolicyArns));
            return this;
        }

        public final Boolean getUserAuthorizationRequired() {
            return userAuthorizationRequired;
        }

        public final void setUserAuthorizationRequired(Boolean userAuthorizationRequired) {
            this.userAuthorizationRequired = userAuthorizationRequired;
        }

        @Override
        public final Builder userAuthorizationRequired(Boolean userAuthorizationRequired) {
            this.userAuthorizationRequired = userAuthorizationRequired;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        public final String getStateReason() {
            return stateReason;
        }

        public final void setStateReason(String stateReason) {
            this.stateReason = stateReason;
        }

        @Override
        public final Builder stateReason(String stateReason) {
            this.stateReason = stateReason;
            return this;
        }

        @Override
        public SlackChannelConfiguration build() {
            return new SlackChannelConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
