/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chime.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The Amazon Chime account details. An AWS account can have multiple Amazon Chime accounts.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Account implements SdkPojo, Serializable, ToCopyableBuilder<Account.Builder, Account> {
    private static final SdkField<String> AWS_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AwsAccountId").getter(getter(Account::awsAccountId)).setter(setter(Builder::awsAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AwsAccountId").build()).build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountId").getter(getter(Account::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Account::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> ACCOUNT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountType").getter(getter(Account::accountTypeAsString)).setter(setter(Builder::accountType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountType").build()).build();

    private static final SdkField<Instant> CREATED_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTimestamp")
            .getter(getter(Account::createdTimestamp))
            .setter(setter(Builder::createdTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTimestamp").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> DEFAULT_LICENSE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DefaultLicense").getter(getter(Account::defaultLicenseAsString)).setter(setter(Builder::defaultLicense))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultLicense").build()).build();

    private static final SdkField<List<String>> SUPPORTED_LICENSES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SupportedLicenses")
            .getter(getter(Account::supportedLicensesAsStrings))
            .setter(setter(Builder::supportedLicensesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SupportedLicenses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ACCOUNT_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountStatus").getter(getter(Account::accountStatusAsString)).setter(setter(Builder::accountStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountStatus").build()).build();

    private static final SdkField<List<SigninDelegateGroup>> SIGNIN_DELEGATE_GROUPS_FIELD = SdkField
            .<List<SigninDelegateGroup>> builder(MarshallingType.LIST)
            .memberName("SigninDelegateGroups")
            .getter(getter(Account::signinDelegateGroups))
            .setter(setter(Builder::signinDelegateGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SigninDelegateGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SigninDelegateGroup> builder(MarshallingType.SDK_POJO)
                                            .constructor(SigninDelegateGroup::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AWS_ACCOUNT_ID_FIELD,
            ACCOUNT_ID_FIELD, NAME_FIELD, ACCOUNT_TYPE_FIELD, CREATED_TIMESTAMP_FIELD, DEFAULT_LICENSE_FIELD,
            SUPPORTED_LICENSES_FIELD, ACCOUNT_STATUS_FIELD, SIGNIN_DELEGATE_GROUPS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String awsAccountId;

    private final String accountId;

    private final String name;

    private final String accountType;

    private final Instant createdTimestamp;

    private final String defaultLicense;

    private final List<String> supportedLicenses;

    private final String accountStatus;

    private final List<SigninDelegateGroup> signinDelegateGroups;

    private Account(BuilderImpl builder) {
        this.awsAccountId = builder.awsAccountId;
        this.accountId = builder.accountId;
        this.name = builder.name;
        this.accountType = builder.accountType;
        this.createdTimestamp = builder.createdTimestamp;
        this.defaultLicense = builder.defaultLicense;
        this.supportedLicenses = builder.supportedLicenses;
        this.accountStatus = builder.accountStatus;
        this.signinDelegateGroups = builder.signinDelegateGroups;
    }

    /**
     * <p>
     * The AWS account ID.
     * </p>
     * 
     * @return The AWS account ID.
     */
    public final String awsAccountId() {
        return awsAccountId;
    }

    /**
     * <p>
     * The Amazon Chime account ID.
     * </p>
     * 
     * @return The Amazon Chime account ID.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The Amazon Chime account name.
     * </p>
     * 
     * @return The Amazon Chime account name.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Chime account type. For more information about different account types, see <a
     * href="https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html">Managing Your Amazon Chime
     * Accounts</a> in the <i>Amazon Chime Administration Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accountType} will
     * return {@link AccountType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #accountTypeAsString}.
     * </p>
     * 
     * @return The Amazon Chime account type. For more information about different account types, see <a
     *         href="https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html">Managing Your Amazon Chime
     *         Accounts</a> in the <i>Amazon Chime Administration Guide</i>.
     * @see AccountType
     */
    public final AccountType accountType() {
        return AccountType.fromValue(accountType);
    }

    /**
     * <p>
     * The Amazon Chime account type. For more information about different account types, see <a
     * href="https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html">Managing Your Amazon Chime
     * Accounts</a> in the <i>Amazon Chime Administration Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accountType} will
     * return {@link AccountType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #accountTypeAsString}.
     * </p>
     * 
     * @return The Amazon Chime account type. For more information about different account types, see <a
     *         href="https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html">Managing Your Amazon Chime
     *         Accounts</a> in the <i>Amazon Chime Administration Guide</i>.
     * @see AccountType
     */
    public final String accountTypeAsString() {
        return accountType;
    }

    /**
     * <p>
     * The Amazon Chime account creation timestamp, in ISO 8601 format.
     * </p>
     * 
     * @return The Amazon Chime account creation timestamp, in ISO 8601 format.
     */
    public final Instant createdTimestamp() {
        return createdTimestamp;
    }

    /**
     * <p>
     * The default license for the Amazon Chime account.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #defaultLicense}
     * will return {@link License#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #defaultLicenseAsString}.
     * </p>
     * 
     * @return The default license for the Amazon Chime account.
     * @see License
     */
    public final License defaultLicense() {
        return License.fromValue(defaultLicense);
    }

    /**
     * <p>
     * The default license for the Amazon Chime account.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #defaultLicense}
     * will return {@link License#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #defaultLicenseAsString}.
     * </p>
     * 
     * @return The default license for the Amazon Chime account.
     * @see License
     */
    public final String defaultLicenseAsString() {
        return defaultLicense;
    }

    /**
     * <p>
     * Supported licenses for the Amazon Chime account.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSupportedLicenses()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Supported licenses for the Amazon Chime account.
     */
    public final List<License> supportedLicenses() {
        return LicenseListCopier.copyStringToEnum(supportedLicenses);
    }

    /**
     * Returns true if the SupportedLicenses property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasSupportedLicenses() {
        return supportedLicenses != null && !(supportedLicenses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Supported licenses for the Amazon Chime account.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSupportedLicenses()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Supported licenses for the Amazon Chime account.
     */
    public final List<String> supportedLicensesAsStrings() {
        return supportedLicenses;
    }

    /**
     * <p>
     * The status of the account, <code>Suspended</code> or <code>Active</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accountStatus}
     * will return {@link AccountStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #accountStatusAsString}.
     * </p>
     * 
     * @return The status of the account, <code>Suspended</code> or <code>Active</code>.
     * @see AccountStatus
     */
    public final AccountStatus accountStatus() {
        return AccountStatus.fromValue(accountStatus);
    }

    /**
     * <p>
     * The status of the account, <code>Suspended</code> or <code>Active</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #accountStatus}
     * will return {@link AccountStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #accountStatusAsString}.
     * </p>
     * 
     * @return The status of the account, <code>Suspended</code> or <code>Active</code>.
     * @see AccountStatus
     */
    public final String accountStatusAsString() {
        return accountStatus;
    }

    /**
     * Returns true if the SigninDelegateGroups property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasSigninDelegateGroups() {
        return signinDelegateGroups != null && !(signinDelegateGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The sign-in delegate groups associated with the account.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSigninDelegateGroups()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The sign-in delegate groups associated with the account.
     */
    public final List<SigninDelegateGroup> signinDelegateGroups() {
        return signinDelegateGroups;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(awsAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(accountTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(defaultLicenseAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasSupportedLicenses() ? supportedLicensesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(accountStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasSigninDelegateGroups() ? signinDelegateGroups() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Account)) {
            return false;
        }
        Account other = (Account) obj;
        return Objects.equals(awsAccountId(), other.awsAccountId()) && Objects.equals(accountId(), other.accountId())
                && Objects.equals(name(), other.name()) && Objects.equals(accountTypeAsString(), other.accountTypeAsString())
                && Objects.equals(createdTimestamp(), other.createdTimestamp())
                && Objects.equals(defaultLicenseAsString(), other.defaultLicenseAsString())
                && hasSupportedLicenses() == other.hasSupportedLicenses()
                && Objects.equals(supportedLicensesAsStrings(), other.supportedLicensesAsStrings())
                && Objects.equals(accountStatusAsString(), other.accountStatusAsString())
                && hasSigninDelegateGroups() == other.hasSigninDelegateGroups()
                && Objects.equals(signinDelegateGroups(), other.signinDelegateGroups());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Account").add("AwsAccountId", awsAccountId()).add("AccountId", accountId()).add("Name", name())
                .add("AccountType", accountTypeAsString()).add("CreatedTimestamp", createdTimestamp())
                .add("DefaultLicense", defaultLicenseAsString())
                .add("SupportedLicenses", hasSupportedLicenses() ? supportedLicensesAsStrings() : null)
                .add("AccountStatus", accountStatusAsString())
                .add("SigninDelegateGroups", hasSigninDelegateGroups() ? signinDelegateGroups() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AwsAccountId":
            return Optional.ofNullable(clazz.cast(awsAccountId()));
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "AccountType":
            return Optional.ofNullable(clazz.cast(accountTypeAsString()));
        case "CreatedTimestamp":
            return Optional.ofNullable(clazz.cast(createdTimestamp()));
        case "DefaultLicense":
            return Optional.ofNullable(clazz.cast(defaultLicenseAsString()));
        case "SupportedLicenses":
            return Optional.ofNullable(clazz.cast(supportedLicensesAsStrings()));
        case "AccountStatus":
            return Optional.ofNullable(clazz.cast(accountStatusAsString()));
        case "SigninDelegateGroups":
            return Optional.ofNullable(clazz.cast(signinDelegateGroups()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Account, T> g) {
        return obj -> g.apply((Account) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Account> {
        /**
         * <p>
         * The AWS account ID.
         * </p>
         * 
         * @param awsAccountId
         *        The AWS account ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsAccountId(String awsAccountId);

        /**
         * <p>
         * The Amazon Chime account ID.
         * </p>
         * 
         * @param accountId
         *        The Amazon Chime account ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The Amazon Chime account name.
         * </p>
         * 
         * @param name
         *        The Amazon Chime account name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Chime account type. For more information about different account types, see <a
         * href="https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html">Managing Your Amazon Chime
         * Accounts</a> in the <i>Amazon Chime Administration Guide</i>.
         * </p>
         * 
         * @param accountType
         *        The Amazon Chime account type. For more information about different account types, see <a
         *        href="https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html">Managing Your Amazon
         *        Chime Accounts</a> in the <i>Amazon Chime Administration Guide</i>.
         * @see AccountType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccountType
         */
        Builder accountType(String accountType);

        /**
         * <p>
         * The Amazon Chime account type. For more information about different account types, see <a
         * href="https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html">Managing Your Amazon Chime
         * Accounts</a> in the <i>Amazon Chime Administration Guide</i>.
         * </p>
         * 
         * @param accountType
         *        The Amazon Chime account type. For more information about different account types, see <a
         *        href="https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html">Managing Your Amazon
         *        Chime Accounts</a> in the <i>Amazon Chime Administration Guide</i>.
         * @see AccountType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccountType
         */
        Builder accountType(AccountType accountType);

        /**
         * <p>
         * The Amazon Chime account creation timestamp, in ISO 8601 format.
         * </p>
         * 
         * @param createdTimestamp
         *        The Amazon Chime account creation timestamp, in ISO 8601 format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTimestamp(Instant createdTimestamp);

        /**
         * <p>
         * The default license for the Amazon Chime account.
         * </p>
         * 
         * @param defaultLicense
         *        The default license for the Amazon Chime account.
         * @see License
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see License
         */
        Builder defaultLicense(String defaultLicense);

        /**
         * <p>
         * The default license for the Amazon Chime account.
         * </p>
         * 
         * @param defaultLicense
         *        The default license for the Amazon Chime account.
         * @see License
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see License
         */
        Builder defaultLicense(License defaultLicense);

        /**
         * <p>
         * Supported licenses for the Amazon Chime account.
         * </p>
         * 
         * @param supportedLicenses
         *        Supported licenses for the Amazon Chime account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedLicensesWithStrings(Collection<String> supportedLicenses);

        /**
         * <p>
         * Supported licenses for the Amazon Chime account.
         * </p>
         * 
         * @param supportedLicenses
         *        Supported licenses for the Amazon Chime account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedLicensesWithStrings(String... supportedLicenses);

        /**
         * <p>
         * Supported licenses for the Amazon Chime account.
         * </p>
         * 
         * @param supportedLicenses
         *        Supported licenses for the Amazon Chime account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedLicenses(Collection<License> supportedLicenses);

        /**
         * <p>
         * Supported licenses for the Amazon Chime account.
         * </p>
         * 
         * @param supportedLicenses
         *        Supported licenses for the Amazon Chime account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder supportedLicenses(License... supportedLicenses);

        /**
         * <p>
         * The status of the account, <code>Suspended</code> or <code>Active</code>.
         * </p>
         * 
         * @param accountStatus
         *        The status of the account, <code>Suspended</code> or <code>Active</code>.
         * @see AccountStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccountStatus
         */
        Builder accountStatus(String accountStatus);

        /**
         * <p>
         * The status of the account, <code>Suspended</code> or <code>Active</code>.
         * </p>
         * 
         * @param accountStatus
         *        The status of the account, <code>Suspended</code> or <code>Active</code>.
         * @see AccountStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccountStatus
         */
        Builder accountStatus(AccountStatus accountStatus);

        /**
         * <p>
         * The sign-in delegate groups associated with the account.
         * </p>
         * 
         * @param signinDelegateGroups
         *        The sign-in delegate groups associated with the account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signinDelegateGroups(Collection<SigninDelegateGroup> signinDelegateGroups);

        /**
         * <p>
         * The sign-in delegate groups associated with the account.
         * </p>
         * 
         * @param signinDelegateGroups
         *        The sign-in delegate groups associated with the account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signinDelegateGroups(SigninDelegateGroup... signinDelegateGroups);

        /**
         * <p>
         * The sign-in delegate groups associated with the account.
         * </p>
         * This is a convenience that creates an instance of the {@link List<SigninDelegateGroup>.Builder} avoiding the
         * need to create one manually via {@link List<SigninDelegateGroup>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<SigninDelegateGroup>.Builder#build()} is called immediately
         * and its result is passed to {@link #signinDelegateGroups(List<SigninDelegateGroup>)}.
         * 
         * @param signinDelegateGroups
         *        a consumer that will call methods on {@link List<SigninDelegateGroup>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #signinDelegateGroups(List<SigninDelegateGroup>)
         */
        Builder signinDelegateGroups(Consumer<SigninDelegateGroup.Builder>... signinDelegateGroups);
    }

    static final class BuilderImpl implements Builder {
        private String awsAccountId;

        private String accountId;

        private String name;

        private String accountType;

        private Instant createdTimestamp;

        private String defaultLicense;

        private List<String> supportedLicenses = DefaultSdkAutoConstructList.getInstance();

        private String accountStatus;

        private List<SigninDelegateGroup> signinDelegateGroups = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Account model) {
            awsAccountId(model.awsAccountId);
            accountId(model.accountId);
            name(model.name);
            accountType(model.accountType);
            createdTimestamp(model.createdTimestamp);
            defaultLicense(model.defaultLicense);
            supportedLicensesWithStrings(model.supportedLicenses);
            accountStatus(model.accountStatus);
            signinDelegateGroups(model.signinDelegateGroups);
        }

        public final String getAwsAccountId() {
            return awsAccountId;
        }

        @Override
        public final Builder awsAccountId(String awsAccountId) {
            this.awsAccountId = awsAccountId;
            return this;
        }

        public final void setAwsAccountId(String awsAccountId) {
            this.awsAccountId = awsAccountId;
        }

        public final String getAccountId() {
            return accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getAccountType() {
            return accountType;
        }

        @Override
        public final Builder accountType(String accountType) {
            this.accountType = accountType;
            return this;
        }

        @Override
        public final Builder accountType(AccountType accountType) {
            this.accountType(accountType == null ? null : accountType.toString());
            return this;
        }

        public final void setAccountType(String accountType) {
            this.accountType = accountType;
        }

        public final Instant getCreatedTimestamp() {
            return createdTimestamp;
        }

        @Override
        public final Builder createdTimestamp(Instant createdTimestamp) {
            this.createdTimestamp = createdTimestamp;
            return this;
        }

        public final void setCreatedTimestamp(Instant createdTimestamp) {
            this.createdTimestamp = createdTimestamp;
        }

        public final String getDefaultLicense() {
            return defaultLicense;
        }

        @Override
        public final Builder defaultLicense(String defaultLicense) {
            this.defaultLicense = defaultLicense;
            return this;
        }

        @Override
        public final Builder defaultLicense(License defaultLicense) {
            this.defaultLicense(defaultLicense == null ? null : defaultLicense.toString());
            return this;
        }

        public final void setDefaultLicense(String defaultLicense) {
            this.defaultLicense = defaultLicense;
        }

        public final Collection<String> getSupportedLicenses() {
            if (supportedLicenses instanceof SdkAutoConstructList) {
                return null;
            }
            return supportedLicenses;
        }

        @Override
        public final Builder supportedLicensesWithStrings(Collection<String> supportedLicenses) {
            this.supportedLicenses = LicenseListCopier.copy(supportedLicenses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedLicensesWithStrings(String... supportedLicenses) {
            supportedLicensesWithStrings(Arrays.asList(supportedLicenses));
            return this;
        }

        @Override
        public final Builder supportedLicenses(Collection<License> supportedLicenses) {
            this.supportedLicenses = LicenseListCopier.copyEnumToString(supportedLicenses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder supportedLicenses(License... supportedLicenses) {
            supportedLicenses(Arrays.asList(supportedLicenses));
            return this;
        }

        public final void setSupportedLicenses(Collection<String> supportedLicenses) {
            this.supportedLicenses = LicenseListCopier.copy(supportedLicenses);
        }

        public final String getAccountStatus() {
            return accountStatus;
        }

        @Override
        public final Builder accountStatus(String accountStatus) {
            this.accountStatus = accountStatus;
            return this;
        }

        @Override
        public final Builder accountStatus(AccountStatus accountStatus) {
            this.accountStatus(accountStatus == null ? null : accountStatus.toString());
            return this;
        }

        public final void setAccountStatus(String accountStatus) {
            this.accountStatus = accountStatus;
        }

        public final List<SigninDelegateGroup.Builder> getSigninDelegateGroups() {
            List<SigninDelegateGroup.Builder> result = SigninDelegateGroupListCopier.copyToBuilder(this.signinDelegateGroups);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder signinDelegateGroups(Collection<SigninDelegateGroup> signinDelegateGroups) {
            this.signinDelegateGroups = SigninDelegateGroupListCopier.copy(signinDelegateGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder signinDelegateGroups(SigninDelegateGroup... signinDelegateGroups) {
            signinDelegateGroups(Arrays.asList(signinDelegateGroups));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder signinDelegateGroups(Consumer<SigninDelegateGroup.Builder>... signinDelegateGroups) {
            signinDelegateGroups(Stream.of(signinDelegateGroups).map(c -> SigninDelegateGroup.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setSigninDelegateGroups(Collection<SigninDelegateGroup.BuilderImpl> signinDelegateGroups) {
            this.signinDelegateGroups = SigninDelegateGroupListCopier.copyFromBuilder(signinDelegateGroups);
        }

        @Override
        public Account build() {
            return new Account(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
