/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chime.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The proxy session for an Amazon Chime Voice Connector.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProxySession implements SdkPojo, Serializable, ToCopyableBuilder<ProxySession.Builder, ProxySession> {
    private static final SdkField<String> VOICE_CONNECTOR_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VoiceConnectorId").getter(getter(ProxySession::voiceConnectorId))
            .setter(setter(Builder::voiceConnectorId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VoiceConnectorId").build()).build();

    private static final SdkField<String> PROXY_SESSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProxySessionId").getter(getter(ProxySession::proxySessionId)).setter(setter(Builder::proxySessionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProxySessionId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ProxySession::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ProxySession::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Integer> EXPIRY_MINUTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ExpiryMinutes").getter(getter(ProxySession::expiryMinutes)).setter(setter(Builder::expiryMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpiryMinutes").build()).build();

    private static final SdkField<List<String>> CAPABILITIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Capabilities")
            .getter(getter(ProxySession::capabilitiesAsStrings))
            .setter(setter(Builder::capabilitiesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Capabilities").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATED_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTimestamp")
            .getter(getter(ProxySession::createdTimestamp))
            .setter(setter(Builder::createdTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTimestamp").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> UPDATED_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("UpdatedTimestamp")
            .getter(getter(ProxySession::updatedTimestamp))
            .setter(setter(Builder::updatedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpdatedTimestamp").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> ENDED_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndedTimestamp")
            .getter(getter(ProxySession::endedTimestamp))
            .setter(setter(Builder::endedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndedTimestamp").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<List<Participant>> PARTICIPANTS_FIELD = SdkField
            .<List<Participant>> builder(MarshallingType.LIST)
            .memberName("Participants")
            .getter(getter(ProxySession::participants))
            .setter(setter(Builder::participants))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Participants").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Participant> builder(MarshallingType.SDK_POJO)
                                            .constructor(Participant::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NUMBER_SELECTION_BEHAVIOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NumberSelectionBehavior").getter(getter(ProxySession::numberSelectionBehaviorAsString))
            .setter(setter(Builder::numberSelectionBehavior))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumberSelectionBehavior").build())
            .build();

    private static final SdkField<String> GEO_MATCH_LEVEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GeoMatchLevel").getter(getter(ProxySession::geoMatchLevelAsString))
            .setter(setter(Builder::geoMatchLevel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GeoMatchLevel").build()).build();

    private static final SdkField<GeoMatchParams> GEO_MATCH_PARAMS_FIELD = SdkField
            .<GeoMatchParams> builder(MarshallingType.SDK_POJO).memberName("GeoMatchParams")
            .getter(getter(ProxySession::geoMatchParams)).setter(setter(Builder::geoMatchParams))
            .constructor(GeoMatchParams::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GeoMatchParams").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VOICE_CONNECTOR_ID_FIELD,
            PROXY_SESSION_ID_FIELD, NAME_FIELD, STATUS_FIELD, EXPIRY_MINUTES_FIELD, CAPABILITIES_FIELD, CREATED_TIMESTAMP_FIELD,
            UPDATED_TIMESTAMP_FIELD, ENDED_TIMESTAMP_FIELD, PARTICIPANTS_FIELD, NUMBER_SELECTION_BEHAVIOR_FIELD,
            GEO_MATCH_LEVEL_FIELD, GEO_MATCH_PARAMS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String voiceConnectorId;

    private final String proxySessionId;

    private final String name;

    private final String status;

    private final Integer expiryMinutes;

    private final List<String> capabilities;

    private final Instant createdTimestamp;

    private final Instant updatedTimestamp;

    private final Instant endedTimestamp;

    private final List<Participant> participants;

    private final String numberSelectionBehavior;

    private final String geoMatchLevel;

    private final GeoMatchParams geoMatchParams;

    private ProxySession(BuilderImpl builder) {
        this.voiceConnectorId = builder.voiceConnectorId;
        this.proxySessionId = builder.proxySessionId;
        this.name = builder.name;
        this.status = builder.status;
        this.expiryMinutes = builder.expiryMinutes;
        this.capabilities = builder.capabilities;
        this.createdTimestamp = builder.createdTimestamp;
        this.updatedTimestamp = builder.updatedTimestamp;
        this.endedTimestamp = builder.endedTimestamp;
        this.participants = builder.participants;
        this.numberSelectionBehavior = builder.numberSelectionBehavior;
        this.geoMatchLevel = builder.geoMatchLevel;
        this.geoMatchParams = builder.geoMatchParams;
    }

    /**
     * <p>
     * The Amazon Chime voice connector ID.
     * </p>
     * 
     * @return The Amazon Chime voice connector ID.
     */
    public final String voiceConnectorId() {
        return voiceConnectorId;
    }

    /**
     * <p>
     * The proxy session ID.
     * </p>
     * 
     * @return The proxy session ID.
     */
    public final String proxySessionId() {
        return proxySessionId;
    }

    /**
     * <p>
     * The name of the proxy session.
     * </p>
     * 
     * @return The name of the proxy session.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The status of the proxy session.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ProxySessionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the proxy session.
     * @see ProxySessionStatus
     */
    public final ProxySessionStatus status() {
        return ProxySessionStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the proxy session.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ProxySessionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the proxy session.
     * @see ProxySessionStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The number of minutes allowed for the proxy session.
     * </p>
     * 
     * @return The number of minutes allowed for the proxy session.
     */
    public final Integer expiryMinutes() {
        return expiryMinutes;
    }

    /**
     * <p>
     * The proxy session capabilities.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCapabilities} method.
     * </p>
     * 
     * @return The proxy session capabilities.
     */
    public final List<Capability> capabilities() {
        return CapabilityListCopier.copyStringToEnum(capabilities);
    }

    /**
     * For responses, this returns true if the service returned a value for the Capabilities property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCapabilities() {
        return capabilities != null && !(capabilities instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The proxy session capabilities.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCapabilities} method.
     * </p>
     * 
     * @return The proxy session capabilities.
     */
    public final List<String> capabilitiesAsStrings() {
        return capabilities;
    }

    /**
     * <p>
     * The created time stamp, in ISO 8601 format.
     * </p>
     * 
     * @return The created time stamp, in ISO 8601 format.
     */
    public final Instant createdTimestamp() {
        return createdTimestamp;
    }

    /**
     * <p>
     * The updated time stamp, in ISO 8601 format.
     * </p>
     * 
     * @return The updated time stamp, in ISO 8601 format.
     */
    public final Instant updatedTimestamp() {
        return updatedTimestamp;
    }

    /**
     * <p>
     * The ended time stamp, in ISO 8601 format.
     * </p>
     * 
     * @return The ended time stamp, in ISO 8601 format.
     */
    public final Instant endedTimestamp() {
        return endedTimestamp;
    }

    /**
     * For responses, this returns true if the service returned a value for the Participants property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasParticipants() {
        return participants != null && !(participants instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The proxy session participants.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParticipants} method.
     * </p>
     * 
     * @return The proxy session participants.
     */
    public final List<Participant> participants() {
        return participants;
    }

    /**
     * <p>
     * The preference for proxy phone number reuse, or stickiness, between the same participants across sessions.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #numberSelectionBehavior} will return {@link NumberSelectionBehavior#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #numberSelectionBehaviorAsString}.
     * </p>
     * 
     * @return The preference for proxy phone number reuse, or stickiness, between the same participants across
     *         sessions.
     * @see NumberSelectionBehavior
     */
    public final NumberSelectionBehavior numberSelectionBehavior() {
        return NumberSelectionBehavior.fromValue(numberSelectionBehavior);
    }

    /**
     * <p>
     * The preference for proxy phone number reuse, or stickiness, between the same participants across sessions.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #numberSelectionBehavior} will return {@link NumberSelectionBehavior#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #numberSelectionBehaviorAsString}.
     * </p>
     * 
     * @return The preference for proxy phone number reuse, or stickiness, between the same participants across
     *         sessions.
     * @see NumberSelectionBehavior
     */
    public final String numberSelectionBehaviorAsString() {
        return numberSelectionBehavior;
    }

    /**
     * <p>
     * The preference for matching the country or area code of the proxy phone number with that of the first
     * participant.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #geoMatchLevel}
     * will return {@link GeoMatchLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #geoMatchLevelAsString}.
     * </p>
     * 
     * @return The preference for matching the country or area code of the proxy phone number with that of the first
     *         participant.
     * @see GeoMatchLevel
     */
    public final GeoMatchLevel geoMatchLevel() {
        return GeoMatchLevel.fromValue(geoMatchLevel);
    }

    /**
     * <p>
     * The preference for matching the country or area code of the proxy phone number with that of the first
     * participant.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #geoMatchLevel}
     * will return {@link GeoMatchLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #geoMatchLevelAsString}.
     * </p>
     * 
     * @return The preference for matching the country or area code of the proxy phone number with that of the first
     *         participant.
     * @see GeoMatchLevel
     */
    public final String geoMatchLevelAsString() {
        return geoMatchLevel;
    }

    /**
     * <p>
     * The country and area code for the proxy phone number.
     * </p>
     * 
     * @return The country and area code for the proxy phone number.
     */
    public final GeoMatchParams geoMatchParams() {
        return geoMatchParams;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(voiceConnectorId());
        hashCode = 31 * hashCode + Objects.hashCode(proxySessionId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(expiryMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(hasCapabilities() ? capabilitiesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createdTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(updatedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(endedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(hasParticipants() ? participants() : null);
        hashCode = 31 * hashCode + Objects.hashCode(numberSelectionBehaviorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(geoMatchLevelAsString());
        hashCode = 31 * hashCode + Objects.hashCode(geoMatchParams());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProxySession)) {
            return false;
        }
        ProxySession other = (ProxySession) obj;
        return Objects.equals(voiceConnectorId(), other.voiceConnectorId())
                && Objects.equals(proxySessionId(), other.proxySessionId()) && Objects.equals(name(), other.name())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(expiryMinutes(), other.expiryMinutes()) && hasCapabilities() == other.hasCapabilities()
                && Objects.equals(capabilitiesAsStrings(), other.capabilitiesAsStrings())
                && Objects.equals(createdTimestamp(), other.createdTimestamp())
                && Objects.equals(updatedTimestamp(), other.updatedTimestamp())
                && Objects.equals(endedTimestamp(), other.endedTimestamp()) && hasParticipants() == other.hasParticipants()
                && Objects.equals(participants(), other.participants())
                && Objects.equals(numberSelectionBehaviorAsString(), other.numberSelectionBehaviorAsString())
                && Objects.equals(geoMatchLevelAsString(), other.geoMatchLevelAsString())
                && Objects.equals(geoMatchParams(), other.geoMatchParams());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProxySession").add("VoiceConnectorId", voiceConnectorId())
                .add("ProxySessionId", proxySessionId()).add("Name", name()).add("Status", statusAsString())
                .add("ExpiryMinutes", expiryMinutes()).add("Capabilities", hasCapabilities() ? capabilitiesAsStrings() : null)
                .add("CreatedTimestamp", createdTimestamp()).add("UpdatedTimestamp", updatedTimestamp())
                .add("EndedTimestamp", endedTimestamp()).add("Participants", hasParticipants() ? participants() : null)
                .add("NumberSelectionBehavior", numberSelectionBehaviorAsString()).add("GeoMatchLevel", geoMatchLevelAsString())
                .add("GeoMatchParams", geoMatchParams()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VoiceConnectorId":
            return Optional.ofNullable(clazz.cast(voiceConnectorId()));
        case "ProxySessionId":
            return Optional.ofNullable(clazz.cast(proxySessionId()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "ExpiryMinutes":
            return Optional.ofNullable(clazz.cast(expiryMinutes()));
        case "Capabilities":
            return Optional.ofNullable(clazz.cast(capabilitiesAsStrings()));
        case "CreatedTimestamp":
            return Optional.ofNullable(clazz.cast(createdTimestamp()));
        case "UpdatedTimestamp":
            return Optional.ofNullable(clazz.cast(updatedTimestamp()));
        case "EndedTimestamp":
            return Optional.ofNullable(clazz.cast(endedTimestamp()));
        case "Participants":
            return Optional.ofNullable(clazz.cast(participants()));
        case "NumberSelectionBehavior":
            return Optional.ofNullable(clazz.cast(numberSelectionBehaviorAsString()));
        case "GeoMatchLevel":
            return Optional.ofNullable(clazz.cast(geoMatchLevelAsString()));
        case "GeoMatchParams":
            return Optional.ofNullable(clazz.cast(geoMatchParams()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProxySession, T> g) {
        return obj -> g.apply((ProxySession) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProxySession> {
        /**
         * <p>
         * The Amazon Chime voice connector ID.
         * </p>
         * 
         * @param voiceConnectorId
         *        The Amazon Chime voice connector ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder voiceConnectorId(String voiceConnectorId);

        /**
         * <p>
         * The proxy session ID.
         * </p>
         * 
         * @param proxySessionId
         *        The proxy session ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder proxySessionId(String proxySessionId);

        /**
         * <p>
         * The name of the proxy session.
         * </p>
         * 
         * @param name
         *        The name of the proxy session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The status of the proxy session.
         * </p>
         * 
         * @param status
         *        The status of the proxy session.
         * @see ProxySessionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProxySessionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the proxy session.
         * </p>
         * 
         * @param status
         *        The status of the proxy session.
         * @see ProxySessionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProxySessionStatus
         */
        Builder status(ProxySessionStatus status);

        /**
         * <p>
         * The number of minutes allowed for the proxy session.
         * </p>
         * 
         * @param expiryMinutes
         *        The number of minutes allowed for the proxy session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expiryMinutes(Integer expiryMinutes);

        /**
         * <p>
         * The proxy session capabilities.
         * </p>
         * 
         * @param capabilities
         *        The proxy session capabilities.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilitiesWithStrings(Collection<String> capabilities);

        /**
         * <p>
         * The proxy session capabilities.
         * </p>
         * 
         * @param capabilities
         *        The proxy session capabilities.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilitiesWithStrings(String... capabilities);

        /**
         * <p>
         * The proxy session capabilities.
         * </p>
         * 
         * @param capabilities
         *        The proxy session capabilities.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(Collection<Capability> capabilities);

        /**
         * <p>
         * The proxy session capabilities.
         * </p>
         * 
         * @param capabilities
         *        The proxy session capabilities.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(Capability... capabilities);

        /**
         * <p>
         * The created time stamp, in ISO 8601 format.
         * </p>
         * 
         * @param createdTimestamp
         *        The created time stamp, in ISO 8601 format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTimestamp(Instant createdTimestamp);

        /**
         * <p>
         * The updated time stamp, in ISO 8601 format.
         * </p>
         * 
         * @param updatedTimestamp
         *        The updated time stamp, in ISO 8601 format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedTimestamp(Instant updatedTimestamp);

        /**
         * <p>
         * The ended time stamp, in ISO 8601 format.
         * </p>
         * 
         * @param endedTimestamp
         *        The ended time stamp, in ISO 8601 format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endedTimestamp(Instant endedTimestamp);

        /**
         * <p>
         * The proxy session participants.
         * </p>
         * 
         * @param participants
         *        The proxy session participants.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder participants(Collection<Participant> participants);

        /**
         * <p>
         * The proxy session participants.
         * </p>
         * 
         * @param participants
         *        The proxy session participants.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder participants(Participant... participants);

        /**
         * <p>
         * The proxy session participants.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Participant>.Builder} avoiding the need to
         * create one manually via {@link List<Participant>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Participant>.Builder#build()} is called immediately and its
         * result is passed to {@link #participants(List<Participant>)}.
         * 
         * @param participants
         *        a consumer that will call methods on {@link List<Participant>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #participants(List<Participant>)
         */
        Builder participants(Consumer<Participant.Builder>... participants);

        /**
         * <p>
         * The preference for proxy phone number reuse, or stickiness, between the same participants across sessions.
         * </p>
         * 
         * @param numberSelectionBehavior
         *        The preference for proxy phone number reuse, or stickiness, between the same participants across
         *        sessions.
         * @see NumberSelectionBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NumberSelectionBehavior
         */
        Builder numberSelectionBehavior(String numberSelectionBehavior);

        /**
         * <p>
         * The preference for proxy phone number reuse, or stickiness, between the same participants across sessions.
         * </p>
         * 
         * @param numberSelectionBehavior
         *        The preference for proxy phone number reuse, or stickiness, between the same participants across
         *        sessions.
         * @see NumberSelectionBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NumberSelectionBehavior
         */
        Builder numberSelectionBehavior(NumberSelectionBehavior numberSelectionBehavior);

        /**
         * <p>
         * The preference for matching the country or area code of the proxy phone number with that of the first
         * participant.
         * </p>
         * 
         * @param geoMatchLevel
         *        The preference for matching the country or area code of the proxy phone number with that of the first
         *        participant.
         * @see GeoMatchLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GeoMatchLevel
         */
        Builder geoMatchLevel(String geoMatchLevel);

        /**
         * <p>
         * The preference for matching the country or area code of the proxy phone number with that of the first
         * participant.
         * </p>
         * 
         * @param geoMatchLevel
         *        The preference for matching the country or area code of the proxy phone number with that of the first
         *        participant.
         * @see GeoMatchLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GeoMatchLevel
         */
        Builder geoMatchLevel(GeoMatchLevel geoMatchLevel);

        /**
         * <p>
         * The country and area code for the proxy phone number.
         * </p>
         * 
         * @param geoMatchParams
         *        The country and area code for the proxy phone number.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder geoMatchParams(GeoMatchParams geoMatchParams);

        /**
         * <p>
         * The country and area code for the proxy phone number.
         * </p>
         * This is a convenience that creates an instance of the {@link GeoMatchParams.Builder} avoiding the need to
         * create one manually via {@link GeoMatchParams#builder()}.
         *
         * When the {@link Consumer} completes, {@link GeoMatchParams.Builder#build()} is called immediately and its
         * result is passed to {@link #geoMatchParams(GeoMatchParams)}.
         * 
         * @param geoMatchParams
         *        a consumer that will call methods on {@link GeoMatchParams.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #geoMatchParams(GeoMatchParams)
         */
        default Builder geoMatchParams(Consumer<GeoMatchParams.Builder> geoMatchParams) {
            return geoMatchParams(GeoMatchParams.builder().applyMutation(geoMatchParams).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String voiceConnectorId;

        private String proxySessionId;

        private String name;

        private String status;

        private Integer expiryMinutes;

        private List<String> capabilities = DefaultSdkAutoConstructList.getInstance();

        private Instant createdTimestamp;

        private Instant updatedTimestamp;

        private Instant endedTimestamp;

        private List<Participant> participants = DefaultSdkAutoConstructList.getInstance();

        private String numberSelectionBehavior;

        private String geoMatchLevel;

        private GeoMatchParams geoMatchParams;

        private BuilderImpl() {
        }

        private BuilderImpl(ProxySession model) {
            voiceConnectorId(model.voiceConnectorId);
            proxySessionId(model.proxySessionId);
            name(model.name);
            status(model.status);
            expiryMinutes(model.expiryMinutes);
            capabilitiesWithStrings(model.capabilities);
            createdTimestamp(model.createdTimestamp);
            updatedTimestamp(model.updatedTimestamp);
            endedTimestamp(model.endedTimestamp);
            participants(model.participants);
            numberSelectionBehavior(model.numberSelectionBehavior);
            geoMatchLevel(model.geoMatchLevel);
            geoMatchParams(model.geoMatchParams);
        }

        public final String getVoiceConnectorId() {
            return voiceConnectorId;
        }

        public final void setVoiceConnectorId(String voiceConnectorId) {
            this.voiceConnectorId = voiceConnectorId;
        }

        @Override
        @Transient
        public final Builder voiceConnectorId(String voiceConnectorId) {
            this.voiceConnectorId = voiceConnectorId;
            return this;
        }

        public final String getProxySessionId() {
            return proxySessionId;
        }

        public final void setProxySessionId(String proxySessionId) {
            this.proxySessionId = proxySessionId;
        }

        @Override
        @Transient
        public final Builder proxySessionId(String proxySessionId) {
            this.proxySessionId = proxySessionId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        @Transient
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        @Transient
        public final Builder status(ProxySessionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Integer getExpiryMinutes() {
            return expiryMinutes;
        }

        public final void setExpiryMinutes(Integer expiryMinutes) {
            this.expiryMinutes = expiryMinutes;
        }

        @Override
        @Transient
        public final Builder expiryMinutes(Integer expiryMinutes) {
            this.expiryMinutes = expiryMinutes;
            return this;
        }

        public final Collection<String> getCapabilities() {
            if (capabilities instanceof SdkAutoConstructList) {
                return null;
            }
            return capabilities;
        }

        public final void setCapabilities(Collection<String> capabilities) {
            this.capabilities = CapabilityListCopier.copy(capabilities);
        }

        @Override
        @Transient
        public final Builder capabilitiesWithStrings(Collection<String> capabilities) {
            this.capabilities = CapabilityListCopier.copy(capabilities);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder capabilitiesWithStrings(String... capabilities) {
            capabilitiesWithStrings(Arrays.asList(capabilities));
            return this;
        }

        @Override
        @Transient
        public final Builder capabilities(Collection<Capability> capabilities) {
            this.capabilities = CapabilityListCopier.copyEnumToString(capabilities);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder capabilities(Capability... capabilities) {
            capabilities(Arrays.asList(capabilities));
            return this;
        }

        public final Instant getCreatedTimestamp() {
            return createdTimestamp;
        }

        public final void setCreatedTimestamp(Instant createdTimestamp) {
            this.createdTimestamp = createdTimestamp;
        }

        @Override
        @Transient
        public final Builder createdTimestamp(Instant createdTimestamp) {
            this.createdTimestamp = createdTimestamp;
            return this;
        }

        public final Instant getUpdatedTimestamp() {
            return updatedTimestamp;
        }

        public final void setUpdatedTimestamp(Instant updatedTimestamp) {
            this.updatedTimestamp = updatedTimestamp;
        }

        @Override
        @Transient
        public final Builder updatedTimestamp(Instant updatedTimestamp) {
            this.updatedTimestamp = updatedTimestamp;
            return this;
        }

        public final Instant getEndedTimestamp() {
            return endedTimestamp;
        }

        public final void setEndedTimestamp(Instant endedTimestamp) {
            this.endedTimestamp = endedTimestamp;
        }

        @Override
        @Transient
        public final Builder endedTimestamp(Instant endedTimestamp) {
            this.endedTimestamp = endedTimestamp;
            return this;
        }

        public final List<Participant.Builder> getParticipants() {
            List<Participant.Builder> result = ParticipantsCopier.copyToBuilder(this.participants);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setParticipants(Collection<Participant.BuilderImpl> participants) {
            this.participants = ParticipantsCopier.copyFromBuilder(participants);
        }

        @Override
        @Transient
        public final Builder participants(Collection<Participant> participants) {
            this.participants = ParticipantsCopier.copy(participants);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder participants(Participant... participants) {
            participants(Arrays.asList(participants));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder participants(Consumer<Participant.Builder>... participants) {
            participants(Stream.of(participants).map(c -> Participant.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getNumberSelectionBehavior() {
            return numberSelectionBehavior;
        }

        public final void setNumberSelectionBehavior(String numberSelectionBehavior) {
            this.numberSelectionBehavior = numberSelectionBehavior;
        }

        @Override
        @Transient
        public final Builder numberSelectionBehavior(String numberSelectionBehavior) {
            this.numberSelectionBehavior = numberSelectionBehavior;
            return this;
        }

        @Override
        @Transient
        public final Builder numberSelectionBehavior(NumberSelectionBehavior numberSelectionBehavior) {
            this.numberSelectionBehavior(numberSelectionBehavior == null ? null : numberSelectionBehavior.toString());
            return this;
        }

        public final String getGeoMatchLevel() {
            return geoMatchLevel;
        }

        public final void setGeoMatchLevel(String geoMatchLevel) {
            this.geoMatchLevel = geoMatchLevel;
        }

        @Override
        @Transient
        public final Builder geoMatchLevel(String geoMatchLevel) {
            this.geoMatchLevel = geoMatchLevel;
            return this;
        }

        @Override
        @Transient
        public final Builder geoMatchLevel(GeoMatchLevel geoMatchLevel) {
            this.geoMatchLevel(geoMatchLevel == null ? null : geoMatchLevel.toString());
            return this;
        }

        public final GeoMatchParams.Builder getGeoMatchParams() {
            return geoMatchParams != null ? geoMatchParams.toBuilder() : null;
        }

        public final void setGeoMatchParams(GeoMatchParams.BuilderImpl geoMatchParams) {
            this.geoMatchParams = geoMatchParams != null ? geoMatchParams.build() : null;
        }

        @Override
        @Transient
        public final Builder geoMatchParams(GeoMatchParams geoMatchParams) {
            this.geoMatchParams = geoMatchParams;
            return this;
        }

        @Override
        public ProxySession build() {
            return new ProxySession(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
