/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chime.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Settings specific for Amazon Transcribe as the live transcription engine.
 * </p>
 * <p>
 * If you specify an invalid combination of parameters, a <code>TranscriptFailed</code> event will be sent with the
 * contents of the <code>BadRequestException</code> generated by Amazon Transcribe. For more information on each
 * parameter and which combinations are valid, refer to the <a
 * href="https://docs.aws.amazon.com/transcribe/latest/APIReference/API_streaming_StartStreamTranscription.html"
 * >StartStreamTranscription</a> API in the <i>Amazon Transcribe Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EngineTranscribeSettings implements SdkPojo, Serializable,
        ToCopyableBuilder<EngineTranscribeSettings.Builder, EngineTranscribeSettings> {
    private static final SdkField<String> LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageCode").getter(getter(EngineTranscribeSettings::languageCodeAsString))
            .setter(setter(Builder::languageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageCode").build()).build();

    private static final SdkField<String> VOCABULARY_FILTER_METHOD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VocabularyFilterMethod").getter(getter(EngineTranscribeSettings::vocabularyFilterMethodAsString))
            .setter(setter(Builder::vocabularyFilterMethod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VocabularyFilterMethod").build())
            .build();

    private static final SdkField<String> VOCABULARY_FILTER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VocabularyFilterName").getter(getter(EngineTranscribeSettings::vocabularyFilterName))
            .setter(setter(Builder::vocabularyFilterName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VocabularyFilterName").build())
            .build();

    private static final SdkField<String> VOCABULARY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VocabularyName").getter(getter(EngineTranscribeSettings::vocabularyName))
            .setter(setter(Builder::vocabularyName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VocabularyName").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Region")
            .getter(getter(EngineTranscribeSettings::regionAsString)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Region").build()).build();

    private static final SdkField<Boolean> ENABLE_PARTIAL_RESULTS_STABILIZATION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnablePartialResultsStabilization")
            .getter(getter(EngineTranscribeSettings::enablePartialResultsStabilization))
            .setter(setter(Builder::enablePartialResultsStabilization))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnablePartialResultsStabilization")
                    .build()).build();

    private static final SdkField<String> PARTIAL_RESULTS_STABILITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PartialResultsStability").getter(getter(EngineTranscribeSettings::partialResultsStabilityAsString))
            .setter(setter(Builder::partialResultsStability))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PartialResultsStability").build())
            .build();

    private static final SdkField<String> CONTENT_IDENTIFICATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContentIdentificationType").getter(getter(EngineTranscribeSettings::contentIdentificationTypeAsString))
            .setter(setter(Builder::contentIdentificationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentIdentificationType").build())
            .build();

    private static final SdkField<String> CONTENT_REDACTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContentRedactionType").getter(getter(EngineTranscribeSettings::contentRedactionTypeAsString))
            .setter(setter(Builder::contentRedactionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentRedactionType").build())
            .build();

    private static final SdkField<String> PII_ENTITY_TYPES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PiiEntityTypes").getter(getter(EngineTranscribeSettings::piiEntityTypes))
            .setter(setter(Builder::piiEntityTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PiiEntityTypes").build()).build();

    private static final SdkField<String> LANGUAGE_MODEL_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageModelName").getter(getter(EngineTranscribeSettings::languageModelName))
            .setter(setter(Builder::languageModelName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageModelName").build()).build();

    private static final SdkField<Boolean> IDENTIFY_LANGUAGE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IdentifyLanguage").getter(getter(EngineTranscribeSettings::identifyLanguage))
            .setter(setter(Builder::identifyLanguage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdentifyLanguage").build()).build();

    private static final SdkField<String> LANGUAGE_OPTIONS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageOptions").getter(getter(EngineTranscribeSettings::languageOptions))
            .setter(setter(Builder::languageOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageOptions").build()).build();

    private static final SdkField<String> PREFERRED_LANGUAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreferredLanguage").getter(getter(EngineTranscribeSettings::preferredLanguageAsString))
            .setter(setter(Builder::preferredLanguage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredLanguage").build()).build();

    private static final SdkField<String> VOCABULARY_NAMES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VocabularyNames").getter(getter(EngineTranscribeSettings::vocabularyNames))
            .setter(setter(Builder::vocabularyNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VocabularyNames").build()).build();

    private static final SdkField<String> VOCABULARY_FILTER_NAMES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VocabularyFilterNames").getter(getter(EngineTranscribeSettings::vocabularyFilterNames))
            .setter(setter(Builder::vocabularyFilterNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VocabularyFilterNames").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LANGUAGE_CODE_FIELD,
            VOCABULARY_FILTER_METHOD_FIELD, VOCABULARY_FILTER_NAME_FIELD, VOCABULARY_NAME_FIELD, REGION_FIELD,
            ENABLE_PARTIAL_RESULTS_STABILIZATION_FIELD, PARTIAL_RESULTS_STABILITY_FIELD, CONTENT_IDENTIFICATION_TYPE_FIELD,
            CONTENT_REDACTION_TYPE_FIELD, PII_ENTITY_TYPES_FIELD, LANGUAGE_MODEL_NAME_FIELD, IDENTIFY_LANGUAGE_FIELD,
            LANGUAGE_OPTIONS_FIELD, PREFERRED_LANGUAGE_FIELD, VOCABULARY_NAMES_FIELD, VOCABULARY_FILTER_NAMES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String languageCode;

    private final String vocabularyFilterMethod;

    private final String vocabularyFilterName;

    private final String vocabularyName;

    private final String region;

    private final Boolean enablePartialResultsStabilization;

    private final String partialResultsStability;

    private final String contentIdentificationType;

    private final String contentRedactionType;

    private final String piiEntityTypes;

    private final String languageModelName;

    private final Boolean identifyLanguage;

    private final String languageOptions;

    private final String preferredLanguage;

    private final String vocabularyNames;

    private final String vocabularyFilterNames;

    private EngineTranscribeSettings(BuilderImpl builder) {
        this.languageCode = builder.languageCode;
        this.vocabularyFilterMethod = builder.vocabularyFilterMethod;
        this.vocabularyFilterName = builder.vocabularyFilterName;
        this.vocabularyName = builder.vocabularyName;
        this.region = builder.region;
        this.enablePartialResultsStabilization = builder.enablePartialResultsStabilization;
        this.partialResultsStability = builder.partialResultsStability;
        this.contentIdentificationType = builder.contentIdentificationType;
        this.contentRedactionType = builder.contentRedactionType;
        this.piiEntityTypes = builder.piiEntityTypes;
        this.languageModelName = builder.languageModelName;
        this.identifyLanguage = builder.identifyLanguage;
        this.languageOptions = builder.languageOptions;
        this.preferredLanguage = builder.preferredLanguage;
        this.vocabularyNames = builder.vocabularyNames;
        this.vocabularyFilterNames = builder.vocabularyFilterNames;
    }

    /**
     * <p>
     * Specify the language code that represents the language spoken.
     * </p>
     * <p>
     * If you're unsure of the language spoken in your audio, consider using <code>IdentifyLanguage</code> to enable
     * automatic language identification.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link TranscribeLanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #languageCodeAsString}.
     * </p>
     * 
     * @return Specify the language code that represents the language spoken.</p>
     *         <p>
     *         If you're unsure of the language spoken in your audio, consider using <code>IdentifyLanguage</code> to
     *         enable automatic language identification.
     * @see TranscribeLanguageCode
     */
    public final TranscribeLanguageCode languageCode() {
        return TranscribeLanguageCode.fromValue(languageCode);
    }

    /**
     * <p>
     * Specify the language code that represents the language spoken.
     * </p>
     * <p>
     * If you're unsure of the language spoken in your audio, consider using <code>IdentifyLanguage</code> to enable
     * automatic language identification.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link TranscribeLanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #languageCodeAsString}.
     * </p>
     * 
     * @return Specify the language code that represents the language spoken.</p>
     *         <p>
     *         If you're unsure of the language spoken in your audio, consider using <code>IdentifyLanguage</code> to
     *         enable automatic language identification.
     * @see TranscribeLanguageCode
     */
    public final String languageCodeAsString() {
        return languageCode;
    }

    /**
     * <p>
     * Specify how you want your vocabulary filter applied to your transcript.
     * </p>
     * <p>
     * To replace words with <code>***</code>, choose <code>mask</code>.
     * </p>
     * <p>
     * To delete words, choose <code>remove</code>.
     * </p>
     * <p>
     * To flag words without changing them, choose <code>tag</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #vocabularyFilterMethod} will return {@link TranscribeVocabularyFilterMethod#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #vocabularyFilterMethodAsString}.
     * </p>
     * 
     * @return Specify how you want your vocabulary filter applied to your transcript.</p>
     *         <p>
     *         To replace words with <code>***</code>, choose <code>mask</code>.
     *         </p>
     *         <p>
     *         To delete words, choose <code>remove</code>.
     *         </p>
     *         <p>
     *         To flag words without changing them, choose <code>tag</code>.
     * @see TranscribeVocabularyFilterMethod
     */
    public final TranscribeVocabularyFilterMethod vocabularyFilterMethod() {
        return TranscribeVocabularyFilterMethod.fromValue(vocabularyFilterMethod);
    }

    /**
     * <p>
     * Specify how you want your vocabulary filter applied to your transcript.
     * </p>
     * <p>
     * To replace words with <code>***</code>, choose <code>mask</code>.
     * </p>
     * <p>
     * To delete words, choose <code>remove</code>.
     * </p>
     * <p>
     * To flag words without changing them, choose <code>tag</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #vocabularyFilterMethod} will return {@link TranscribeVocabularyFilterMethod#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #vocabularyFilterMethodAsString}.
     * </p>
     * 
     * @return Specify how you want your vocabulary filter applied to your transcript.</p>
     *         <p>
     *         To replace words with <code>***</code>, choose <code>mask</code>.
     *         </p>
     *         <p>
     *         To delete words, choose <code>remove</code>.
     *         </p>
     *         <p>
     *         To flag words without changing them, choose <code>tag</code>.
     * @see TranscribeVocabularyFilterMethod
     */
    public final String vocabularyFilterMethodAsString() {
        return vocabularyFilterMethod;
    }

    /**
     * <p>
     * Specify the name of the custom vocabulary filter that you want to use when processing your transcription. Note
     * that vocabulary filter names are case sensitive.
     * </p>
     * <p>
     * If you use Amazon Transcribe in multiple Regions, the vocabulary filter must be available in Amazon Transcribe in
     * each Region.
     * </p>
     * <p>
     * If you include <code>IdentifyLanguage</code> and want to use one or more vocabulary filters with your
     * transcription, use the <code>VocabularyFilterNames</code> parameter instead.
     * </p>
     * 
     * @return Specify the name of the custom vocabulary filter that you want to use when processing your transcription.
     *         Note that vocabulary filter names are case sensitive. </p>
     *         <p>
     *         If you use Amazon Transcribe in multiple Regions, the vocabulary filter must be available in Amazon
     *         Transcribe in each Region.
     *         </p>
     *         <p>
     *         If you include <code>IdentifyLanguage</code> and want to use one or more vocabulary filters with your
     *         transcription, use the <code>VocabularyFilterNames</code> parameter instead.
     */
    public final String vocabularyFilterName() {
        return vocabularyFilterName;
    }

    /**
     * <p>
     * Specify the name of the custom vocabulary that you want to use when processing your transcription. Note that
     * vocabulary names are case sensitive.
     * </p>
     * <p>
     * If you use Amazon Transcribe multiple Regions, the vocabulary must be available in Amazon Transcribe in each
     * Region.
     * </p>
     * <p>
     * If you include <code>IdentifyLanguage</code> and want to use one or more custom vocabularies with your
     * transcription, use the <code>VocabularyNames</code> parameter instead.
     * </p>
     * 
     * @return Specify the name of the custom vocabulary that you want to use when processing your transcription. Note
     *         that vocabulary names are case sensitive.</p>
     *         <p>
     *         If you use Amazon Transcribe multiple Regions, the vocabulary must be available in Amazon Transcribe in
     *         each Region.
     *         </p>
     *         <p>
     *         If you include <code>IdentifyLanguage</code> and want to use one or more custom vocabularies with your
     *         transcription, use the <code>VocabularyNames</code> parameter instead.
     */
    public final String vocabularyName() {
        return vocabularyName;
    }

    /**
     * <p>
     * The AWS Region in which to use Amazon Transcribe.
     * </p>
     * <p>
     * If you don't specify a Region, then the <code>MediaRegion</code> parameter of the <a
     * href="https://docs.aws.amazon.com/chime-sdk/latest/APIReference/API_meeting-chime_CreateMeeting.html"
     * >CreateMeeting.html</a> API will be used. However, if Amazon Transcribe is not available in the
     * <code>MediaRegion</code>, then a TranscriptFailed event is sent.
     * </p>
     * <p>
     * Use <code>auto</code> to use Amazon Transcribe in a Region near the meeting’s <code>MediaRegion</code>. For more
     * information, refer to <a
     * href="https://docs.aws.amazon.com/chime-sdk/latest/dg/transcription-options.html#choose-region">Choosing a
     * transcription Region</a> in the <i>Amazon Chime SDK Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #region} will
     * return {@link TranscribeRegion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #regionAsString}.
     * </p>
     * 
     * @return The AWS Region in which to use Amazon Transcribe.</p>
     *         <p>
     *         If you don't specify a Region, then the <code>MediaRegion</code> parameter of the <a
     *         href="https://docs.aws.amazon.com/chime-sdk/latest/APIReference/API_meeting-chime_CreateMeeting.html"
     *         >CreateMeeting.html</a> API will be used. However, if Amazon Transcribe is not available in the
     *         <code>MediaRegion</code>, then a TranscriptFailed event is sent.
     *         </p>
     *         <p>
     *         Use <code>auto</code> to use Amazon Transcribe in a Region near the meeting’s <code>MediaRegion</code>.
     *         For more information, refer to <a
     *         href="https://docs.aws.amazon.com/chime-sdk/latest/dg/transcription-options.html#choose-region">Choosing
     *         a transcription Region</a> in the <i>Amazon Chime SDK Developer Guide</i>.
     * @see TranscribeRegion
     */
    public final TranscribeRegion region() {
        return TranscribeRegion.fromValue(region);
    }

    /**
     * <p>
     * The AWS Region in which to use Amazon Transcribe.
     * </p>
     * <p>
     * If you don't specify a Region, then the <code>MediaRegion</code> parameter of the <a
     * href="https://docs.aws.amazon.com/chime-sdk/latest/APIReference/API_meeting-chime_CreateMeeting.html"
     * >CreateMeeting.html</a> API will be used. However, if Amazon Transcribe is not available in the
     * <code>MediaRegion</code>, then a TranscriptFailed event is sent.
     * </p>
     * <p>
     * Use <code>auto</code> to use Amazon Transcribe in a Region near the meeting’s <code>MediaRegion</code>. For more
     * information, refer to <a
     * href="https://docs.aws.amazon.com/chime-sdk/latest/dg/transcription-options.html#choose-region">Choosing a
     * transcription Region</a> in the <i>Amazon Chime SDK Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #region} will
     * return {@link TranscribeRegion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #regionAsString}.
     * </p>
     * 
     * @return The AWS Region in which to use Amazon Transcribe.</p>
     *         <p>
     *         If you don't specify a Region, then the <code>MediaRegion</code> parameter of the <a
     *         href="https://docs.aws.amazon.com/chime-sdk/latest/APIReference/API_meeting-chime_CreateMeeting.html"
     *         >CreateMeeting.html</a> API will be used. However, if Amazon Transcribe is not available in the
     *         <code>MediaRegion</code>, then a TranscriptFailed event is sent.
     *         </p>
     *         <p>
     *         Use <code>auto</code> to use Amazon Transcribe in a Region near the meeting’s <code>MediaRegion</code>.
     *         For more information, refer to <a
     *         href="https://docs.aws.amazon.com/chime-sdk/latest/dg/transcription-options.html#choose-region">Choosing
     *         a transcription Region</a> in the <i>Amazon Chime SDK Developer Guide</i>.
     * @see TranscribeRegion
     */
    public final String regionAsString() {
        return region;
    }

    /**
     * <p>
     * Enables partial result stabilization for your transcription. Partial result stabilization can reduce latency in
     * your output, but may impact accuracy.
     * </p>
     * 
     * @return Enables partial result stabilization for your transcription. Partial result stabilization can reduce
     *         latency in your output, but may impact accuracy.
     */
    public final Boolean enablePartialResultsStabilization() {
        return enablePartialResultsStabilization;
    }

    /**
     * <p>
     * Specify the level of stability to use when you enable partial results stabilization (
     * <code>EnablePartialResultsStabilization</code>).
     * </p>
     * <p>
     * Low stability provides the highest accuracy. High stability transcribes faster, but with slightly lower accuracy.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #partialResultsStability} will return {@link TranscribePartialResultsStability#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #partialResultsStabilityAsString}.
     * </p>
     * 
     * @return Specify the level of stability to use when you enable partial results stabilization (
     *         <code>EnablePartialResultsStabilization</code>).</p>
     *         <p>
     *         Low stability provides the highest accuracy. High stability transcribes faster, but with slightly lower
     *         accuracy.
     * @see TranscribePartialResultsStability
     */
    public final TranscribePartialResultsStability partialResultsStability() {
        return TranscribePartialResultsStability.fromValue(partialResultsStability);
    }

    /**
     * <p>
     * Specify the level of stability to use when you enable partial results stabilization (
     * <code>EnablePartialResultsStabilization</code>).
     * </p>
     * <p>
     * Low stability provides the highest accuracy. High stability transcribes faster, but with slightly lower accuracy.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #partialResultsStability} will return {@link TranscribePartialResultsStability#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #partialResultsStabilityAsString}.
     * </p>
     * 
     * @return Specify the level of stability to use when you enable partial results stabilization (
     *         <code>EnablePartialResultsStabilization</code>).</p>
     *         <p>
     *         Low stability provides the highest accuracy. High stability transcribes faster, but with slightly lower
     *         accuracy.
     * @see TranscribePartialResultsStability
     */
    public final String partialResultsStabilityAsString() {
        return partialResultsStability;
    }

    /**
     * <p>
     * Labels all personally identifiable information (PII) identified in your transcript. If you don't include
     * <code>PiiEntityTypes</code>, all PII is identified.
     * </p>
     * <p>
     * You can’t set <code>ContentIdentificationType</code> and <code>ContentRedactionType</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #contentIdentificationType} will return {@link TranscribeContentIdentificationType#UNKNOWN_TO_SDK_VERSION}
     * . The raw value returned by the service is available from {@link #contentIdentificationTypeAsString}.
     * </p>
     * 
     * @return Labels all personally identifiable information (PII) identified in your transcript. If you don't include
     *         <code>PiiEntityTypes</code>, all PII is identified.</p>
     *         <p>
     *         You can’t set <code>ContentIdentificationType</code> and <code>ContentRedactionType</code>.
     * @see TranscribeContentIdentificationType
     */
    public final TranscribeContentIdentificationType contentIdentificationType() {
        return TranscribeContentIdentificationType.fromValue(contentIdentificationType);
    }

    /**
     * <p>
     * Labels all personally identifiable information (PII) identified in your transcript. If you don't include
     * <code>PiiEntityTypes</code>, all PII is identified.
     * </p>
     * <p>
     * You can’t set <code>ContentIdentificationType</code> and <code>ContentRedactionType</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #contentIdentificationType} will return {@link TranscribeContentIdentificationType#UNKNOWN_TO_SDK_VERSION}
     * . The raw value returned by the service is available from {@link #contentIdentificationTypeAsString}.
     * </p>
     * 
     * @return Labels all personally identifiable information (PII) identified in your transcript. If you don't include
     *         <code>PiiEntityTypes</code>, all PII is identified.</p>
     *         <p>
     *         You can’t set <code>ContentIdentificationType</code> and <code>ContentRedactionType</code>.
     * @see TranscribeContentIdentificationType
     */
    public final String contentIdentificationTypeAsString() {
        return contentIdentificationType;
    }

    /**
     * <p>
     * Content redaction is performed at the segment level. If you don't include <code>PiiEntityTypes</code>, all PII is
     * redacted.
     * </p>
     * <p>
     * You can’t set <code>ContentIdentificationType</code> and <code>ContentRedactionType</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #contentRedactionType} will return {@link TranscribeContentRedactionType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #contentRedactionTypeAsString}.
     * </p>
     * 
     * @return Content redaction is performed at the segment level. If you don't include <code>PiiEntityTypes</code>,
     *         all PII is redacted.</p>
     *         <p>
     *         You can’t set <code>ContentIdentificationType</code> and <code>ContentRedactionType</code>.
     * @see TranscribeContentRedactionType
     */
    public final TranscribeContentRedactionType contentRedactionType() {
        return TranscribeContentRedactionType.fromValue(contentRedactionType);
    }

    /**
     * <p>
     * Content redaction is performed at the segment level. If you don't include <code>PiiEntityTypes</code>, all PII is
     * redacted.
     * </p>
     * <p>
     * You can’t set <code>ContentIdentificationType</code> and <code>ContentRedactionType</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #contentRedactionType} will return {@link TranscribeContentRedactionType#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #contentRedactionTypeAsString}.
     * </p>
     * 
     * @return Content redaction is performed at the segment level. If you don't include <code>PiiEntityTypes</code>,
     *         all PII is redacted.</p>
     *         <p>
     *         You can’t set <code>ContentIdentificationType</code> and <code>ContentRedactionType</code>.
     * @see TranscribeContentRedactionType
     */
    public final String contentRedactionTypeAsString() {
        return contentRedactionType;
    }

    /**
     * <p>
     * Specify which types of personally identifiable information (PII) you want to redact in your transcript. You can
     * include as many types as you'd like, or you can select <code>ALL</code>.
     * </p>
     * <p>
     * Values must be comma-separated and can include: <code>ADDRESS</code>, <code>BANK_ACCOUNT_NUMBER</code>,
     * <code>BANK_ROUTING</code>, <code>CREDIT_DEBIT_CVV</code>, <code>CREDIT_DEBIT_EXPIRY</code>
     * <code>CREDIT_DEBIT_NUMBER</code>, <code>EMAIL</code>,<code>NAME</code>, <code>PHONE</code>, <code>PIN</code>,
     * <code>SSN</code>, or <code>ALL</code>.
     * </p>
     * <p>
     * Note that if you include <code>PiiEntityTypes</code>, you must also include
     * <code>ContentIdentificationType</code> or <code>ContentRedactionType</code>.
     * </p>
     * <p>
     * If you include <code>ContentRedactionType</code> or <code>ContentIdentificationType</code>, but do not include
     * <code>PiiEntityTypes</code>, all PII is redacted or identified.
     * </p>
     * 
     * @return Specify which types of personally identifiable information (PII) you want to redact in your transcript.
     *         You can include as many types as you'd like, or you can select <code>ALL</code>.</p>
     *         <p>
     *         Values must be comma-separated and can include: <code>ADDRESS</code>, <code>BANK_ACCOUNT_NUMBER</code>,
     *         <code>BANK_ROUTING</code>, <code>CREDIT_DEBIT_CVV</code>, <code>CREDIT_DEBIT_EXPIRY</code>
     *         <code>CREDIT_DEBIT_NUMBER</code>, <code>EMAIL</code>,<code>NAME</code>, <code>PHONE</code>,
     *         <code>PIN</code>, <code>SSN</code>, or <code>ALL</code>.
     *         </p>
     *         <p>
     *         Note that if you include <code>PiiEntityTypes</code>, you must also include
     *         <code>ContentIdentificationType</code> or <code>ContentRedactionType</code>.
     *         </p>
     *         <p>
     *         If you include <code>ContentRedactionType</code> or <code>ContentIdentificationType</code>, but do not
     *         include <code>PiiEntityTypes</code>, all PII is redacted or identified.
     */
    public final String piiEntityTypes() {
        return piiEntityTypes;
    }

    /**
     * <p>
     * Specify the name of the custom language model that you want to use when processing your transcription. Note that
     * language model names are case sensitive.
     * </p>
     * <p>
     * The language of the specified language model must match the language code. If the languages don't match, the
     * custom language model isn't applied. There are no errors or warnings associated with a language mismatch.
     * </p>
     * <p>
     * If you use Amazon Transcribe in multiple Regions, the custom language model must be available in Amazon
     * Transcribe in each Region.
     * </p>
     * 
     * @return Specify the name of the custom language model that you want to use when processing your transcription.
     *         Note that language model names are case sensitive.</p>
     *         <p>
     *         The language of the specified language model must match the language code. If the languages don't match,
     *         the custom language model isn't applied. There are no errors or warnings associated with a language
     *         mismatch.
     *         </p>
     *         <p>
     *         If you use Amazon Transcribe in multiple Regions, the custom language model must be available in Amazon
     *         Transcribe in each Region.
     */
    public final String languageModelName() {
        return languageModelName;
    }

    /**
     * <p>
     * Enables automatic language identification for your transcription.
     * </p>
     * <p>
     * If you include <code>IdentifyLanguage</code>, you can optionally use <code>LanguageOptions</code> to include a
     * list of language codes that you think may be present in your audio stream. Including language options can improve
     * transcription accuracy.
     * </p>
     * <p>
     * You can also use <code>PreferredLanguage</code> to include a preferred language. Doing so can help Amazon
     * Transcribe identify the language faster.
     * </p>
     * <p>
     * You must include either <code>LanguageCode</code> or <code>IdentifyLanguage</code>.
     * </p>
     * <p>
     * Language identification can't be combined with custom language models or redaction.
     * </p>
     * 
     * @return Enables automatic language identification for your transcription.</p>
     *         <p>
     *         If you include <code>IdentifyLanguage</code>, you can optionally use <code>LanguageOptions</code> to
     *         include a list of language codes that you think may be present in your audio stream. Including language
     *         options can improve transcription accuracy.
     *         </p>
     *         <p>
     *         You can also use <code>PreferredLanguage</code> to include a preferred language. Doing so can help Amazon
     *         Transcribe identify the language faster.
     *         </p>
     *         <p>
     *         You must include either <code>LanguageCode</code> or <code>IdentifyLanguage</code>.
     *         </p>
     *         <p>
     *         Language identification can't be combined with custom language models or redaction.
     */
    public final Boolean identifyLanguage() {
        return identifyLanguage;
    }

    /**
     * <p>
     * Specify two or more language codes that represent the languages you think may be present in your media; including
     * more than five is not recommended. If you're unsure what languages are present, do not include this parameter.
     * </p>
     * <p>
     * Including language options can improve the accuracy of language identification.
     * </p>
     * <p>
     * If you include <code>LanguageOptions</code>, you must also include <code>IdentifyLanguage</code>.
     * </p>
     * <important>
     * <p>
     * You can only include one language dialect per language. For example, you cannot include <code>en-US</code> and
     * <code>en-AU</code>.
     * </p>
     * </important>
     * 
     * @return Specify two or more language codes that represent the languages you think may be present in your media;
     *         including more than five is not recommended. If you're unsure what languages are present, do not include
     *         this parameter.</p>
     *         <p>
     *         Including language options can improve the accuracy of language identification.
     *         </p>
     *         <p>
     *         If you include <code>LanguageOptions</code>, you must also include <code>IdentifyLanguage</code>.
     *         </p>
     *         <important>
     *         <p>
     *         You can only include one language dialect per language. For example, you cannot include
     *         <code>en-US</code> and <code>en-AU</code>.
     *         </p>
     */
    public final String languageOptions() {
        return languageOptions;
    }

    /**
     * <p>
     * Specify a preferred language from the subset of languages codes you specified in <code>LanguageOptions</code>.
     * </p>
     * <p>
     * You can only use this parameter if you include <code>IdentifyLanguage</code> and <code>LanguageOptions</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #preferredLanguage}
     * will return {@link TranscribeLanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #preferredLanguageAsString}.
     * </p>
     * 
     * @return Specify a preferred language from the subset of languages codes you specified in
     *         <code>LanguageOptions</code>.</p>
     *         <p>
     *         You can only use this parameter if you include <code>IdentifyLanguage</code> and
     *         <code>LanguageOptions</code>.
     * @see TranscribeLanguageCode
     */
    public final TranscribeLanguageCode preferredLanguage() {
        return TranscribeLanguageCode.fromValue(preferredLanguage);
    }

    /**
     * <p>
     * Specify a preferred language from the subset of languages codes you specified in <code>LanguageOptions</code>.
     * </p>
     * <p>
     * You can only use this parameter if you include <code>IdentifyLanguage</code> and <code>LanguageOptions</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #preferredLanguage}
     * will return {@link TranscribeLanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #preferredLanguageAsString}.
     * </p>
     * 
     * @return Specify a preferred language from the subset of languages codes you specified in
     *         <code>LanguageOptions</code>.</p>
     *         <p>
     *         You can only use this parameter if you include <code>IdentifyLanguage</code> and
     *         <code>LanguageOptions</code>.
     * @see TranscribeLanguageCode
     */
    public final String preferredLanguageAsString() {
        return preferredLanguage;
    }

    /**
     * <p>
     * Specify the names of the custom vocabularies that you want to use when processing your transcription. Note that
     * vocabulary names are case sensitive.
     * </p>
     * <p>
     * If you use Amazon Transcribe in multiple Regions, the vocabulary must be available in Amazon Transcribe in each
     * Region.
     * </p>
     * <p>
     * If you don't include <code>IdentifyLanguage</code> and want to use a custom vocabulary with your transcription,
     * use the <code>VocabularyName</code> parameter instead.
     * </p>
     * 
     * @return Specify the names of the custom vocabularies that you want to use when processing your transcription.
     *         Note that vocabulary names are case sensitive.</p>
     *         <p>
     *         If you use Amazon Transcribe in multiple Regions, the vocabulary must be available in Amazon Transcribe
     *         in each Region.
     *         </p>
     *         <p>
     *         If you don't include <code>IdentifyLanguage</code> and want to use a custom vocabulary with your
     *         transcription, use the <code>VocabularyName</code> parameter instead.
     */
    public final String vocabularyNames() {
        return vocabularyNames;
    }

    /**
     * <p>
     * Specify the names of the custom vocabulary filters that you want to use when processing your transcription. Note
     * that vocabulary filter names are case sensitive.
     * </p>
     * <p>
     * If you use Amazon Transcribe in multiple Regions, the vocabulary filter must be available in Amazon Transcribe in
     * each Region.
     * </p>
     * <p>
     * If you're <i>not</i> including <code>IdentifyLanguage</code> and want to use a custom vocabulary filter with your
     * transcription, use the <code>VocabularyFilterName</code> parameter instead.
     * </p>
     * 
     * @return Specify the names of the custom vocabulary filters that you want to use when processing your
     *         transcription. Note that vocabulary filter names are case sensitive.</p>
     *         <p>
     *         If you use Amazon Transcribe in multiple Regions, the vocabulary filter must be available in Amazon
     *         Transcribe in each Region.
     *         </p>
     *         <p>
     *         If you're <i>not</i> including <code>IdentifyLanguage</code> and want to use a custom vocabulary filter
     *         with your transcription, use the <code>VocabularyFilterName</code> parameter instead.
     */
    public final String vocabularyFilterNames() {
        return vocabularyFilterNames;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(languageCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vocabularyFilterMethodAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vocabularyFilterName());
        hashCode = 31 * hashCode + Objects.hashCode(vocabularyName());
        hashCode = 31 * hashCode + Objects.hashCode(regionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(enablePartialResultsStabilization());
        hashCode = 31 * hashCode + Objects.hashCode(partialResultsStabilityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(contentIdentificationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(contentRedactionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(piiEntityTypes());
        hashCode = 31 * hashCode + Objects.hashCode(languageModelName());
        hashCode = 31 * hashCode + Objects.hashCode(identifyLanguage());
        hashCode = 31 * hashCode + Objects.hashCode(languageOptions());
        hashCode = 31 * hashCode + Objects.hashCode(preferredLanguageAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vocabularyNames());
        hashCode = 31 * hashCode + Objects.hashCode(vocabularyFilterNames());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EngineTranscribeSettings)) {
            return false;
        }
        EngineTranscribeSettings other = (EngineTranscribeSettings) obj;
        return Objects.equals(languageCodeAsString(), other.languageCodeAsString())
                && Objects.equals(vocabularyFilterMethodAsString(), other.vocabularyFilterMethodAsString())
                && Objects.equals(vocabularyFilterName(), other.vocabularyFilterName())
                && Objects.equals(vocabularyName(), other.vocabularyName())
                && Objects.equals(regionAsString(), other.regionAsString())
                && Objects.equals(enablePartialResultsStabilization(), other.enablePartialResultsStabilization())
                && Objects.equals(partialResultsStabilityAsString(), other.partialResultsStabilityAsString())
                && Objects.equals(contentIdentificationTypeAsString(), other.contentIdentificationTypeAsString())
                && Objects.equals(contentRedactionTypeAsString(), other.contentRedactionTypeAsString())
                && Objects.equals(piiEntityTypes(), other.piiEntityTypes())
                && Objects.equals(languageModelName(), other.languageModelName())
                && Objects.equals(identifyLanguage(), other.identifyLanguage())
                && Objects.equals(languageOptions(), other.languageOptions())
                && Objects.equals(preferredLanguageAsString(), other.preferredLanguageAsString())
                && Objects.equals(vocabularyNames(), other.vocabularyNames())
                && Objects.equals(vocabularyFilterNames(), other.vocabularyFilterNames());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EngineTranscribeSettings").add("LanguageCode", languageCodeAsString())
                .add("VocabularyFilterMethod", vocabularyFilterMethodAsString())
                .add("VocabularyFilterName", vocabularyFilterName()).add("VocabularyName", vocabularyName())
                .add("Region", regionAsString()).add("EnablePartialResultsStabilization", enablePartialResultsStabilization())
                .add("PartialResultsStability", partialResultsStabilityAsString())
                .add("ContentIdentificationType", contentIdentificationTypeAsString())
                .add("ContentRedactionType", contentRedactionTypeAsString()).add("PiiEntityTypes", piiEntityTypes())
                .add("LanguageModelName", languageModelName()).add("IdentifyLanguage", identifyLanguage())
                .add("LanguageOptions", languageOptions()).add("PreferredLanguage", preferredLanguageAsString())
                .add("VocabularyNames", vocabularyNames()).add("VocabularyFilterNames", vocabularyFilterNames()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LanguageCode":
            return Optional.ofNullable(clazz.cast(languageCodeAsString()));
        case "VocabularyFilterMethod":
            return Optional.ofNullable(clazz.cast(vocabularyFilterMethodAsString()));
        case "VocabularyFilterName":
            return Optional.ofNullable(clazz.cast(vocabularyFilterName()));
        case "VocabularyName":
            return Optional.ofNullable(clazz.cast(vocabularyName()));
        case "Region":
            return Optional.ofNullable(clazz.cast(regionAsString()));
        case "EnablePartialResultsStabilization":
            return Optional.ofNullable(clazz.cast(enablePartialResultsStabilization()));
        case "PartialResultsStability":
            return Optional.ofNullable(clazz.cast(partialResultsStabilityAsString()));
        case "ContentIdentificationType":
            return Optional.ofNullable(clazz.cast(contentIdentificationTypeAsString()));
        case "ContentRedactionType":
            return Optional.ofNullable(clazz.cast(contentRedactionTypeAsString()));
        case "PiiEntityTypes":
            return Optional.ofNullable(clazz.cast(piiEntityTypes()));
        case "LanguageModelName":
            return Optional.ofNullable(clazz.cast(languageModelName()));
        case "IdentifyLanguage":
            return Optional.ofNullable(clazz.cast(identifyLanguage()));
        case "LanguageOptions":
            return Optional.ofNullable(clazz.cast(languageOptions()));
        case "PreferredLanguage":
            return Optional.ofNullable(clazz.cast(preferredLanguageAsString()));
        case "VocabularyNames":
            return Optional.ofNullable(clazz.cast(vocabularyNames()));
        case "VocabularyFilterNames":
            return Optional.ofNullable(clazz.cast(vocabularyFilterNames()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("LanguageCode", LANGUAGE_CODE_FIELD);
        map.put("VocabularyFilterMethod", VOCABULARY_FILTER_METHOD_FIELD);
        map.put("VocabularyFilterName", VOCABULARY_FILTER_NAME_FIELD);
        map.put("VocabularyName", VOCABULARY_NAME_FIELD);
        map.put("Region", REGION_FIELD);
        map.put("EnablePartialResultsStabilization", ENABLE_PARTIAL_RESULTS_STABILIZATION_FIELD);
        map.put("PartialResultsStability", PARTIAL_RESULTS_STABILITY_FIELD);
        map.put("ContentIdentificationType", CONTENT_IDENTIFICATION_TYPE_FIELD);
        map.put("ContentRedactionType", CONTENT_REDACTION_TYPE_FIELD);
        map.put("PiiEntityTypes", PII_ENTITY_TYPES_FIELD);
        map.put("LanguageModelName", LANGUAGE_MODEL_NAME_FIELD);
        map.put("IdentifyLanguage", IDENTIFY_LANGUAGE_FIELD);
        map.put("LanguageOptions", LANGUAGE_OPTIONS_FIELD);
        map.put("PreferredLanguage", PREFERRED_LANGUAGE_FIELD);
        map.put("VocabularyNames", VOCABULARY_NAMES_FIELD);
        map.put("VocabularyFilterNames", VOCABULARY_FILTER_NAMES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EngineTranscribeSettings, T> g) {
        return obj -> g.apply((EngineTranscribeSettings) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EngineTranscribeSettings> {
        /**
         * <p>
         * Specify the language code that represents the language spoken.
         * </p>
         * <p>
         * If you're unsure of the language spoken in your audio, consider using <code>IdentifyLanguage</code> to enable
         * automatic language identification.
         * </p>
         * 
         * @param languageCode
         *        Specify the language code that represents the language spoken.</p>
         *        <p>
         *        If you're unsure of the language spoken in your audio, consider using <code>IdentifyLanguage</code> to
         *        enable automatic language identification.
         * @see TranscribeLanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscribeLanguageCode
         */
        Builder languageCode(String languageCode);

        /**
         * <p>
         * Specify the language code that represents the language spoken.
         * </p>
         * <p>
         * If you're unsure of the language spoken in your audio, consider using <code>IdentifyLanguage</code> to enable
         * automatic language identification.
         * </p>
         * 
         * @param languageCode
         *        Specify the language code that represents the language spoken.</p>
         *        <p>
         *        If you're unsure of the language spoken in your audio, consider using <code>IdentifyLanguage</code> to
         *        enable automatic language identification.
         * @see TranscribeLanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscribeLanguageCode
         */
        Builder languageCode(TranscribeLanguageCode languageCode);

        /**
         * <p>
         * Specify how you want your vocabulary filter applied to your transcript.
         * </p>
         * <p>
         * To replace words with <code>***</code>, choose <code>mask</code>.
         * </p>
         * <p>
         * To delete words, choose <code>remove</code>.
         * </p>
         * <p>
         * To flag words without changing them, choose <code>tag</code>.
         * </p>
         * 
         * @param vocabularyFilterMethod
         *        Specify how you want your vocabulary filter applied to your transcript.</p>
         *        <p>
         *        To replace words with <code>***</code>, choose <code>mask</code>.
         *        </p>
         *        <p>
         *        To delete words, choose <code>remove</code>.
         *        </p>
         *        <p>
         *        To flag words without changing them, choose <code>tag</code>.
         * @see TranscribeVocabularyFilterMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscribeVocabularyFilterMethod
         */
        Builder vocabularyFilterMethod(String vocabularyFilterMethod);

        /**
         * <p>
         * Specify how you want your vocabulary filter applied to your transcript.
         * </p>
         * <p>
         * To replace words with <code>***</code>, choose <code>mask</code>.
         * </p>
         * <p>
         * To delete words, choose <code>remove</code>.
         * </p>
         * <p>
         * To flag words without changing them, choose <code>tag</code>.
         * </p>
         * 
         * @param vocabularyFilterMethod
         *        Specify how you want your vocabulary filter applied to your transcript.</p>
         *        <p>
         *        To replace words with <code>***</code>, choose <code>mask</code>.
         *        </p>
         *        <p>
         *        To delete words, choose <code>remove</code>.
         *        </p>
         *        <p>
         *        To flag words without changing them, choose <code>tag</code>.
         * @see TranscribeVocabularyFilterMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscribeVocabularyFilterMethod
         */
        Builder vocabularyFilterMethod(TranscribeVocabularyFilterMethod vocabularyFilterMethod);

        /**
         * <p>
         * Specify the name of the custom vocabulary filter that you want to use when processing your transcription.
         * Note that vocabulary filter names are case sensitive.
         * </p>
         * <p>
         * If you use Amazon Transcribe in multiple Regions, the vocabulary filter must be available in Amazon
         * Transcribe in each Region.
         * </p>
         * <p>
         * If you include <code>IdentifyLanguage</code> and want to use one or more vocabulary filters with your
         * transcription, use the <code>VocabularyFilterNames</code> parameter instead.
         * </p>
         * 
         * @param vocabularyFilterName
         *        Specify the name of the custom vocabulary filter that you want to use when processing your
         *        transcription. Note that vocabulary filter names are case sensitive. </p>
         *        <p>
         *        If you use Amazon Transcribe in multiple Regions, the vocabulary filter must be available in Amazon
         *        Transcribe in each Region.
         *        </p>
         *        <p>
         *        If you include <code>IdentifyLanguage</code> and want to use one or more vocabulary filters with your
         *        transcription, use the <code>VocabularyFilterNames</code> parameter instead.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vocabularyFilterName(String vocabularyFilterName);

        /**
         * <p>
         * Specify the name of the custom vocabulary that you want to use when processing your transcription. Note that
         * vocabulary names are case sensitive.
         * </p>
         * <p>
         * If you use Amazon Transcribe multiple Regions, the vocabulary must be available in Amazon Transcribe in each
         * Region.
         * </p>
         * <p>
         * If you include <code>IdentifyLanguage</code> and want to use one or more custom vocabularies with your
         * transcription, use the <code>VocabularyNames</code> parameter instead.
         * </p>
         * 
         * @param vocabularyName
         *        Specify the name of the custom vocabulary that you want to use when processing your transcription.
         *        Note that vocabulary names are case sensitive.</p>
         *        <p>
         *        If you use Amazon Transcribe multiple Regions, the vocabulary must be available in Amazon Transcribe
         *        in each Region.
         *        </p>
         *        <p>
         *        If you include <code>IdentifyLanguage</code> and want to use one or more custom vocabularies with your
         *        transcription, use the <code>VocabularyNames</code> parameter instead.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vocabularyName(String vocabularyName);

        /**
         * <p>
         * The AWS Region in which to use Amazon Transcribe.
         * </p>
         * <p>
         * If you don't specify a Region, then the <code>MediaRegion</code> parameter of the <a
         * href="https://docs.aws.amazon.com/chime-sdk/latest/APIReference/API_meeting-chime_CreateMeeting.html"
         * >CreateMeeting.html</a> API will be used. However, if Amazon Transcribe is not available in the
         * <code>MediaRegion</code>, then a TranscriptFailed event is sent.
         * </p>
         * <p>
         * Use <code>auto</code> to use Amazon Transcribe in a Region near the meeting’s <code>MediaRegion</code>. For
         * more information, refer to <a
         * href="https://docs.aws.amazon.com/chime-sdk/latest/dg/transcription-options.html#choose-region">Choosing a
         * transcription Region</a> in the <i>Amazon Chime SDK Developer Guide</i>.
         * </p>
         * 
         * @param region
         *        The AWS Region in which to use Amazon Transcribe.</p>
         *        <p>
         *        If you don't specify a Region, then the <code>MediaRegion</code> parameter of the <a
         *        href="https://docs.aws.amazon.com/chime-sdk/latest/APIReference/API_meeting-chime_CreateMeeting.html"
         *        >CreateMeeting.html</a> API will be used. However, if Amazon Transcribe is not available in the
         *        <code>MediaRegion</code>, then a TranscriptFailed event is sent.
         *        </p>
         *        <p>
         *        Use <code>auto</code> to use Amazon Transcribe in a Region near the meeting’s <code>MediaRegion</code>
         *        . For more information, refer to <a
         *        href="https://docs.aws.amazon.com/chime-sdk/latest/dg/transcription-options.html#choose-region"
         *        >Choosing a transcription Region</a> in the <i>Amazon Chime SDK Developer Guide</i>.
         * @see TranscribeRegion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscribeRegion
         */
        Builder region(String region);

        /**
         * <p>
         * The AWS Region in which to use Amazon Transcribe.
         * </p>
         * <p>
         * If you don't specify a Region, then the <code>MediaRegion</code> parameter of the <a
         * href="https://docs.aws.amazon.com/chime-sdk/latest/APIReference/API_meeting-chime_CreateMeeting.html"
         * >CreateMeeting.html</a> API will be used. However, if Amazon Transcribe is not available in the
         * <code>MediaRegion</code>, then a TranscriptFailed event is sent.
         * </p>
         * <p>
         * Use <code>auto</code> to use Amazon Transcribe in a Region near the meeting’s <code>MediaRegion</code>. For
         * more information, refer to <a
         * href="https://docs.aws.amazon.com/chime-sdk/latest/dg/transcription-options.html#choose-region">Choosing a
         * transcription Region</a> in the <i>Amazon Chime SDK Developer Guide</i>.
         * </p>
         * 
         * @param region
         *        The AWS Region in which to use Amazon Transcribe.</p>
         *        <p>
         *        If you don't specify a Region, then the <code>MediaRegion</code> parameter of the <a
         *        href="https://docs.aws.amazon.com/chime-sdk/latest/APIReference/API_meeting-chime_CreateMeeting.html"
         *        >CreateMeeting.html</a> API will be used. However, if Amazon Transcribe is not available in the
         *        <code>MediaRegion</code>, then a TranscriptFailed event is sent.
         *        </p>
         *        <p>
         *        Use <code>auto</code> to use Amazon Transcribe in a Region near the meeting’s <code>MediaRegion</code>
         *        . For more information, refer to <a
         *        href="https://docs.aws.amazon.com/chime-sdk/latest/dg/transcription-options.html#choose-region"
         *        >Choosing a transcription Region</a> in the <i>Amazon Chime SDK Developer Guide</i>.
         * @see TranscribeRegion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscribeRegion
         */
        Builder region(TranscribeRegion region);

        /**
         * <p>
         * Enables partial result stabilization for your transcription. Partial result stabilization can reduce latency
         * in your output, but may impact accuracy.
         * </p>
         * 
         * @param enablePartialResultsStabilization
         *        Enables partial result stabilization for your transcription. Partial result stabilization can reduce
         *        latency in your output, but may impact accuracy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enablePartialResultsStabilization(Boolean enablePartialResultsStabilization);

        /**
         * <p>
         * Specify the level of stability to use when you enable partial results stabilization (
         * <code>EnablePartialResultsStabilization</code>).
         * </p>
         * <p>
         * Low stability provides the highest accuracy. High stability transcribes faster, but with slightly lower
         * accuracy.
         * </p>
         * 
         * @param partialResultsStability
         *        Specify the level of stability to use when you enable partial results stabilization (
         *        <code>EnablePartialResultsStabilization</code>).</p>
         *        <p>
         *        Low stability provides the highest accuracy. High stability transcribes faster, but with slightly
         *        lower accuracy.
         * @see TranscribePartialResultsStability
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscribePartialResultsStability
         */
        Builder partialResultsStability(String partialResultsStability);

        /**
         * <p>
         * Specify the level of stability to use when you enable partial results stabilization (
         * <code>EnablePartialResultsStabilization</code>).
         * </p>
         * <p>
         * Low stability provides the highest accuracy. High stability transcribes faster, but with slightly lower
         * accuracy.
         * </p>
         * 
         * @param partialResultsStability
         *        Specify the level of stability to use when you enable partial results stabilization (
         *        <code>EnablePartialResultsStabilization</code>).</p>
         *        <p>
         *        Low stability provides the highest accuracy. High stability transcribes faster, but with slightly
         *        lower accuracy.
         * @see TranscribePartialResultsStability
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscribePartialResultsStability
         */
        Builder partialResultsStability(TranscribePartialResultsStability partialResultsStability);

        /**
         * <p>
         * Labels all personally identifiable information (PII) identified in your transcript. If you don't include
         * <code>PiiEntityTypes</code>, all PII is identified.
         * </p>
         * <p>
         * You can’t set <code>ContentIdentificationType</code> and <code>ContentRedactionType</code>.
         * </p>
         * 
         * @param contentIdentificationType
         *        Labels all personally identifiable information (PII) identified in your transcript. If you don't
         *        include <code>PiiEntityTypes</code>, all PII is identified.</p>
         *        <p>
         *        You can’t set <code>ContentIdentificationType</code> and <code>ContentRedactionType</code>.
         * @see TranscribeContentIdentificationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscribeContentIdentificationType
         */
        Builder contentIdentificationType(String contentIdentificationType);

        /**
         * <p>
         * Labels all personally identifiable information (PII) identified in your transcript. If you don't include
         * <code>PiiEntityTypes</code>, all PII is identified.
         * </p>
         * <p>
         * You can’t set <code>ContentIdentificationType</code> and <code>ContentRedactionType</code>.
         * </p>
         * 
         * @param contentIdentificationType
         *        Labels all personally identifiable information (PII) identified in your transcript. If you don't
         *        include <code>PiiEntityTypes</code>, all PII is identified.</p>
         *        <p>
         *        You can’t set <code>ContentIdentificationType</code> and <code>ContentRedactionType</code>.
         * @see TranscribeContentIdentificationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscribeContentIdentificationType
         */
        Builder contentIdentificationType(TranscribeContentIdentificationType contentIdentificationType);

        /**
         * <p>
         * Content redaction is performed at the segment level. If you don't include <code>PiiEntityTypes</code>, all
         * PII is redacted.
         * </p>
         * <p>
         * You can’t set <code>ContentIdentificationType</code> and <code>ContentRedactionType</code>.
         * </p>
         * 
         * @param contentRedactionType
         *        Content redaction is performed at the segment level. If you don't include <code>PiiEntityTypes</code>,
         *        all PII is redacted.</p>
         *        <p>
         *        You can’t set <code>ContentIdentificationType</code> and <code>ContentRedactionType</code>.
         * @see TranscribeContentRedactionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscribeContentRedactionType
         */
        Builder contentRedactionType(String contentRedactionType);

        /**
         * <p>
         * Content redaction is performed at the segment level. If you don't include <code>PiiEntityTypes</code>, all
         * PII is redacted.
         * </p>
         * <p>
         * You can’t set <code>ContentIdentificationType</code> and <code>ContentRedactionType</code>.
         * </p>
         * 
         * @param contentRedactionType
         *        Content redaction is performed at the segment level. If you don't include <code>PiiEntityTypes</code>,
         *        all PII is redacted.</p>
         *        <p>
         *        You can’t set <code>ContentIdentificationType</code> and <code>ContentRedactionType</code>.
         * @see TranscribeContentRedactionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscribeContentRedactionType
         */
        Builder contentRedactionType(TranscribeContentRedactionType contentRedactionType);

        /**
         * <p>
         * Specify which types of personally identifiable information (PII) you want to redact in your transcript. You
         * can include as many types as you'd like, or you can select <code>ALL</code>.
         * </p>
         * <p>
         * Values must be comma-separated and can include: <code>ADDRESS</code>, <code>BANK_ACCOUNT_NUMBER</code>,
         * <code>BANK_ROUTING</code>, <code>CREDIT_DEBIT_CVV</code>, <code>CREDIT_DEBIT_EXPIRY</code>
         * <code>CREDIT_DEBIT_NUMBER</code>, <code>EMAIL</code>,<code>NAME</code>, <code>PHONE</code>, <code>PIN</code>,
         * <code>SSN</code>, or <code>ALL</code>.
         * </p>
         * <p>
         * Note that if you include <code>PiiEntityTypes</code>, you must also include
         * <code>ContentIdentificationType</code> or <code>ContentRedactionType</code>.
         * </p>
         * <p>
         * If you include <code>ContentRedactionType</code> or <code>ContentIdentificationType</code>, but do not
         * include <code>PiiEntityTypes</code>, all PII is redacted or identified.
         * </p>
         * 
         * @param piiEntityTypes
         *        Specify which types of personally identifiable information (PII) you want to redact in your
         *        transcript. You can include as many types as you'd like, or you can select <code>ALL</code>.</p>
         *        <p>
         *        Values must be comma-separated and can include: <code>ADDRESS</code>, <code>BANK_ACCOUNT_NUMBER</code>, <code>BANK_ROUTING</code>, <code>CREDIT_DEBIT_CVV</code>, <code>CREDIT_DEBIT_EXPIRY</code>
         *        <code>CREDIT_DEBIT_NUMBER</code>, <code>EMAIL</code>,<code>NAME</code>, <code>PHONE</code>,
         *        <code>PIN</code>, <code>SSN</code>, or <code>ALL</code>.
         *        </p>
         *        <p>
         *        Note that if you include <code>PiiEntityTypes</code>, you must also include
         *        <code>ContentIdentificationType</code> or <code>ContentRedactionType</code>.
         *        </p>
         *        <p>
         *        If you include <code>ContentRedactionType</code> or <code>ContentIdentificationType</code>, but do not
         *        include <code>PiiEntityTypes</code>, all PII is redacted or identified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder piiEntityTypes(String piiEntityTypes);

        /**
         * <p>
         * Specify the name of the custom language model that you want to use when processing your transcription. Note
         * that language model names are case sensitive.
         * </p>
         * <p>
         * The language of the specified language model must match the language code. If the languages don't match, the
         * custom language model isn't applied. There are no errors or warnings associated with a language mismatch.
         * </p>
         * <p>
         * If you use Amazon Transcribe in multiple Regions, the custom language model must be available in Amazon
         * Transcribe in each Region.
         * </p>
         * 
         * @param languageModelName
         *        Specify the name of the custom language model that you want to use when processing your transcription.
         *        Note that language model names are case sensitive.</p>
         *        <p>
         *        The language of the specified language model must match the language code. If the languages don't
         *        match, the custom language model isn't applied. There are no errors or warnings associated with a
         *        language mismatch.
         *        </p>
         *        <p>
         *        If you use Amazon Transcribe in multiple Regions, the custom language model must be available in
         *        Amazon Transcribe in each Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageModelName(String languageModelName);

        /**
         * <p>
         * Enables automatic language identification for your transcription.
         * </p>
         * <p>
         * If you include <code>IdentifyLanguage</code>, you can optionally use <code>LanguageOptions</code> to include
         * a list of language codes that you think may be present in your audio stream. Including language options can
         * improve transcription accuracy.
         * </p>
         * <p>
         * You can also use <code>PreferredLanguage</code> to include a preferred language. Doing so can help Amazon
         * Transcribe identify the language faster.
         * </p>
         * <p>
         * You must include either <code>LanguageCode</code> or <code>IdentifyLanguage</code>.
         * </p>
         * <p>
         * Language identification can't be combined with custom language models or redaction.
         * </p>
         * 
         * @param identifyLanguage
         *        Enables automatic language identification for your transcription.</p>
         *        <p>
         *        If you include <code>IdentifyLanguage</code>, you can optionally use <code>LanguageOptions</code> to
         *        include a list of language codes that you think may be present in your audio stream. Including
         *        language options can improve transcription accuracy.
         *        </p>
         *        <p>
         *        You can also use <code>PreferredLanguage</code> to include a preferred language. Doing so can help
         *        Amazon Transcribe identify the language faster.
         *        </p>
         *        <p>
         *        You must include either <code>LanguageCode</code> or <code>IdentifyLanguage</code>.
         *        </p>
         *        <p>
         *        Language identification can't be combined with custom language models or redaction.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identifyLanguage(Boolean identifyLanguage);

        /**
         * <p>
         * Specify two or more language codes that represent the languages you think may be present in your media;
         * including more than five is not recommended. If you're unsure what languages are present, do not include this
         * parameter.
         * </p>
         * <p>
         * Including language options can improve the accuracy of language identification.
         * </p>
         * <p>
         * If you include <code>LanguageOptions</code>, you must also include <code>IdentifyLanguage</code>.
         * </p>
         * <important>
         * <p>
         * You can only include one language dialect per language. For example, you cannot include <code>en-US</code>
         * and <code>en-AU</code>.
         * </p>
         * </important>
         * 
         * @param languageOptions
         *        Specify two or more language codes that represent the languages you think may be present in your
         *        media; including more than five is not recommended. If you're unsure what languages are present, do
         *        not include this parameter.</p>
         *        <p>
         *        Including language options can improve the accuracy of language identification.
         *        </p>
         *        <p>
         *        If you include <code>LanguageOptions</code>, you must also include <code>IdentifyLanguage</code>.
         *        </p>
         *        <important>
         *        <p>
         *        You can only include one language dialect per language. For example, you cannot include
         *        <code>en-US</code> and <code>en-AU</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder languageOptions(String languageOptions);

        /**
         * <p>
         * Specify a preferred language from the subset of languages codes you specified in <code>LanguageOptions</code>
         * .
         * </p>
         * <p>
         * You can only use this parameter if you include <code>IdentifyLanguage</code> and <code>LanguageOptions</code>
         * .
         * </p>
         * 
         * @param preferredLanguage
         *        Specify a preferred language from the subset of languages codes you specified in
         *        <code>LanguageOptions</code>.</p>
         *        <p>
         *        You can only use this parameter if you include <code>IdentifyLanguage</code> and
         *        <code>LanguageOptions</code>.
         * @see TranscribeLanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscribeLanguageCode
         */
        Builder preferredLanguage(String preferredLanguage);

        /**
         * <p>
         * Specify a preferred language from the subset of languages codes you specified in <code>LanguageOptions</code>
         * .
         * </p>
         * <p>
         * You can only use this parameter if you include <code>IdentifyLanguage</code> and <code>LanguageOptions</code>
         * .
         * </p>
         * 
         * @param preferredLanguage
         *        Specify a preferred language from the subset of languages codes you specified in
         *        <code>LanguageOptions</code>.</p>
         *        <p>
         *        You can only use this parameter if you include <code>IdentifyLanguage</code> and
         *        <code>LanguageOptions</code>.
         * @see TranscribeLanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TranscribeLanguageCode
         */
        Builder preferredLanguage(TranscribeLanguageCode preferredLanguage);

        /**
         * <p>
         * Specify the names of the custom vocabularies that you want to use when processing your transcription. Note
         * that vocabulary names are case sensitive.
         * </p>
         * <p>
         * If you use Amazon Transcribe in multiple Regions, the vocabulary must be available in Amazon Transcribe in
         * each Region.
         * </p>
         * <p>
         * If you don't include <code>IdentifyLanguage</code> and want to use a custom vocabulary with your
         * transcription, use the <code>VocabularyName</code> parameter instead.
         * </p>
         * 
         * @param vocabularyNames
         *        Specify the names of the custom vocabularies that you want to use when processing your transcription.
         *        Note that vocabulary names are case sensitive.</p>
         *        <p>
         *        If you use Amazon Transcribe in multiple Regions, the vocabulary must be available in Amazon
         *        Transcribe in each Region.
         *        </p>
         *        <p>
         *        If you don't include <code>IdentifyLanguage</code> and want to use a custom vocabulary with your
         *        transcription, use the <code>VocabularyName</code> parameter instead.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vocabularyNames(String vocabularyNames);

        /**
         * <p>
         * Specify the names of the custom vocabulary filters that you want to use when processing your transcription.
         * Note that vocabulary filter names are case sensitive.
         * </p>
         * <p>
         * If you use Amazon Transcribe in multiple Regions, the vocabulary filter must be available in Amazon
         * Transcribe in each Region.
         * </p>
         * <p>
         * If you're <i>not</i> including <code>IdentifyLanguage</code> and want to use a custom vocabulary filter with
         * your transcription, use the <code>VocabularyFilterName</code> parameter instead.
         * </p>
         * 
         * @param vocabularyFilterNames
         *        Specify the names of the custom vocabulary filters that you want to use when processing your
         *        transcription. Note that vocabulary filter names are case sensitive.</p>
         *        <p>
         *        If you use Amazon Transcribe in multiple Regions, the vocabulary filter must be available in Amazon
         *        Transcribe in each Region.
         *        </p>
         *        <p>
         *        If you're <i>not</i> including <code>IdentifyLanguage</code> and want to use a custom vocabulary
         *        filter with your transcription, use the <code>VocabularyFilterName</code> parameter instead.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vocabularyFilterNames(String vocabularyFilterNames);
    }

    static final class BuilderImpl implements Builder {
        private String languageCode;

        private String vocabularyFilterMethod;

        private String vocabularyFilterName;

        private String vocabularyName;

        private String region;

        private Boolean enablePartialResultsStabilization;

        private String partialResultsStability;

        private String contentIdentificationType;

        private String contentRedactionType;

        private String piiEntityTypes;

        private String languageModelName;

        private Boolean identifyLanguage;

        private String languageOptions;

        private String preferredLanguage;

        private String vocabularyNames;

        private String vocabularyFilterNames;

        private BuilderImpl() {
        }

        private BuilderImpl(EngineTranscribeSettings model) {
            languageCode(model.languageCode);
            vocabularyFilterMethod(model.vocabularyFilterMethod);
            vocabularyFilterName(model.vocabularyFilterName);
            vocabularyName(model.vocabularyName);
            region(model.region);
            enablePartialResultsStabilization(model.enablePartialResultsStabilization);
            partialResultsStability(model.partialResultsStability);
            contentIdentificationType(model.contentIdentificationType);
            contentRedactionType(model.contentRedactionType);
            piiEntityTypes(model.piiEntityTypes);
            languageModelName(model.languageModelName);
            identifyLanguage(model.identifyLanguage);
            languageOptions(model.languageOptions);
            preferredLanguage(model.preferredLanguage);
            vocabularyNames(model.vocabularyNames);
            vocabularyFilterNames(model.vocabularyFilterNames);
        }

        public final String getLanguageCode() {
            return languageCode;
        }

        public final void setLanguageCode(String languageCode) {
            this.languageCode = languageCode;
        }

        @Override
        public final Builder languageCode(String languageCode) {
            this.languageCode = languageCode;
            return this;
        }

        @Override
        public final Builder languageCode(TranscribeLanguageCode languageCode) {
            this.languageCode(languageCode == null ? null : languageCode.toString());
            return this;
        }

        public final String getVocabularyFilterMethod() {
            return vocabularyFilterMethod;
        }

        public final void setVocabularyFilterMethod(String vocabularyFilterMethod) {
            this.vocabularyFilterMethod = vocabularyFilterMethod;
        }

        @Override
        public final Builder vocabularyFilterMethod(String vocabularyFilterMethod) {
            this.vocabularyFilterMethod = vocabularyFilterMethod;
            return this;
        }

        @Override
        public final Builder vocabularyFilterMethod(TranscribeVocabularyFilterMethod vocabularyFilterMethod) {
            this.vocabularyFilterMethod(vocabularyFilterMethod == null ? null : vocabularyFilterMethod.toString());
            return this;
        }

        public final String getVocabularyFilterName() {
            return vocabularyFilterName;
        }

        public final void setVocabularyFilterName(String vocabularyFilterName) {
            this.vocabularyFilterName = vocabularyFilterName;
        }

        @Override
        public final Builder vocabularyFilterName(String vocabularyFilterName) {
            this.vocabularyFilterName = vocabularyFilterName;
            return this;
        }

        public final String getVocabularyName() {
            return vocabularyName;
        }

        public final void setVocabularyName(String vocabularyName) {
            this.vocabularyName = vocabularyName;
        }

        @Override
        public final Builder vocabularyName(String vocabularyName) {
            this.vocabularyName = vocabularyName;
            return this;
        }

        public final String getRegion() {
            return region;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        @Override
        public final Builder region(TranscribeRegion region) {
            this.region(region == null ? null : region.toString());
            return this;
        }

        public final Boolean getEnablePartialResultsStabilization() {
            return enablePartialResultsStabilization;
        }

        public final void setEnablePartialResultsStabilization(Boolean enablePartialResultsStabilization) {
            this.enablePartialResultsStabilization = enablePartialResultsStabilization;
        }

        @Override
        public final Builder enablePartialResultsStabilization(Boolean enablePartialResultsStabilization) {
            this.enablePartialResultsStabilization = enablePartialResultsStabilization;
            return this;
        }

        public final String getPartialResultsStability() {
            return partialResultsStability;
        }

        public final void setPartialResultsStability(String partialResultsStability) {
            this.partialResultsStability = partialResultsStability;
        }

        @Override
        public final Builder partialResultsStability(String partialResultsStability) {
            this.partialResultsStability = partialResultsStability;
            return this;
        }

        @Override
        public final Builder partialResultsStability(TranscribePartialResultsStability partialResultsStability) {
            this.partialResultsStability(partialResultsStability == null ? null : partialResultsStability.toString());
            return this;
        }

        public final String getContentIdentificationType() {
            return contentIdentificationType;
        }

        public final void setContentIdentificationType(String contentIdentificationType) {
            this.contentIdentificationType = contentIdentificationType;
        }

        @Override
        public final Builder contentIdentificationType(String contentIdentificationType) {
            this.contentIdentificationType = contentIdentificationType;
            return this;
        }

        @Override
        public final Builder contentIdentificationType(TranscribeContentIdentificationType contentIdentificationType) {
            this.contentIdentificationType(contentIdentificationType == null ? null : contentIdentificationType.toString());
            return this;
        }

        public final String getContentRedactionType() {
            return contentRedactionType;
        }

        public final void setContentRedactionType(String contentRedactionType) {
            this.contentRedactionType = contentRedactionType;
        }

        @Override
        public final Builder contentRedactionType(String contentRedactionType) {
            this.contentRedactionType = contentRedactionType;
            return this;
        }

        @Override
        public final Builder contentRedactionType(TranscribeContentRedactionType contentRedactionType) {
            this.contentRedactionType(contentRedactionType == null ? null : contentRedactionType.toString());
            return this;
        }

        public final String getPiiEntityTypes() {
            return piiEntityTypes;
        }

        public final void setPiiEntityTypes(String piiEntityTypes) {
            this.piiEntityTypes = piiEntityTypes;
        }

        @Override
        public final Builder piiEntityTypes(String piiEntityTypes) {
            this.piiEntityTypes = piiEntityTypes;
            return this;
        }

        public final String getLanguageModelName() {
            return languageModelName;
        }

        public final void setLanguageModelName(String languageModelName) {
            this.languageModelName = languageModelName;
        }

        @Override
        public final Builder languageModelName(String languageModelName) {
            this.languageModelName = languageModelName;
            return this;
        }

        public final Boolean getIdentifyLanguage() {
            return identifyLanguage;
        }

        public final void setIdentifyLanguage(Boolean identifyLanguage) {
            this.identifyLanguage = identifyLanguage;
        }

        @Override
        public final Builder identifyLanguage(Boolean identifyLanguage) {
            this.identifyLanguage = identifyLanguage;
            return this;
        }

        public final String getLanguageOptions() {
            return languageOptions;
        }

        public final void setLanguageOptions(String languageOptions) {
            this.languageOptions = languageOptions;
        }

        @Override
        public final Builder languageOptions(String languageOptions) {
            this.languageOptions = languageOptions;
            return this;
        }

        public final String getPreferredLanguage() {
            return preferredLanguage;
        }

        public final void setPreferredLanguage(String preferredLanguage) {
            this.preferredLanguage = preferredLanguage;
        }

        @Override
        public final Builder preferredLanguage(String preferredLanguage) {
            this.preferredLanguage = preferredLanguage;
            return this;
        }

        @Override
        public final Builder preferredLanguage(TranscribeLanguageCode preferredLanguage) {
            this.preferredLanguage(preferredLanguage == null ? null : preferredLanguage.toString());
            return this;
        }

        public final String getVocabularyNames() {
            return vocabularyNames;
        }

        public final void setVocabularyNames(String vocabularyNames) {
            this.vocabularyNames = vocabularyNames;
        }

        @Override
        public final Builder vocabularyNames(String vocabularyNames) {
            this.vocabularyNames = vocabularyNames;
            return this;
        }

        public final String getVocabularyFilterNames() {
            return vocabularyFilterNames;
        }

        public final void setVocabularyFilterNames(String vocabularyFilterNames) {
            this.vocabularyFilterNames = vocabularyFilterNames;
        }

        @Override
        public final Builder vocabularyFilterNames(String vocabularyFilterNames) {
            this.vocabularyFilterNames = vocabularyFilterNames;
            return this;
        }

        @Override
        public EngineTranscribeSettings build() {
            return new EngineTranscribeSettings(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
