/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chimesdkidentity;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.chimesdkidentity.model.BadRequestException;
import software.amazon.awssdk.services.chimesdkidentity.model.ChimeSdkIdentityException;
import software.amazon.awssdk.services.chimesdkidentity.model.ChimeSdkIdentityRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.ConflictException;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceAdminRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceAdminResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceUserRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceUserResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceAdminRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceAdminResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceUserRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceUserResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceAdminRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceAdminResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceUserRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceUserResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ForbiddenException;
import software.amazon.awssdk.services.chimesdkidentity.model.GetAppInstanceRetentionSettingsRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.GetAppInstanceRetentionSettingsResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceAdminsRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceAdminsResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUsersRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUsersResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstancesRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstancesResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.PutAppInstanceRetentionSettingsRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.PutAppInstanceRetentionSettingsResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.chimesdkidentity.model.ServiceFailureException;
import software.amazon.awssdk.services.chimesdkidentity.model.ServiceUnavailableException;
import software.amazon.awssdk.services.chimesdkidentity.model.ThrottledClientException;
import software.amazon.awssdk.services.chimesdkidentity.model.UnauthorizedClientException;
import software.amazon.awssdk.services.chimesdkidentity.model.UpdateAppInstanceRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.UpdateAppInstanceResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.UpdateAppInstanceUserRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.UpdateAppInstanceUserResponse;
import software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstanceAdminsIterable;
import software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstanceUsersIterable;
import software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstancesIterable;
import software.amazon.awssdk.services.chimesdkidentity.transform.CreateAppInstanceAdminRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.CreateAppInstanceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.CreateAppInstanceUserRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DeleteAppInstanceAdminRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DeleteAppInstanceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DeleteAppInstanceUserRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DescribeAppInstanceAdminRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DescribeAppInstanceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DescribeAppInstanceUserRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.GetAppInstanceRetentionSettingsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.ListAppInstanceAdminsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.ListAppInstanceUsersRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.ListAppInstancesRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.PutAppInstanceRetentionSettingsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.UpdateAppInstanceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.UpdateAppInstanceUserRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ChimeSdkIdentityClient}.
 *
 * @see ChimeSdkIdentityClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultChimeSdkIdentityClient implements ChimeSdkIdentityClient {
    private static final Logger log = Logger.loggerFor(DefaultChimeSdkIdentityClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultChimeSdkIdentityClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates an Amazon Chime SDK messaging <code>AppInstance</code> under an AWS account. Only SDK messaging customers
     * use this API. <code>CreateAppInstance</code> supports idempotency behavior as described in the AWS API Standard.
     * </p>
     * <p>
     * identity
     * </p>
     *
     * @param createAppInstanceRequest
     * @return Result of the CreateAppInstance operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ResourceLimitExceededException
     *         The request exceeds the resource limit.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.CreateAppInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/CreateAppInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateAppInstanceResponse createAppInstance(CreateAppInstanceRequest createAppInstanceRequest)
            throws BadRequestException, ConflictException, ForbiddenException, ResourceLimitExceededException,
            ThrottledClientException, UnauthorizedClientException, ServiceUnavailableException, ServiceFailureException,
            AwsServiceException, SdkClientException, ChimeSdkIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAppInstanceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAppInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAppInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAppInstance");

            return clientHandler.execute(new ClientExecutionParams<CreateAppInstanceRequest, CreateAppInstanceResponse>()
                    .withOperationName("CreateAppInstance").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createAppInstanceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateAppInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Promotes an <code>AppInstanceUser</code> to an <code>AppInstanceAdmin</code>. The promoted user can perform the
     * following actions.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ChannelModerator</code> actions across all channels in the <code>AppInstance</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DeleteChannelMessage</code> actions.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Only an <code>AppInstanceUser</code> can be promoted to an <code>AppInstanceAdmin</code> role.
     * </p>
     *
     * @param createAppInstanceAdminRequest
     * @return Result of the CreateAppInstanceAdmin operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ResourceLimitExceededException
     *         The request exceeds the resource limit.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.CreateAppInstanceAdmin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/CreateAppInstanceAdmin"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateAppInstanceAdminResponse createAppInstanceAdmin(CreateAppInstanceAdminRequest createAppInstanceAdminRequest)
            throws BadRequestException, ConflictException, ForbiddenException, ResourceLimitExceededException,
            ThrottledClientException, UnauthorizedClientException, ServiceUnavailableException, ServiceFailureException,
            AwsServiceException, SdkClientException, ChimeSdkIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAppInstanceAdminResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateAppInstanceAdminResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAppInstanceAdminRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAppInstanceAdmin");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateAppInstanceAdminRequest, CreateAppInstanceAdminResponse>()
                            .withOperationName("CreateAppInstanceAdmin").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createAppInstanceAdminRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateAppInstanceAdminRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a user under an Amazon Chime <code>AppInstance</code>. The request consists of a unique
     * <code>appInstanceUserId</code> and <code>Name</code> for that user.
     * </p>
     *
     * @param createAppInstanceUserRequest
     * @return Result of the CreateAppInstanceUser operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ResourceLimitExceededException
     *         The request exceeds the resource limit.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.CreateAppInstanceUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/CreateAppInstanceUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateAppInstanceUserResponse createAppInstanceUser(CreateAppInstanceUserRequest createAppInstanceUserRequest)
            throws BadRequestException, ConflictException, ForbiddenException, ResourceLimitExceededException,
            ThrottledClientException, UnauthorizedClientException, ServiceUnavailableException, ServiceFailureException,
            AwsServiceException, SdkClientException, ChimeSdkIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAppInstanceUserResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateAppInstanceUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAppInstanceUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAppInstanceUser");

            return clientHandler.execute(new ClientExecutionParams<CreateAppInstanceUserRequest, CreateAppInstanceUserResponse>()
                    .withOperationName("CreateAppInstanceUser").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createAppInstanceUserRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateAppInstanceUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an <code>AppInstance</code> and all associated data asynchronously.
     * </p>
     *
     * @param deleteAppInstanceRequest
     * @return Result of the DeleteAppInstance operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ResourceLimitExceededException
     *         The request exceeds the resource limit.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.DeleteAppInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DeleteAppInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAppInstanceResponse deleteAppInstance(DeleteAppInstanceRequest deleteAppInstanceRequest)
            throws BadRequestException, ForbiddenException, ResourceLimitExceededException, ThrottledClientException,
            UnauthorizedClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeSdkIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAppInstanceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAppInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAppInstance");

            return clientHandler.execute(new ClientExecutionParams<DeleteAppInstanceRequest, DeleteAppInstanceResponse>()
                    .withOperationName("DeleteAppInstance").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteAppInstanceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAppInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Demotes an <code>AppInstanceAdmin</code> to an <code>AppInstanceUser</code>. This action does not delete the
     * user.
     * </p>
     *
     * @param deleteAppInstanceAdminRequest
     * @return Result of the DeleteAppInstanceAdmin operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ResourceLimitExceededException
     *         The request exceeds the resource limit.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.DeleteAppInstanceAdmin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DeleteAppInstanceAdmin"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAppInstanceAdminResponse deleteAppInstanceAdmin(DeleteAppInstanceAdminRequest deleteAppInstanceAdminRequest)
            throws BadRequestException, ConflictException, ForbiddenException, ResourceLimitExceededException,
            ThrottledClientException, UnauthorizedClientException, ServiceUnavailableException, ServiceFailureException,
            AwsServiceException, SdkClientException, ChimeSdkIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAppInstanceAdminResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteAppInstanceAdminResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppInstanceAdminRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAppInstanceAdmin");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteAppInstanceAdminRequest, DeleteAppInstanceAdminResponse>()
                            .withOperationName("DeleteAppInstanceAdmin").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteAppInstanceAdminRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteAppInstanceAdminRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an <code>AppInstanceUser</code>.
     * </p>
     *
     * @param deleteAppInstanceUserRequest
     * @return Result of the DeleteAppInstanceUser operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ResourceLimitExceededException
     *         The request exceeds the resource limit.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.DeleteAppInstanceUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DeleteAppInstanceUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAppInstanceUserResponse deleteAppInstanceUser(DeleteAppInstanceUserRequest deleteAppInstanceUserRequest)
            throws BadRequestException, ConflictException, ForbiddenException, ResourceLimitExceededException,
            ThrottledClientException, UnauthorizedClientException, ServiceUnavailableException, ServiceFailureException,
            AwsServiceException, SdkClientException, ChimeSdkIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAppInstanceUserResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteAppInstanceUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppInstanceUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAppInstanceUser");

            return clientHandler.execute(new ClientExecutionParams<DeleteAppInstanceUserRequest, DeleteAppInstanceUserResponse>()
                    .withOperationName("DeleteAppInstanceUser").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteAppInstanceUserRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAppInstanceUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the full details of an <code>AppInstance</code>.
     * </p>
     *
     * @param describeAppInstanceRequest
     * @return Result of the DescribeAppInstance operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.DescribeAppInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DescribeAppInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAppInstanceResponse describeAppInstance(DescribeAppInstanceRequest describeAppInstanceRequest)
            throws BadRequestException, ForbiddenException, ThrottledClientException, UnauthorizedClientException,
            ServiceUnavailableException, ServiceFailureException, AwsServiceException, SdkClientException,
            ChimeSdkIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAppInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAppInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAppInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAppInstance");

            return clientHandler.execute(new ClientExecutionParams<DescribeAppInstanceRequest, DescribeAppInstanceResponse>()
                    .withOperationName("DescribeAppInstance").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeAppInstanceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAppInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the full details of an <code>AppInstanceAdmin</code>.
     * </p>
     *
     * @param describeAppInstanceAdminRequest
     * @return Result of the DescribeAppInstanceAdmin operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.DescribeAppInstanceAdmin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DescribeAppInstanceAdmin"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAppInstanceAdminResponse describeAppInstanceAdmin(
            DescribeAppInstanceAdminRequest describeAppInstanceAdminRequest) throws BadRequestException, ForbiddenException,
            ThrottledClientException, UnauthorizedClientException, ServiceUnavailableException, ServiceFailureException,
            AwsServiceException, SdkClientException, ChimeSdkIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAppInstanceAdminResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAppInstanceAdminResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAppInstanceAdminRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAppInstanceAdmin");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAppInstanceAdminRequest, DescribeAppInstanceAdminResponse>()
                            .withOperationName("DescribeAppInstanceAdmin").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeAppInstanceAdminRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAppInstanceAdminRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the full details of an <code>AppInstanceUser</code>.
     * </p>
     *
     * @param describeAppInstanceUserRequest
     * @return Result of the DescribeAppInstanceUser operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.DescribeAppInstanceUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DescribeAppInstanceUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAppInstanceUserResponse describeAppInstanceUser(DescribeAppInstanceUserRequest describeAppInstanceUserRequest)
            throws BadRequestException, ForbiddenException, ThrottledClientException, UnauthorizedClientException,
            ServiceUnavailableException, ServiceFailureException, AwsServiceException, SdkClientException,
            ChimeSdkIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeAppInstanceUserResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeAppInstanceUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAppInstanceUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAppInstanceUser");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAppInstanceUserRequest, DescribeAppInstanceUserResponse>()
                            .withOperationName("DescribeAppInstanceUser").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeAppInstanceUserRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAppInstanceUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the retention settings for an <code>AppInstance</code>.
     * </p>
     *
     * @param getAppInstanceRetentionSettingsRequest
     * @return Result of the GetAppInstanceRetentionSettings operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.GetAppInstanceRetentionSettings
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/GetAppInstanceRetentionSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAppInstanceRetentionSettingsResponse getAppInstanceRetentionSettings(
            GetAppInstanceRetentionSettingsRequest getAppInstanceRetentionSettingsRequest) throws BadRequestException,
            ForbiddenException, ThrottledClientException, UnauthorizedClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeSdkIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAppInstanceRetentionSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAppInstanceRetentionSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getAppInstanceRetentionSettingsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAppInstanceRetentionSettings");

            return clientHandler
                    .execute(new ClientExecutionParams<GetAppInstanceRetentionSettingsRequest, GetAppInstanceRetentionSettingsResponse>()
                            .withOperationName("GetAppInstanceRetentionSettings").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getAppInstanceRetentionSettingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetAppInstanceRetentionSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the administrators in the <code>AppInstance</code>.
     * </p>
     *
     * @param listAppInstanceAdminsRequest
     * @return Result of the ListAppInstanceAdmins operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ResourceLimitExceededException
     *         The request exceeds the resource limit.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.ListAppInstanceAdmins
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstanceAdmins"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAppInstanceAdminsResponse listAppInstanceAdmins(ListAppInstanceAdminsRequest listAppInstanceAdminsRequest)
            throws BadRequestException, ForbiddenException, ResourceLimitExceededException, ThrottledClientException,
            UnauthorizedClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeSdkIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAppInstanceAdminsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAppInstanceAdminsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppInstanceAdminsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppInstanceAdmins");

            return clientHandler.execute(new ClientExecutionParams<ListAppInstanceAdminsRequest, ListAppInstanceAdminsResponse>()
                    .withOperationName("ListAppInstanceAdmins").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listAppInstanceAdminsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAppInstanceAdminsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the administrators in the <code>AppInstance</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAppInstanceAdmins(software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceAdminsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstanceAdminsIterable responses = client.listAppInstanceAdminsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstanceAdminsIterable responses = client
     *             .listAppInstanceAdminsPaginator(request);
     *     for (software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceAdminsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstanceAdminsIterable responses = client.listAppInstanceAdminsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAppInstanceAdmins(software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceAdminsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAppInstanceAdminsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ResourceLimitExceededException
     *         The request exceeds the resource limit.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.ListAppInstanceAdmins
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstanceAdmins"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAppInstanceAdminsIterable listAppInstanceAdminsPaginator(ListAppInstanceAdminsRequest listAppInstanceAdminsRequest)
            throws BadRequestException, ForbiddenException, ResourceLimitExceededException, ThrottledClientException,
            UnauthorizedClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeSdkIdentityException {
        return new ListAppInstanceAdminsIterable(this, applyPaginatorUserAgent(listAppInstanceAdminsRequest));
    }

    /**
     * <p>
     * List all <code>AppInstanceUsers</code> created under a single <code>AppInstance</code>.
     * </p>
     *
     * @param listAppInstanceUsersRequest
     * @return Result of the ListAppInstanceUsers operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.ListAppInstanceUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstanceUsers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAppInstanceUsersResponse listAppInstanceUsers(ListAppInstanceUsersRequest listAppInstanceUsersRequest)
            throws BadRequestException, ForbiddenException, ThrottledClientException, UnauthorizedClientException,
            ServiceUnavailableException, ServiceFailureException, AwsServiceException, SdkClientException,
            ChimeSdkIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAppInstanceUsersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListAppInstanceUsersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppInstanceUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppInstanceUsers");

            return clientHandler.execute(new ClientExecutionParams<ListAppInstanceUsersRequest, ListAppInstanceUsersResponse>()
                    .withOperationName("ListAppInstanceUsers").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listAppInstanceUsersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAppInstanceUsersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List all <code>AppInstanceUsers</code> created under a single <code>AppInstance</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAppInstanceUsers(software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUsersRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstanceUsersIterable responses = client.listAppInstanceUsersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstanceUsersIterable responses = client
     *             .listAppInstanceUsersPaginator(request);
     *     for (software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUsersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstanceUsersIterable responses = client.listAppInstanceUsersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAppInstanceUsers(software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUsersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAppInstanceUsersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.ListAppInstanceUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstanceUsers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAppInstanceUsersIterable listAppInstanceUsersPaginator(ListAppInstanceUsersRequest listAppInstanceUsersRequest)
            throws BadRequestException, ForbiddenException, ThrottledClientException, UnauthorizedClientException,
            ServiceUnavailableException, ServiceFailureException, AwsServiceException, SdkClientException,
            ChimeSdkIdentityException {
        return new ListAppInstanceUsersIterable(this, applyPaginatorUserAgent(listAppInstanceUsersRequest));
    }

    /**
     * <p>
     * Lists all Amazon Chime <code>AppInstance</code>s created under a single AWS account.
     * </p>
     *
     * @param listAppInstancesRequest
     * @return Result of the ListAppInstances operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.ListAppInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAppInstancesResponse listAppInstances(ListAppInstancesRequest listAppInstancesRequest) throws BadRequestException,
            ForbiddenException, ThrottledClientException, UnauthorizedClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeSdkIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAppInstancesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAppInstancesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppInstances");

            return clientHandler.execute(new ClientExecutionParams<ListAppInstancesRequest, ListAppInstancesResponse>()
                    .withOperationName("ListAppInstances").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listAppInstancesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAppInstancesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all Amazon Chime <code>AppInstance</code>s created under a single AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAppInstances(software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstancesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstancesIterable responses = client.listAppInstancesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstancesIterable responses = client
     *             .listAppInstancesPaginator(request);
     *     for (software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstancesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstancesIterable responses = client.listAppInstancesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAppInstances(software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstancesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAppInstancesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.ListAppInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAppInstancesIterable listAppInstancesPaginator(ListAppInstancesRequest listAppInstancesRequest)
            throws BadRequestException, ForbiddenException, ThrottledClientException, UnauthorizedClientException,
            ServiceUnavailableException, ServiceFailureException, AwsServiceException, SdkClientException,
            ChimeSdkIdentityException {
        return new ListAppInstancesIterable(this, applyPaginatorUserAgent(listAppInstancesRequest));
    }

    /**
     * <p>
     * Sets the amount of time in days that a given <code>AppInstance</code> retains data.
     * </p>
     *
     * @param putAppInstanceRetentionSettingsRequest
     * @return Result of the PutAppInstanceRetentionSettings operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.PutAppInstanceRetentionSettings
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/PutAppInstanceRetentionSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutAppInstanceRetentionSettingsResponse putAppInstanceRetentionSettings(
            PutAppInstanceRetentionSettingsRequest putAppInstanceRetentionSettingsRequest) throws BadRequestException,
            ForbiddenException, ThrottledClientException, UnauthorizedClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeSdkIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutAppInstanceRetentionSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutAppInstanceRetentionSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putAppInstanceRetentionSettingsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAppInstanceRetentionSettings");

            return clientHandler
                    .execute(new ClientExecutionParams<PutAppInstanceRetentionSettingsRequest, PutAppInstanceRetentionSettingsResponse>()
                            .withOperationName("PutAppInstanceRetentionSettings").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(putAppInstanceRetentionSettingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutAppInstanceRetentionSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates <code>AppInstance</code> metadata.
     * </p>
     *
     * @param updateAppInstanceRequest
     * @return Result of the UpdateAppInstance operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.UpdateAppInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/UpdateAppInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateAppInstanceResponse updateAppInstance(UpdateAppInstanceRequest updateAppInstanceRequest)
            throws BadRequestException, ConflictException, ForbiddenException, ThrottledClientException,
            UnauthorizedClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeSdkIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAppInstanceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateAppInstanceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAppInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAppInstance");

            return clientHandler.execute(new ClientExecutionParams<UpdateAppInstanceRequest, UpdateAppInstanceResponse>()
                    .withOperationName("UpdateAppInstance").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateAppInstanceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateAppInstanceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the details of an <code>AppInstanceUser</code>. You can update names and metadata.
     * </p>
     *
     * @param updateAppInstanceUserRequest
     * @return Result of the UpdateAppInstanceUser operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ResourceLimitExceededException
     *         The request exceeds the resource limit.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkIdentityException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkIdentityClient.UpdateAppInstanceUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/UpdateAppInstanceUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateAppInstanceUserResponse updateAppInstanceUser(UpdateAppInstanceUserRequest updateAppInstanceUserRequest)
            throws BadRequestException, ConflictException, ForbiddenException, ResourceLimitExceededException,
            ThrottledClientException, UnauthorizedClientException, ServiceUnavailableException, ServiceFailureException,
            AwsServiceException, SdkClientException, ChimeSdkIdentityException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAppInstanceUserResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateAppInstanceUserResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAppInstanceUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAppInstanceUser");

            return clientHandler.execute(new ClientExecutionParams<UpdateAppInstanceUserRequest, UpdateAppInstanceUserResponse>()
                    .withOperationName("UpdateAppInstanceUser").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateAppInstanceUserRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateAppInstanceUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ChimeSdkIdentityException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedClientException")
                                .exceptionBuilderSupplier(UnauthorizedClientException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottledClientException")
                                .exceptionBuilderSupplier(ThrottledClientException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceFailureException")
                                .exceptionBuilderSupplier(ServiceFailureException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends ChimeSdkIdentityRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
