/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chimesdkidentity;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.chimesdkidentity.model.BadRequestException;
import software.amazon.awssdk.services.chimesdkidentity.model.ChimeSdkIdentityException;
import software.amazon.awssdk.services.chimesdkidentity.model.ChimeSdkIdentityRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.ConflictException;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceAdminRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceAdminResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceUserRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.CreateAppInstanceUserResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceAdminRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceAdminResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceUserRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DeleteAppInstanceUserResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DeregisterAppInstanceUserEndpointRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DeregisterAppInstanceUserEndpointResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceAdminRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceAdminResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceUserEndpointRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceUserEndpointResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceUserRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.DescribeAppInstanceUserResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ForbiddenException;
import software.amazon.awssdk.services.chimesdkidentity.model.GetAppInstanceRetentionSettingsRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.GetAppInstanceRetentionSettingsResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceAdminsRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceAdminsResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUserEndpointsRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUserEndpointsResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUsersRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUsersResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstancesRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstancesResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.PutAppInstanceRetentionSettingsRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.PutAppInstanceRetentionSettingsResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.RegisterAppInstanceUserEndpointRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.RegisterAppInstanceUserEndpointResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.chimesdkidentity.model.ServiceFailureException;
import software.amazon.awssdk.services.chimesdkidentity.model.ServiceUnavailableException;
import software.amazon.awssdk.services.chimesdkidentity.model.TagResourceRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.TagResourceResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.ThrottledClientException;
import software.amazon.awssdk.services.chimesdkidentity.model.UnauthorizedClientException;
import software.amazon.awssdk.services.chimesdkidentity.model.UntagResourceRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.UntagResourceResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.UpdateAppInstanceRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.UpdateAppInstanceResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.UpdateAppInstanceUserEndpointRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.UpdateAppInstanceUserEndpointResponse;
import software.amazon.awssdk.services.chimesdkidentity.model.UpdateAppInstanceUserRequest;
import software.amazon.awssdk.services.chimesdkidentity.model.UpdateAppInstanceUserResponse;
import software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstanceAdminsPublisher;
import software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstanceUserEndpointsPublisher;
import software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstanceUsersPublisher;
import software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstancesPublisher;
import software.amazon.awssdk.services.chimesdkidentity.transform.CreateAppInstanceAdminRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.CreateAppInstanceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.CreateAppInstanceUserRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DeleteAppInstanceAdminRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DeleteAppInstanceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DeleteAppInstanceUserRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DeregisterAppInstanceUserEndpointRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DescribeAppInstanceAdminRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DescribeAppInstanceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DescribeAppInstanceUserEndpointRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.DescribeAppInstanceUserRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.GetAppInstanceRetentionSettingsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.ListAppInstanceAdminsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.ListAppInstanceUserEndpointsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.ListAppInstanceUsersRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.ListAppInstancesRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.PutAppInstanceRetentionSettingsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.RegisterAppInstanceUserEndpointRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.UpdateAppInstanceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.UpdateAppInstanceUserEndpointRequestMarshaller;
import software.amazon.awssdk.services.chimesdkidentity.transform.UpdateAppInstanceUserRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ChimeSdkIdentityAsyncClient}.
 *
 * @see ChimeSdkIdentityAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultChimeSdkIdentityAsyncClient implements ChimeSdkIdentityAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultChimeSdkIdentityAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultChimeSdkIdentityAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an Amazon Chime SDK messaging <code>AppInstance</code> under an AWS account. Only SDK messaging customers
     * use this API. <code>CreateAppInstance</code> supports idempotency behavior as described in the AWS API Standard.
     * </p>
     * <p>
     * identity
     * </p>
     *
     * @param createAppInstanceRequest
     * @return A Java Future containing the result of the CreateAppInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.CreateAppInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/CreateAppInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAppInstanceResponse> createAppInstance(CreateAppInstanceRequest createAppInstanceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAppInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAppInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAppInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAppInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAppInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAppInstanceRequest, CreateAppInstanceResponse>()
                            .withOperationName("CreateAppInstance")
                            .withMarshaller(new CreateAppInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAppInstanceRequest));
            CompletableFuture<CreateAppInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Promotes an <code>AppInstanceUser</code> to an <code>AppInstanceAdmin</code>. The promoted user can perform the
     * following actions.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ChannelModerator</code> actions across all channels in the <code>AppInstance</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DeleteChannelMessage</code> actions.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Only an <code>AppInstanceUser</code> can be promoted to an <code>AppInstanceAdmin</code> role.
     * </p>
     *
     * @param createAppInstanceAdminRequest
     * @return A Java Future containing the result of the CreateAppInstanceAdmin operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.CreateAppInstanceAdmin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/CreateAppInstanceAdmin"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAppInstanceAdminResponse> createAppInstanceAdmin(
            CreateAppInstanceAdminRequest createAppInstanceAdminRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAppInstanceAdminRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAppInstanceAdmin");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAppInstanceAdminResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAppInstanceAdminResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAppInstanceAdminResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAppInstanceAdminRequest, CreateAppInstanceAdminResponse>()
                            .withOperationName("CreateAppInstanceAdmin")
                            .withMarshaller(new CreateAppInstanceAdminRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAppInstanceAdminRequest));
            CompletableFuture<CreateAppInstanceAdminResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a user under an Amazon Chime <code>AppInstance</code>. The request consists of a unique
     * <code>appInstanceUserId</code> and <code>Name</code> for that user.
     * </p>
     *
     * @param createAppInstanceUserRequest
     * @return A Java Future containing the result of the CreateAppInstanceUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.CreateAppInstanceUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/CreateAppInstanceUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateAppInstanceUserResponse> createAppInstanceUser(
            CreateAppInstanceUserRequest createAppInstanceUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAppInstanceUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAppInstanceUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateAppInstanceUserResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateAppInstanceUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateAppInstanceUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateAppInstanceUserRequest, CreateAppInstanceUserResponse>()
                            .withOperationName("CreateAppInstanceUser")
                            .withMarshaller(new CreateAppInstanceUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createAppInstanceUserRequest));
            CompletableFuture<CreateAppInstanceUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an <code>AppInstance</code> and all associated data asynchronously.
     * </p>
     *
     * @param deleteAppInstanceRequest
     * @return A Java Future containing the result of the DeleteAppInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.DeleteAppInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DeleteAppInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAppInstanceResponse> deleteAppInstance(DeleteAppInstanceRequest deleteAppInstanceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAppInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAppInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAppInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAppInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAppInstanceRequest, DeleteAppInstanceResponse>()
                            .withOperationName("DeleteAppInstance")
                            .withMarshaller(new DeleteAppInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAppInstanceRequest));
            CompletableFuture<DeleteAppInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Demotes an <code>AppInstanceAdmin</code> to an <code>AppInstanceUser</code>. This action does not delete the
     * user.
     * </p>
     *
     * @param deleteAppInstanceAdminRequest
     * @return A Java Future containing the result of the DeleteAppInstanceAdmin operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.DeleteAppInstanceAdmin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DeleteAppInstanceAdmin"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAppInstanceAdminResponse> deleteAppInstanceAdmin(
            DeleteAppInstanceAdminRequest deleteAppInstanceAdminRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppInstanceAdminRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAppInstanceAdmin");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAppInstanceAdminResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAppInstanceAdminResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAppInstanceAdminResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAppInstanceAdminRequest, DeleteAppInstanceAdminResponse>()
                            .withOperationName("DeleteAppInstanceAdmin")
                            .withMarshaller(new DeleteAppInstanceAdminRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAppInstanceAdminRequest));
            CompletableFuture<DeleteAppInstanceAdminResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an <code>AppInstanceUser</code>.
     * </p>
     *
     * @param deleteAppInstanceUserRequest
     * @return A Java Future containing the result of the DeleteAppInstanceUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.DeleteAppInstanceUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DeleteAppInstanceUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteAppInstanceUserResponse> deleteAppInstanceUser(
            DeleteAppInstanceUserRequest deleteAppInstanceUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAppInstanceUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAppInstanceUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteAppInstanceUserResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteAppInstanceUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteAppInstanceUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteAppInstanceUserRequest, DeleteAppInstanceUserResponse>()
                            .withOperationName("DeleteAppInstanceUser")
                            .withMarshaller(new DeleteAppInstanceUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteAppInstanceUserRequest));
            CompletableFuture<DeleteAppInstanceUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deregisters an <code>AppInstanceUserEndpoint</code>.
     * </p>
     *
     * @param deregisterAppInstanceUserEndpointRequest
     * @return A Java Future containing the result of the DeregisterAppInstanceUserEndpoint operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.DeregisterAppInstanceUserEndpoint
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DeregisterAppInstanceUserEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeregisterAppInstanceUserEndpointResponse> deregisterAppInstanceUserEndpoint(
            DeregisterAppInstanceUserEndpointRequest deregisterAppInstanceUserEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deregisterAppInstanceUserEndpointRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeregisterAppInstanceUserEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeregisterAppInstanceUserEndpointResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeregisterAppInstanceUserEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeregisterAppInstanceUserEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeregisterAppInstanceUserEndpointRequest, DeregisterAppInstanceUserEndpointResponse>()
                            .withOperationName("DeregisterAppInstanceUserEndpoint")
                            .withMarshaller(new DeregisterAppInstanceUserEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deregisterAppInstanceUserEndpointRequest));
            CompletableFuture<DeregisterAppInstanceUserEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the full details of an <code>AppInstance</code>.
     * </p>
     *
     * @param describeAppInstanceRequest
     * @return A Java Future containing the result of the DescribeAppInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.DescribeAppInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DescribeAppInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAppInstanceResponse> describeAppInstance(
            DescribeAppInstanceRequest describeAppInstanceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAppInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAppInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAppInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAppInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAppInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAppInstanceRequest, DescribeAppInstanceResponse>()
                            .withOperationName("DescribeAppInstance")
                            .withMarshaller(new DescribeAppInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAppInstanceRequest));
            CompletableFuture<DescribeAppInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the full details of an <code>AppInstanceAdmin</code>.
     * </p>
     *
     * @param describeAppInstanceAdminRequest
     * @return A Java Future containing the result of the DescribeAppInstanceAdmin operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.DescribeAppInstanceAdmin
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DescribeAppInstanceAdmin"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAppInstanceAdminResponse> describeAppInstanceAdmin(
            DescribeAppInstanceAdminRequest describeAppInstanceAdminRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAppInstanceAdminRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAppInstanceAdmin");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAppInstanceAdminResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAppInstanceAdminResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAppInstanceAdminResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAppInstanceAdminRequest, DescribeAppInstanceAdminResponse>()
                            .withOperationName("DescribeAppInstanceAdmin")
                            .withMarshaller(new DescribeAppInstanceAdminRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAppInstanceAdminRequest));
            CompletableFuture<DescribeAppInstanceAdminResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the full details of an <code>AppInstanceUser</code>.
     * </p>
     *
     * @param describeAppInstanceUserRequest
     * @return A Java Future containing the result of the DescribeAppInstanceUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.DescribeAppInstanceUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DescribeAppInstanceUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAppInstanceUserResponse> describeAppInstanceUser(
            DescribeAppInstanceUserRequest describeAppInstanceUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAppInstanceUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAppInstanceUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAppInstanceUserResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAppInstanceUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAppInstanceUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAppInstanceUserRequest, DescribeAppInstanceUserResponse>()
                            .withOperationName("DescribeAppInstanceUser")
                            .withMarshaller(new DescribeAppInstanceUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAppInstanceUserRequest));
            CompletableFuture<DescribeAppInstanceUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the full details of an <code>AppInstanceUserEndpoint</code>.
     * </p>
     *
     * @param describeAppInstanceUserEndpointRequest
     * @return A Java Future containing the result of the DescribeAppInstanceUserEndpoint operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.DescribeAppInstanceUserEndpoint
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/DescribeAppInstanceUserEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAppInstanceUserEndpointResponse> describeAppInstanceUserEndpoint(
            DescribeAppInstanceUserEndpointRequest describeAppInstanceUserEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeAppInstanceUserEndpointRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAppInstanceUserEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAppInstanceUserEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAppInstanceUserEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAppInstanceUserEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAppInstanceUserEndpointRequest, DescribeAppInstanceUserEndpointResponse>()
                            .withOperationName("DescribeAppInstanceUserEndpoint")
                            .withMarshaller(new DescribeAppInstanceUserEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAppInstanceUserEndpointRequest));
            CompletableFuture<DescribeAppInstanceUserEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the retention settings for an <code>AppInstance</code>.
     * </p>
     *
     * @param getAppInstanceRetentionSettingsRequest
     * @return A Java Future containing the result of the GetAppInstanceRetentionSettings operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.GetAppInstanceRetentionSettings
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/GetAppInstanceRetentionSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAppInstanceRetentionSettingsResponse> getAppInstanceRetentionSettings(
            GetAppInstanceRetentionSettingsRequest getAppInstanceRetentionSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getAppInstanceRetentionSettingsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAppInstanceRetentionSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAppInstanceRetentionSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAppInstanceRetentionSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAppInstanceRetentionSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAppInstanceRetentionSettingsRequest, GetAppInstanceRetentionSettingsResponse>()
                            .withOperationName("GetAppInstanceRetentionSettings")
                            .withMarshaller(new GetAppInstanceRetentionSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAppInstanceRetentionSettingsRequest));
            CompletableFuture<GetAppInstanceRetentionSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the administrators in the <code>AppInstance</code>.
     * </p>
     *
     * @param listAppInstanceAdminsRequest
     * @return A Java Future containing the result of the ListAppInstanceAdmins operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.ListAppInstanceAdmins
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstanceAdmins"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAppInstanceAdminsResponse> listAppInstanceAdmins(
            ListAppInstanceAdminsRequest listAppInstanceAdminsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppInstanceAdminsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppInstanceAdmins");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAppInstanceAdminsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAppInstanceAdminsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAppInstanceAdminsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAppInstanceAdminsRequest, ListAppInstanceAdminsResponse>()
                            .withOperationName("ListAppInstanceAdmins")
                            .withMarshaller(new ListAppInstanceAdminsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAppInstanceAdminsRequest));
            CompletableFuture<ListAppInstanceAdminsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the administrators in the <code>AppInstance</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAppInstanceAdmins(software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceAdminsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstanceAdminsPublisher publisher = client.listAppInstanceAdminsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstanceAdminsPublisher publisher = client.listAppInstanceAdminsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceAdminsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceAdminsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAppInstanceAdmins(software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceAdminsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAppInstanceAdminsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.ListAppInstanceAdmins
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstanceAdmins"
     *      target="_top">AWS API Documentation</a>
     */
    public ListAppInstanceAdminsPublisher listAppInstanceAdminsPaginator(ListAppInstanceAdminsRequest listAppInstanceAdminsRequest) {
        return new ListAppInstanceAdminsPublisher(this, applyPaginatorUserAgent(listAppInstanceAdminsRequest));
    }

    /**
     * <p>
     * Lists all the <code>AppInstanceUserEndpoints</code> created under a single <code>AppInstanceUser</code>.
     * </p>
     *
     * @param listAppInstanceUserEndpointsRequest
     * @return A Java Future containing the result of the ListAppInstanceUserEndpoints operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.ListAppInstanceUserEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstanceUserEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAppInstanceUserEndpointsResponse> listAppInstanceUserEndpoints(
            ListAppInstanceUserEndpointsRequest listAppInstanceUserEndpointsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppInstanceUserEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppInstanceUserEndpoints");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAppInstanceUserEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAppInstanceUserEndpointsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAppInstanceUserEndpointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAppInstanceUserEndpointsRequest, ListAppInstanceUserEndpointsResponse>()
                            .withOperationName("ListAppInstanceUserEndpoints")
                            .withMarshaller(new ListAppInstanceUserEndpointsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAppInstanceUserEndpointsRequest));
            CompletableFuture<ListAppInstanceUserEndpointsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the <code>AppInstanceUserEndpoints</code> created under a single <code>AppInstanceUser</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAppInstanceUserEndpoints(software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUserEndpointsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstanceUserEndpointsPublisher publisher = client.listAppInstanceUserEndpointsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstanceUserEndpointsPublisher publisher = client.listAppInstanceUserEndpointsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUserEndpointsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUserEndpointsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAppInstanceUserEndpoints(software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUserEndpointsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAppInstanceUserEndpointsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.ListAppInstanceUserEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstanceUserEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    public ListAppInstanceUserEndpointsPublisher listAppInstanceUserEndpointsPaginator(
            ListAppInstanceUserEndpointsRequest listAppInstanceUserEndpointsRequest) {
        return new ListAppInstanceUserEndpointsPublisher(this, applyPaginatorUserAgent(listAppInstanceUserEndpointsRequest));
    }

    /**
     * <p>
     * List all <code>AppInstanceUsers</code> created under a single <code>AppInstance</code>.
     * </p>
     *
     * @param listAppInstanceUsersRequest
     * @return A Java Future containing the result of the ListAppInstanceUsers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.ListAppInstanceUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstanceUsers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAppInstanceUsersResponse> listAppInstanceUsers(
            ListAppInstanceUsersRequest listAppInstanceUsersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppInstanceUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppInstanceUsers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAppInstanceUsersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAppInstanceUsersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAppInstanceUsersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAppInstanceUsersRequest, ListAppInstanceUsersResponse>()
                            .withOperationName("ListAppInstanceUsers")
                            .withMarshaller(new ListAppInstanceUsersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAppInstanceUsersRequest));
            CompletableFuture<ListAppInstanceUsersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all <code>AppInstanceUsers</code> created under a single <code>AppInstance</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAppInstanceUsers(software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUsersRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstanceUsersPublisher publisher = client.listAppInstanceUsersPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstanceUsersPublisher publisher = client.listAppInstanceUsersPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUsersResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUsersResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAppInstanceUsers(software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstanceUsersRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAppInstanceUsersRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.ListAppInstanceUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstanceUsers"
     *      target="_top">AWS API Documentation</a>
     */
    public ListAppInstanceUsersPublisher listAppInstanceUsersPaginator(ListAppInstanceUsersRequest listAppInstanceUsersRequest) {
        return new ListAppInstanceUsersPublisher(this, applyPaginatorUserAgent(listAppInstanceUsersRequest));
    }

    /**
     * <p>
     * Lists all Amazon Chime <code>AppInstance</code>s created under a single AWS account.
     * </p>
     *
     * @param listAppInstancesRequest
     * @return A Java Future containing the result of the ListAppInstances operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.ListAppInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstances"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAppInstancesResponse> listAppInstances(ListAppInstancesRequest listAppInstancesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAppInstancesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAppInstances");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAppInstancesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAppInstancesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAppInstancesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAppInstancesRequest, ListAppInstancesResponse>()
                            .withOperationName("ListAppInstances")
                            .withMarshaller(new ListAppInstancesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAppInstancesRequest));
            CompletableFuture<ListAppInstancesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all Amazon Chime <code>AppInstance</code>s created under a single AWS account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAppInstances(software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstancesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstancesPublisher publisher = client.listAppInstancesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkidentity.paginators.ListAppInstancesPublisher publisher = client.listAppInstancesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstancesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstancesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAppInstances(software.amazon.awssdk.services.chimesdkidentity.model.ListAppInstancesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAppInstancesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.ListAppInstances
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListAppInstances"
     *      target="_top">AWS API Documentation</a>
     */
    public ListAppInstancesPublisher listAppInstancesPaginator(ListAppInstancesRequest listAppInstancesRequest) {
        return new ListAppInstancesPublisher(this, applyPaginatorUserAgent(listAppInstancesRequest));
    }

    /**
     * <p>
     * Lists the tags applied to an Amazon Chime SDK identity resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the amount of time in days that a given <code>AppInstance</code> retains data.
     * </p>
     *
     * @param putAppInstanceRetentionSettingsRequest
     * @return A Java Future containing the result of the PutAppInstanceRetentionSettings operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.PutAppInstanceRetentionSettings
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/PutAppInstanceRetentionSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutAppInstanceRetentionSettingsResponse> putAppInstanceRetentionSettings(
            PutAppInstanceRetentionSettingsRequest putAppInstanceRetentionSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putAppInstanceRetentionSettingsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAppInstanceRetentionSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutAppInstanceRetentionSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutAppInstanceRetentionSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutAppInstanceRetentionSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutAppInstanceRetentionSettingsRequest, PutAppInstanceRetentionSettingsResponse>()
                            .withOperationName("PutAppInstanceRetentionSettings")
                            .withMarshaller(new PutAppInstanceRetentionSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putAppInstanceRetentionSettingsRequest));
            CompletableFuture<PutAppInstanceRetentionSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Registers an endpoint under an Amazon Chime <code>AppInstanceUser</code>. The endpoint receives messages for a
     * user. For push notifications, the endpoint is a mobile device used to receive mobile push notifications for a
     * user.
     * </p>
     *
     * @param registerAppInstanceUserEndpointRequest
     * @return A Java Future containing the result of the RegisterAppInstanceUserEndpoint operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.RegisterAppInstanceUserEndpoint
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/RegisterAppInstanceUserEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RegisterAppInstanceUserEndpointResponse> registerAppInstanceUserEndpoint(
            RegisterAppInstanceUserEndpointRequest registerAppInstanceUserEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                registerAppInstanceUserEndpointRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegisterAppInstanceUserEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RegisterAppInstanceUserEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RegisterAppInstanceUserEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RegisterAppInstanceUserEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RegisterAppInstanceUserEndpointRequest, RegisterAppInstanceUserEndpointResponse>()
                            .withOperationName("RegisterAppInstanceUserEndpoint")
                            .withMarshaller(new RegisterAppInstanceUserEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(registerAppInstanceUserEndpointRequest));
            CompletableFuture<RegisterAppInstanceUserEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Applies the specified tags to the specified Amazon Chime SDK identity resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified Amazon Chime SDK identity resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates <code>AppInstance</code> metadata.
     * </p>
     *
     * @param updateAppInstanceRequest
     * @return A Java Future containing the result of the UpdateAppInstance operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.UpdateAppInstance
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/UpdateAppInstance"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAppInstanceResponse> updateAppInstance(UpdateAppInstanceRequest updateAppInstanceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAppInstanceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAppInstance");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAppInstanceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAppInstanceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAppInstanceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAppInstanceRequest, UpdateAppInstanceResponse>()
                            .withOperationName("UpdateAppInstance")
                            .withMarshaller(new UpdateAppInstanceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateAppInstanceRequest));
            CompletableFuture<UpdateAppInstanceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the details of an <code>AppInstanceUser</code>. You can update names and metadata.
     * </p>
     *
     * @param updateAppInstanceUserRequest
     * @return A Java Future containing the result of the UpdateAppInstanceUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.UpdateAppInstanceUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/UpdateAppInstanceUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAppInstanceUserResponse> updateAppInstanceUser(
            UpdateAppInstanceUserRequest updateAppInstanceUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAppInstanceUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAppInstanceUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAppInstanceUserResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAppInstanceUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAppInstanceUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAppInstanceUserRequest, UpdateAppInstanceUserResponse>()
                            .withOperationName("UpdateAppInstanceUser")
                            .withMarshaller(new UpdateAppInstanceUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateAppInstanceUserRequest));
            CompletableFuture<UpdateAppInstanceUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the details of an <code>AppInstanceUserEndpoint</code>. You can update the name and
     * <code>AllowMessage</code> values.
     * </p>
     *
     * @param updateAppInstanceUserEndpointRequest
     * @return A Java Future containing the result of the UpdateAppInstanceUserEndpoint operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkIdentityException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkIdentityAsyncClient.UpdateAppInstanceUserEndpoint
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-identity-2021-04-20/UpdateAppInstanceUserEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAppInstanceUserEndpointResponse> updateAppInstanceUserEndpoint(
            UpdateAppInstanceUserEndpointRequest updateAppInstanceUserEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateAppInstanceUserEndpointRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Identity");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAppInstanceUserEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAppInstanceUserEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAppInstanceUserEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateAppInstanceUserEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAppInstanceUserEndpointRequest, UpdateAppInstanceUserEndpointResponse>()
                            .withOperationName("UpdateAppInstanceUserEndpoint")
                            .withMarshaller(new UpdateAppInstanceUserEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateAppInstanceUserEndpointRequest));
            CompletableFuture<UpdateAppInstanceUserEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ChimeSdkIdentityException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedClientException")
                                .exceptionBuilderSupplier(UnauthorizedClientException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottledClientException")
                                .exceptionBuilderSupplier(ThrottledClientException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceFailureException")
                                .exceptionBuilderSupplier(ServiceFailureException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends ChimeSdkIdentityRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
