/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chimesdkmediapipelines;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.BadRequestException;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.ChimeSdkMediaPipelinesException;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.ChimeSdkMediaPipelinesRequest;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.CreateMediaCapturePipelineRequest;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.CreateMediaCapturePipelineResponse;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.CreateMediaConcatenationPipelineRequest;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.CreateMediaConcatenationPipelineResponse;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.CreateMediaLiveConnectorPipelineRequest;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.CreateMediaLiveConnectorPipelineResponse;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.DeleteMediaCapturePipelineRequest;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.DeleteMediaCapturePipelineResponse;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.DeleteMediaPipelineRequest;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.DeleteMediaPipelineResponse;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.ForbiddenException;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.GetMediaCapturePipelineRequest;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.GetMediaCapturePipelineResponse;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.GetMediaPipelineRequest;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.GetMediaPipelineResponse;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.ListMediaCapturePipelinesRequest;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.ListMediaCapturePipelinesResponse;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.ListMediaPipelinesRequest;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.ListMediaPipelinesResponse;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.NotFoundException;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.ServiceFailureException;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.ServiceUnavailableException;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.TagResourceRequest;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.TagResourceResponse;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.ThrottledClientException;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.UnauthorizedClientException;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.UntagResourceRequest;
import software.amazon.awssdk.services.chimesdkmediapipelines.model.UntagResourceResponse;
import software.amazon.awssdk.services.chimesdkmediapipelines.paginators.ListMediaCapturePipelinesPublisher;
import software.amazon.awssdk.services.chimesdkmediapipelines.paginators.ListMediaPipelinesPublisher;
import software.amazon.awssdk.services.chimesdkmediapipelines.transform.CreateMediaCapturePipelineRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmediapipelines.transform.CreateMediaConcatenationPipelineRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmediapipelines.transform.CreateMediaLiveConnectorPipelineRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmediapipelines.transform.DeleteMediaCapturePipelineRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmediapipelines.transform.DeleteMediaPipelineRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmediapipelines.transform.GetMediaCapturePipelineRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmediapipelines.transform.GetMediaPipelineRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmediapipelines.transform.ListMediaCapturePipelinesRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmediapipelines.transform.ListMediaPipelinesRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmediapipelines.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmediapipelines.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmediapipelines.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ChimeSdkMediaPipelinesAsyncClient}.
 *
 * @see ChimeSdkMediaPipelinesAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultChimeSdkMediaPipelinesAsyncClient implements ChimeSdkMediaPipelinesAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultChimeSdkMediaPipelinesAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultChimeSdkMediaPipelinesAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a media pipeline.
     * </p>
     *
     * @param createMediaCapturePipelineRequest
     * @return A Java Future containing the result of the CreateMediaCapturePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMediaPipelinesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMediaPipelinesAsyncClient.CreateMediaCapturePipeline
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-media-pipelines-2021-07-15/CreateMediaCapturePipeline"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMediaCapturePipelineResponse> createMediaCapturePipeline(
            CreateMediaCapturePipelineRequest createMediaCapturePipelineRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMediaCapturePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Media Pipelines");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMediaCapturePipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMediaCapturePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateMediaCapturePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMediaCapturePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMediaCapturePipelineRequest, CreateMediaCapturePipelineResponse>()
                            .withOperationName("CreateMediaCapturePipeline")
                            .withMarshaller(new CreateMediaCapturePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createMediaCapturePipelineRequest));
            CompletableFuture<CreateMediaCapturePipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a media concatenation pipeline.
     * </p>
     *
     * @param createMediaConcatenationPipelineRequest
     * @return A Java Future containing the result of the CreateMediaConcatenationPipeline operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMediaPipelinesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMediaPipelinesAsyncClient.CreateMediaConcatenationPipeline
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-media-pipelines-2021-07-15/CreateMediaConcatenationPipeline"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMediaConcatenationPipelineResponse> createMediaConcatenationPipeline(
            CreateMediaConcatenationPipelineRequest createMediaConcatenationPipelineRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createMediaConcatenationPipelineRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Media Pipelines");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMediaConcatenationPipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMediaConcatenationPipelineResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateMediaConcatenationPipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMediaConcatenationPipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMediaConcatenationPipelineRequest, CreateMediaConcatenationPipelineResponse>()
                            .withOperationName("CreateMediaConcatenationPipeline")
                            .withMarshaller(new CreateMediaConcatenationPipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createMediaConcatenationPipelineRequest));
            CompletableFuture<CreateMediaConcatenationPipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a streaming media pipeline in an Amazon Chime SDK meeting.
     * </p>
     *
     * @param createMediaLiveConnectorPipelineRequest
     * @return A Java Future containing the result of the CreateMediaLiveConnectorPipeline operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMediaPipelinesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMediaPipelinesAsyncClient.CreateMediaLiveConnectorPipeline
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-media-pipelines-2021-07-15/CreateMediaLiveConnectorPipeline"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMediaLiveConnectorPipelineResponse> createMediaLiveConnectorPipeline(
            CreateMediaLiveConnectorPipelineRequest createMediaLiveConnectorPipelineRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createMediaLiveConnectorPipelineRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Media Pipelines");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMediaLiveConnectorPipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMediaLiveConnectorPipelineResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateMediaLiveConnectorPipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMediaLiveConnectorPipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMediaLiveConnectorPipelineRequest, CreateMediaLiveConnectorPipelineResponse>()
                            .withOperationName("CreateMediaLiveConnectorPipeline")
                            .withMarshaller(new CreateMediaLiveConnectorPipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createMediaLiveConnectorPipelineRequest));
            CompletableFuture<CreateMediaLiveConnectorPipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the media pipeline.
     * </p>
     *
     * @param deleteMediaCapturePipelineRequest
     * @return A Java Future containing the result of the DeleteMediaCapturePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMediaPipelinesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMediaPipelinesAsyncClient.DeleteMediaCapturePipeline
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-media-pipelines-2021-07-15/DeleteMediaCapturePipeline"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMediaCapturePipelineResponse> deleteMediaCapturePipeline(
            DeleteMediaCapturePipelineRequest deleteMediaCapturePipelineRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMediaCapturePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Media Pipelines");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMediaCapturePipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMediaCapturePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMediaCapturePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMediaCapturePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMediaCapturePipelineRequest, DeleteMediaCapturePipelineResponse>()
                            .withOperationName("DeleteMediaCapturePipeline")
                            .withMarshaller(new DeleteMediaCapturePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteMediaCapturePipelineRequest));
            CompletableFuture<DeleteMediaCapturePipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the media pipeline.
     * </p>
     *
     * @param deleteMediaPipelineRequest
     * @return A Java Future containing the result of the DeleteMediaPipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMediaPipelinesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMediaPipelinesAsyncClient.DeleteMediaPipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-media-pipelines-2021-07-15/DeleteMediaPipeline"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMediaPipelineResponse> deleteMediaPipeline(
            DeleteMediaPipelineRequest deleteMediaPipelineRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMediaPipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Media Pipelines");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMediaPipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMediaPipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMediaPipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMediaPipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMediaPipelineRequest, DeleteMediaPipelineResponse>()
                            .withOperationName("DeleteMediaPipeline")
                            .withMarshaller(new DeleteMediaPipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteMediaPipelineRequest));
            CompletableFuture<DeleteMediaPipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets an existing media pipeline.
     * </p>
     *
     * @param getMediaCapturePipelineRequest
     * @return A Java Future containing the result of the GetMediaCapturePipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMediaPipelinesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMediaPipelinesAsyncClient.GetMediaCapturePipeline
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-media-pipelines-2021-07-15/GetMediaCapturePipeline"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMediaCapturePipelineResponse> getMediaCapturePipeline(
            GetMediaCapturePipelineRequest getMediaCapturePipelineRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMediaCapturePipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Media Pipelines");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMediaCapturePipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMediaCapturePipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMediaCapturePipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMediaCapturePipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMediaCapturePipelineRequest, GetMediaCapturePipelineResponse>()
                            .withOperationName("GetMediaCapturePipeline")
                            .withMarshaller(new GetMediaCapturePipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getMediaCapturePipelineRequest));
            CompletableFuture<GetMediaCapturePipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets an existing media pipeline.
     * </p>
     *
     * @param getMediaPipelineRequest
     * @return A Java Future containing the result of the GetMediaPipeline operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMediaPipelinesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMediaPipelinesAsyncClient.GetMediaPipeline
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-media-pipelines-2021-07-15/GetMediaPipeline"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMediaPipelineResponse> getMediaPipeline(GetMediaPipelineRequest getMediaPipelineRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMediaPipelineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Media Pipelines");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMediaPipeline");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMediaPipelineResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMediaPipelineResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMediaPipelineResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMediaPipelineRequest, GetMediaPipelineResponse>()
                            .withOperationName("GetMediaPipeline")
                            .withMarshaller(new GetMediaPipelineRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getMediaPipelineRequest));
            CompletableFuture<GetMediaPipelineResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of media pipelines.
     * </p>
     *
     * @param listMediaCapturePipelinesRequest
     * @return A Java Future containing the result of the ListMediaCapturePipelines operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMediaPipelinesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMediaPipelinesAsyncClient.ListMediaCapturePipelines
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-media-pipelines-2021-07-15/ListMediaCapturePipelines"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMediaCapturePipelinesResponse> listMediaCapturePipelines(
            ListMediaCapturePipelinesRequest listMediaCapturePipelinesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMediaCapturePipelinesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Media Pipelines");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMediaCapturePipelines");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMediaCapturePipelinesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMediaCapturePipelinesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMediaCapturePipelinesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMediaCapturePipelinesRequest, ListMediaCapturePipelinesResponse>()
                            .withOperationName("ListMediaCapturePipelines")
                            .withMarshaller(new ListMediaCapturePipelinesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listMediaCapturePipelinesRequest));
            CompletableFuture<ListMediaCapturePipelinesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of media pipelines.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMediaCapturePipelines(software.amazon.awssdk.services.chimesdkmediapipelines.model.ListMediaCapturePipelinesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmediapipelines.paginators.ListMediaCapturePipelinesPublisher publisher = client.listMediaCapturePipelinesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmediapipelines.paginators.ListMediaCapturePipelinesPublisher publisher = client.listMediaCapturePipelinesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.chimesdkmediapipelines.model.ListMediaCapturePipelinesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.chimesdkmediapipelines.model.ListMediaCapturePipelinesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMediaCapturePipelines(software.amazon.awssdk.services.chimesdkmediapipelines.model.ListMediaCapturePipelinesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listMediaCapturePipelinesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMediaPipelinesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMediaPipelinesAsyncClient.ListMediaCapturePipelines
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-media-pipelines-2021-07-15/ListMediaCapturePipelines"
     *      target="_top">AWS API Documentation</a>
     */
    public ListMediaCapturePipelinesPublisher listMediaCapturePipelinesPaginator(
            ListMediaCapturePipelinesRequest listMediaCapturePipelinesRequest) {
        return new ListMediaCapturePipelinesPublisher(this, applyPaginatorUserAgent(listMediaCapturePipelinesRequest));
    }

    /**
     * <p>
     * Returns a list of media pipelines.
     * </p>
     *
     * @param listMediaPipelinesRequest
     * @return A Java Future containing the result of the ListMediaPipelines operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMediaPipelinesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMediaPipelinesAsyncClient.ListMediaPipelines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-media-pipelines-2021-07-15/ListMediaPipelines"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMediaPipelinesResponse> listMediaPipelines(ListMediaPipelinesRequest listMediaPipelinesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMediaPipelinesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Media Pipelines");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMediaPipelines");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMediaPipelinesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMediaPipelinesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMediaPipelinesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMediaPipelinesRequest, ListMediaPipelinesResponse>()
                            .withOperationName("ListMediaPipelines")
                            .withMarshaller(new ListMediaPipelinesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listMediaPipelinesRequest));
            CompletableFuture<ListMediaPipelinesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of media pipelines.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMediaPipelines(software.amazon.awssdk.services.chimesdkmediapipelines.model.ListMediaPipelinesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmediapipelines.paginators.ListMediaPipelinesPublisher publisher = client.listMediaPipelinesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmediapipelines.paginators.ListMediaPipelinesPublisher publisher = client.listMediaPipelinesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.chimesdkmediapipelines.model.ListMediaPipelinesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.chimesdkmediapipelines.model.ListMediaPipelinesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMediaPipelines(software.amazon.awssdk.services.chimesdkmediapipelines.model.ListMediaPipelinesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listMediaPipelinesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMediaPipelinesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMediaPipelinesAsyncClient.ListMediaPipelines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-media-pipelines-2021-07-15/ListMediaPipelines"
     *      target="_top">AWS API Documentation</a>
     */
    public ListMediaPipelinesPublisher listMediaPipelinesPaginator(ListMediaPipelinesRequest listMediaPipelinesRequest) {
        return new ListMediaPipelinesPublisher(this, applyPaginatorUserAgent(listMediaPipelinesRequest));
    }

    /**
     * <p>
     * Lists the tags available for a media pipeline.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMediaPipelinesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMediaPipelinesAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-media-pipelines-2021-07-15/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Media Pipelines");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The ARN of the media pipeline that you want to tag. Consists of he pipeline's endpoint region, resource ID, and
     * pipeline ID.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMediaPipelinesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMediaPipelinesAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-media-pipelines-2021-07-15/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Media Pipelines");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes any tags from a media pipeline.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMediaPipelinesException Base class for all service exceptions. Unknown exceptions will be
     *         thrown as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMediaPipelinesAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-media-pipelines-2021-07-15/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Media Pipelines");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ChimeSdkMediaPipelinesException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedClientException")
                                .exceptionBuilderSupplier(UnauthorizedClientException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottledClientException")
                                .exceptionBuilderSupplier(ThrottledClientException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceFailureException")
                                .exceptionBuilderSupplier(ServiceFailureException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends ChimeSdkMediaPipelinesRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
