/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chimesdkmediapipelines.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A media pipeline that streams call analytics data.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MediaInsightsPipeline implements SdkPojo, Serializable,
        ToCopyableBuilder<MediaInsightsPipeline.Builder, MediaInsightsPipeline> {
    private static final SdkField<String> MEDIA_PIPELINE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MediaPipelineId").getter(getter(MediaInsightsPipeline::mediaPipelineId))
            .setter(setter(Builder::mediaPipelineId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MediaPipelineId").build()).build();

    private static final SdkField<String> MEDIA_PIPELINE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MediaPipelineArn").getter(getter(MediaInsightsPipeline::mediaPipelineArn))
            .setter(setter(Builder::mediaPipelineArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MediaPipelineArn").build()).build();

    private static final SdkField<String> MEDIA_INSIGHTS_PIPELINE_CONFIGURATION_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("MediaInsightsPipelineConfigurationArn")
            .getter(getter(MediaInsightsPipeline::mediaInsightsPipelineConfigurationArn))
            .setter(setter(Builder::mediaInsightsPipelineConfigurationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("MediaInsightsPipelineConfigurationArn").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(MediaInsightsPipeline::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<KinesisVideoStreamSourceRuntimeConfiguration> KINESIS_VIDEO_STREAM_SOURCE_RUNTIME_CONFIGURATION_FIELD = SdkField
            .<KinesisVideoStreamSourceRuntimeConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("KinesisVideoStreamSourceRuntimeConfiguration")
            .getter(getter(MediaInsightsPipeline::kinesisVideoStreamSourceRuntimeConfiguration))
            .setter(setter(Builder::kinesisVideoStreamSourceRuntimeConfiguration))
            .constructor(KinesisVideoStreamSourceRuntimeConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("KinesisVideoStreamSourceRuntimeConfiguration").build()).build();

    private static final SdkField<Map<String, String>> MEDIA_INSIGHTS_RUNTIME_METADATA_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("MediaInsightsRuntimeMetadata")
            .getter(getter(MediaInsightsPipeline::mediaInsightsRuntimeMetadata))
            .setter(setter(Builder::mediaInsightsRuntimeMetadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MediaInsightsRuntimeMetadata")
                    .build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<KinesisVideoStreamRecordingSourceRuntimeConfiguration> KINESIS_VIDEO_STREAM_RECORDING_SOURCE_RUNTIME_CONFIGURATION_FIELD = SdkField
            .<KinesisVideoStreamRecordingSourceRuntimeConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("KinesisVideoStreamRecordingSourceRuntimeConfiguration")
            .getter(getter(MediaInsightsPipeline::kinesisVideoStreamRecordingSourceRuntimeConfiguration))
            .setter(setter(Builder::kinesisVideoStreamRecordingSourceRuntimeConfiguration))
            .constructor(KinesisVideoStreamRecordingSourceRuntimeConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("KinesisVideoStreamRecordingSourceRuntimeConfiguration").build()).build();

    private static final SdkField<S3RecordingSinkRuntimeConfiguration> S3_RECORDING_SINK_RUNTIME_CONFIGURATION_FIELD = SdkField
            .<S3RecordingSinkRuntimeConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("S3RecordingSinkRuntimeConfiguration")
            .getter(getter(MediaInsightsPipeline::s3RecordingSinkRuntimeConfiguration))
            .setter(setter(Builder::s3RecordingSinkRuntimeConfiguration))
            .constructor(S3RecordingSinkRuntimeConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("S3RecordingSinkRuntimeConfiguration").build()).build();

    private static final SdkField<Instant> CREATED_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTimestamp")
            .getter(getter(MediaInsightsPipeline::createdTimestamp))
            .setter(setter(Builder::createdTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTimestamp").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<List<MediaInsightsPipelineElementStatus>> ELEMENT_STATUSES_FIELD = SdkField
            .<List<MediaInsightsPipelineElementStatus>> builder(MarshallingType.LIST)
            .memberName("ElementStatuses")
            .getter(getter(MediaInsightsPipeline::elementStatuses))
            .setter(setter(Builder::elementStatuses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ElementStatuses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MediaInsightsPipelineElementStatus> builder(MarshallingType.SDK_POJO)
                                            .constructor(MediaInsightsPipelineElementStatus::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MEDIA_PIPELINE_ID_FIELD,
            MEDIA_PIPELINE_ARN_FIELD, MEDIA_INSIGHTS_PIPELINE_CONFIGURATION_ARN_FIELD, STATUS_FIELD,
            KINESIS_VIDEO_STREAM_SOURCE_RUNTIME_CONFIGURATION_FIELD, MEDIA_INSIGHTS_RUNTIME_METADATA_FIELD,
            KINESIS_VIDEO_STREAM_RECORDING_SOURCE_RUNTIME_CONFIGURATION_FIELD, S3_RECORDING_SINK_RUNTIME_CONFIGURATION_FIELD,
            CREATED_TIMESTAMP_FIELD, ELEMENT_STATUSES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("MediaPipelineId", MEDIA_PIPELINE_ID_FIELD);
                    put("MediaPipelineArn", MEDIA_PIPELINE_ARN_FIELD);
                    put("MediaInsightsPipelineConfigurationArn", MEDIA_INSIGHTS_PIPELINE_CONFIGURATION_ARN_FIELD);
                    put("Status", STATUS_FIELD);
                    put("KinesisVideoStreamSourceRuntimeConfiguration", KINESIS_VIDEO_STREAM_SOURCE_RUNTIME_CONFIGURATION_FIELD);
                    put("MediaInsightsRuntimeMetadata", MEDIA_INSIGHTS_RUNTIME_METADATA_FIELD);
                    put("KinesisVideoStreamRecordingSourceRuntimeConfiguration",
                            KINESIS_VIDEO_STREAM_RECORDING_SOURCE_RUNTIME_CONFIGURATION_FIELD);
                    put("S3RecordingSinkRuntimeConfiguration", S3_RECORDING_SINK_RUNTIME_CONFIGURATION_FIELD);
                    put("CreatedTimestamp", CREATED_TIMESTAMP_FIELD);
                    put("ElementStatuses", ELEMENT_STATUSES_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String mediaPipelineId;

    private final String mediaPipelineArn;

    private final String mediaInsightsPipelineConfigurationArn;

    private final String status;

    private final KinesisVideoStreamSourceRuntimeConfiguration kinesisVideoStreamSourceRuntimeConfiguration;

    private final Map<String, String> mediaInsightsRuntimeMetadata;

    private final KinesisVideoStreamRecordingSourceRuntimeConfiguration kinesisVideoStreamRecordingSourceRuntimeConfiguration;

    private final S3RecordingSinkRuntimeConfiguration s3RecordingSinkRuntimeConfiguration;

    private final Instant createdTimestamp;

    private final List<MediaInsightsPipelineElementStatus> elementStatuses;

    private MediaInsightsPipeline(BuilderImpl builder) {
        this.mediaPipelineId = builder.mediaPipelineId;
        this.mediaPipelineArn = builder.mediaPipelineArn;
        this.mediaInsightsPipelineConfigurationArn = builder.mediaInsightsPipelineConfigurationArn;
        this.status = builder.status;
        this.kinesisVideoStreamSourceRuntimeConfiguration = builder.kinesisVideoStreamSourceRuntimeConfiguration;
        this.mediaInsightsRuntimeMetadata = builder.mediaInsightsRuntimeMetadata;
        this.kinesisVideoStreamRecordingSourceRuntimeConfiguration = builder.kinesisVideoStreamRecordingSourceRuntimeConfiguration;
        this.s3RecordingSinkRuntimeConfiguration = builder.s3RecordingSinkRuntimeConfiguration;
        this.createdTimestamp = builder.createdTimestamp;
        this.elementStatuses = builder.elementStatuses;
    }

    /**
     * <p>
     * The ID of a media insights pipeline.
     * </p>
     * 
     * @return The ID of a media insights pipeline.
     */
    public final String mediaPipelineId() {
        return mediaPipelineId;
    }

    /**
     * <p>
     * The ARN of a media insights pipeline.
     * </p>
     * 
     * @return The ARN of a media insights pipeline.
     */
    public final String mediaPipelineArn() {
        return mediaPipelineArn;
    }

    /**
     * <p>
     * The ARN of a media insight pipeline's configuration settings.
     * </p>
     * 
     * @return The ARN of a media insight pipeline's configuration settings.
     */
    public final String mediaInsightsPipelineConfigurationArn() {
        return mediaInsightsPipelineConfigurationArn;
    }

    /**
     * <p>
     * The status of a media insights pipeline.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MediaPipelineStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of a media insights pipeline.
     * @see MediaPipelineStatus
     */
    public final MediaPipelineStatus status() {
        return MediaPipelineStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of a media insights pipeline.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MediaPipelineStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of a media insights pipeline.
     * @see MediaPipelineStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The configuration settings for a Kinesis runtime video stream in a media insights pipeline.
     * </p>
     * 
     * @return The configuration settings for a Kinesis runtime video stream in a media insights pipeline.
     */
    public final KinesisVideoStreamSourceRuntimeConfiguration kinesisVideoStreamSourceRuntimeConfiguration() {
        return kinesisVideoStreamSourceRuntimeConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the MediaInsightsRuntimeMetadata property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasMediaInsightsRuntimeMetadata() {
        return mediaInsightsRuntimeMetadata != null && !(mediaInsightsRuntimeMetadata instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The runtime metadata of a media insights pipeline.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMediaInsightsRuntimeMetadata} method.
     * </p>
     * 
     * @return The runtime metadata of a media insights pipeline.
     */
    public final Map<String, String> mediaInsightsRuntimeMetadata() {
        return mediaInsightsRuntimeMetadata;
    }

    /**
     * <p>
     * The runtime configuration settings for a Kinesis recording video stream in a media insights pipeline.
     * </p>
     * 
     * @return The runtime configuration settings for a Kinesis recording video stream in a media insights pipeline.
     */
    public final KinesisVideoStreamRecordingSourceRuntimeConfiguration kinesisVideoStreamRecordingSourceRuntimeConfiguration() {
        return kinesisVideoStreamRecordingSourceRuntimeConfiguration;
    }

    /**
     * <p>
     * The runtime configuration of the Amazon S3 bucket that stores recordings in a media insights pipeline.
     * </p>
     * 
     * @return The runtime configuration of the Amazon S3 bucket that stores recordings in a media insights pipeline.
     */
    public final S3RecordingSinkRuntimeConfiguration s3RecordingSinkRuntimeConfiguration() {
        return s3RecordingSinkRuntimeConfiguration;
    }

    /**
     * <p>
     * The time at which the media insights pipeline was created.
     * </p>
     * 
     * @return The time at which the media insights pipeline was created.
     */
    public final Instant createdTimestamp() {
        return createdTimestamp;
    }

    /**
     * For responses, this returns true if the service returned a value for the ElementStatuses property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasElementStatuses() {
        return elementStatuses != null && !(elementStatuses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The statuses that the elements in a media insights pipeline can have during data processing.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasElementStatuses} method.
     * </p>
     * 
     * @return The statuses that the elements in a media insights pipeline can have during data processing.
     */
    public final List<MediaInsightsPipelineElementStatus> elementStatuses() {
        return elementStatuses;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(mediaPipelineId());
        hashCode = 31 * hashCode + Objects.hashCode(mediaPipelineArn());
        hashCode = 31 * hashCode + Objects.hashCode(mediaInsightsPipelineConfigurationArn());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(kinesisVideoStreamSourceRuntimeConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasMediaInsightsRuntimeMetadata() ? mediaInsightsRuntimeMetadata() : null);
        hashCode = 31 * hashCode + Objects.hashCode(kinesisVideoStreamRecordingSourceRuntimeConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(s3RecordingSinkRuntimeConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(createdTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(hasElementStatuses() ? elementStatuses() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MediaInsightsPipeline)) {
            return false;
        }
        MediaInsightsPipeline other = (MediaInsightsPipeline) obj;
        return Objects.equals(mediaPipelineId(), other.mediaPipelineId())
                && Objects.equals(mediaPipelineArn(), other.mediaPipelineArn())
                && Objects.equals(mediaInsightsPipelineConfigurationArn(), other.mediaInsightsPipelineConfigurationArn())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(kinesisVideoStreamSourceRuntimeConfiguration(),
                        other.kinesisVideoStreamSourceRuntimeConfiguration())
                && hasMediaInsightsRuntimeMetadata() == other.hasMediaInsightsRuntimeMetadata()
                && Objects.equals(mediaInsightsRuntimeMetadata(), other.mediaInsightsRuntimeMetadata())
                && Objects.equals(kinesisVideoStreamRecordingSourceRuntimeConfiguration(),
                        other.kinesisVideoStreamRecordingSourceRuntimeConfiguration())
                && Objects.equals(s3RecordingSinkRuntimeConfiguration(), other.s3RecordingSinkRuntimeConfiguration())
                && Objects.equals(createdTimestamp(), other.createdTimestamp())
                && hasElementStatuses() == other.hasElementStatuses()
                && Objects.equals(elementStatuses(), other.elementStatuses());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("MediaInsightsPipeline")
                .add("MediaPipelineId", mediaPipelineId())
                .add("MediaPipelineArn", mediaPipelineArn() == null ? null : "*** Sensitive Data Redacted ***")
                .add("MediaInsightsPipelineConfigurationArn",
                        mediaInsightsPipelineConfigurationArn() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Status", statusAsString())
                .add("KinesisVideoStreamSourceRuntimeConfiguration", kinesisVideoStreamSourceRuntimeConfiguration())
                .add("MediaInsightsRuntimeMetadata",
                        mediaInsightsRuntimeMetadata() == null ? null : "*** Sensitive Data Redacted ***")
                .add("KinesisVideoStreamRecordingSourceRuntimeConfiguration",
                        kinesisVideoStreamRecordingSourceRuntimeConfiguration())
                .add("S3RecordingSinkRuntimeConfiguration", s3RecordingSinkRuntimeConfiguration())
                .add("CreatedTimestamp", createdTimestamp())
                .add("ElementStatuses", hasElementStatuses() ? elementStatuses() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MediaPipelineId":
            return Optional.ofNullable(clazz.cast(mediaPipelineId()));
        case "MediaPipelineArn":
            return Optional.ofNullable(clazz.cast(mediaPipelineArn()));
        case "MediaInsightsPipelineConfigurationArn":
            return Optional.ofNullable(clazz.cast(mediaInsightsPipelineConfigurationArn()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "KinesisVideoStreamSourceRuntimeConfiguration":
            return Optional.ofNullable(clazz.cast(kinesisVideoStreamSourceRuntimeConfiguration()));
        case "MediaInsightsRuntimeMetadata":
            return Optional.ofNullable(clazz.cast(mediaInsightsRuntimeMetadata()));
        case "KinesisVideoStreamRecordingSourceRuntimeConfiguration":
            return Optional.ofNullable(clazz.cast(kinesisVideoStreamRecordingSourceRuntimeConfiguration()));
        case "S3RecordingSinkRuntimeConfiguration":
            return Optional.ofNullable(clazz.cast(s3RecordingSinkRuntimeConfiguration()));
        case "CreatedTimestamp":
            return Optional.ofNullable(clazz.cast(createdTimestamp()));
        case "ElementStatuses":
            return Optional.ofNullable(clazz.cast(elementStatuses()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<MediaInsightsPipeline, T> g) {
        return obj -> g.apply((MediaInsightsPipeline) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MediaInsightsPipeline> {
        /**
         * <p>
         * The ID of a media insights pipeline.
         * </p>
         * 
         * @param mediaPipelineId
         *        The ID of a media insights pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mediaPipelineId(String mediaPipelineId);

        /**
         * <p>
         * The ARN of a media insights pipeline.
         * </p>
         * 
         * @param mediaPipelineArn
         *        The ARN of a media insights pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mediaPipelineArn(String mediaPipelineArn);

        /**
         * <p>
         * The ARN of a media insight pipeline's configuration settings.
         * </p>
         * 
         * @param mediaInsightsPipelineConfigurationArn
         *        The ARN of a media insight pipeline's configuration settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mediaInsightsPipelineConfigurationArn(String mediaInsightsPipelineConfigurationArn);

        /**
         * <p>
         * The status of a media insights pipeline.
         * </p>
         * 
         * @param status
         *        The status of a media insights pipeline.
         * @see MediaPipelineStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MediaPipelineStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of a media insights pipeline.
         * </p>
         * 
         * @param status
         *        The status of a media insights pipeline.
         * @see MediaPipelineStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MediaPipelineStatus
         */
        Builder status(MediaPipelineStatus status);

        /**
         * <p>
         * The configuration settings for a Kinesis runtime video stream in a media insights pipeline.
         * </p>
         * 
         * @param kinesisVideoStreamSourceRuntimeConfiguration
         *        The configuration settings for a Kinesis runtime video stream in a media insights pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kinesisVideoStreamSourceRuntimeConfiguration(
                KinesisVideoStreamSourceRuntimeConfiguration kinesisVideoStreamSourceRuntimeConfiguration);

        /**
         * <p>
         * The configuration settings for a Kinesis runtime video stream in a media insights pipeline.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link KinesisVideoStreamSourceRuntimeConfiguration.Builder} avoiding the need to create one manually via
         * {@link KinesisVideoStreamSourceRuntimeConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link KinesisVideoStreamSourceRuntimeConfiguration.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #kinesisVideoStreamSourceRuntimeConfiguration(KinesisVideoStreamSourceRuntimeConfiguration)}.
         * 
         * @param kinesisVideoStreamSourceRuntimeConfiguration
         *        a consumer that will call methods on {@link KinesisVideoStreamSourceRuntimeConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #kinesisVideoStreamSourceRuntimeConfiguration(KinesisVideoStreamSourceRuntimeConfiguration)
         */
        default Builder kinesisVideoStreamSourceRuntimeConfiguration(
                Consumer<KinesisVideoStreamSourceRuntimeConfiguration.Builder> kinesisVideoStreamSourceRuntimeConfiguration) {
            return kinesisVideoStreamSourceRuntimeConfiguration(KinesisVideoStreamSourceRuntimeConfiguration.builder()
                    .applyMutation(kinesisVideoStreamSourceRuntimeConfiguration).build());
        }

        /**
         * <p>
         * The runtime metadata of a media insights pipeline.
         * </p>
         * 
         * @param mediaInsightsRuntimeMetadata
         *        The runtime metadata of a media insights pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mediaInsightsRuntimeMetadata(Map<String, String> mediaInsightsRuntimeMetadata);

        /**
         * <p>
         * The runtime configuration settings for a Kinesis recording video stream in a media insights pipeline.
         * </p>
         * 
         * @param kinesisVideoStreamRecordingSourceRuntimeConfiguration
         *        The runtime configuration settings for a Kinesis recording video stream in a media insights pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kinesisVideoStreamRecordingSourceRuntimeConfiguration(
                KinesisVideoStreamRecordingSourceRuntimeConfiguration kinesisVideoStreamRecordingSourceRuntimeConfiguration);

        /**
         * <p>
         * The runtime configuration settings for a Kinesis recording video stream in a media insights pipeline.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link KinesisVideoStreamRecordingSourceRuntimeConfiguration.Builder} avoiding the need to create one
         * manually via {@link KinesisVideoStreamRecordingSourceRuntimeConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link KinesisVideoStreamRecordingSourceRuntimeConfiguration.Builder#build()} is called immediately and its
         * result is passed to
         * {@link #kinesisVideoStreamRecordingSourceRuntimeConfiguration(KinesisVideoStreamRecordingSourceRuntimeConfiguration)}.
         * 
         * @param kinesisVideoStreamRecordingSourceRuntimeConfiguration
         *        a consumer that will call methods on
         *        {@link KinesisVideoStreamRecordingSourceRuntimeConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #kinesisVideoStreamRecordingSourceRuntimeConfiguration(KinesisVideoStreamRecordingSourceRuntimeConfiguration)
         */
        default Builder kinesisVideoStreamRecordingSourceRuntimeConfiguration(
                Consumer<KinesisVideoStreamRecordingSourceRuntimeConfiguration.Builder> kinesisVideoStreamRecordingSourceRuntimeConfiguration) {
            return kinesisVideoStreamRecordingSourceRuntimeConfiguration(KinesisVideoStreamRecordingSourceRuntimeConfiguration
                    .builder().applyMutation(kinesisVideoStreamRecordingSourceRuntimeConfiguration).build());
        }

        /**
         * <p>
         * The runtime configuration of the Amazon S3 bucket that stores recordings in a media insights pipeline.
         * </p>
         * 
         * @param s3RecordingSinkRuntimeConfiguration
         *        The runtime configuration of the Amazon S3 bucket that stores recordings in a media insights pipeline.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3RecordingSinkRuntimeConfiguration(S3RecordingSinkRuntimeConfiguration s3RecordingSinkRuntimeConfiguration);

        /**
         * <p>
         * The runtime configuration of the Amazon S3 bucket that stores recordings in a media insights pipeline.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link S3RecordingSinkRuntimeConfiguration.Builder} avoiding the need to create one manually via
         * {@link S3RecordingSinkRuntimeConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link S3RecordingSinkRuntimeConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #s3RecordingSinkRuntimeConfiguration(S3RecordingSinkRuntimeConfiguration)}.
         * 
         * @param s3RecordingSinkRuntimeConfiguration
         *        a consumer that will call methods on {@link S3RecordingSinkRuntimeConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3RecordingSinkRuntimeConfiguration(S3RecordingSinkRuntimeConfiguration)
         */
        default Builder s3RecordingSinkRuntimeConfiguration(
                Consumer<S3RecordingSinkRuntimeConfiguration.Builder> s3RecordingSinkRuntimeConfiguration) {
            return s3RecordingSinkRuntimeConfiguration(S3RecordingSinkRuntimeConfiguration.builder()
                    .applyMutation(s3RecordingSinkRuntimeConfiguration).build());
        }

        /**
         * <p>
         * The time at which the media insights pipeline was created.
         * </p>
         * 
         * @param createdTimestamp
         *        The time at which the media insights pipeline was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTimestamp(Instant createdTimestamp);

        /**
         * <p>
         * The statuses that the elements in a media insights pipeline can have during data processing.
         * </p>
         * 
         * @param elementStatuses
         *        The statuses that the elements in a media insights pipeline can have during data processing.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder elementStatuses(Collection<MediaInsightsPipelineElementStatus> elementStatuses);

        /**
         * <p>
         * The statuses that the elements in a media insights pipeline can have during data processing.
         * </p>
         * 
         * @param elementStatuses
         *        The statuses that the elements in a media insights pipeline can have during data processing.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder elementStatuses(MediaInsightsPipelineElementStatus... elementStatuses);

        /**
         * <p>
         * The statuses that the elements in a media insights pipeline can have during data processing.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.chimesdkmediapipelines.model.MediaInsightsPipelineElementStatus.Builder}
         * avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.chimesdkmediapipelines.model.MediaInsightsPipelineElementStatus#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.chimesdkmediapipelines.model.MediaInsightsPipelineElementStatus.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #elementStatuses(List<MediaInsightsPipelineElementStatus>)}.
         * 
         * @param elementStatuses
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.chimesdkmediapipelines.model.MediaInsightsPipelineElementStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #elementStatuses(java.util.Collection<MediaInsightsPipelineElementStatus>)
         */
        Builder elementStatuses(Consumer<MediaInsightsPipelineElementStatus.Builder>... elementStatuses);
    }

    static final class BuilderImpl implements Builder {
        private String mediaPipelineId;

        private String mediaPipelineArn;

        private String mediaInsightsPipelineConfigurationArn;

        private String status;

        private KinesisVideoStreamSourceRuntimeConfiguration kinesisVideoStreamSourceRuntimeConfiguration;

        private Map<String, String> mediaInsightsRuntimeMetadata = DefaultSdkAutoConstructMap.getInstance();

        private KinesisVideoStreamRecordingSourceRuntimeConfiguration kinesisVideoStreamRecordingSourceRuntimeConfiguration;

        private S3RecordingSinkRuntimeConfiguration s3RecordingSinkRuntimeConfiguration;

        private Instant createdTimestamp;

        private List<MediaInsightsPipelineElementStatus> elementStatuses = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MediaInsightsPipeline model) {
            mediaPipelineId(model.mediaPipelineId);
            mediaPipelineArn(model.mediaPipelineArn);
            mediaInsightsPipelineConfigurationArn(model.mediaInsightsPipelineConfigurationArn);
            status(model.status);
            kinesisVideoStreamSourceRuntimeConfiguration(model.kinesisVideoStreamSourceRuntimeConfiguration);
            mediaInsightsRuntimeMetadata(model.mediaInsightsRuntimeMetadata);
            kinesisVideoStreamRecordingSourceRuntimeConfiguration(model.kinesisVideoStreamRecordingSourceRuntimeConfiguration);
            s3RecordingSinkRuntimeConfiguration(model.s3RecordingSinkRuntimeConfiguration);
            createdTimestamp(model.createdTimestamp);
            elementStatuses(model.elementStatuses);
        }

        public final String getMediaPipelineId() {
            return mediaPipelineId;
        }

        public final void setMediaPipelineId(String mediaPipelineId) {
            this.mediaPipelineId = mediaPipelineId;
        }

        @Override
        public final Builder mediaPipelineId(String mediaPipelineId) {
            this.mediaPipelineId = mediaPipelineId;
            return this;
        }

        public final String getMediaPipelineArn() {
            return mediaPipelineArn;
        }

        public final void setMediaPipelineArn(String mediaPipelineArn) {
            this.mediaPipelineArn = mediaPipelineArn;
        }

        @Override
        public final Builder mediaPipelineArn(String mediaPipelineArn) {
            this.mediaPipelineArn = mediaPipelineArn;
            return this;
        }

        public final String getMediaInsightsPipelineConfigurationArn() {
            return mediaInsightsPipelineConfigurationArn;
        }

        public final void setMediaInsightsPipelineConfigurationArn(String mediaInsightsPipelineConfigurationArn) {
            this.mediaInsightsPipelineConfigurationArn = mediaInsightsPipelineConfigurationArn;
        }

        @Override
        public final Builder mediaInsightsPipelineConfigurationArn(String mediaInsightsPipelineConfigurationArn) {
            this.mediaInsightsPipelineConfigurationArn = mediaInsightsPipelineConfigurationArn;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(MediaPipelineStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final KinesisVideoStreamSourceRuntimeConfiguration.Builder getKinesisVideoStreamSourceRuntimeConfiguration() {
            return kinesisVideoStreamSourceRuntimeConfiguration != null ? kinesisVideoStreamSourceRuntimeConfiguration
                    .toBuilder() : null;
        }

        public final void setKinesisVideoStreamSourceRuntimeConfiguration(
                KinesisVideoStreamSourceRuntimeConfiguration.BuilderImpl kinesisVideoStreamSourceRuntimeConfiguration) {
            this.kinesisVideoStreamSourceRuntimeConfiguration = kinesisVideoStreamSourceRuntimeConfiguration != null ? kinesisVideoStreamSourceRuntimeConfiguration
                    .build() : null;
        }

        @Override
        public final Builder kinesisVideoStreamSourceRuntimeConfiguration(
                KinesisVideoStreamSourceRuntimeConfiguration kinesisVideoStreamSourceRuntimeConfiguration) {
            this.kinesisVideoStreamSourceRuntimeConfiguration = kinesisVideoStreamSourceRuntimeConfiguration;
            return this;
        }

        public final Map<String, String> getMediaInsightsRuntimeMetadata() {
            if (mediaInsightsRuntimeMetadata instanceof SdkAutoConstructMap) {
                return null;
            }
            return mediaInsightsRuntimeMetadata;
        }

        public final void setMediaInsightsRuntimeMetadata(Map<String, String> mediaInsightsRuntimeMetadata) {
            this.mediaInsightsRuntimeMetadata = MediaInsightsRuntimeMetadataCopier.copy(mediaInsightsRuntimeMetadata);
        }

        @Override
        public final Builder mediaInsightsRuntimeMetadata(Map<String, String> mediaInsightsRuntimeMetadata) {
            this.mediaInsightsRuntimeMetadata = MediaInsightsRuntimeMetadataCopier.copy(mediaInsightsRuntimeMetadata);
            return this;
        }

        public final KinesisVideoStreamRecordingSourceRuntimeConfiguration.Builder getKinesisVideoStreamRecordingSourceRuntimeConfiguration() {
            return kinesisVideoStreamRecordingSourceRuntimeConfiguration != null ? kinesisVideoStreamRecordingSourceRuntimeConfiguration
                    .toBuilder() : null;
        }

        public final void setKinesisVideoStreamRecordingSourceRuntimeConfiguration(
                KinesisVideoStreamRecordingSourceRuntimeConfiguration.BuilderImpl kinesisVideoStreamRecordingSourceRuntimeConfiguration) {
            this.kinesisVideoStreamRecordingSourceRuntimeConfiguration = kinesisVideoStreamRecordingSourceRuntimeConfiguration != null ? kinesisVideoStreamRecordingSourceRuntimeConfiguration
                    .build() : null;
        }

        @Override
        public final Builder kinesisVideoStreamRecordingSourceRuntimeConfiguration(
                KinesisVideoStreamRecordingSourceRuntimeConfiguration kinesisVideoStreamRecordingSourceRuntimeConfiguration) {
            this.kinesisVideoStreamRecordingSourceRuntimeConfiguration = kinesisVideoStreamRecordingSourceRuntimeConfiguration;
            return this;
        }

        public final S3RecordingSinkRuntimeConfiguration.Builder getS3RecordingSinkRuntimeConfiguration() {
            return s3RecordingSinkRuntimeConfiguration != null ? s3RecordingSinkRuntimeConfiguration.toBuilder() : null;
        }

        public final void setS3RecordingSinkRuntimeConfiguration(
                S3RecordingSinkRuntimeConfiguration.BuilderImpl s3RecordingSinkRuntimeConfiguration) {
            this.s3RecordingSinkRuntimeConfiguration = s3RecordingSinkRuntimeConfiguration != null ? s3RecordingSinkRuntimeConfiguration
                    .build() : null;
        }

        @Override
        public final Builder s3RecordingSinkRuntimeConfiguration(
                S3RecordingSinkRuntimeConfiguration s3RecordingSinkRuntimeConfiguration) {
            this.s3RecordingSinkRuntimeConfiguration = s3RecordingSinkRuntimeConfiguration;
            return this;
        }

        public final Instant getCreatedTimestamp() {
            return createdTimestamp;
        }

        public final void setCreatedTimestamp(Instant createdTimestamp) {
            this.createdTimestamp = createdTimestamp;
        }

        @Override
        public final Builder createdTimestamp(Instant createdTimestamp) {
            this.createdTimestamp = createdTimestamp;
            return this;
        }

        public final List<MediaInsightsPipelineElementStatus.Builder> getElementStatuses() {
            List<MediaInsightsPipelineElementStatus.Builder> result = MediaInsightsPipelineElementStatusesCopier
                    .copyToBuilder(this.elementStatuses);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setElementStatuses(Collection<MediaInsightsPipelineElementStatus.BuilderImpl> elementStatuses) {
            this.elementStatuses = MediaInsightsPipelineElementStatusesCopier.copyFromBuilder(elementStatuses);
        }

        @Override
        public final Builder elementStatuses(Collection<MediaInsightsPipelineElementStatus> elementStatuses) {
            this.elementStatuses = MediaInsightsPipelineElementStatusesCopier.copy(elementStatuses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder elementStatuses(MediaInsightsPipelineElementStatus... elementStatuses) {
            elementStatuses(Arrays.asList(elementStatuses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder elementStatuses(Consumer<MediaInsightsPipelineElementStatus.Builder>... elementStatuses) {
            elementStatuses(Stream.of(elementStatuses)
                    .map(c -> MediaInsightsPipelineElementStatus.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public MediaInsightsPipeline build() {
            return new MediaInsightsPipeline(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
