/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chimesdkmeetings;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.chimesdkmeetings.model.BadRequestException;
import software.amazon.awssdk.services.chimesdkmeetings.model.BatchCreateAttendeeRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.BatchCreateAttendeeResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.ChimeSdkMeetingsException;
import software.amazon.awssdk.services.chimesdkmeetings.model.ChimeSdkMeetingsRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.CreateAttendeeRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.CreateAttendeeResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.CreateMeetingRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.CreateMeetingResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.CreateMeetingWithAttendeesRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.CreateMeetingWithAttendeesResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.DeleteAttendeeRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.DeleteAttendeeResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.DeleteMeetingRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.DeleteMeetingResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.ForbiddenException;
import software.amazon.awssdk.services.chimesdkmeetings.model.GetAttendeeRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.GetAttendeeResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.GetMeetingRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.GetMeetingResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.LimitExceededException;
import software.amazon.awssdk.services.chimesdkmeetings.model.ListAttendeesRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.ListAttendeesResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.NotFoundException;
import software.amazon.awssdk.services.chimesdkmeetings.model.ServiceUnavailableException;
import software.amazon.awssdk.services.chimesdkmeetings.model.StartMeetingTranscriptionRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.StartMeetingTranscriptionResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.StopMeetingTranscriptionRequest;
import software.amazon.awssdk.services.chimesdkmeetings.model.StopMeetingTranscriptionResponse;
import software.amazon.awssdk.services.chimesdkmeetings.model.UnauthorizedException;
import software.amazon.awssdk.services.chimesdkmeetings.model.UnprocessableEntityException;
import software.amazon.awssdk.services.chimesdkmeetings.paginators.ListAttendeesIterable;
import software.amazon.awssdk.services.chimesdkmeetings.transform.BatchCreateAttendeeRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.CreateAttendeeRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.CreateMeetingRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.CreateMeetingWithAttendeesRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.DeleteAttendeeRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.DeleteMeetingRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.GetAttendeeRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.GetMeetingRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.ListAttendeesRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.StartMeetingTranscriptionRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmeetings.transform.StopMeetingTranscriptionRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ChimeSdkMeetingsClient}.
 *
 * @see ChimeSdkMeetingsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultChimeSdkMeetingsClient implements ChimeSdkMeetingsClient {
    private static final Logger log = Logger.loggerFor(DefaultChimeSdkMeetingsClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultChimeSdkMeetingsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates up to 100 attendees for an active Amazon Chime SDK meeting. For more information about the Amazon Chime
     * SDK, see <a href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the Amazon Chime SDK</a>
     * in the <i>Amazon Chime Developer Guide</i>.
     * </p>
     *
     * @param batchCreateAttendeeRequest
     * @return Result of the BatchCreateAttendee operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws UnauthorizedException
     *         The user isn't authorized to request a resource.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws LimitExceededException
     *         The request exceeds the resource limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkMeetingsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkMeetingsClient.BatchCreateAttendee
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/BatchCreateAttendee"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchCreateAttendeeResponse batchCreateAttendee(BatchCreateAttendeeRequest batchCreateAttendeeRequest)
            throws BadRequestException, UnauthorizedException, ServiceUnavailableException, NotFoundException,
            ForbiddenException, LimitExceededException, AwsServiceException, SdkClientException, ChimeSdkMeetingsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchCreateAttendeeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchCreateAttendeeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchCreateAttendeeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchCreateAttendee");

            return clientHandler.execute(new ClientExecutionParams<BatchCreateAttendeeRequest, BatchCreateAttendeeResponse>()
                    .withOperationName("BatchCreateAttendee").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(batchCreateAttendeeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchCreateAttendeeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new attendee for an active Amazon Chime SDK meeting. For more information about the Amazon Chime SDK,
     * see <a href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the Amazon Chime SDK</a> in the
     * <i>Amazon Chime Developer Guide</i>.
     * </p>
     *
     * @param createAttendeeRequest
     * @return Result of the CreateAttendee operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws UnauthorizedException
     *         The user isn't authorized to request a resource.
     * @throws UnprocessableEntityException
     *         The request was well-formed but was unable to be followed due to semantic errors.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws LimitExceededException
     *         The request exceeds the resource limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkMeetingsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkMeetingsClient.CreateAttendee
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/CreateAttendee"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateAttendeeResponse createAttendee(CreateAttendeeRequest createAttendeeRequest) throws BadRequestException,
            ForbiddenException, UnauthorizedException, UnprocessableEntityException, ServiceUnavailableException,
            NotFoundException, LimitExceededException, AwsServiceException, SdkClientException, ChimeSdkMeetingsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAttendeeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAttendeeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAttendeeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAttendee");

            return clientHandler.execute(new ClientExecutionParams<CreateAttendeeRequest, CreateAttendeeResponse>()
                    .withOperationName("CreateAttendee").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createAttendeeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateAttendeeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new Amazon Chime SDK meeting in the specified media Region with no initial attendees. For more
     * information about specifying media Regions, see <a
     * href="https://docs.aws.amazon.com/chime/latest/dg/chime-sdk-meetings-regions.html">Amazon Chime SDK Media
     * Regions</a> in the <i>Amazon Chime Developer Guide</i>. For more information about the Amazon Chime SDK, see <a
     * href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the Amazon Chime SDK</a> in the
     * <i>Amazon Chime Developer Guide</i>.
     * </p>
     *
     * @param createMeetingRequest
     * @return Result of the CreateMeeting operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws UnauthorizedException
     *         The user isn't authorized to request a resource.
     * @throws LimitExceededException
     *         The request exceeds the resource limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkMeetingsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkMeetingsClient.CreateMeeting
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/CreateMeeting"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateMeetingResponse createMeeting(CreateMeetingRequest createMeetingRequest) throws BadRequestException,
            ServiceUnavailableException, UnauthorizedException, LimitExceededException, AwsServiceException, SdkClientException,
            ChimeSdkMeetingsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateMeetingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateMeetingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMeetingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMeeting");

            return clientHandler.execute(new ClientExecutionParams<CreateMeetingRequest, CreateMeetingResponse>()
                    .withOperationName("CreateMeeting").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createMeetingRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateMeetingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new Amazon Chime SDK meeting in the specified media Region, with attendees. For more information about
     * specifying media Regions, see <a
     * href="https://docs.aws.amazon.com/chime/latest/dg/chime-sdk-meetings-regions.html">Amazon Chime SDK Media
     * Regions</a> in the <i>Amazon Chime Developer Guide</i>. For more information about the Amazon Chime SDK, see <a
     * href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the Amazon Chime SDK</a> in the
     * <i>Amazon Chime Developer Guide</i>.
     * </p>
     *
     * @param createMeetingWithAttendeesRequest
     * @return Result of the CreateMeetingWithAttendees operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws UnauthorizedException
     *         The user isn't authorized to request a resource.
     * @throws LimitExceededException
     *         The request exceeds the resource limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkMeetingsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkMeetingsClient.CreateMeetingWithAttendees
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/CreateMeetingWithAttendees"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateMeetingWithAttendeesResponse createMeetingWithAttendees(
            CreateMeetingWithAttendeesRequest createMeetingWithAttendeesRequest) throws BadRequestException,
            ServiceUnavailableException, UnauthorizedException, LimitExceededException, AwsServiceException, SdkClientException,
            ChimeSdkMeetingsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateMeetingWithAttendeesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateMeetingWithAttendeesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMeetingWithAttendeesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMeetingWithAttendees");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateMeetingWithAttendeesRequest, CreateMeetingWithAttendeesResponse>()
                            .withOperationName("CreateMeetingWithAttendees").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createMeetingWithAttendeesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateMeetingWithAttendeesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an attendee from the specified Amazon Chime SDK meeting and deletes their <code>JoinToken</code>.
     * Attendees are automatically deleted when a Amazon Chime SDK meeting is deleted. For more information about the
     * Amazon Chime SDK, see <a href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the Amazon
     * Chime SDK</a> in the <i>Amazon Chime Developer Guide</i>.
     * </p>
     *
     * @param deleteAttendeeRequest
     * @return Result of the DeleteAttendee operation returned by the service.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkMeetingsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkMeetingsClient.DeleteAttendee
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/DeleteAttendee"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAttendeeResponse deleteAttendee(DeleteAttendeeRequest deleteAttendeeRequest) throws ForbiddenException,
            NotFoundException, AwsServiceException, SdkClientException, ChimeSdkMeetingsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAttendeeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAttendeeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAttendeeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAttendee");

            return clientHandler.execute(new ClientExecutionParams<DeleteAttendeeRequest, DeleteAttendeeResponse>()
                    .withOperationName("DeleteAttendee").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteAttendeeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAttendeeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified Amazon Chime SDK meeting. The operation deletes all attendees, disconnects all clients, and
     * prevents new clients from joining the meeting. For more information about the Amazon Chime SDK, see <a
     * href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the Amazon Chime SDK</a> in the
     * <i>Amazon Chime Developer Guide</i>.
     * </p>
     *
     * @param deleteMeetingRequest
     * @return Result of the DeleteMeeting operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkMeetingsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkMeetingsClient.DeleteMeeting
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/DeleteMeeting"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteMeetingResponse deleteMeeting(DeleteMeetingRequest deleteMeetingRequest) throws BadRequestException,
            ForbiddenException, AwsServiceException, SdkClientException, ChimeSdkMeetingsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteMeetingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteMeetingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMeetingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMeeting");

            return clientHandler.execute(new ClientExecutionParams<DeleteMeetingRequest, DeleteMeetingResponse>()
                    .withOperationName("DeleteMeeting").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteMeetingRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteMeetingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the Amazon Chime SDK attendee details for a specified meeting ID and attendee ID. For more information about
     * the Amazon Chime SDK, see <a href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the
     * Amazon Chime SDK</a> in the <i>Amazon Chime Developer Guide</i>.
     * </p>
     *
     * @param getAttendeeRequest
     * @return Result of the GetAttendee operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkMeetingsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkMeetingsClient.GetAttendee
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/GetAttendee"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAttendeeResponse getAttendee(GetAttendeeRequest getAttendeeRequest) throws BadRequestException, NotFoundException,
            ForbiddenException, AwsServiceException, SdkClientException, ChimeSdkMeetingsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAttendeeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAttendeeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAttendeeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAttendee");

            return clientHandler.execute(new ClientExecutionParams<GetAttendeeRequest, GetAttendeeResponse>()
                    .withOperationName("GetAttendee").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getAttendeeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAttendeeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the Amazon Chime SDK meeting details for the specified meeting ID. For more information about the Amazon
     * Chime SDK, see <a href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the Amazon Chime
     * SDK</a> in the <i>Amazon Chime Developer Guide</i>.
     * </p>
     *
     * @param getMeetingRequest
     * @return Result of the GetMeeting operation returned by the service.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkMeetingsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkMeetingsClient.GetMeeting
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/GetMeeting" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetMeetingResponse getMeeting(GetMeetingRequest getMeetingRequest) throws NotFoundException, BadRequestException,
            ForbiddenException, AwsServiceException, SdkClientException, ChimeSdkMeetingsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetMeetingResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetMeetingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMeetingRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMeeting");

            return clientHandler
                    .execute(new ClientExecutionParams<GetMeetingRequest, GetMeetingResponse>().withOperationName("GetMeeting")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getMeetingRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetMeetingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the attendees for the specified Amazon Chime SDK meeting. For more information about the Amazon Chime SDK,
     * see <a href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the Amazon Chime SDK</a> in the
     * <i>Amazon Chime Developer Guide</i>.
     * </p>
     *
     * @param listAttendeesRequest
     * @return Result of the ListAttendees operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkMeetingsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkMeetingsClient.ListAttendees
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/ListAttendees"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAttendeesResponse listAttendees(ListAttendeesRequest listAttendeesRequest) throws BadRequestException,
            NotFoundException, ForbiddenException, AwsServiceException, SdkClientException, ChimeSdkMeetingsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAttendeesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAttendeesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAttendeesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAttendees");

            return clientHandler.execute(new ClientExecutionParams<ListAttendeesRequest, ListAttendeesResponse>()
                    .withOperationName("ListAttendees").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listAttendeesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAttendeesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the attendees for the specified Amazon Chime SDK meeting. For more information about the Amazon Chime SDK,
     * see <a href="https://docs.aws.amazon.com/chime/latest/dg/meetings-sdk.html">Using the Amazon Chime SDK</a> in the
     * <i>Amazon Chime Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAttendees(software.amazon.awssdk.services.chimesdkmeetings.model.ListAttendeesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmeetings.paginators.ListAttendeesIterable responses = client.listAttendeesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.chimesdkmeetings.paginators.ListAttendeesIterable responses = client
     *             .listAttendeesPaginator(request);
     *     for (software.amazon.awssdk.services.chimesdkmeetings.model.ListAttendeesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmeetings.paginators.ListAttendeesIterable responses = client.listAttendeesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAttendees(software.amazon.awssdk.services.chimesdkmeetings.model.ListAttendeesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAttendeesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkMeetingsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkMeetingsClient.ListAttendees
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/ListAttendees"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListAttendeesIterable listAttendeesPaginator(ListAttendeesRequest listAttendeesRequest) throws BadRequestException,
            NotFoundException, ForbiddenException, AwsServiceException, SdkClientException, ChimeSdkMeetingsException {
        return new ListAttendeesIterable(this, applyPaginatorUserAgent(listAttendeesRequest));
    }

    /**
     * <p>
     * Starts transcription for the specified <code>meetingId</code>.
     * </p>
     *
     * @param startMeetingTranscriptionRequest
     * @return Result of the StartMeetingTranscription operation returned by the service.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws UnauthorizedException
     *         The user isn't authorized to request a resource.
     * @throws LimitExceededException
     *         The request exceeds the resource limit.
     * @throws UnprocessableEntityException
     *         The request was well-formed but was unable to be followed due to semantic errors.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkMeetingsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkMeetingsClient.StartMeetingTranscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/StartMeetingTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartMeetingTranscriptionResponse startMeetingTranscription(
            StartMeetingTranscriptionRequest startMeetingTranscriptionRequest) throws NotFoundException, ForbiddenException,
            BadRequestException, UnauthorizedException, LimitExceededException, UnprocessableEntityException,
            ServiceUnavailableException, AwsServiceException, SdkClientException, ChimeSdkMeetingsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartMeetingTranscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartMeetingTranscriptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMeetingTranscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMeetingTranscription");

            return clientHandler
                    .execute(new ClientExecutionParams<StartMeetingTranscriptionRequest, StartMeetingTranscriptionResponse>()
                            .withOperationName("StartMeetingTranscription").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(startMeetingTranscriptionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartMeetingTranscriptionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops transcription for the specified <code>meetingId</code>.
     * </p>
     *
     * @param stopMeetingTranscriptionRequest
     * @return Result of the StopMeetingTranscription operation returned by the service.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws UnauthorizedException
     *         The user isn't authorized to request a resource.
     * @throws UnprocessableEntityException
     *         The request was well-formed but was unable to be followed due to semantic errors.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeSdkMeetingsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeSdkMeetingsClient.StopMeetingTranscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-meetings-2021-07-15/StopMeetingTranscription"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StopMeetingTranscriptionResponse stopMeetingTranscription(
            StopMeetingTranscriptionRequest stopMeetingTranscriptionRequest) throws ForbiddenException, NotFoundException,
            BadRequestException, UnauthorizedException, UnprocessableEntityException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, ChimeSdkMeetingsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopMeetingTranscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StopMeetingTranscriptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopMeetingTranscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Meetings");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopMeetingTranscription");

            return clientHandler
                    .execute(new ClientExecutionParams<StopMeetingTranscriptionRequest, StopMeetingTranscriptionResponse>()
                            .withOperationName("StopMeetingTranscription").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(stopMeetingTranscriptionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StopMeetingTranscriptionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ChimeSdkMeetingsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnprocessableEntityException")
                                .exceptionBuilderSupplier(UnprocessableEntityException::builder).httpStatusCode(422).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends ChimeSdkMeetingsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
