/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chimesdkmessaging;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.chimesdkmessaging.model.AssociateChannelFlowRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.AssociateChannelFlowResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.BadRequestException;
import software.amazon.awssdk.services.chimesdkmessaging.model.BatchCreateChannelMembershipRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.BatchCreateChannelMembershipResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.ChannelFlowCallbackRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.ChannelFlowCallbackResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.ChimeSdkMessagingException;
import software.amazon.awssdk.services.chimesdkmessaging.model.ChimeSdkMessagingRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.ConflictException;
import software.amazon.awssdk.services.chimesdkmessaging.model.CreateChannelBanRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.CreateChannelBanResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.CreateChannelFlowRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.CreateChannelFlowResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.CreateChannelMembershipRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.CreateChannelMembershipResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.CreateChannelModeratorRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.CreateChannelModeratorResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.CreateChannelRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.CreateChannelResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.DeleteChannelBanRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.DeleteChannelBanResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.DeleteChannelFlowRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.DeleteChannelFlowResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.DeleteChannelMembershipRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.DeleteChannelMembershipResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.DeleteChannelMessageRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.DeleteChannelMessageResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.DeleteChannelModeratorRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.DeleteChannelModeratorResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.DeleteChannelRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.DeleteChannelResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.DescribeChannelBanRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.DescribeChannelBanResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.DescribeChannelFlowRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.DescribeChannelFlowResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.DescribeChannelMembershipForAppInstanceUserRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.DescribeChannelMembershipForAppInstanceUserResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.DescribeChannelMembershipRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.DescribeChannelMembershipResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.DescribeChannelModeratedByAppInstanceUserRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.DescribeChannelModeratedByAppInstanceUserResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.DescribeChannelModeratorRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.DescribeChannelModeratorResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.DescribeChannelRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.DescribeChannelResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.DisassociateChannelFlowRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.DisassociateChannelFlowResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.ForbiddenException;
import software.amazon.awssdk.services.chimesdkmessaging.model.GetChannelMembershipPreferencesRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.GetChannelMembershipPreferencesResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.GetChannelMessageRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.GetChannelMessageResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.GetChannelMessageStatusRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.GetChannelMessageStatusResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.GetMessagingSessionEndpointRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.GetMessagingSessionEndpointResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelBansRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelBansResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelFlowsRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelFlowsResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelMembershipsForAppInstanceUserRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelMembershipsForAppInstanceUserResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelMembershipsRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelMembershipsResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelMessagesRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelMessagesResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelModeratorsRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelModeratorsResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelsAssociatedWithChannelFlowRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelsAssociatedWithChannelFlowResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelsModeratedByAppInstanceUserRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelsModeratedByAppInstanceUserResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelsRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelsResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListSubChannelsRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListSubChannelsResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.NotFoundException;
import software.amazon.awssdk.services.chimesdkmessaging.model.PutChannelMembershipPreferencesRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.PutChannelMembershipPreferencesResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.RedactChannelMessageRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.RedactChannelMessageResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.chimesdkmessaging.model.SearchChannelsRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.SearchChannelsResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.SendChannelMessageRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.SendChannelMessageResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.ServiceFailureException;
import software.amazon.awssdk.services.chimesdkmessaging.model.ServiceUnavailableException;
import software.amazon.awssdk.services.chimesdkmessaging.model.TagResourceRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.TagResourceResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.ThrottledClientException;
import software.amazon.awssdk.services.chimesdkmessaging.model.UnauthorizedClientException;
import software.amazon.awssdk.services.chimesdkmessaging.model.UntagResourceRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.UntagResourceResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.UpdateChannelFlowRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.UpdateChannelFlowResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.UpdateChannelMessageRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.UpdateChannelMessageResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.UpdateChannelReadMarkerRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.UpdateChannelReadMarkerResponse;
import software.amazon.awssdk.services.chimesdkmessaging.model.UpdateChannelRequest;
import software.amazon.awssdk.services.chimesdkmessaging.model.UpdateChannelResponse;
import software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelBansPublisher;
import software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelFlowsPublisher;
import software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelMembershipsForAppInstanceUserPublisher;
import software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelMembershipsPublisher;
import software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelMessagesPublisher;
import software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelModeratorsPublisher;
import software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelsAssociatedWithChannelFlowPublisher;
import software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelsModeratedByAppInstanceUserPublisher;
import software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelsPublisher;
import software.amazon.awssdk.services.chimesdkmessaging.paginators.ListSubChannelsPublisher;
import software.amazon.awssdk.services.chimesdkmessaging.paginators.SearchChannelsPublisher;
import software.amazon.awssdk.services.chimesdkmessaging.transform.AssociateChannelFlowRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.BatchCreateChannelMembershipRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.ChannelFlowCallbackRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.CreateChannelBanRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.CreateChannelFlowRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.CreateChannelMembershipRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.CreateChannelModeratorRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.CreateChannelRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.DeleteChannelBanRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.DeleteChannelFlowRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.DeleteChannelMembershipRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.DeleteChannelMessageRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.DeleteChannelModeratorRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.DeleteChannelRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.DescribeChannelBanRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.DescribeChannelFlowRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.DescribeChannelMembershipForAppInstanceUserRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.DescribeChannelMembershipRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.DescribeChannelModeratedByAppInstanceUserRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.DescribeChannelModeratorRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.DescribeChannelRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.DisassociateChannelFlowRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.GetChannelMembershipPreferencesRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.GetChannelMessageRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.GetChannelMessageStatusRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.GetMessagingSessionEndpointRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.ListChannelBansRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.ListChannelFlowsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.ListChannelMembershipsForAppInstanceUserRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.ListChannelMembershipsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.ListChannelMessagesRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.ListChannelModeratorsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.ListChannelsAssociatedWithChannelFlowRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.ListChannelsModeratedByAppInstanceUserRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.ListChannelsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.ListSubChannelsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.PutChannelMembershipPreferencesRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.RedactChannelMessageRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.SearchChannelsRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.SendChannelMessageRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.UpdateChannelFlowRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.UpdateChannelMessageRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.UpdateChannelReadMarkerRequestMarshaller;
import software.amazon.awssdk.services.chimesdkmessaging.transform.UpdateChannelRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ChimeSdkMessagingAsyncClient}.
 *
 * @see ChimeSdkMessagingAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultChimeSdkMessagingAsyncClient implements ChimeSdkMessagingAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultChimeSdkMessagingAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultChimeSdkMessagingAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associates a channel flow with a channel. Once associated, all messages to that channel go through channel flow
     * processors. To stop processing, use the <code>DisassociateChannelFlow</code> API.
     * </p>
     * <note>
     * <p>
     * Only administrators or channel moderators can associate a channel flow. The <code>x-amz-chime-bearer</code>
     * request header is mandatory. Use the <code>AppInstanceUserArn</code> of the user that makes the API call as the
     * value in the header.
     * </p>
     * </note>
     *
     * @param associateChannelFlowRequest
     * @return A Java Future containing the result of the AssociateChannelFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.AssociateChannelFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/AssociateChannelFlow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateChannelFlowResponse> associateChannelFlow(
            AssociateChannelFlowRequest associateChannelFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateChannelFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateChannelFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateChannelFlowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateChannelFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateChannelFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateChannelFlowRequest, AssociateChannelFlowResponse>()
                            .withOperationName("AssociateChannelFlow")
                            .withMarshaller(new AssociateChannelFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(associateChannelFlowRequest));
            CompletableFuture<AssociateChannelFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a specified number of users to a channel.
     * </p>
     *
     * @param batchCreateChannelMembershipRequest
     * @return A Java Future containing the result of the BatchCreateChannelMembership operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.BatchCreateChannelMembership
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/BatchCreateChannelMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchCreateChannelMembershipResponse> batchCreateChannelMembership(
            BatchCreateChannelMembershipRequest batchCreateChannelMembershipRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchCreateChannelMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchCreateChannelMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchCreateChannelMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchCreateChannelMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchCreateChannelMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchCreateChannelMembershipRequest, BatchCreateChannelMembershipResponse>()
                            .withOperationName("BatchCreateChannelMembership")
                            .withMarshaller(new BatchCreateChannelMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchCreateChannelMembershipRequest));
            CompletableFuture<BatchCreateChannelMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Calls back Chime SDK Messaging with a processing response message. This should be invoked from the processor
     * Lambda. This is a developer API.
     * </p>
     * <p>
     * You can return one of the following processing responses:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Update message content or metadata
     * </p>
     * </li>
     * <li>
     * <p>
     * Deny a message
     * </p>
     * </li>
     * <li>
     * <p>
     * Make no changes to the message
     * </p>
     * </li>
     * </ul>
     *
     * @param channelFlowCallbackRequest
     * @return A Java Future containing the result of the ChannelFlowCallback operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ChannelFlowCallback
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ChannelFlowCallback"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ChannelFlowCallbackResponse> channelFlowCallback(
            ChannelFlowCallbackRequest channelFlowCallbackRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, channelFlowCallbackRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ChannelFlowCallback");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ChannelFlowCallbackResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ChannelFlowCallbackResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ChannelFlowCallbackResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ChannelFlowCallbackRequest, ChannelFlowCallbackResponse>()
                            .withOperationName("ChannelFlowCallback")
                            .withMarshaller(new ChannelFlowCallbackRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(channelFlowCallbackRequest));
            CompletableFuture<ChannelFlowCallbackResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a channel to which you can add users and send messages.
     * </p>
     * <p>
     * <b>Restriction</b>: You can't change a channel's privacy.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param createChannelRequest
     * @return A Java Future containing the result of the CreateChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.CreateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/CreateChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateChannelResponse> createChannel(CreateChannelRequest createChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateChannelRequest, CreateChannelResponse>()
                            .withOperationName("CreateChannel")
                            .withMarshaller(new CreateChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createChannelRequest));
            CompletableFuture<CreateChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Permanently bans a member from a channel. Moderators can't add banned members to a channel. To undo a ban, you
     * first have to <code>DeleteChannelBan</code>, and then <code>CreateChannelMembership</code>. Bans are cleaned up
     * when you delete users or channels.
     * </p>
     * <p>
     * If you ban a user who is already part of a channel, that user is automatically kicked from the channel.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param createChannelBanRequest
     * @return A Java Future containing the result of the CreateChannelBan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.CreateChannelBan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/CreateChannelBan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateChannelBanResponse> createChannelBan(CreateChannelBanRequest createChannelBanRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChannelBanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChannelBan");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateChannelBanResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateChannelBanResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateChannelBanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateChannelBanRequest, CreateChannelBanResponse>()
                            .withOperationName("CreateChannelBan")
                            .withMarshaller(new CreateChannelBanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createChannelBanRequest));
            CompletableFuture<CreateChannelBanResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a channel flow, a container for processors. Processors are AWS Lambda functions that perform actions on
     * chat messages, such as stripping out profanity. You can associate channel flows with channels, and the processors
     * in the channel flow then take action on all messages sent to that channel. This is a developer API.
     * </p>
     * <p>
     * Channel flows process the following items:
     * </p>
     * <ol>
     * <li>
     * <p>
     * New and updated messages
     * </p>
     * </li>
     * <li>
     * <p>
     * Persistent and non-persistent messages
     * </p>
     * </li>
     * <li>
     * <p>
     * The Standard message type
     * </p>
     * </li>
     * </ol>
     * <note>
     * <p>
     * Channel flows don't process Control or System messages. For more information about the message types provided by
     * Chime SDK Messaging, refer to <a
     * href="https://docs.aws.amazon.com/chime/latest/dg/using-the-messaging-sdk.html#msg-types">Message types</a> in
     * the <i>Amazon Chime developer guide</i>.
     * </p>
     * </note>
     *
     * @param createChannelFlowRequest
     * @return A Java Future containing the result of the CreateChannelFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.CreateChannelFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/CreateChannelFlow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateChannelFlowResponse> createChannelFlow(CreateChannelFlowRequest createChannelFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChannelFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChannelFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateChannelFlowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateChannelFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateChannelFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateChannelFlowRequest, CreateChannelFlowResponse>()
                            .withOperationName("CreateChannelFlow")
                            .withMarshaller(new CreateChannelFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createChannelFlowRequest));
            CompletableFuture<CreateChannelFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a user to a channel. The <code>InvitedBy</code> field in <code>ChannelMembership</code> is derived from the
     * request header. A channel member can:
     * </p>
     * <ul>
     * <li>
     * <p>
     * List messages
     * </p>
     * </li>
     * <li>
     * <p>
     * Send messages
     * </p>
     * </li>
     * <li>
     * <p>
     * Receive messages
     * </p>
     * </li>
     * <li>
     * <p>
     * Edit their own messages
     * </p>
     * </li>
     * <li>
     * <p>
     * Leave the channel
     * </p>
     * </li>
     * </ul>
     * <p>
     * Privacy settings impact this action as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Public Channels: You do not need to be a member to list messages, but you must be a member to send messages.
     * </p>
     * </li>
     * <li>
     * <p>
     * Private Channels: You must be a member to list or send messages.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param createChannelMembershipRequest
     * @return A Java Future containing the result of the CreateChannelMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.CreateChannelMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/CreateChannelMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateChannelMembershipResponse> createChannelMembership(
            CreateChannelMembershipRequest createChannelMembershipRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChannelMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChannelMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateChannelMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateChannelMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateChannelMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateChannelMembershipRequest, CreateChannelMembershipResponse>()
                            .withOperationName("CreateChannelMembership")
                            .withMarshaller(new CreateChannelMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createChannelMembershipRequest));
            CompletableFuture<CreateChannelMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new <code>ChannelModerator</code>. A channel moderator can:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Add and remove other members of the channel.
     * </p>
     * </li>
     * <li>
     * <p>
     * Add and remove other moderators of the channel.
     * </p>
     * </li>
     * <li>
     * <p>
     * Add and remove user bans for the channel.
     * </p>
     * </li>
     * <li>
     * <p>
     * Redact messages in the channel.
     * </p>
     * </li>
     * <li>
     * <p>
     * List messages in the channel.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param createChannelModeratorRequest
     * @return A Java Future containing the result of the CreateChannelModerator operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.CreateChannelModerator
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/CreateChannelModerator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateChannelModeratorResponse> createChannelModerator(
            CreateChannelModeratorRequest createChannelModeratorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createChannelModeratorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChannelModerator");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateChannelModeratorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateChannelModeratorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateChannelModeratorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateChannelModeratorRequest, CreateChannelModeratorResponse>()
                            .withOperationName("CreateChannelModerator")
                            .withMarshaller(new CreateChannelModeratorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createChannelModeratorRequest));
            CompletableFuture<CreateChannelModeratorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Immediately makes a channel and its memberships inaccessible and marks them for deletion. This is an irreversible
     * process.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param deleteChannelRequest
     * @return A Java Future containing the result of the DeleteChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.DeleteChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/DeleteChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteChannelResponse> deleteChannel(DeleteChannelRequest deleteChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteChannelRequest, DeleteChannelResponse>()
                            .withOperationName("DeleteChannel")
                            .withMarshaller(new DeleteChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteChannelRequest));
            CompletableFuture<DeleteChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a user from a channel's ban list.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param deleteChannelBanRequest
     * @return A Java Future containing the result of the DeleteChannelBan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.DeleteChannelBan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/DeleteChannelBan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteChannelBanResponse> deleteChannelBan(DeleteChannelBanRequest deleteChannelBanRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelBanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannelBan");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteChannelBanResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteChannelBanResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteChannelBanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteChannelBanRequest, DeleteChannelBanResponse>()
                            .withOperationName("DeleteChannelBan")
                            .withMarshaller(new DeleteChannelBanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteChannelBanRequest));
            CompletableFuture<DeleteChannelBanResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a channel flow, an irreversible process. This is a developer API.
     * </p>
     * <note>
     * <p>
     * This API works only when the channel flow is not associated with any channel. To get a list of all channels that
     * a channel flow is associated with, use the <code>ListChannelsAssociatedWithChannelFlow</code> API. Use the
     * <code>DisassociateChannelFlow</code> API to disassociate a channel flow from all channels.
     * </p>
     * </note>
     *
     * @param deleteChannelFlowRequest
     * @return A Java Future containing the result of the DeleteChannelFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.DeleteChannelFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/DeleteChannelFlow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteChannelFlowResponse> deleteChannelFlow(DeleteChannelFlowRequest deleteChannelFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannelFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteChannelFlowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteChannelFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteChannelFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteChannelFlowRequest, DeleteChannelFlowResponse>()
                            .withOperationName("DeleteChannelFlow")
                            .withMarshaller(new DeleteChannelFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteChannelFlowRequest));
            CompletableFuture<DeleteChannelFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a member from a channel.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param deleteChannelMembershipRequest
     * @return A Java Future containing the result of the DeleteChannelMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.DeleteChannelMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/DeleteChannelMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteChannelMembershipResponse> deleteChannelMembership(
            DeleteChannelMembershipRequest deleteChannelMembershipRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannelMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteChannelMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteChannelMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteChannelMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteChannelMembershipRequest, DeleteChannelMembershipResponse>()
                            .withOperationName("DeleteChannelMembership")
                            .withMarshaller(new DeleteChannelMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteChannelMembershipRequest));
            CompletableFuture<DeleteChannelMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a channel message. Only admins can perform this action. Deletion makes messages inaccessible immediately.
     * A background process deletes any revisions created by <code>UpdateChannelMessage</code>.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param deleteChannelMessageRequest
     * @return A Java Future containing the result of the DeleteChannelMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.DeleteChannelMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/DeleteChannelMessage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteChannelMessageResponse> deleteChannelMessage(
            DeleteChannelMessageRequest deleteChannelMessageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannelMessage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteChannelMessageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteChannelMessageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteChannelMessageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteChannelMessageRequest, DeleteChannelMessageResponse>()
                            .withOperationName("DeleteChannelMessage")
                            .withMarshaller(new DeleteChannelMessageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteChannelMessageRequest));
            CompletableFuture<DeleteChannelMessageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a channel moderator.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param deleteChannelModeratorRequest
     * @return A Java Future containing the result of the DeleteChannelModerator operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.DeleteChannelModerator
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/DeleteChannelModerator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteChannelModeratorResponse> deleteChannelModerator(
            DeleteChannelModeratorRequest deleteChannelModeratorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteChannelModeratorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChannelModerator");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteChannelModeratorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteChannelModeratorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteChannelModeratorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteChannelModeratorRequest, DeleteChannelModeratorResponse>()
                            .withOperationName("DeleteChannelModerator")
                            .withMarshaller(new DeleteChannelModeratorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteChannelModeratorRequest));
            CompletableFuture<DeleteChannelModeratorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the full details of a channel in an Amazon Chime <code>AppInstance</code>.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param describeChannelRequest
     * @return A Java Future containing the result of the DescribeChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.DescribeChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/DescribeChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeChannelResponse> describeChannel(DescribeChannelRequest describeChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeChannelRequest, DescribeChannelResponse>()
                            .withOperationName("DescribeChannel")
                            .withMarshaller(new DescribeChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeChannelRequest));
            CompletableFuture<DescribeChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the full details of a channel ban.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param describeChannelBanRequest
     * @return A Java Future containing the result of the DescribeChannelBan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.DescribeChannelBan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/DescribeChannelBan"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeChannelBanResponse> describeChannelBan(DescribeChannelBanRequest describeChannelBanRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeChannelBanRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeChannelBan");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeChannelBanResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeChannelBanResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeChannelBanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeChannelBanRequest, DescribeChannelBanResponse>()
                            .withOperationName("DescribeChannelBan")
                            .withMarshaller(new DescribeChannelBanRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeChannelBanRequest));
            CompletableFuture<DescribeChannelBanResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the full details of a channel flow in an Amazon Chime <code>AppInstance</code>. This is a developer API.
     * </p>
     *
     * @param describeChannelFlowRequest
     * @return A Java Future containing the result of the DescribeChannelFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.DescribeChannelFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/DescribeChannelFlow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeChannelFlowResponse> describeChannelFlow(
            DescribeChannelFlowRequest describeChannelFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeChannelFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeChannelFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeChannelFlowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeChannelFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeChannelFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeChannelFlowRequest, DescribeChannelFlowResponse>()
                            .withOperationName("DescribeChannelFlow")
                            .withMarshaller(new DescribeChannelFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeChannelFlowRequest));
            CompletableFuture<DescribeChannelFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the full details of a user's channel membership.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param describeChannelMembershipRequest
     * @return A Java Future containing the result of the DescribeChannelMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.DescribeChannelMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/DescribeChannelMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeChannelMembershipResponse> describeChannelMembership(
            DescribeChannelMembershipRequest describeChannelMembershipRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeChannelMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeChannelMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeChannelMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeChannelMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeChannelMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeChannelMembershipRequest, DescribeChannelMembershipResponse>()
                            .withOperationName("DescribeChannelMembership")
                            .withMarshaller(new DescribeChannelMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeChannelMembershipRequest));
            CompletableFuture<DescribeChannelMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the details of a channel based on the membership of the specified <code>AppInstanceUser</code>.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param describeChannelMembershipForAppInstanceUserRequest
     * @return A Java Future containing the result of the DescribeChannelMembershipForAppInstanceUser operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.DescribeChannelMembershipForAppInstanceUser
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/DescribeChannelMembershipForAppInstanceUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeChannelMembershipForAppInstanceUserResponse> describeChannelMembershipForAppInstanceUser(
            DescribeChannelMembershipForAppInstanceUserRequest describeChannelMembershipForAppInstanceUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeChannelMembershipForAppInstanceUserRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeChannelMembershipForAppInstanceUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeChannelMembershipForAppInstanceUserResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeChannelMembershipForAppInstanceUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeChannelMembershipForAppInstanceUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeChannelMembershipForAppInstanceUserRequest, DescribeChannelMembershipForAppInstanceUserResponse>()
                            .withOperationName("DescribeChannelMembershipForAppInstanceUser")
                            .withMarshaller(new DescribeChannelMembershipForAppInstanceUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(describeChannelMembershipForAppInstanceUserRequest));
            CompletableFuture<DescribeChannelMembershipForAppInstanceUserResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the full details of a channel moderated by the specified <code>AppInstanceUser</code>.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param describeChannelModeratedByAppInstanceUserRequest
     * @return A Java Future containing the result of the DescribeChannelModeratedByAppInstanceUser operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.DescribeChannelModeratedByAppInstanceUser
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/DescribeChannelModeratedByAppInstanceUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeChannelModeratedByAppInstanceUserResponse> describeChannelModeratedByAppInstanceUser(
            DescribeChannelModeratedByAppInstanceUserRequest describeChannelModeratedByAppInstanceUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeChannelModeratedByAppInstanceUserRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeChannelModeratedByAppInstanceUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeChannelModeratedByAppInstanceUserResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeChannelModeratedByAppInstanceUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeChannelModeratedByAppInstanceUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeChannelModeratedByAppInstanceUserRequest, DescribeChannelModeratedByAppInstanceUserResponse>()
                            .withOperationName("DescribeChannelModeratedByAppInstanceUser")
                            .withMarshaller(new DescribeChannelModeratedByAppInstanceUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(describeChannelModeratedByAppInstanceUserRequest));
            CompletableFuture<DescribeChannelModeratedByAppInstanceUserResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the full details of a single ChannelModerator.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param describeChannelModeratorRequest
     * @return A Java Future containing the result of the DescribeChannelModerator operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.DescribeChannelModerator
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/DescribeChannelModerator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeChannelModeratorResponse> describeChannelModerator(
            DescribeChannelModeratorRequest describeChannelModeratorRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeChannelModeratorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeChannelModerator");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeChannelModeratorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeChannelModeratorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeChannelModeratorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeChannelModeratorRequest, DescribeChannelModeratorResponse>()
                            .withOperationName("DescribeChannelModerator")
                            .withMarshaller(new DescribeChannelModeratorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeChannelModeratorRequest));
            CompletableFuture<DescribeChannelModeratorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disassociates a channel flow from all its channels. Once disassociated, all messages to that channel stop going
     * through the channel flow processor.
     * </p>
     * <note>
     * <p>
     * Only administrators or channel moderators can disassociate a channel flow. The <code>x-amz-chime-bearer</code>
     * request header is mandatory. Use the <code>AppInstanceUserArn</code> of the user that makes the API call as the
     * value in the header.
     * </p>
     * </note>
     *
     * @param disassociateChannelFlowRequest
     * @return A Java Future containing the result of the DisassociateChannelFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.DisassociateChannelFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/DisassociateChannelFlow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateChannelFlowResponse> disassociateChannelFlow(
            DisassociateChannelFlowRequest disassociateChannelFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateChannelFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateChannelFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateChannelFlowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateChannelFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateChannelFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateChannelFlowRequest, DisassociateChannelFlowResponse>()
                            .withOperationName("DisassociateChannelFlow")
                            .withMarshaller(new DisassociateChannelFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateChannelFlowRequest));
            CompletableFuture<DisassociateChannelFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the membership preferences of an <code>AppInstanceUser</code> for the specified channel. The
     * <code>AppInstanceUser</code> must be a member of the channel. Only the <code>AppInstanceUser</code> who owns the
     * membership can retrieve preferences. Users in the <code>AppInstanceAdmin</code> and channel moderator roles can't
     * retrieve preferences for other users. Banned users can't retrieve membership preferences for the channel from
     * which they are banned.
     * </p>
     *
     * @param getChannelMembershipPreferencesRequest
     * @return A Java Future containing the result of the GetChannelMembershipPreferences operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.GetChannelMembershipPreferences
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/GetChannelMembershipPreferences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetChannelMembershipPreferencesResponse> getChannelMembershipPreferences(
            GetChannelMembershipPreferencesRequest getChannelMembershipPreferencesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getChannelMembershipPreferencesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChannelMembershipPreferences");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetChannelMembershipPreferencesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetChannelMembershipPreferencesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetChannelMembershipPreferencesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetChannelMembershipPreferencesRequest, GetChannelMembershipPreferencesResponse>()
                            .withOperationName("GetChannelMembershipPreferences")
                            .withMarshaller(new GetChannelMembershipPreferencesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getChannelMembershipPreferencesRequest));
            CompletableFuture<GetChannelMembershipPreferencesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the full details of a channel message.
     * </p>
     * <note>
     * <p>
     * The x-amz-chime-bearer request header is mandatory. Use the <code>AppInstanceUserArn</code> of the user that
     * makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param getChannelMessageRequest
     * @return A Java Future containing the result of the GetChannelMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>NotFoundException One or more of the resources in the request does not exist in the system.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.GetChannelMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/GetChannelMessage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetChannelMessageResponse> getChannelMessage(GetChannelMessageRequest getChannelMessageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChannelMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChannelMessage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetChannelMessageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetChannelMessageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetChannelMessageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetChannelMessageRequest, GetChannelMessageResponse>()
                            .withOperationName("GetChannelMessage")
                            .withMarshaller(new GetChannelMessageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getChannelMessageRequest));
            CompletableFuture<GetChannelMessageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets message status for a specified <code>messageId</code>. Use this API to determine the intermediate status of
     * messages going through channel flow processing. The API provides an alternative to retrieving message status if
     * the event was not received because a client wasn't connected to a websocket.
     * </p>
     * <p>
     * Messages can have any one of these statuses.
     * </p>
     * <dl>
     * <dt>SENT</dt>
     * <dd>
     * <p>
     * Message processed successfully
     * </p>
     * </dd>
     * <dt>PENDING</dt>
     * <dd>
     * <p>
     * Ongoing processing
     * </p>
     * </dd>
     * <dt>FAILED</dt>
     * <dd>
     * <p>
     * Processing failed
     * </p>
     * </dd>
     * <dt>DENIED</dt>
     * <dd>
     * <p>
     * Messasge denied by the processor
     * </p>
     * </dd>
     * </dl>
     * <note>
     * <ul>
     * <li>
     * <p>
     * This API does not return statuses for denied messages, because we don't store them once the processor denies
     * them.
     * </p>
     * </li>
     * <li>
     * <p>
     * Only the message sender can invoke this API.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header
     * </p>
     * </li>
     * </ul>
     * </note>
     *
     * @param getChannelMessageStatusRequest
     * @return A Java Future containing the result of the GetChannelMessageStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.GetChannelMessageStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/GetChannelMessageStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetChannelMessageStatusResponse> getChannelMessageStatus(
            GetChannelMessageStatusRequest getChannelMessageStatusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getChannelMessageStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetChannelMessageStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetChannelMessageStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetChannelMessageStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetChannelMessageStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetChannelMessageStatusRequest, GetChannelMessageStatusResponse>()
                            .withOperationName("GetChannelMessageStatus")
                            .withMarshaller(new GetChannelMessageStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getChannelMessageStatusRequest));
            CompletableFuture<GetChannelMessageStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The details of the endpoint for the messaging session.
     * </p>
     *
     * @param getMessagingSessionEndpointRequest
     * @return A Java Future containing the result of the GetMessagingSessionEndpoint operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.GetMessagingSessionEndpoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/GetMessagingSessionEndpoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMessagingSessionEndpointResponse> getMessagingSessionEndpoint(
            GetMessagingSessionEndpointRequest getMessagingSessionEndpointRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMessagingSessionEndpointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMessagingSessionEndpoint");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMessagingSessionEndpointResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetMessagingSessionEndpointResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMessagingSessionEndpointResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMessagingSessionEndpointRequest, GetMessagingSessionEndpointResponse>()
                            .withOperationName("GetMessagingSessionEndpoint")
                            .withMarshaller(new GetMessagingSessionEndpointRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getMessagingSessionEndpointRequest));
            CompletableFuture<GetMessagingSessionEndpointResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the users banned from a particular channel.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param listChannelBansRequest
     * @return A Java Future containing the result of the ListChannelBans operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListChannelBans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListChannelBans"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListChannelBansResponse> listChannelBans(ListChannelBansRequest listChannelBansRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelBansRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannelBans");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChannelBansResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListChannelBansResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChannelBansResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChannelBansRequest, ListChannelBansResponse>()
                            .withOperationName("ListChannelBans")
                            .withMarshaller(new ListChannelBansRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listChannelBansRequest));
            CompletableFuture<ListChannelBansResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the users banned from a particular channel.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listChannelBans(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelBansRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelBansPublisher publisher = client.listChannelBansPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelBansPublisher publisher = client.listChannelBansPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelBansResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelBansResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChannelBans(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelBansRequest)}
     * operation.</b>
     * </p>
     *
     * @param listChannelBansRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListChannelBans
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListChannelBans"
     *      target="_top">AWS API Documentation</a>
     */
    public ListChannelBansPublisher listChannelBansPaginator(ListChannelBansRequest listChannelBansRequest) {
        return new ListChannelBansPublisher(this, applyPaginatorUserAgent(listChannelBansRequest));
    }

    /**
     * <p>
     * Returns a paginated lists of all the channel flows created under a single Chime. This is a developer API.
     * </p>
     *
     * @param listChannelFlowsRequest
     * @return A Java Future containing the result of the ListChannelFlows operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListChannelFlows
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListChannelFlows"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListChannelFlowsResponse> listChannelFlows(ListChannelFlowsRequest listChannelFlowsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelFlowsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannelFlows");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChannelFlowsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListChannelFlowsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChannelFlowsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChannelFlowsRequest, ListChannelFlowsResponse>()
                            .withOperationName("ListChannelFlows")
                            .withMarshaller(new ListChannelFlowsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listChannelFlowsRequest));
            CompletableFuture<ListChannelFlowsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a paginated lists of all the channel flows created under a single Chime. This is a developer API.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listChannelFlows(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelFlowsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelFlowsPublisher publisher = client.listChannelFlowsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelFlowsPublisher publisher = client.listChannelFlowsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelFlowsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelFlowsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChannelFlows(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelFlowsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listChannelFlowsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListChannelFlows
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListChannelFlows"
     *      target="_top">AWS API Documentation</a>
     */
    public ListChannelFlowsPublisher listChannelFlowsPaginator(ListChannelFlowsRequest listChannelFlowsRequest) {
        return new ListChannelFlowsPublisher(this, applyPaginatorUserAgent(listChannelFlowsRequest));
    }

    /**
     * <p>
     * Lists all channel memberships in a channel.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     * <p>
     * If you want to list the channels to which a specific app instance user belongs, see the <a href=
     * "https://docs.aws.amazon.com/chime/latest/APIReference/API_messaging-chime_ListChannelMembershipsForAppInstanceUser.html"
     * >ListChannelMembershipsForAppInstanceUser</a> API.
     * </p>
     *
     * @param listChannelMembershipsRequest
     * @return A Java Future containing the result of the ListChannelMemberships operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListChannelMemberships
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListChannelMemberships"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListChannelMembershipsResponse> listChannelMemberships(
            ListChannelMembershipsRequest listChannelMembershipsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelMembershipsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannelMemberships");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChannelMembershipsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListChannelMembershipsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChannelMembershipsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChannelMembershipsRequest, ListChannelMembershipsResponse>()
                            .withOperationName("ListChannelMemberships")
                            .withMarshaller(new ListChannelMembershipsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listChannelMembershipsRequest));
            CompletableFuture<ListChannelMembershipsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all channels that a particular <code>AppInstanceUser</code> is a part of. Only an
     * <code>AppInstanceAdmin</code> can call the API with a user ARN that is not their own.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param listChannelMembershipsForAppInstanceUserRequest
     * @return A Java Future containing the result of the ListChannelMembershipsForAppInstanceUser operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListChannelMembershipsForAppInstanceUser
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListChannelMembershipsForAppInstanceUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListChannelMembershipsForAppInstanceUserResponse> listChannelMembershipsForAppInstanceUser(
            ListChannelMembershipsForAppInstanceUserRequest listChannelMembershipsForAppInstanceUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listChannelMembershipsForAppInstanceUserRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannelMembershipsForAppInstanceUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChannelMembershipsForAppInstanceUserResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListChannelMembershipsForAppInstanceUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChannelMembershipsForAppInstanceUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChannelMembershipsForAppInstanceUserRequest, ListChannelMembershipsForAppInstanceUserResponse>()
                            .withOperationName("ListChannelMembershipsForAppInstanceUser")
                            .withMarshaller(new ListChannelMembershipsForAppInstanceUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(listChannelMembershipsForAppInstanceUserRequest));
            CompletableFuture<ListChannelMembershipsForAppInstanceUserResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all channels that a particular <code>AppInstanceUser</code> is a part of. Only an
     * <code>AppInstanceAdmin</code> can call the API with a user ARN that is not their own.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listChannelMembershipsForAppInstanceUser(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelMembershipsForAppInstanceUserRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelMembershipsForAppInstanceUserPublisher publisher = client.listChannelMembershipsForAppInstanceUserPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelMembershipsForAppInstanceUserPublisher publisher = client.listChannelMembershipsForAppInstanceUserPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelMembershipsForAppInstanceUserResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelMembershipsForAppInstanceUserResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChannelMembershipsForAppInstanceUser(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelMembershipsForAppInstanceUserRequest)}
     * operation.</b>
     * </p>
     *
     * @param listChannelMembershipsForAppInstanceUserRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListChannelMembershipsForAppInstanceUser
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListChannelMembershipsForAppInstanceUser"
     *      target="_top">AWS API Documentation</a>
     */
    public ListChannelMembershipsForAppInstanceUserPublisher listChannelMembershipsForAppInstanceUserPaginator(
            ListChannelMembershipsForAppInstanceUserRequest listChannelMembershipsForAppInstanceUserRequest) {
        return new ListChannelMembershipsForAppInstanceUserPublisher(this,
                applyPaginatorUserAgent(listChannelMembershipsForAppInstanceUserRequest));
    }

    /**
     * <p>
     * Lists all channel memberships in a channel.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     * <p>
     * If you want to list the channels to which a specific app instance user belongs, see the <a href=
     * "https://docs.aws.amazon.com/chime/latest/APIReference/API_messaging-chime_ListChannelMembershipsForAppInstanceUser.html"
     * >ListChannelMembershipsForAppInstanceUser</a> API.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listChannelMemberships(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelMembershipsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelMembershipsPublisher publisher = client.listChannelMembershipsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelMembershipsPublisher publisher = client.listChannelMembershipsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelMembershipsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelMembershipsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChannelMemberships(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelMembershipsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listChannelMembershipsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListChannelMemberships
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListChannelMemberships"
     *      target="_top">AWS API Documentation</a>
     */
    public ListChannelMembershipsPublisher listChannelMembershipsPaginator(
            ListChannelMembershipsRequest listChannelMembershipsRequest) {
        return new ListChannelMembershipsPublisher(this, applyPaginatorUserAgent(listChannelMembershipsRequest));
    }

    /**
     * <p>
     * List all the messages in a channel. Returns a paginated list of <code>ChannelMessages</code>. By default, sorted
     * by creation timestamp in descending order.
     * </p>
     * <note>
     * <p>
     * Redacted messages appear in the results as empty, since they are only redacted, not deleted. Deleted messages do
     * not appear in the results. This action always returns the latest version of an edited message.
     * </p>
     * <p>
     * Also, the x-amz-chime-bearer request header is mandatory. Use the <code>AppInstanceUserArn</code> of the user
     * that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param listChannelMessagesRequest
     * @return A Java Future containing the result of the ListChannelMessages operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListChannelMessages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListChannelMessages"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListChannelMessagesResponse> listChannelMessages(
            ListChannelMessagesRequest listChannelMessagesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelMessagesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannelMessages");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChannelMessagesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListChannelMessagesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChannelMessagesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChannelMessagesRequest, ListChannelMessagesResponse>()
                            .withOperationName("ListChannelMessages")
                            .withMarshaller(new ListChannelMessagesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listChannelMessagesRequest));
            CompletableFuture<ListChannelMessagesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all the messages in a channel. Returns a paginated list of <code>ChannelMessages</code>. By default, sorted
     * by creation timestamp in descending order.
     * </p>
     * <note>
     * <p>
     * Redacted messages appear in the results as empty, since they are only redacted, not deleted. Deleted messages do
     * not appear in the results. This action always returns the latest version of an edited message.
     * </p>
     * <p>
     * Also, the x-amz-chime-bearer request header is mandatory. Use the <code>AppInstanceUserArn</code> of the user
     * that makes the API call as the value in the header.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listChannelMessages(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelMessagesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelMessagesPublisher publisher = client.listChannelMessagesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelMessagesPublisher publisher = client.listChannelMessagesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelMessagesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelMessagesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChannelMessages(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelMessagesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listChannelMessagesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListChannelMessages
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListChannelMessages"
     *      target="_top">AWS API Documentation</a>
     */
    public ListChannelMessagesPublisher listChannelMessagesPaginator(ListChannelMessagesRequest listChannelMessagesRequest) {
        return new ListChannelMessagesPublisher(this, applyPaginatorUserAgent(listChannelMessagesRequest));
    }

    /**
     * <p>
     * Lists all the moderators for a channel.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param listChannelModeratorsRequest
     * @return A Java Future containing the result of the ListChannelModerators operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListChannelModerators
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListChannelModerators"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListChannelModeratorsResponse> listChannelModerators(
            ListChannelModeratorsRequest listChannelModeratorsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelModeratorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannelModerators");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChannelModeratorsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListChannelModeratorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChannelModeratorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChannelModeratorsRequest, ListChannelModeratorsResponse>()
                            .withOperationName("ListChannelModerators")
                            .withMarshaller(new ListChannelModeratorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listChannelModeratorsRequest));
            CompletableFuture<ListChannelModeratorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the moderators for a channel.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listChannelModerators(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelModeratorsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelModeratorsPublisher publisher = client.listChannelModeratorsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelModeratorsPublisher publisher = client.listChannelModeratorsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelModeratorsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelModeratorsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChannelModerators(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelModeratorsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listChannelModeratorsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListChannelModerators
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListChannelModerators"
     *      target="_top">AWS API Documentation</a>
     */
    public ListChannelModeratorsPublisher listChannelModeratorsPaginator(ListChannelModeratorsRequest listChannelModeratorsRequest) {
        return new ListChannelModeratorsPublisher(this, applyPaginatorUserAgent(listChannelModeratorsRequest));
    }

    /**
     * <p>
     * Lists all Channels created under a single Chime App as a paginated list. You can specify filters to narrow
     * results.
     * </p>
     * <p class="title">
     * <b>Functionality &amp; restrictions</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use privacy = <code>PUBLIC</code> to retrieve all public channels in the account.
     * </p>
     * </li>
     * <li>
     * <p>
     * Only an <code>AppInstanceAdmin</code> can set privacy = <code>PRIVATE</code> to list the private channels in an
     * account.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param listChannelsRequest
     * @return A Java Future containing the result of the ListChannels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListChannels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListChannelsResponse> listChannels(ListChannelsRequest listChannelsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChannelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListChannelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChannelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChannelsRequest, ListChannelsResponse>()
                            .withOperationName("ListChannels").withMarshaller(new ListChannelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listChannelsRequest));
            CompletableFuture<ListChannelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all channels associated with a specified channel flow. You can associate a channel flow with multiple
     * channels, but you can only associate a channel with one channel flow. This is a developer API.
     * </p>
     *
     * @param listChannelsAssociatedWithChannelFlowRequest
     * @return A Java Future containing the result of the ListChannelsAssociatedWithChannelFlow operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListChannelsAssociatedWithChannelFlow
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListChannelsAssociatedWithChannelFlow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListChannelsAssociatedWithChannelFlowResponse> listChannelsAssociatedWithChannelFlow(
            ListChannelsAssociatedWithChannelFlowRequest listChannelsAssociatedWithChannelFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listChannelsAssociatedWithChannelFlowRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannelsAssociatedWithChannelFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChannelsAssociatedWithChannelFlowResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListChannelsAssociatedWithChannelFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChannelsAssociatedWithChannelFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChannelsAssociatedWithChannelFlowRequest, ListChannelsAssociatedWithChannelFlowResponse>()
                            .withOperationName("ListChannelsAssociatedWithChannelFlow")
                            .withMarshaller(new ListChannelsAssociatedWithChannelFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listChannelsAssociatedWithChannelFlowRequest));
            CompletableFuture<ListChannelsAssociatedWithChannelFlowResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all channels associated with a specified channel flow. You can associate a channel flow with multiple
     * channels, but you can only associate a channel with one channel flow. This is a developer API.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listChannelsAssociatedWithChannelFlow(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelsAssociatedWithChannelFlowRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelsAssociatedWithChannelFlowPublisher publisher = client.listChannelsAssociatedWithChannelFlowPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelsAssociatedWithChannelFlowPublisher publisher = client.listChannelsAssociatedWithChannelFlowPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelsAssociatedWithChannelFlowResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelsAssociatedWithChannelFlowResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChannelsAssociatedWithChannelFlow(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelsAssociatedWithChannelFlowRequest)}
     * operation.</b>
     * </p>
     *
     * @param listChannelsAssociatedWithChannelFlowRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListChannelsAssociatedWithChannelFlow
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListChannelsAssociatedWithChannelFlow"
     *      target="_top">AWS API Documentation</a>
     */
    public ListChannelsAssociatedWithChannelFlowPublisher listChannelsAssociatedWithChannelFlowPaginator(
            ListChannelsAssociatedWithChannelFlowRequest listChannelsAssociatedWithChannelFlowRequest) {
        return new ListChannelsAssociatedWithChannelFlowPublisher(this,
                applyPaginatorUserAgent(listChannelsAssociatedWithChannelFlowRequest));
    }

    /**
     * <p>
     * A list of the channels moderated by an <code>AppInstanceUser</code>.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param listChannelsModeratedByAppInstanceUserRequest
     * @return A Java Future containing the result of the ListChannelsModeratedByAppInstanceUser operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListChannelsModeratedByAppInstanceUser
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListChannelsModeratedByAppInstanceUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListChannelsModeratedByAppInstanceUserResponse> listChannelsModeratedByAppInstanceUser(
            ListChannelsModeratedByAppInstanceUserRequest listChannelsModeratedByAppInstanceUserRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listChannelsModeratedByAppInstanceUserRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListChannelsModeratedByAppInstanceUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListChannelsModeratedByAppInstanceUserResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListChannelsModeratedByAppInstanceUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListChannelsModeratedByAppInstanceUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListChannelsModeratedByAppInstanceUserRequest, ListChannelsModeratedByAppInstanceUserResponse>()
                            .withOperationName("ListChannelsModeratedByAppInstanceUser")
                            .withMarshaller(new ListChannelsModeratedByAppInstanceUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listChannelsModeratedByAppInstanceUserRequest));
            CompletableFuture<ListChannelsModeratedByAppInstanceUserResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A list of the channels moderated by an <code>AppInstanceUser</code>.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listChannelsModeratedByAppInstanceUser(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelsModeratedByAppInstanceUserRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelsModeratedByAppInstanceUserPublisher publisher = client.listChannelsModeratedByAppInstanceUserPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelsModeratedByAppInstanceUserPublisher publisher = client.listChannelsModeratedByAppInstanceUserPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelsModeratedByAppInstanceUserResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelsModeratedByAppInstanceUserResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChannelsModeratedByAppInstanceUser(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelsModeratedByAppInstanceUserRequest)}
     * operation.</b>
     * </p>
     *
     * @param listChannelsModeratedByAppInstanceUserRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListChannelsModeratedByAppInstanceUser
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListChannelsModeratedByAppInstanceUser"
     *      target="_top">AWS API Documentation</a>
     */
    public ListChannelsModeratedByAppInstanceUserPublisher listChannelsModeratedByAppInstanceUserPaginator(
            ListChannelsModeratedByAppInstanceUserRequest listChannelsModeratedByAppInstanceUserRequest) {
        return new ListChannelsModeratedByAppInstanceUserPublisher(this,
                applyPaginatorUserAgent(listChannelsModeratedByAppInstanceUserRequest));
    }

    /**
     * <p>
     * Lists all Channels created under a single Chime App as a paginated list. You can specify filters to narrow
     * results.
     * </p>
     * <p class="title">
     * <b>Functionality &amp; restrictions</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * Use privacy = <code>PUBLIC</code> to retrieve all public channels in the account.
     * </p>
     * </li>
     * <li>
     * <p>
     * Only an <code>AppInstanceAdmin</code> can set privacy = <code>PRIVATE</code> to list the private channels in an
     * account.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #listChannels(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelsPublisher publisher = client.listChannelsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListChannelsPublisher publisher = client.listChannelsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChannels(software.amazon.awssdk.services.chimesdkmessaging.model.ListChannelsRequest)} operation.</b>
     * </p>
     *
     * @param listChannelsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListChannels"
     *      target="_top">AWS API Documentation</a>
     */
    public ListChannelsPublisher listChannelsPaginator(ListChannelsRequest listChannelsRequest) {
        return new ListChannelsPublisher(this, applyPaginatorUserAgent(listChannelsRequest));
    }

    /**
     * <p>
     * Lists all the SubChannels in an elastic channel when given a channel ID. Available only to the app instance
     * admins and channel moderators of elastic channels.
     * </p>
     *
     * @param listSubChannelsRequest
     * @return A Java Future containing the result of the ListSubChannels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListSubChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListSubChannels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListSubChannelsResponse> listSubChannels(ListSubChannelsRequest listSubChannelsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSubChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSubChannels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSubChannelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListSubChannelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSubChannelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSubChannelsRequest, ListSubChannelsResponse>()
                            .withOperationName("ListSubChannels")
                            .withMarshaller(new ListSubChannelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSubChannelsRequest));
            CompletableFuture<ListSubChannelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the SubChannels in an elastic channel when given a channel ID. Available only to the app instance
     * admins and channel moderators of elastic channels.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSubChannels(software.amazon.awssdk.services.chimesdkmessaging.model.ListSubChannelsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListSubChannelsPublisher publisher = client.listSubChannelsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.ListSubChannelsPublisher publisher = client.listSubChannelsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.chimesdkmessaging.model.ListSubChannelsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.chimesdkmessaging.model.ListSubChannelsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSubChannels(software.amazon.awssdk.services.chimesdkmessaging.model.ListSubChannelsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listSubChannelsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListSubChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListSubChannels"
     *      target="_top">AWS API Documentation</a>
     */
    public ListSubChannelsPublisher listSubChannelsPaginator(ListSubChannelsRequest listSubChannelsRequest) {
        return new ListSubChannelsPublisher(this, applyPaginatorUserAgent(listSubChannelsRequest));
    }

    /**
     * <p>
     * Lists the tags applied to an Amazon Chime SDK messaging resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Sets the membership preferences of an <code>AppInstanceUser</code> for the specified channel. The
     * <code>AppInstanceUser</code> must be a member of the channel. Only the <code>AppInstanceUser</code> who owns the
     * membership can set preferences. Users in the <code>AppInstanceAdmin</code> and channel moderator roles can't set
     * preferences for other users. Banned users can't set membership preferences for the channel from which they are
     * banned.
     * </p>
     *
     * @param putChannelMembershipPreferencesRequest
     * @return A Java Future containing the result of the PutChannelMembershipPreferences operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.PutChannelMembershipPreferences
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/PutChannelMembershipPreferences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutChannelMembershipPreferencesResponse> putChannelMembershipPreferences(
            PutChannelMembershipPreferencesRequest putChannelMembershipPreferencesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putChannelMembershipPreferencesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutChannelMembershipPreferences");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutChannelMembershipPreferencesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutChannelMembershipPreferencesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutChannelMembershipPreferencesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutChannelMembershipPreferencesRequest, PutChannelMembershipPreferencesResponse>()
                            .withOperationName("PutChannelMembershipPreferences")
                            .withMarshaller(new PutChannelMembershipPreferencesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putChannelMembershipPreferencesRequest));
            CompletableFuture<PutChannelMembershipPreferencesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Redacts message content, but not metadata. The message exists in the back end, but the action returns null
     * content, and the state shows as redacted.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param redactChannelMessageRequest
     * @return A Java Future containing the result of the RedactChannelMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.RedactChannelMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/RedactChannelMessage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RedactChannelMessageResponse> redactChannelMessage(
            RedactChannelMessageRequest redactChannelMessageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, redactChannelMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RedactChannelMessage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RedactChannelMessageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RedactChannelMessageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RedactChannelMessageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RedactChannelMessageRequest, RedactChannelMessageResponse>()
                            .withOperationName("RedactChannelMessage")
                            .withMarshaller(new RedactChannelMessageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(redactChannelMessageRequest));
            CompletableFuture<RedactChannelMessageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows <code>ChimeBearer</code> to search channels by channel members. AppInstanceUsers can search across the
     * channels that they belong to. AppInstanceAdmins can search across all channels.
     * </p>
     *
     * @param searchChannelsRequest
     * @return A Java Future containing the result of the SearchChannels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.SearchChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/SearchChannels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SearchChannelsResponse> searchChannels(SearchChannelsRequest searchChannelsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchChannels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchChannelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchChannelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchChannelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchChannelsRequest, SearchChannelsResponse>()
                            .withOperationName("SearchChannels")
                            .withMarshaller(new SearchChannelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(searchChannelsRequest));
            CompletableFuture<SearchChannelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows <code>ChimeBearer</code> to search channels by channel members. AppInstanceUsers can search across the
     * channels that they belong to. AppInstanceAdmins can search across all channels.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #searchChannels(software.amazon.awssdk.services.chimesdkmessaging.model.SearchChannelsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.SearchChannelsPublisher publisher = client.searchChannelsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.chimesdkmessaging.paginators.SearchChannelsPublisher publisher = client.searchChannelsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.chimesdkmessaging.model.SearchChannelsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.chimesdkmessaging.model.SearchChannelsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchChannels(software.amazon.awssdk.services.chimesdkmessaging.model.SearchChannelsRequest)}
     * operation.</b>
     * </p>
     *
     * @param searchChannelsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.SearchChannels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/SearchChannels"
     *      target="_top">AWS API Documentation</a>
     */
    public SearchChannelsPublisher searchChannelsPaginator(SearchChannelsRequest searchChannelsRequest) {
        return new SearchChannelsPublisher(this, applyPaginatorUserAgent(searchChannelsRequest));
    }

    /**
     * <p>
     * Sends a message to a particular channel that the member is a part of.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * <p>
     * Also, <code>STANDARD</code> messages can contain 4KB of data and the 1KB of metadata. <code>CONTROL</code>
     * messages can contain 30 bytes of data and no metadata.
     * </p>
     * </note>
     *
     * @param sendChannelMessageRequest
     * @return A Java Future containing the result of the SendChannelMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.SendChannelMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/SendChannelMessage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<SendChannelMessageResponse> sendChannelMessage(SendChannelMessageRequest sendChannelMessageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, sendChannelMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SendChannelMessage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SendChannelMessageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SendChannelMessageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SendChannelMessageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SendChannelMessageRequest, SendChannelMessageResponse>()
                            .withOperationName("SendChannelMessage")
                            .withMarshaller(new SendChannelMessageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(sendChannelMessageRequest));
            CompletableFuture<SendChannelMessageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Applies the specified tags to the specified Amazon Chime SDK messaging resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ResourceLimitExceededException The request exceeds the resource limit.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified Amazon Chime SDK messaging resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update a channel's attributes.
     * </p>
     * <p>
     * <b>Restriction</b>: You can't change a channel's privacy.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param updateChannelRequest
     * @return A Java Future containing the result of the UpdateChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.UpdateChannel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/UpdateChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateChannelResponse> updateChannel(UpdateChannelRequest updateChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateChannelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateChannelRequest, UpdateChannelResponse>()
                            .withOperationName("UpdateChannel")
                            .withMarshaller(new UpdateChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateChannelRequest));
            CompletableFuture<UpdateChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates channel flow attributes. This is a developer API.
     * </p>
     *
     * @param updateChannelFlowRequest
     * @return A Java Future containing the result of the UpdateChannelFlow operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.UpdateChannelFlow
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/UpdateChannelFlow"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateChannelFlowResponse> updateChannelFlow(UpdateChannelFlowRequest updateChannelFlowRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateChannelFlowRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateChannelFlow");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateChannelFlowResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateChannelFlowResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateChannelFlowResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateChannelFlowRequest, UpdateChannelFlowResponse>()
                            .withOperationName("UpdateChannelFlow")
                            .withMarshaller(new UpdateChannelFlowRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateChannelFlowRequest));
            CompletableFuture<UpdateChannelFlowResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the content of a message.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param updateChannelMessageRequest
     * @return A Java Future containing the result of the UpdateChannelMessage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.UpdateChannelMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/UpdateChannelMessage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateChannelMessageResponse> updateChannelMessage(
            UpdateChannelMessageRequest updateChannelMessageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateChannelMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateChannelMessage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateChannelMessageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateChannelMessageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateChannelMessageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateChannelMessageRequest, UpdateChannelMessageResponse>()
                            .withOperationName("UpdateChannelMessage")
                            .withMarshaller(new UpdateChannelMessageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateChannelMessageRequest));
            CompletableFuture<UpdateChannelMessageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The details of the time when a user last read messages in a channel.
     * </p>
     * <note>
     * <p>
     * The <code>x-amz-chime-bearer</code> request header is mandatory. Use the <code>AppInstanceUserArn</code> of the
     * user that makes the API call as the value in the header.
     * </p>
     * </note>
     *
     * @param updateChannelReadMarkerRequest
     * @return A Java Future containing the result of the UpdateChannelReadMarker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The input parameters don't match the service's restrictions.</li>
     *         <li>ForbiddenException The client is permanently forbidden from making the request.</li>
     *         <li>ConflictException The request could not be processed because of conflict in the current state of the
     *         resource.</li>
     *         <li>UnauthorizedClientException The client is not currently authorized to make the request.</li>
     *         <li>ThrottledClientException The client exceeded its request rate limit.</li>
     *         <li>ServiceUnavailableException The service is currently unavailable.</li>
     *         <li>ServiceFailureException The service encountered an unexpected error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChimeSdkMessagingException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ChimeSdkMessagingAsyncClient.UpdateChannelReadMarker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-sdk-messaging-2021-05-15/UpdateChannelReadMarker"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateChannelReadMarkerResponse> updateChannelReadMarker(
            UpdateChannelReadMarkerRequest updateChannelReadMarkerRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateChannelReadMarkerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime SDK Messaging");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateChannelReadMarker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateChannelReadMarkerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateChannelReadMarkerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateChannelReadMarkerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateChannelReadMarkerRequest, UpdateChannelReadMarkerResponse>()
                            .withOperationName("UpdateChannelReadMarker")
                            .withMarshaller(new UpdateChannelReadMarkerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateChannelReadMarkerRequest));
            CompletableFuture<UpdateChannelReadMarkerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ChimeSdkMessagingException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedClientException")
                                .exceptionBuilderSupplier(UnauthorizedClientException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceFailureException")
                                .exceptionBuilderSupplier(ServiceFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottledClientException")
                                .exceptionBuilderSupplier(ThrottledClientException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(503).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends ChimeSdkMessagingRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
