/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chimesdkmessaging.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Summary of the messages in a <code>Channel</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ChannelMessageSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<ChannelMessageSummary.Builder, ChannelMessageSummary> {
    private static final SdkField<String> MESSAGE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MessageId").getter(getter(ChannelMessageSummary::messageId)).setter(setter(Builder::messageId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageId").build()).build();

    private static final SdkField<String> CONTENT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Content")
            .getter(getter(ChannelMessageSummary::content)).setter(setter(Builder::content))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Content").build()).build();

    private static final SdkField<String> METADATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Metadata").getter(getter(ChannelMessageSummary::metadata)).setter(setter(Builder::metadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Metadata").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(ChannelMessageSummary::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<Instant> CREATED_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTimestamp").getter(getter(ChannelMessageSummary::createdTimestamp))
            .setter(setter(Builder::createdTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTimestamp").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedTimestamp").getter(getter(ChannelMessageSummary::lastUpdatedTimestamp))
            .setter(setter(Builder::lastUpdatedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedTimestamp").build())
            .build();

    private static final SdkField<Instant> LAST_EDITED_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastEditedTimestamp").getter(getter(ChannelMessageSummary::lastEditedTimestamp))
            .setter(setter(Builder::lastEditedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastEditedTimestamp").build())
            .build();

    private static final SdkField<Identity> SENDER_FIELD = SdkField.<Identity> builder(MarshallingType.SDK_POJO)
            .memberName("Sender").getter(getter(ChannelMessageSummary::sender)).setter(setter(Builder::sender))
            .constructor(Identity::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Sender").build()).build();

    private static final SdkField<Boolean> REDACTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Redacted").getter(getter(ChannelMessageSummary::redacted)).setter(setter(Builder::redacted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Redacted").build()).build();

    private static final SdkField<ChannelMessageStatusStructure> STATUS_FIELD = SdkField
            .<ChannelMessageStatusStructure> builder(MarshallingType.SDK_POJO).memberName("Status")
            .getter(getter(ChannelMessageSummary::status)).setter(setter(Builder::status))
            .constructor(ChannelMessageStatusStructure::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Map<String, MessageAttributeValue>> MESSAGE_ATTRIBUTES_FIELD = SdkField
            .<Map<String, MessageAttributeValue>> builder(MarshallingType.MAP)
            .memberName("MessageAttributes")
            .getter(getter(ChannelMessageSummary::messageAttributes))
            .setter(setter(Builder::messageAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageAttributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<MessageAttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(MessageAttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> CONTENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ContentType").getter(getter(ChannelMessageSummary::contentType)).setter(setter(Builder::contentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ContentType").build()).build();

    private static final SdkField<List<Target>> TARGET_FIELD = SdkField
            .<List<Target>> builder(MarshallingType.LIST)
            .memberName("Target")
            .getter(getter(ChannelMessageSummary::target))
            .setter(setter(Builder::target))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Target").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Target> builder(MarshallingType.SDK_POJO)
                                            .constructor(Target::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MESSAGE_ID_FIELD,
            CONTENT_FIELD, METADATA_FIELD, TYPE_FIELD, CREATED_TIMESTAMP_FIELD, LAST_UPDATED_TIMESTAMP_FIELD,
            LAST_EDITED_TIMESTAMP_FIELD, SENDER_FIELD, REDACTED_FIELD, STATUS_FIELD, MESSAGE_ATTRIBUTES_FIELD,
            CONTENT_TYPE_FIELD, TARGET_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String messageId;

    private final String content;

    private final String metadata;

    private final String type;

    private final Instant createdTimestamp;

    private final Instant lastUpdatedTimestamp;

    private final Instant lastEditedTimestamp;

    private final Identity sender;

    private final Boolean redacted;

    private final ChannelMessageStatusStructure status;

    private final Map<String, MessageAttributeValue> messageAttributes;

    private final String contentType;

    private final List<Target> target;

    private ChannelMessageSummary(BuilderImpl builder) {
        this.messageId = builder.messageId;
        this.content = builder.content;
        this.metadata = builder.metadata;
        this.type = builder.type;
        this.createdTimestamp = builder.createdTimestamp;
        this.lastUpdatedTimestamp = builder.lastUpdatedTimestamp;
        this.lastEditedTimestamp = builder.lastEditedTimestamp;
        this.sender = builder.sender;
        this.redacted = builder.redacted;
        this.status = builder.status;
        this.messageAttributes = builder.messageAttributes;
        this.contentType = builder.contentType;
        this.target = builder.target;
    }

    /**
     * <p>
     * The ID of the message.
     * </p>
     * 
     * @return The ID of the message.
     */
    public final String messageId() {
        return messageId;
    }

    /**
     * <p>
     * The content of the channel message. For Amazon Lex V2 bot responses, this field holds a list of messages
     * originating from the bot. For more information, refer to <a
     * href="https://docs.aws.amazon.com/chime-sdk/latest/dg/appinstance-bots#process-response.html">Processing
     * responses from an AppInstanceBot</a> in the <i>Amazon Chime SDK Messaging Developer Guide</i>.
     * </p>
     * 
     * @return The content of the channel message. For Amazon Lex V2 bot responses, this field holds a list of messages
     *         originating from the bot. For more information, refer to <a
     *         href="https://docs.aws.amazon.com/chime-sdk/latest/dg/appinstance-bots#process-response.html">Processing
     *         responses from an AppInstanceBot</a> in the <i>Amazon Chime SDK Messaging Developer Guide</i>.
     */
    public final String content() {
        return content;
    }

    /**
     * <p>
     * The metadata of the message.
     * </p>
     * 
     * @return The metadata of the message.
     */
    public final String metadata() {
        return metadata;
    }

    /**
     * <p>
     * The type of message.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ChannelMessageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of message.
     * @see ChannelMessageType
     */
    public final ChannelMessageType type() {
        return ChannelMessageType.fromValue(type);
    }

    /**
     * <p>
     * The type of message.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ChannelMessageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of message.
     * @see ChannelMessageType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The time at which the message summary was created.
     * </p>
     * 
     * @return The time at which the message summary was created.
     */
    public final Instant createdTimestamp() {
        return createdTimestamp;
    }

    /**
     * <p>
     * The time at which a message was last updated.
     * </p>
     * 
     * @return The time at which a message was last updated.
     */
    public final Instant lastUpdatedTimestamp() {
        return lastUpdatedTimestamp;
    }

    /**
     * <p>
     * The time at which a message was last edited.
     * </p>
     * 
     * @return The time at which a message was last edited.
     */
    public final Instant lastEditedTimestamp() {
        return lastEditedTimestamp;
    }

    /**
     * <p>
     * The message sender.
     * </p>
     * 
     * @return The message sender.
     */
    public final Identity sender() {
        return sender;
    }

    /**
     * <p>
     * Indicates whether a message was redacted.
     * </p>
     * 
     * @return Indicates whether a message was redacted.
     */
    public final Boolean redacted() {
        return redacted;
    }

    /**
     * <p>
     * The message status. The status value is <code>SENT</code> for messages sent to a channel without a channel flow.
     * For channels associated with channel flow, the value determines the processing stage.
     * </p>
     * 
     * @return The message status. The status value is <code>SENT</code> for messages sent to a channel without a
     *         channel flow. For channels associated with channel flow, the value determines the processing stage.
     */
    public final ChannelMessageStatusStructure status() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the MessageAttributes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMessageAttributes() {
        return messageAttributes != null && !(messageAttributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The attributes for the channel message. For Amazon Lex V2 bot responses, the attributes are mapped to specific
     * fields from the bot. For more information, refer to <a
     * href="https://docs.aws.amazon.com/chime-sdk/latest/dg/appinstance-bots#process-response.html">Processing
     * responses from an AppInstanceBot</a> in the <i>Amazon Chime SDK Messaging Developer Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMessageAttributes} method.
     * </p>
     * 
     * @return The attributes for the channel message. For Amazon Lex V2 bot responses, the attributes are mapped to
     *         specific fields from the bot. For more information, refer to <a
     *         href="https://docs.aws.amazon.com/chime-sdk/latest/dg/appinstance-bots#process-response.html">Processing
     *         responses from an AppInstanceBot</a> in the <i>Amazon Chime SDK Messaging Developer Guide</i>.
     */
    public final Map<String, MessageAttributeValue> messageAttributes() {
        return messageAttributes;
    }

    /**
     * <p>
     * The content type of the channel message listed in the summary. For Amazon Lex V2 bot responses, the content type
     * is <code>application/amz-chime-lex-msgs</code> for success responses and
     * <code>application/amz-chime-lex-error</code> for failure responses. For more information, refer to <a
     * href="https://docs.aws.amazon.com/chime-sdk/latest/dg/appinstance-bots#process-response.html">Processing
     * responses from an AppInstanceBot</a> in the <i>Amazon Chime SDK Messaging Developer Guide</i>.
     * </p>
     * 
     * @return The content type of the channel message listed in the summary. For Amazon Lex V2 bot responses, the
     *         content type is <code>application/amz-chime-lex-msgs</code> for success responses and
     *         <code>application/amz-chime-lex-error</code> for failure responses. For more information, refer to <a
     *         href="https://docs.aws.amazon.com/chime-sdk/latest/dg/appinstance-bots#process-response.html">Processing
     *         responses from an AppInstanceBot</a> in the <i>Amazon Chime SDK Messaging Developer Guide</i>.
     */
    public final String contentType() {
        return contentType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Target property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTarget() {
        return target != null && !(target instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The target of a message, a sender, a user, or a bot. Only the target and the sender can view targeted messages.
     * Only users who can see targeted messages can take actions on them. However, administrators can delete targeted
     * messages that they can’t see.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTarget} method.
     * </p>
     * 
     * @return The target of a message, a sender, a user, or a bot. Only the target and the sender can view targeted
     *         messages. Only users who can see targeted messages can take actions on them. However, administrators can
     *         delete targeted messages that they can’t see.
     */
    public final List<Target> target() {
        return target;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(messageId());
        hashCode = 31 * hashCode + Objects.hashCode(content());
        hashCode = 31 * hashCode + Objects.hashCode(metadata());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(lastEditedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(sender());
        hashCode = 31 * hashCode + Objects.hashCode(redacted());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(hasMessageAttributes() ? messageAttributes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(contentType());
        hashCode = 31 * hashCode + Objects.hashCode(hasTarget() ? target() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ChannelMessageSummary)) {
            return false;
        }
        ChannelMessageSummary other = (ChannelMessageSummary) obj;
        return Objects.equals(messageId(), other.messageId()) && Objects.equals(content(), other.content())
                && Objects.equals(metadata(), other.metadata()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(createdTimestamp(), other.createdTimestamp())
                && Objects.equals(lastUpdatedTimestamp(), other.lastUpdatedTimestamp())
                && Objects.equals(lastEditedTimestamp(), other.lastEditedTimestamp()) && Objects.equals(sender(), other.sender())
                && Objects.equals(redacted(), other.redacted()) && Objects.equals(status(), other.status())
                && hasMessageAttributes() == other.hasMessageAttributes()
                && Objects.equals(messageAttributes(), other.messageAttributes())
                && Objects.equals(contentType(), other.contentType()) && hasTarget() == other.hasTarget()
                && Objects.equals(target(), other.target());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ChannelMessageSummary").add("MessageId", messageId())
                .add("Content", content() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Metadata", metadata() == null ? null : "*** Sensitive Data Redacted ***").add("Type", typeAsString())
                .add("CreatedTimestamp", createdTimestamp()).add("LastUpdatedTimestamp", lastUpdatedTimestamp())
                .add("LastEditedTimestamp", lastEditedTimestamp()).add("Sender", sender()).add("Redacted", redacted())
                .add("Status", status())
                .add("MessageAttributes", messageAttributes() == null ? null : "*** Sensitive Data Redacted ***")
                .add("ContentType", contentType() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Target", hasTarget() ? target() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MessageId":
            return Optional.ofNullable(clazz.cast(messageId()));
        case "Content":
            return Optional.ofNullable(clazz.cast(content()));
        case "Metadata":
            return Optional.ofNullable(clazz.cast(metadata()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "CreatedTimestamp":
            return Optional.ofNullable(clazz.cast(createdTimestamp()));
        case "LastUpdatedTimestamp":
            return Optional.ofNullable(clazz.cast(lastUpdatedTimestamp()));
        case "LastEditedTimestamp":
            return Optional.ofNullable(clazz.cast(lastEditedTimestamp()));
        case "Sender":
            return Optional.ofNullable(clazz.cast(sender()));
        case "Redacted":
            return Optional.ofNullable(clazz.cast(redacted()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "MessageAttributes":
            return Optional.ofNullable(clazz.cast(messageAttributes()));
        case "ContentType":
            return Optional.ofNullable(clazz.cast(contentType()));
        case "Target":
            return Optional.ofNullable(clazz.cast(target()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("MessageId", MESSAGE_ID_FIELD);
        map.put("Content", CONTENT_FIELD);
        map.put("Metadata", METADATA_FIELD);
        map.put("Type", TYPE_FIELD);
        map.put("CreatedTimestamp", CREATED_TIMESTAMP_FIELD);
        map.put("LastUpdatedTimestamp", LAST_UPDATED_TIMESTAMP_FIELD);
        map.put("LastEditedTimestamp", LAST_EDITED_TIMESTAMP_FIELD);
        map.put("Sender", SENDER_FIELD);
        map.put("Redacted", REDACTED_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("MessageAttributes", MESSAGE_ATTRIBUTES_FIELD);
        map.put("ContentType", CONTENT_TYPE_FIELD);
        map.put("Target", TARGET_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ChannelMessageSummary, T> g) {
        return obj -> g.apply((ChannelMessageSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ChannelMessageSummary> {
        /**
         * <p>
         * The ID of the message.
         * </p>
         * 
         * @param messageId
         *        The ID of the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageId(String messageId);

        /**
         * <p>
         * The content of the channel message. For Amazon Lex V2 bot responses, this field holds a list of messages
         * originating from the bot. For more information, refer to <a
         * href="https://docs.aws.amazon.com/chime-sdk/latest/dg/appinstance-bots#process-response.html">Processing
         * responses from an AppInstanceBot</a> in the <i>Amazon Chime SDK Messaging Developer Guide</i>.
         * </p>
         * 
         * @param content
         *        The content of the channel message. For Amazon Lex V2 bot responses, this field holds a list of
         *        messages originating from the bot. For more information, refer to <a
         *        href="https://docs.aws.amazon.com/chime-sdk/latest/dg/appinstance-bots#process-response.html"
         *        >Processing responses from an AppInstanceBot</a> in the <i>Amazon Chime SDK Messaging Developer
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder content(String content);

        /**
         * <p>
         * The metadata of the message.
         * </p>
         * 
         * @param metadata
         *        The metadata of the message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metadata(String metadata);

        /**
         * <p>
         * The type of message.
         * </p>
         * 
         * @param type
         *        The type of message.
         * @see ChannelMessageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChannelMessageType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of message.
         * </p>
         * 
         * @param type
         *        The type of message.
         * @see ChannelMessageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChannelMessageType
         */
        Builder type(ChannelMessageType type);

        /**
         * <p>
         * The time at which the message summary was created.
         * </p>
         * 
         * @param createdTimestamp
         *        The time at which the message summary was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTimestamp(Instant createdTimestamp);

        /**
         * <p>
         * The time at which a message was last updated.
         * </p>
         * 
         * @param lastUpdatedTimestamp
         *        The time at which a message was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTimestamp(Instant lastUpdatedTimestamp);

        /**
         * <p>
         * The time at which a message was last edited.
         * </p>
         * 
         * @param lastEditedTimestamp
         *        The time at which a message was last edited.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastEditedTimestamp(Instant lastEditedTimestamp);

        /**
         * <p>
         * The message sender.
         * </p>
         * 
         * @param sender
         *        The message sender.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sender(Identity sender);

        /**
         * <p>
         * The message sender.
         * </p>
         * This is a convenience method that creates an instance of the {@link Identity.Builder} avoiding the need to
         * create one manually via {@link Identity#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Identity.Builder#build()} is called immediately and its result is
         * passed to {@link #sender(Identity)}.
         * 
         * @param sender
         *        a consumer that will call methods on {@link Identity.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sender(Identity)
         */
        default Builder sender(Consumer<Identity.Builder> sender) {
            return sender(Identity.builder().applyMutation(sender).build());
        }

        /**
         * <p>
         * Indicates whether a message was redacted.
         * </p>
         * 
         * @param redacted
         *        Indicates whether a message was redacted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder redacted(Boolean redacted);

        /**
         * <p>
         * The message status. The status value is <code>SENT</code> for messages sent to a channel without a channel
         * flow. For channels associated with channel flow, the value determines the processing stage.
         * </p>
         * 
         * @param status
         *        The message status. The status value is <code>SENT</code> for messages sent to a channel without a
         *        channel flow. For channels associated with channel flow, the value determines the processing stage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(ChannelMessageStatusStructure status);

        /**
         * <p>
         * The message status. The status value is <code>SENT</code> for messages sent to a channel without a channel
         * flow. For channels associated with channel flow, the value determines the processing stage.
         * </p>
         * This is a convenience method that creates an instance of the {@link ChannelMessageStatusStructure.Builder}
         * avoiding the need to create one manually via {@link ChannelMessageStatusStructure#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ChannelMessageStatusStructure.Builder#build()} is called
         * immediately and its result is passed to {@link #status(ChannelMessageStatusStructure)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link ChannelMessageStatusStructure.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(ChannelMessageStatusStructure)
         */
        default Builder status(Consumer<ChannelMessageStatusStructure.Builder> status) {
            return status(ChannelMessageStatusStructure.builder().applyMutation(status).build());
        }

        /**
         * <p>
         * The attributes for the channel message. For Amazon Lex V2 bot responses, the attributes are mapped to
         * specific fields from the bot. For more information, refer to <a
         * href="https://docs.aws.amazon.com/chime-sdk/latest/dg/appinstance-bots#process-response.html">Processing
         * responses from an AppInstanceBot</a> in the <i>Amazon Chime SDK Messaging Developer Guide</i>.
         * </p>
         * 
         * @param messageAttributes
         *        The attributes for the channel message. For Amazon Lex V2 bot responses, the attributes are mapped to
         *        specific fields from the bot. For more information, refer to <a
         *        href="https://docs.aws.amazon.com/chime-sdk/latest/dg/appinstance-bots#process-response.html"
         *        >Processing responses from an AppInstanceBot</a> in the <i>Amazon Chime SDK Messaging Developer
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageAttributes(Map<String, MessageAttributeValue> messageAttributes);

        /**
         * <p>
         * The content type of the channel message listed in the summary. For Amazon Lex V2 bot responses, the content
         * type is <code>application/amz-chime-lex-msgs</code> for success responses and
         * <code>application/amz-chime-lex-error</code> for failure responses. For more information, refer to <a
         * href="https://docs.aws.amazon.com/chime-sdk/latest/dg/appinstance-bots#process-response.html">Processing
         * responses from an AppInstanceBot</a> in the <i>Amazon Chime SDK Messaging Developer Guide</i>.
         * </p>
         * 
         * @param contentType
         *        The content type of the channel message listed in the summary. For Amazon Lex V2 bot responses, the
         *        content type is <code>application/amz-chime-lex-msgs</code> for success responses and
         *        <code>application/amz-chime-lex-error</code> for failure responses. For more information, refer to <a
         *        href
         *        ="https://docs.aws.amazon.com/chime-sdk/latest/dg/appinstance-bots#process-response.html">Processing
         *        responses from an AppInstanceBot</a> in the <i>Amazon Chime SDK Messaging Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contentType(String contentType);

        /**
         * <p>
         * The target of a message, a sender, a user, or a bot. Only the target and the sender can view targeted
         * messages. Only users who can see targeted messages can take actions on them. However, administrators can
         * delete targeted messages that they can’t see.
         * </p>
         * 
         * @param target
         *        The target of a message, a sender, a user, or a bot. Only the target and the sender can view targeted
         *        messages. Only users who can see targeted messages can take actions on them. However, administrators
         *        can delete targeted messages that they can’t see.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder target(Collection<Target> target);

        /**
         * <p>
         * The target of a message, a sender, a user, or a bot. Only the target and the sender can view targeted
         * messages. Only users who can see targeted messages can take actions on them. However, administrators can
         * delete targeted messages that they can’t see.
         * </p>
         * 
         * @param target
         *        The target of a message, a sender, a user, or a bot. Only the target and the sender can view targeted
         *        messages. Only users who can see targeted messages can take actions on them. However, administrators
         *        can delete targeted messages that they can’t see.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder target(Target... target);

        /**
         * <p>
         * The target of a message, a sender, a user, or a bot. Only the target and the sender can view targeted
         * messages. Only users who can see targeted messages can take actions on them. However, administrators can
         * delete targeted messages that they can’t see.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.chimesdkmessaging.model.Target.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.chimesdkmessaging.model.Target#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.chimesdkmessaging.model.Target.Builder#build()} is called immediately
         * and its result is passed to {@link #target(List<Target>)}.
         * 
         * @param target
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.chimesdkmessaging.model.Target.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #target(java.util.Collection<Target>)
         */
        Builder target(Consumer<Target.Builder>... target);
    }

    static final class BuilderImpl implements Builder {
        private String messageId;

        private String content;

        private String metadata;

        private String type;

        private Instant createdTimestamp;

        private Instant lastUpdatedTimestamp;

        private Instant lastEditedTimestamp;

        private Identity sender;

        private Boolean redacted;

        private ChannelMessageStatusStructure status;

        private Map<String, MessageAttributeValue> messageAttributes = DefaultSdkAutoConstructMap.getInstance();

        private String contentType;

        private List<Target> target = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ChannelMessageSummary model) {
            messageId(model.messageId);
            content(model.content);
            metadata(model.metadata);
            type(model.type);
            createdTimestamp(model.createdTimestamp);
            lastUpdatedTimestamp(model.lastUpdatedTimestamp);
            lastEditedTimestamp(model.lastEditedTimestamp);
            sender(model.sender);
            redacted(model.redacted);
            status(model.status);
            messageAttributes(model.messageAttributes);
            contentType(model.contentType);
            target(model.target);
        }

        public final String getMessageId() {
            return messageId;
        }

        public final void setMessageId(String messageId) {
            this.messageId = messageId;
        }

        @Override
        public final Builder messageId(String messageId) {
            this.messageId = messageId;
            return this;
        }

        public final String getContent() {
            return content;
        }

        public final void setContent(String content) {
            this.content = content;
        }

        @Override
        public final Builder content(String content) {
            this.content = content;
            return this;
        }

        public final String getMetadata() {
            return metadata;
        }

        public final void setMetadata(String metadata) {
            this.metadata = metadata;
        }

        @Override
        public final Builder metadata(String metadata) {
            this.metadata = metadata;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ChannelMessageType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final Instant getCreatedTimestamp() {
            return createdTimestamp;
        }

        public final void setCreatedTimestamp(Instant createdTimestamp) {
            this.createdTimestamp = createdTimestamp;
        }

        @Override
        public final Builder createdTimestamp(Instant createdTimestamp) {
            this.createdTimestamp = createdTimestamp;
            return this;
        }

        public final Instant getLastUpdatedTimestamp() {
            return lastUpdatedTimestamp;
        }

        public final void setLastUpdatedTimestamp(Instant lastUpdatedTimestamp) {
            this.lastUpdatedTimestamp = lastUpdatedTimestamp;
        }

        @Override
        public final Builder lastUpdatedTimestamp(Instant lastUpdatedTimestamp) {
            this.lastUpdatedTimestamp = lastUpdatedTimestamp;
            return this;
        }

        public final Instant getLastEditedTimestamp() {
            return lastEditedTimestamp;
        }

        public final void setLastEditedTimestamp(Instant lastEditedTimestamp) {
            this.lastEditedTimestamp = lastEditedTimestamp;
        }

        @Override
        public final Builder lastEditedTimestamp(Instant lastEditedTimestamp) {
            this.lastEditedTimestamp = lastEditedTimestamp;
            return this;
        }

        public final Identity.Builder getSender() {
            return sender != null ? sender.toBuilder() : null;
        }

        public final void setSender(Identity.BuilderImpl sender) {
            this.sender = sender != null ? sender.build() : null;
        }

        @Override
        public final Builder sender(Identity sender) {
            this.sender = sender;
            return this;
        }

        public final Boolean getRedacted() {
            return redacted;
        }

        public final void setRedacted(Boolean redacted) {
            this.redacted = redacted;
        }

        @Override
        public final Builder redacted(Boolean redacted) {
            this.redacted = redacted;
            return this;
        }

        public final ChannelMessageStatusStructure.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        public final void setStatus(ChannelMessageStatusStructure.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        @Override
        public final Builder status(ChannelMessageStatusStructure status) {
            this.status = status;
            return this;
        }

        public final Map<String, MessageAttributeValue.Builder> getMessageAttributes() {
            Map<String, MessageAttributeValue.Builder> result = MessageAttributeMapCopier.copyToBuilder(this.messageAttributes);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setMessageAttributes(Map<String, MessageAttributeValue.BuilderImpl> messageAttributes) {
            this.messageAttributes = MessageAttributeMapCopier.copyFromBuilder(messageAttributes);
        }

        @Override
        public final Builder messageAttributes(Map<String, MessageAttributeValue> messageAttributes) {
            this.messageAttributes = MessageAttributeMapCopier.copy(messageAttributes);
            return this;
        }

        public final String getContentType() {
            return contentType;
        }

        public final void setContentType(String contentType) {
            this.contentType = contentType;
        }

        @Override
        public final Builder contentType(String contentType) {
            this.contentType = contentType;
            return this;
        }

        public final List<Target.Builder> getTarget() {
            List<Target.Builder> result = TargetListCopier.copyToBuilder(this.target);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTarget(Collection<Target.BuilderImpl> target) {
            this.target = TargetListCopier.copyFromBuilder(target);
        }

        @Override
        public final Builder target(Collection<Target> target) {
            this.target = TargetListCopier.copy(target);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder target(Target... target) {
            target(Arrays.asList(target));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder target(Consumer<Target.Builder>... target) {
            target(Stream.of(target).map(c -> Target.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public ChannelMessageSummary build() {
            return new ChannelMessageSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
