/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cleanrooms.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The analysis template within a collaboration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CollaborationAnalysisTemplate implements SdkPojo, Serializable,
        ToCopyableBuilder<CollaborationAnalysisTemplate.Builder, CollaborationAnalysisTemplate> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(CollaborationAnalysisTemplate::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(CollaborationAnalysisTemplate::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> COLLABORATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("collaborationId").getter(getter(CollaborationAnalysisTemplate::collaborationId))
            .setter(setter(Builder::collaborationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("collaborationId").build()).build();

    private static final SdkField<String> COLLABORATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("collaborationArn").getter(getter(CollaborationAnalysisTemplate::collaborationArn))
            .setter(setter(Builder::collaborationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("collaborationArn").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("description").getter(getter(CollaborationAnalysisTemplate::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<String> CREATOR_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("creatorAccountId").getter(getter(CollaborationAnalysisTemplate::creatorAccountId))
            .setter(setter(Builder::creatorAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("creatorAccountId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(CollaborationAnalysisTemplate::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<Instant> CREATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createTime").getter(getter(CollaborationAnalysisTemplate::createTime))
            .setter(setter(Builder::createTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createTime").build()).build();

    private static final SdkField<Instant> UPDATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("updateTime").getter(getter(CollaborationAnalysisTemplate::updateTime))
            .setter(setter(Builder::updateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updateTime").build()).build();

    private static final SdkField<AnalysisSchema> SCHEMA_FIELD = SdkField.<AnalysisSchema> builder(MarshallingType.SDK_POJO)
            .memberName("schema").getter(getter(CollaborationAnalysisTemplate::schema)).setter(setter(Builder::schema))
            .constructor(AnalysisSchema::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("schema").build()).build();

    private static final SdkField<String> FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("format")
            .getter(getter(CollaborationAnalysisTemplate::formatAsString)).setter(setter(Builder::format))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("format").build()).build();

    private static final SdkField<AnalysisSource> SOURCE_FIELD = SdkField.<AnalysisSource> builder(MarshallingType.SDK_POJO)
            .memberName("source").getter(getter(CollaborationAnalysisTemplate::source)).setter(setter(Builder::source))
            .constructor(AnalysisSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("source").build()).build();

    private static final SdkField<AnalysisSourceMetadata> SOURCE_METADATA_FIELD = SdkField
            .<AnalysisSourceMetadata> builder(MarshallingType.SDK_POJO).memberName("sourceMetadata")
            .getter(getter(CollaborationAnalysisTemplate::sourceMetadata)).setter(setter(Builder::sourceMetadata))
            .constructor(AnalysisSourceMetadata::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceMetadata").build()).build();

    private static final SdkField<List<AnalysisParameter>> ANALYSIS_PARAMETERS_FIELD = SdkField
            .<List<AnalysisParameter>> builder(MarshallingType.LIST)
            .memberName("analysisParameters")
            .getter(getter(CollaborationAnalysisTemplate::analysisParameters))
            .setter(setter(Builder::analysisParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("analysisParameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AnalysisParameter> builder(MarshallingType.SDK_POJO)
                                            .constructor(AnalysisParameter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<AnalysisTemplateValidationStatusDetail>> VALIDATIONS_FIELD = SdkField
            .<List<AnalysisTemplateValidationStatusDetail>> builder(MarshallingType.LIST)
            .memberName("validations")
            .getter(getter(CollaborationAnalysisTemplate::validations))
            .setter(setter(Builder::validations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("validations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AnalysisTemplateValidationStatusDetail> builder(MarshallingType.SDK_POJO)
                                            .constructor(AnalysisTemplateValidationStatusDetail::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ErrorMessageConfiguration> ERROR_MESSAGE_CONFIGURATION_FIELD = SdkField
            .<ErrorMessageConfiguration> builder(MarshallingType.SDK_POJO).memberName("errorMessageConfiguration")
            .getter(getter(CollaborationAnalysisTemplate::errorMessageConfiguration))
            .setter(setter(Builder::errorMessageConfiguration)).constructor(ErrorMessageConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("errorMessageConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, ARN_FIELD,
            COLLABORATION_ID_FIELD, COLLABORATION_ARN_FIELD, DESCRIPTION_FIELD, CREATOR_ACCOUNT_ID_FIELD, NAME_FIELD,
            CREATE_TIME_FIELD, UPDATE_TIME_FIELD, SCHEMA_FIELD, FORMAT_FIELD, SOURCE_FIELD, SOURCE_METADATA_FIELD,
            ANALYSIS_PARAMETERS_FIELD, VALIDATIONS_FIELD, ERROR_MESSAGE_CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String arn;

    private final String collaborationId;

    private final String collaborationArn;

    private final String description;

    private final String creatorAccountId;

    private final String name;

    private final Instant createTime;

    private final Instant updateTime;

    private final AnalysisSchema schema;

    private final String format;

    private final AnalysisSource source;

    private final AnalysisSourceMetadata sourceMetadata;

    private final List<AnalysisParameter> analysisParameters;

    private final List<AnalysisTemplateValidationStatusDetail> validations;

    private final ErrorMessageConfiguration errorMessageConfiguration;

    private CollaborationAnalysisTemplate(BuilderImpl builder) {
        this.id = builder.id;
        this.arn = builder.arn;
        this.collaborationId = builder.collaborationId;
        this.collaborationArn = builder.collaborationArn;
        this.description = builder.description;
        this.creatorAccountId = builder.creatorAccountId;
        this.name = builder.name;
        this.createTime = builder.createTime;
        this.updateTime = builder.updateTime;
        this.schema = builder.schema;
        this.format = builder.format;
        this.source = builder.source;
        this.sourceMetadata = builder.sourceMetadata;
        this.analysisParameters = builder.analysisParameters;
        this.validations = builder.validations;
        this.errorMessageConfiguration = builder.errorMessageConfiguration;
    }

    /**
     * <p>
     * The identifier of the analysis template.
     * </p>
     * 
     * @return The identifier of the analysis template.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the analysis template.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the analysis template.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * A unique identifier for the collaboration that the analysis templates belong to. Currently accepts collaboration
     * ID.
     * </p>
     * 
     * @return A unique identifier for the collaboration that the analysis templates belong to. Currently accepts
     *         collaboration ID.
     */
    public final String collaborationId() {
        return collaborationId;
    }

    /**
     * <p>
     * The unique ARN for the analysis template’s associated collaboration.
     * </p>
     * 
     * @return The unique ARN for the analysis template’s associated collaboration.
     */
    public final String collaborationArn() {
        return collaborationArn;
    }

    /**
     * <p>
     * The description of the analysis template.
     * </p>
     * 
     * @return The description of the analysis template.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The identifier used to reference members of the collaboration. Currently only supports Amazon Web Services
     * account ID.
     * </p>
     * 
     * @return The identifier used to reference members of the collaboration. Currently only supports Amazon Web
     *         Services account ID.
     */
    public final String creatorAccountId() {
        return creatorAccountId;
    }

    /**
     * <p>
     * The name of the analysis template.
     * </p>
     * 
     * @return The name of the analysis template.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The time that the analysis template within a collaboration was created.
     * </p>
     * 
     * @return The time that the analysis template within a collaboration was created.
     */
    public final Instant createTime() {
        return createTime;
    }

    /**
     * <p>
     * The time that the analysis template in the collaboration was last updated.
     * </p>
     * 
     * @return The time that the analysis template in the collaboration was last updated.
     */
    public final Instant updateTime() {
        return updateTime;
    }

    /**
     * <p>
     * The entire schema object.
     * </p>
     * 
     * @return The entire schema object.
     */
    public final AnalysisSchema schema() {
        return schema;
    }

    /**
     * <p>
     * The format of the analysis template in the collaboration.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link AnalysisFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the analysis template in the collaboration.
     * @see AnalysisFormat
     */
    public final AnalysisFormat format() {
        return AnalysisFormat.fromValue(format);
    }

    /**
     * <p>
     * The format of the analysis template in the collaboration.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link AnalysisFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the analysis template in the collaboration.
     * @see AnalysisFormat
     */
    public final String formatAsString() {
        return format;
    }

    /**
     * <p>
     * The source of the analysis template within a collaboration.
     * </p>
     * 
     * @return The source of the analysis template within a collaboration.
     */
    public final AnalysisSource source() {
        return source;
    }

    /**
     * <p>
     * The source metadata for the collaboration analysis template.
     * </p>
     * 
     * @return The source metadata for the collaboration analysis template.
     */
    public final AnalysisSourceMetadata sourceMetadata() {
        return sourceMetadata;
    }

    /**
     * For responses, this returns true if the service returned a value for the AnalysisParameters property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAnalysisParameters() {
        return analysisParameters != null && !(analysisParameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The analysis parameters that have been specified in the analysis template.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAnalysisParameters} method.
     * </p>
     * 
     * @return The analysis parameters that have been specified in the analysis template.
     */
    public final List<AnalysisParameter> analysisParameters() {
        return analysisParameters;
    }

    /**
     * For responses, this returns true if the service returned a value for the Validations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasValidations() {
        return validations != null && !(validations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The validations that were performed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValidations} method.
     * </p>
     * 
     * @return The validations that were performed.
     */
    public final List<AnalysisTemplateValidationStatusDetail> validations() {
        return validations;
    }

    /**
     * <p>
     * The configuration that specifies the level of detail in error messages returned by analyses using this template.
     * When set to <code>DETAILED</code>, error messages include more information to help troubleshoot issues with
     * PySpark jobs. Detailed error messages may expose underlying data, including sensitive information. Recommended
     * for faster troubleshooting in development and testing environments.
     * </p>
     * 
     * @return The configuration that specifies the level of detail in error messages returned by analyses using this
     *         template. When set to <code>DETAILED</code>, error messages include more information to help troubleshoot
     *         issues with PySpark jobs. Detailed error messages may expose underlying data, including sensitive
     *         information. Recommended for faster troubleshooting in development and testing environments.
     */
    public final ErrorMessageConfiguration errorMessageConfiguration() {
        return errorMessageConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(collaborationId());
        hashCode = 31 * hashCode + Objects.hashCode(collaborationArn());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(creatorAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(createTime());
        hashCode = 31 * hashCode + Objects.hashCode(updateTime());
        hashCode = 31 * hashCode + Objects.hashCode(schema());
        hashCode = 31 * hashCode + Objects.hashCode(formatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(sourceMetadata());
        hashCode = 31 * hashCode + Objects.hashCode(hasAnalysisParameters() ? analysisParameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasValidations() ? validations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(errorMessageConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CollaborationAnalysisTemplate)) {
            return false;
        }
        CollaborationAnalysisTemplate other = (CollaborationAnalysisTemplate) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(arn(), other.arn())
                && Objects.equals(collaborationId(), other.collaborationId())
                && Objects.equals(collaborationArn(), other.collaborationArn())
                && Objects.equals(description(), other.description())
                && Objects.equals(creatorAccountId(), other.creatorAccountId()) && Objects.equals(name(), other.name())
                && Objects.equals(createTime(), other.createTime()) && Objects.equals(updateTime(), other.updateTime())
                && Objects.equals(schema(), other.schema()) && Objects.equals(formatAsString(), other.formatAsString())
                && Objects.equals(source(), other.source()) && Objects.equals(sourceMetadata(), other.sourceMetadata())
                && hasAnalysisParameters() == other.hasAnalysisParameters()
                && Objects.equals(analysisParameters(), other.analysisParameters()) && hasValidations() == other.hasValidations()
                && Objects.equals(validations(), other.validations())
                && Objects.equals(errorMessageConfiguration(), other.errorMessageConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CollaborationAnalysisTemplate").add("Id", id()).add("Arn", arn())
                .add("CollaborationId", collaborationId()).add("CollaborationArn", collaborationArn())
                .add("Description", description()).add("CreatorAccountId", creatorAccountId()).add("Name", name())
                .add("CreateTime", createTime()).add("UpdateTime", updateTime()).add("Schema", schema())
                .add("Format", formatAsString()).add("Source", source()).add("SourceMetadata", sourceMetadata())
                .add("AnalysisParameters", analysisParameters() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Validations", hasValidations() ? validations() : null)
                .add("ErrorMessageConfiguration", errorMessageConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "collaborationId":
            return Optional.ofNullable(clazz.cast(collaborationId()));
        case "collaborationArn":
            return Optional.ofNullable(clazz.cast(collaborationArn()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "creatorAccountId":
            return Optional.ofNullable(clazz.cast(creatorAccountId()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "createTime":
            return Optional.ofNullable(clazz.cast(createTime()));
        case "updateTime":
            return Optional.ofNullable(clazz.cast(updateTime()));
        case "schema":
            return Optional.ofNullable(clazz.cast(schema()));
        case "format":
            return Optional.ofNullable(clazz.cast(formatAsString()));
        case "source":
            return Optional.ofNullable(clazz.cast(source()));
        case "sourceMetadata":
            return Optional.ofNullable(clazz.cast(sourceMetadata()));
        case "analysisParameters":
            return Optional.ofNullable(clazz.cast(analysisParameters()));
        case "validations":
            return Optional.ofNullable(clazz.cast(validations()));
        case "errorMessageConfiguration":
            return Optional.ofNullable(clazz.cast(errorMessageConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("id", ID_FIELD);
        map.put("arn", ARN_FIELD);
        map.put("collaborationId", COLLABORATION_ID_FIELD);
        map.put("collaborationArn", COLLABORATION_ARN_FIELD);
        map.put("description", DESCRIPTION_FIELD);
        map.put("creatorAccountId", CREATOR_ACCOUNT_ID_FIELD);
        map.put("name", NAME_FIELD);
        map.put("createTime", CREATE_TIME_FIELD);
        map.put("updateTime", UPDATE_TIME_FIELD);
        map.put("schema", SCHEMA_FIELD);
        map.put("format", FORMAT_FIELD);
        map.put("source", SOURCE_FIELD);
        map.put("sourceMetadata", SOURCE_METADATA_FIELD);
        map.put("analysisParameters", ANALYSIS_PARAMETERS_FIELD);
        map.put("validations", VALIDATIONS_FIELD);
        map.put("errorMessageConfiguration", ERROR_MESSAGE_CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CollaborationAnalysisTemplate, T> g) {
        return obj -> g.apply((CollaborationAnalysisTemplate) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CollaborationAnalysisTemplate> {
        /**
         * <p>
         * The identifier of the analysis template.
         * </p>
         * 
         * @param id
         *        The identifier of the analysis template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the analysis template.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the analysis template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * A unique identifier for the collaboration that the analysis templates belong to. Currently accepts
         * collaboration ID.
         * </p>
         * 
         * @param collaborationId
         *        A unique identifier for the collaboration that the analysis templates belong to. Currently accepts
         *        collaboration ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder collaborationId(String collaborationId);

        /**
         * <p>
         * The unique ARN for the analysis template’s associated collaboration.
         * </p>
         * 
         * @param collaborationArn
         *        The unique ARN for the analysis template’s associated collaboration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder collaborationArn(String collaborationArn);

        /**
         * <p>
         * The description of the analysis template.
         * </p>
         * 
         * @param description
         *        The description of the analysis template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The identifier used to reference members of the collaboration. Currently only supports Amazon Web Services
         * account ID.
         * </p>
         * 
         * @param creatorAccountId
         *        The identifier used to reference members of the collaboration. Currently only supports Amazon Web
         *        Services account ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creatorAccountId(String creatorAccountId);

        /**
         * <p>
         * The name of the analysis template.
         * </p>
         * 
         * @param name
         *        The name of the analysis template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The time that the analysis template within a collaboration was created.
         * </p>
         * 
         * @param createTime
         *        The time that the analysis template within a collaboration was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTime(Instant createTime);

        /**
         * <p>
         * The time that the analysis template in the collaboration was last updated.
         * </p>
         * 
         * @param updateTime
         *        The time that the analysis template in the collaboration was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updateTime(Instant updateTime);

        /**
         * <p>
         * The entire schema object.
         * </p>
         * 
         * @param schema
         *        The entire schema object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schema(AnalysisSchema schema);

        /**
         * <p>
         * The entire schema object.
         * </p>
         * This is a convenience method that creates an instance of the {@link AnalysisSchema.Builder} avoiding the need
         * to create one manually via {@link AnalysisSchema#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AnalysisSchema.Builder#build()} is called immediately and its
         * result is passed to {@link #schema(AnalysisSchema)}.
         * 
         * @param schema
         *        a consumer that will call methods on {@link AnalysisSchema.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #schema(AnalysisSchema)
         */
        default Builder schema(Consumer<AnalysisSchema.Builder> schema) {
            return schema(AnalysisSchema.builder().applyMutation(schema).build());
        }

        /**
         * <p>
         * The format of the analysis template in the collaboration.
         * </p>
         * 
         * @param format
         *        The format of the analysis template in the collaboration.
         * @see AnalysisFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnalysisFormat
         */
        Builder format(String format);

        /**
         * <p>
         * The format of the analysis template in the collaboration.
         * </p>
         * 
         * @param format
         *        The format of the analysis template in the collaboration.
         * @see AnalysisFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AnalysisFormat
         */
        Builder format(AnalysisFormat format);

        /**
         * <p>
         * The source of the analysis template within a collaboration.
         * </p>
         * 
         * @param source
         *        The source of the analysis template within a collaboration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(AnalysisSource source);

        /**
         * <p>
         * The source of the analysis template within a collaboration.
         * </p>
         * This is a convenience method that creates an instance of the {@link AnalysisSource.Builder} avoiding the need
         * to create one manually via {@link AnalysisSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AnalysisSource.Builder#build()} is called immediately and its
         * result is passed to {@link #source(AnalysisSource)}.
         * 
         * @param source
         *        a consumer that will call methods on {@link AnalysisSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #source(AnalysisSource)
         */
        default Builder source(Consumer<AnalysisSource.Builder> source) {
            return source(AnalysisSource.builder().applyMutation(source).build());
        }

        /**
         * <p>
         * The source metadata for the collaboration analysis template.
         * </p>
         * 
         * @param sourceMetadata
         *        The source metadata for the collaboration analysis template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceMetadata(AnalysisSourceMetadata sourceMetadata);

        /**
         * <p>
         * The source metadata for the collaboration analysis template.
         * </p>
         * This is a convenience method that creates an instance of the {@link AnalysisSourceMetadata.Builder} avoiding
         * the need to create one manually via {@link AnalysisSourceMetadata#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AnalysisSourceMetadata.Builder#build()} is called immediately and
         * its result is passed to {@link #sourceMetadata(AnalysisSourceMetadata)}.
         * 
         * @param sourceMetadata
         *        a consumer that will call methods on {@link AnalysisSourceMetadata.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceMetadata(AnalysisSourceMetadata)
         */
        default Builder sourceMetadata(Consumer<AnalysisSourceMetadata.Builder> sourceMetadata) {
            return sourceMetadata(AnalysisSourceMetadata.builder().applyMutation(sourceMetadata).build());
        }

        /**
         * <p>
         * The analysis parameters that have been specified in the analysis template.
         * </p>
         * 
         * @param analysisParameters
         *        The analysis parameters that have been specified in the analysis template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder analysisParameters(Collection<AnalysisParameter> analysisParameters);

        /**
         * <p>
         * The analysis parameters that have been specified in the analysis template.
         * </p>
         * 
         * @param analysisParameters
         *        The analysis parameters that have been specified in the analysis template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder analysisParameters(AnalysisParameter... analysisParameters);

        /**
         * <p>
         * The analysis parameters that have been specified in the analysis template.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cleanrooms.model.AnalysisParameter.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.cleanrooms.model.AnalysisParameter#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cleanrooms.model.AnalysisParameter.Builder#build()} is called
         * immediately and its result is passed to {@link #analysisParameters(List<AnalysisParameter>)}.
         * 
         * @param analysisParameters
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cleanrooms.model.AnalysisParameter.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #analysisParameters(java.util.Collection<AnalysisParameter>)
         */
        Builder analysisParameters(Consumer<AnalysisParameter.Builder>... analysisParameters);

        /**
         * <p>
         * The validations that were performed.
         * </p>
         * 
         * @param validations
         *        The validations that were performed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validations(Collection<AnalysisTemplateValidationStatusDetail> validations);

        /**
         * <p>
         * The validations that were performed.
         * </p>
         * 
         * @param validations
         *        The validations that were performed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validations(AnalysisTemplateValidationStatusDetail... validations);

        /**
         * <p>
         * The validations that were performed.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cleanrooms.model.AnalysisTemplateValidationStatusDetail.Builder}
         * avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.cleanrooms.model.AnalysisTemplateValidationStatusDetail#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cleanrooms.model.AnalysisTemplateValidationStatusDetail.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #validations(List<AnalysisTemplateValidationStatusDetail>)}.
         * 
         * @param validations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cleanrooms.model.AnalysisTemplateValidationStatusDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #validations(java.util.Collection<AnalysisTemplateValidationStatusDetail>)
         */
        Builder validations(Consumer<AnalysisTemplateValidationStatusDetail.Builder>... validations);

        /**
         * <p>
         * The configuration that specifies the level of detail in error messages returned by analyses using this
         * template. When set to <code>DETAILED</code>, error messages include more information to help troubleshoot
         * issues with PySpark jobs. Detailed error messages may expose underlying data, including sensitive
         * information. Recommended for faster troubleshooting in development and testing environments.
         * </p>
         * 
         * @param errorMessageConfiguration
         *        The configuration that specifies the level of detail in error messages returned by analyses using this
         *        template. When set to <code>DETAILED</code>, error messages include more information to help
         *        troubleshoot issues with PySpark jobs. Detailed error messages may expose underlying data, including
         *        sensitive information. Recommended for faster troubleshooting in development and testing environments.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorMessageConfiguration(ErrorMessageConfiguration errorMessageConfiguration);

        /**
         * <p>
         * The configuration that specifies the level of detail in error messages returned by analyses using this
         * template. When set to <code>DETAILED</code>, error messages include more information to help troubleshoot
         * issues with PySpark jobs. Detailed error messages may expose underlying data, including sensitive
         * information. Recommended for faster troubleshooting in development and testing environments.
         * </p>
         * This is a convenience method that creates an instance of the {@link ErrorMessageConfiguration.Builder}
         * avoiding the need to create one manually via {@link ErrorMessageConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ErrorMessageConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #errorMessageConfiguration(ErrorMessageConfiguration)}.
         * 
         * @param errorMessageConfiguration
         *        a consumer that will call methods on {@link ErrorMessageConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #errorMessageConfiguration(ErrorMessageConfiguration)
         */
        default Builder errorMessageConfiguration(Consumer<ErrorMessageConfiguration.Builder> errorMessageConfiguration) {
            return errorMessageConfiguration(ErrorMessageConfiguration.builder().applyMutation(errorMessageConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String arn;

        private String collaborationId;

        private String collaborationArn;

        private String description;

        private String creatorAccountId;

        private String name;

        private Instant createTime;

        private Instant updateTime;

        private AnalysisSchema schema;

        private String format;

        private AnalysisSource source;

        private AnalysisSourceMetadata sourceMetadata;

        private List<AnalysisParameter> analysisParameters = DefaultSdkAutoConstructList.getInstance();

        private List<AnalysisTemplateValidationStatusDetail> validations = DefaultSdkAutoConstructList.getInstance();

        private ErrorMessageConfiguration errorMessageConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(CollaborationAnalysisTemplate model) {
            id(model.id);
            arn(model.arn);
            collaborationId(model.collaborationId);
            collaborationArn(model.collaborationArn);
            description(model.description);
            creatorAccountId(model.creatorAccountId);
            name(model.name);
            createTime(model.createTime);
            updateTime(model.updateTime);
            schema(model.schema);
            format(model.format);
            source(model.source);
            sourceMetadata(model.sourceMetadata);
            analysisParameters(model.analysisParameters);
            validations(model.validations);
            errorMessageConfiguration(model.errorMessageConfiguration);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getCollaborationId() {
            return collaborationId;
        }

        public final void setCollaborationId(String collaborationId) {
            this.collaborationId = collaborationId;
        }

        @Override
        public final Builder collaborationId(String collaborationId) {
            this.collaborationId = collaborationId;
            return this;
        }

        public final String getCollaborationArn() {
            return collaborationArn;
        }

        public final void setCollaborationArn(String collaborationArn) {
            this.collaborationArn = collaborationArn;
        }

        @Override
        public final Builder collaborationArn(String collaborationArn) {
            this.collaborationArn = collaborationArn;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getCreatorAccountId() {
            return creatorAccountId;
        }

        public final void setCreatorAccountId(String creatorAccountId) {
            this.creatorAccountId = creatorAccountId;
        }

        @Override
        public final Builder creatorAccountId(String creatorAccountId) {
            this.creatorAccountId = creatorAccountId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Instant getCreateTime() {
            return createTime;
        }

        public final void setCreateTime(Instant createTime) {
            this.createTime = createTime;
        }

        @Override
        public final Builder createTime(Instant createTime) {
            this.createTime = createTime;
            return this;
        }

        public final Instant getUpdateTime() {
            return updateTime;
        }

        public final void setUpdateTime(Instant updateTime) {
            this.updateTime = updateTime;
        }

        @Override
        public final Builder updateTime(Instant updateTime) {
            this.updateTime = updateTime;
            return this;
        }

        public final AnalysisSchema.Builder getSchema() {
            return schema != null ? schema.toBuilder() : null;
        }

        public final void setSchema(AnalysisSchema.BuilderImpl schema) {
            this.schema = schema != null ? schema.build() : null;
        }

        @Override
        public final Builder schema(AnalysisSchema schema) {
            this.schema = schema;
            return this;
        }

        public final String getFormat() {
            return format;
        }

        public final void setFormat(String format) {
            this.format = format;
        }

        @Override
        public final Builder format(String format) {
            this.format = format;
            return this;
        }

        @Override
        public final Builder format(AnalysisFormat format) {
            this.format(format == null ? null : format.toString());
            return this;
        }

        public final AnalysisSource.Builder getSource() {
            return source != null ? source.toBuilder() : null;
        }

        public final void setSource(AnalysisSource.BuilderImpl source) {
            this.source = source != null ? source.build() : null;
        }

        @Override
        public final Builder source(AnalysisSource source) {
            this.source = source;
            return this;
        }

        public final AnalysisSourceMetadata.Builder getSourceMetadata() {
            return sourceMetadata != null ? sourceMetadata.toBuilder() : null;
        }

        public final void setSourceMetadata(AnalysisSourceMetadata.BuilderImpl sourceMetadata) {
            this.sourceMetadata = sourceMetadata != null ? sourceMetadata.build() : null;
        }

        @Override
        public final Builder sourceMetadata(AnalysisSourceMetadata sourceMetadata) {
            this.sourceMetadata = sourceMetadata;
            return this;
        }

        public final List<AnalysisParameter.Builder> getAnalysisParameters() {
            List<AnalysisParameter.Builder> result = AnalysisParameterListCopier.copyToBuilder(this.analysisParameters);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAnalysisParameters(Collection<AnalysisParameter.BuilderImpl> analysisParameters) {
            this.analysisParameters = AnalysisParameterListCopier.copyFromBuilder(analysisParameters);
        }

        @Override
        public final Builder analysisParameters(Collection<AnalysisParameter> analysisParameters) {
            this.analysisParameters = AnalysisParameterListCopier.copy(analysisParameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder analysisParameters(AnalysisParameter... analysisParameters) {
            analysisParameters(Arrays.asList(analysisParameters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder analysisParameters(Consumer<AnalysisParameter.Builder>... analysisParameters) {
            analysisParameters(Stream.of(analysisParameters).map(c -> AnalysisParameter.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<AnalysisTemplateValidationStatusDetail.Builder> getValidations() {
            List<AnalysisTemplateValidationStatusDetail.Builder> result = AnalysisTemplateValidationStatusDetailListCopier
                    .copyToBuilder(this.validations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setValidations(Collection<AnalysisTemplateValidationStatusDetail.BuilderImpl> validations) {
            this.validations = AnalysisTemplateValidationStatusDetailListCopier.copyFromBuilder(validations);
        }

        @Override
        public final Builder validations(Collection<AnalysisTemplateValidationStatusDetail> validations) {
            this.validations = AnalysisTemplateValidationStatusDetailListCopier.copy(validations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validations(AnalysisTemplateValidationStatusDetail... validations) {
            validations(Arrays.asList(validations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder validations(Consumer<AnalysisTemplateValidationStatusDetail.Builder>... validations) {
            validations(Stream.of(validations)
                    .map(c -> AnalysisTemplateValidationStatusDetail.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final ErrorMessageConfiguration.Builder getErrorMessageConfiguration() {
            return errorMessageConfiguration != null ? errorMessageConfiguration.toBuilder() : null;
        }

        public final void setErrorMessageConfiguration(ErrorMessageConfiguration.BuilderImpl errorMessageConfiguration) {
            this.errorMessageConfiguration = errorMessageConfiguration != null ? errorMessageConfiguration.build() : null;
        }

        @Override
        public final Builder errorMessageConfiguration(ErrorMessageConfiguration errorMessageConfiguration) {
            this.errorMessageConfiguration = errorMessageConfiguration;
            return this;
        }

        @Override
        public CollaborationAnalysisTemplate build() {
            return new CollaborationAnalysisTemplate(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
